﻿using System;
using System.ServiceModel;
using Curse.Friends.AuthenticationClient.AuthService;
using System.ServiceModel.Channels;

namespace Curse.Friends.AuthenticationClient
{
    public class AuthenticationProvider
    {

        protected static string _serviceUrl;
        protected static int _siteID;

        protected static StringCipher _outgoingCipher;

        protected const int SubscriptionSeedPremium = 181756351;
        protected const int SubscriptionSeedNonPremium = 76172335;

        public static readonly TimeSpan RenewAfter = TimeSpan.FromDays(1);
        
        protected static NetworkServiceSoapClient CreateServiceClient()
        {
            return new NetworkServiceSoapClient(_serviceUrl.StartsWith("https") ? (Binding) new BasicHttpsBinding() : new BasicHttpBinding(), new EndpointAddress(_serviceUrl));
        }

        public static void Initialize(string serviceUrl, int authSiteID, string authSiteKey)
        {

            if (!Uri.IsWellFormedUriString(serviceUrl, UriKind.Absolute))
            {
                throw new ArgumentException("The service url supplied is invalid: " + serviceUrl);
            }
            _serviceUrl = serviceUrl;
            _siteID = authSiteID;
            _outgoingCipher = new StringCipher(authSiteKey);
        }

        public static int GetUserID(string usernameOrEmail)
        {
            if (usernameOrEmail.Contains("@"))
            {
                using (var authService = CreateServiceClient())
                {
                    return authService.v2GetUserIDFromEmail(_siteID, usernameOrEmail);
                }
            }
            else
            {
                using (var authService = CreateServiceClient())
                {
                    return authService.v2GetUserIDFromUsername(_siteID, usernameOrEmail);
                }
            }

        }

        public static int GetSubscriptionToken(int userID, bool hasPremium)
        {
            return userID ^ (hasPremium ? SubscriptionSeedPremium : SubscriptionSeedNonPremium);
        }

        public static bool IsUsernameAvailable(string username)
        {
            using (var authService = CreateServiceClient())
            {
                return authService.v2IsUsernameAvailable(_siteID, username);
            }
        }

        public static MappedUserResult GetMappedUser(int twitchUserID)
        {
            using (var authService = CreateServiceClient())
            {
                return authService.v2GetMappedUser(_siteID, twitchUserID);
            }
        }

        public static ProvisionMappedUserResult ProvisionMappedUser(int twitchUserID, bool isAutoProvisioned)
        {
            if (isAutoProvisioned)
            {
                using (var authService = CreateServiceClient())
                {
                    return authService.v2AutoProvisionMappedUserAccount(_siteID, twitchUserID);
                }
            }
            else
            {
                using (var authService = CreateServiceClient())
                {
                    return authService.v2ProvisionMappedUserAccount(_siteID, twitchUserID);
                }
            }
           
        }

        public static MergeUserAccountResult MergeUserAccount(string username, string password, int twitchUserID)
        {
            using (var authService = CreateServiceClient())
            {
                return authService.v2MergeUserAccount(_siteID, username, _outgoingCipher.Encrypt(password), twitchUserID);
            }
        }

        public static UnmergeAccountResult UnmergeUserAccount(int twitchUserID)
        {
            using (var authService = CreateServiceClient())
            {
                return authService.v2UnmergeUserAccount(_siteID, twitchUserID);
            }
        }

        public static MergeUserAccountResult MergeTempUserAccount(int userID, int twitchUserID)
        {
            using (var authService = CreateServiceClient())
            {
                return authService.v2MergeTempUserAccount(_siteID, userID, twitchUserID);
            }
        }
    }
}
