﻿using System;
using System.Text;

namespace Curse.Friends.AuthenticationClient
{
    public class StringCipher
    {
        /**
         * Private constants
         */
        private const Int32 SIZEOF_PADDING = 4;
        private const Int32 SIZEOF_STATE = 256;

        /**
         * Private member data
         */
        private Byte[] mState = new Byte[SIZEOF_STATE];

        /**
         * Initialization constructor
         * Initializes the internal base state using a byte array for the key
         *
         * @param pKey the byte array containing the key to initialize base state with
         */
        public StringCipher(Byte[] pKey)
        {
            InitializeState(pKey);
        }

        /**
         * Initialization constructor
         * Initializes the internal base state using a hex string for the key
         *
         * @param pKey the string containing hex data to initialize base state with
         */
        public StringCipher(String pKey)
        {
            InitializeState(HexStringToByteArray(pKey));
        }

        private const Int32 HEX_BASE = 16;

        public static Byte[] HexStringToByteArray(String pHex)
        {
            Int32 length = pHex.Length;
            Byte[] data = new Byte[(length + 1) / 2];
            for (Int32 index = 0;
                 index < length;
                 index += 2)
            {
                data[index / 2] = Convert.ToByte(pHex.Substring(index, 2), HEX_BASE);
            }
            return data;
        }

        /**
         * Initializes the internal base state using a byte array for the key
         *
         * @param pKey the byte array containing the key to initialize base state with
         */
        private void InitializeState(Byte[] pKey)
        {
            for (Byte i = 0; ; )
            {
                mState[i] = i;
                if (++i == 0)
                {
                    break;
                }
            }

            for (Byte i = 0, j = 0, k = 0; ; )
            {
                j += mState[i];
                j += pKey[i % pKey.Length];
                k = mState[i];
                mState[i] = mState[j];
                mState[j] = k;
                i += 2;
                if (i == 0)
                {
                    break;
                }
            }
        }

        /**
         * Publically exposed method for encrypting a string
         * Takes consideration for Base64 encoding
         *
         * @param  pString the string to be encrypted
         * @return         the string after random padding, encrypting and Base64 encoding
         */
        public String Encrypt(String pString)
        {
            Byte[] state = new Byte[SIZEOF_STATE];
            Byte[] junk = new Byte[SIZEOF_PADDING];
            Byte[] bytes = Encoding.UTF8.GetBytes(pString);
            Byte[] encrypted = new Byte[SIZEOF_PADDING + bytes.Length];

            Buffer.BlockCopy(mState, 0, state, 0, state.Length);
            new Random().NextBytes(junk);
            Buffer.BlockCopy(junk, 0, encrypted, 0, SIZEOF_PADDING);
            Buffer.BlockCopy(bytes, 0, encrypted, SIZEOF_PADDING, bytes.Length);

            Encrypt(state, encrypted);
            return Convert.ToBase64String(encrypted);
        }

        /**
         * Publically exposed method for decrypting a string
         * Takes consideration for Base64 encoding
         *
         * @param  pString the string to be decrypted
         * @return         the string after Base64 decoding, decrypting and random padding stripping
         */
        public String Decrypt(String pString)
        {
            Byte[] state = new Byte[mState.Length];
            Byte[] bytes = Convert.FromBase64String(pString);
            Buffer.BlockCopy(mState, 0, state, 0, state.Length);

            Decrypt(state, bytes);
            return Encoding.UTF8.GetString(bytes, SIZEOF_PADDING, bytes.Length - SIZEOF_PADDING);
        }



        /**
         * Encrypts pData inline using a copy of the base state
         *
         * @param pState contains a copy of the base state
         * @param pData  contains the data to be encrypted
         */
        private static void Encrypt(Byte[] pState,
                                    Byte[] pData)
        {
            Byte feedback = 0xFF;
            Byte tmp;
            for (Int32 i = 0, j = 0, k = 0; k < pData.Length; ++k)
            {
                i = (i + 1) % SIZEOF_STATE;
                j = (j + pState[i]) % SIZEOF_STATE;

                tmp = pState[i];
                pState[i] = pState[j];
                pState[j] = tmp;

                pData[k] ^= pState[(pState[i] + pState[j]) % SIZEOF_STATE];
                pData[k] ^= feedback;
                feedback = pData[k];
            }
        }

        /**
         * Decrypts pData inline using a copy of the base state
         *
         * @param pState contains a copy of the base state
         * @param pData  contains the data to be decrypted
         */
        private static void Decrypt(Byte[] pState,
                                    Byte[] pData)
        {
            Byte feedback = 0xFF;
            Byte tmp;
            for (Int32 i = 0, j = 0, k = 0; k < pData.Length; ++k)
            {
                i = (i + 1) % SIZEOF_STATE;
                j = (j + pState[i]) % SIZEOF_STATE;

                tmp = pState[i];
                pState[i] = pState[j];
                pState[j] = tmp;

                tmp = pData[k];
                pData[k] ^= pState[(pState[i] + pState[j]) % SIZEOF_STATE];
                pData[k] ^= feedback;
                feedback = tmp;
            }
        }
    }
}
