﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Serialization;
using Curse.Friends.Enums;

namespace Curse.Friends.AvatarsWebService.Configuration
{
    [XmlType("AvatarService")]
    public class AvatarServiceConfiguration
    {
        public static AvatarServiceConfiguration Current
        {
            get;
            set;
        }         

        [XmlArray]
        [XmlArrayItem("Domain")]
        public HashSet<string> DomainWhitelist { get; set; }

        [XmlElement]
        public string DefaultUserAvatarUrl { get; set; }

        [XmlElement]
        public string DefaultGroupAvatarUrl { get; set; }
 
        [XmlElement]
        public SyncedAvatarUrlMappings DefaultSyncedAvatarUrls { get; set; }

        [XmlElement]
        public int? DefaultMinWidth { get; set; }

        [XmlElement]
        public int? DefaultMaxWidth { get; set; }

        [XmlElement]
        public int? DefaultMinHeight { get; set; }

        [XmlElement]
        public int? DefaultMaxHeight { get; set; }

        [XmlElement]
        public bool CloudFlareEnabled { get; set; }

        [XmlElement]
        public string CloudFlareApiKey { get; set; }

        [XmlElement]
        public string CloudFlareEmail { get; set; }

        [XmlElement]
        public string CloudFlareDomain { get; set; }

        private CustomConstraints[] _customConstraints;

        [XmlElement]
        public CustomConstraints[] CustomConstraints
        {
            get { return _customConstraints; }
            set
            {
                _customConstraints = value ?? new CustomConstraints[0];

                // Preload all AvatarType constraints
                var allConstraints = Enum.GetValues(typeof (AvatarType)).Cast<AvatarType>()
                    .Select(t => _customConstraints.FirstOrDefault(c => c.AvatarType == t) ?? new CustomConstraints {AvatarType = t});

                _constraintsDictionary = allConstraints.ToDictionary(v => v.AvatarType, v => new AvatarConstraints
                {
                    AvatarType = v.AvatarType,
                    IgnoreAspectRatio = v.IgnoreAspectRatio ?? false,
                    MaxWidth = GetMaxDimension(v.MaxWidth, DefaultMaxWidth),
                    MinWidth = GetMinDimension(v.MinWidth, DefaultMinWidth),
                    MinHeight = GetMinDimension(v.MinHeight, DefaultMinHeight),
                    MaxHeight = GetMaxDimension(v.MaxHeight, DefaultMaxHeight),
                    AllowAnimation = v.AllowAnimation ?? false
                });

            }
        }

        private Dictionary<AvatarType, AvatarConstraints> _constraintsDictionary;

        public AvatarConstraints GetConstraints(AvatarType avatarType)
        {
            AvatarConstraints constraints;
            if (_constraintsDictionary.TryGetValue(avatarType, out constraints))
            {
                return constraints;
            }

            return new AvatarConstraints
            {
                AvatarType = avatarType,
                IgnoreAspectRatio = false,
                MaxWidth = GetMaxDimension(null, DefaultMaxWidth),
                MinWidth = GetMinDimension(null, DefaultMinWidth),
                MaxHeight = GetMaxDimension(null, DefaultMaxHeight),
                MinHeight = GetMinDimension(null, DefaultMinHeight),
                AllowAnimation = false
            };
        }

        private int GetMaxDimension(int? maxDimension = null, int? defaultMaxDimension =null)
        {
            var globalMax = ImageManager.ImageManagerConfiguration.Global.MaxImageDimension;
            var globalMin = ImageManager.ImageManagerConfiguration.Global.MinImageDimension;

            var specified = maxDimension ?? defaultMaxDimension?? globalMax;

            if (specified > globalMax)
            {
                throw new Exception(string.Format("Specified Max Image Dimension {0} exceeds the global maximum {1} from ImageManagerConfiguration", specified, globalMax));
            }
            if (specified < globalMin)
            {
                throw new Exception(string.Format("Specified Max Image Dimension {0} is less than the global minimum {1} from ImageManagerConfiguration", specified, globalMax));
            }

            return specified;
        }

        private int GetMinDimension(int? minDimension = null, int? defaultMinDimension = null)
        {
            var globalMax = ImageManager.ImageManagerConfiguration.Global.MaxImageDimension;
            var globalMin = ImageManager.ImageManagerConfiguration.Global.MinImageDimension;

            var specified = minDimension ?? defaultMinDimension ?? globalMin;

            if (specified > globalMax)
            {
                throw new Exception(string.Format("Specified Min Image Dimension {0} exceeds the global maximum {1} from ImageManagerConfiguration", specified, globalMax));
            }
            if (specified < globalMin)
            {
                throw new Exception(string.Format("Specified Min Image Dimension {0} is less than the global minimum {1} from ImageManagerConfiguration", specified, globalMax));
            }

            return specified;
        }
    }

    public class SyncedAvatarUrlMappings
    {
        [XmlElement]
        public string BackupAvatarUrl { get; set; }

        [XmlElement("Mapping")]
        public SyncedAvatarUrlMapping[] Mappings { get; set; }
    }

    public class SyncedAvatarUrlMapping
    {
        [XmlElement]
        public AccountType Type { get;set; }

        [XmlElement]
        public string DefaultAvatarUrl { get; set; }
    }

    public class CustomConstraints
    {
        public AvatarType AvatarType { get; set; }

        public int? MinWidth { get; set; }

        public int? MaxWidth { get; set; }

        public int? MinHeight { get; set; }

        public int? MaxHeight { get; set; }

        public bool? IgnoreAspectRatio { get; set; }

        public bool? AllowAnimation { get; set; }
    }

    public class AvatarConstraints
    {
        public AvatarType AvatarType { get; set; }

        public int MinWidth { get; set; }

        public int MaxWidth { get; set; }

        public int MinHeight { get; set; }

        public int MaxHeight { get; set; }

        public bool IgnoreAspectRatio { get; set; }

        public bool AllowAnimation { get; set; }
    }
}