﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Aerospike;
using Curse.CloudFlare;
using Curse.Extensions;
using Curse.Friends.AvatarsWebService.Configuration;
using Curse.Friends.AvatarsWebService.Contract;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Curse.Friends.Data.Search.FriendSearch;
using Curse.Friends.Enums;
using Curse.Friends.ImageManager;
using Curse.Friends.MicroService;
using Nest;
using System.Threading.Tasks;
using Curse.Logging;

namespace Curse.Friends.AvatarsWebService.Controllers
{
    [RoutePrefix("avatars/api")]
    [AuthenticationFilter(AuthenticationLevel.ApiKey)]
#if !CONFIG_DEBUG && !CONFIG_STAGING
    [ApiExplorerSettings(IgnoreApi = true)]
#endif
    public class AvatarAdminController : MicroServiceController
    {
        [Route("quarantine/users/search")]
        [HttpPost]
        [ResponseType(typeof (IEnumerable<FindUserAvatarSearchResult>))]
        public IHttpActionResult FindUserAvatar(FindUserAvatarSearchRequest search)
        {
            var results = new Dictionary<int, string>();

            // Known User ID
            if (search.AvatarUserID.HasValue)
            {
                var otherUser = GetUserAndRegion(search.AvatarUserID.Value);
                results[otherUser.User.UserID] = otherUser.User.Username;
                return Ok(CreateResults(results));
            }

            // Check the requestor's friendships
            if (search.RequestorUserID.HasValue)
            {
                var requestor = GetUserAndRegion(search.RequestorUserID.Value);
                var friendship = Friendship.GetAll(requestor.Region.RegionID, f => f.UserID, search.RequestorUserID.Value)
                    .FirstOrDefault(f => string.Equals(f.OtherUsername, search.AvatarUsername, StringComparison.InvariantCultureIgnoreCase));
                if (friendship != null)
                {
                    results[friendship.OtherUserID] = friendship.OtherUsername;
                    return Ok(CreateResults(results));
                }
            }

            // Look through the specified group
            if (search.GroupID.HasValue)
            {
                // Search via group members
                var members = GroupMemberManager.SearchGroupMembers(search.GroupID.Value, new GroupMemberSearch
                {
                    Username = search.AvatarUsername,
                    PageSize = 10,
                    PageNumber = 1
                });

                var exactMatchMember = members.FirstOrDefault(m => string.Equals(m.Username, search.AvatarUsername, StringComparison.CurrentCultureIgnoreCase));
                if (exactMatchMember != null)
                {
                    results.Clear();
                    results[exactMatchMember.UserID] = exactMatchMember.Username;
                    return Ok(CreateResults(results));
                }

                foreach (var member in members)
                {
                    results[member.UserID] = member.Username;
                }
            }

            // Fall back to a username search
            var searchResults = UsernameFriendSearchManager.GetClient().Search<UsernameFriendSearchModel>(s => s
                .Filter(f => f.Bool(b => b.Should(
                    bs => bs.Term(t => t.Username, search.AvatarUsername),
                    bs => bs.Term(t => t.Username.Suffix("autocomplete"), search.AvatarUsername)
                    )))
                .Size(20)
                );

            var exactMatchUser = searchResults.Documents.FirstOrDefault(r => string.Equals(r.Username, search.AvatarUsername, StringComparison.InvariantCultureIgnoreCase));
            if (exactMatchUser != null)
            {
                results.Clear();
                results[exactMatchUser.UserID] = exactMatchUser.Username;
                return Ok(CreateResults(results));
            }

            foreach (var searchResult in searchResults.Documents)
            {
                results[searchResult.UserID] = searchResult.Username;
            }

            // return all results
            return Ok(CreateResults(results));
        }

        private IEnumerable<FindUserAvatarSearchResult> CreateResults(Dictionary<int, string> users)
        {
            var avatars = Avatar.MultiGetLocal(users.Select(u => new KeyInfo((int) AvatarType.User, u.Key.ToString()))).ToDictionary(a => int.Parse(a.EntityID));
            var results = users.Select(u => FindUserAvatarSearchResult.Create(u.Key, u.Value, avatars.GetValueOrDefault(u.Key)));
            return results;
        }

        [Route("quarantine/users")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult MarkAvatarAsInappropriate(MarkAvatarInappropriateRequest request)
        {
            var avatar = Avatar.GetByTypeAndID(AvatarType.User, request.UserID.ToString());
            if (avatar == null)
            {
                return NotFound();
            }

            avatar.IsInappropriate = true;
            avatar.Update(a => a.IsInappropriate);

            return Ok();
        }

        [Route("quarantine/users/{userID}")]
        [HttpDelete]
        [ResponseType(typeof(void))]
        public IHttpActionResult UnmarkAvatarAsInappropriate(int userID)
        {
            var avatar = Avatar.GetByTypeAndID(AvatarType.User, userID.ToString());
            if (avatar == null)
            {
                return NotFound();
            }

            avatar.IsInappropriate = false;
            avatar.Update(a => a.IsInappropriate);

            return Ok();
        }

        [Route("{entityID}")]
        [HttpGet]
        [ResponseType(typeof(IEnumerable<GetAvatarResult>))]
        public IHttpActionResult GetAvatar(string entityID)
        {
            var avatars = Avatar.MultiGetLocal(Enum.GetValues(typeof (AvatarType)).Cast<AvatarType>().Select(t => new KeyInfo((int) t, entityID)));
            return Ok(avatars.Select(GetAvatarResult.FromAvatar));
        }

        [Route("{type}/{entityID}")]
        [HttpDelete]
        [ResponseType(typeof(void))]
        public IHttpActionResult PurgeAvatar(AvatarType type, string entityID)
        {
            var avatar = Avatar.GetByTypeAndID(type, entityID);
            if (avatar == null)
            {
                return NotFound();
            }

            if (string.IsNullOrEmpty(avatar.StorageKey))
            {
                return BadRequest("This avatar cannot be purged since it is not an uploaded file");
            }

            var url = GetUrlForAvatar(avatar);

            // keep file info
            var storageKey = avatar.StorageKey;
            var region = avatar.FileRegionID;

            // update avatar
            avatar.Url = string.Empty;
            avatar.Filename = string.Empty;
            avatar.FileRegionID = 0;
            avatar.StorageKey = string.Empty;
            avatar.IsInappropriate = true;
            avatar.Timestamp = DateTime.UtcNow.ToEpochMilliseconds();
            avatar.Update();

            // delete file
            if (!ImageManager.ImageManager.DeleteImage(storageKey, region))
            {
                return NotFound();
            }

            if (AvatarServiceConfiguration.Current.CloudFlareEnabled)
            {
                CloudFlareApi.InvalidateUrl(AvatarServiceConfiguration.Current.CloudFlareDomain, url);
            }

            return Ok();
        }

        private static string GetUrlForAvatar(Avatar avatar)
        {
            switch ((AvatarType)avatar.AvatarType)
            {
                case AvatarType.User:
                    return string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, "users", avatar.EntityID);
                case AvatarType.Group:
                    return string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, "groups", avatar.EntityID);
                case AvatarType.GroupCover:
                    return string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, "groups", avatar.EntityID + "/cover");
                case AvatarType.GroupEmoticon:
                    return string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, "emoticons", avatar.EntityID);
                default:
                    throw new DataValidationException("Unsupported avatar type for purging");
            }
        }

        [HttpPost]
        [Route("{type}/{entityID}/base64")]
        [ResponseType(typeof(ImageMetadata))]
        public IHttpActionResult ChangeAvatarBase64(AvatarType type, string entityID, [FromBody] string value)
        {
            var imageMetadata = AvatarController.SaveAvatar(type, entityID, value);
            return Ok(imageMetadata);
        }

        [HttpPost]
        [Route("{type}/{entityID}/url")]
        [ResponseType(typeof (void))]
        public IHttpActionResult ChangeAvatarUrl(AvatarType type, string entityID, [FromBody] string url)
        {
            if (string.IsNullOrEmpty(url))
            {
                AvatarController.ClearAvatar(Request, type, entityID);
            }
            else
            {
                AvatarController.SaveAvatarUrl(Request, type, entityID, url);
            }
            return StatusCode(HttpStatusCode.NoContent);
        }

        [HttpPost]
        [Route("bulk/urls")]
        [ResponseType(typeof(void))]
        public IHttpActionResult BulkUpdateAvatarUrls(BulkUpdateAvatarUrlsRequest request)
        {
            Task.Run(() =>
            {
                var failures = new List<Tuple<BulkUpdateAvatarUrlsUpdate, string>>();
                foreach (var update in request.Updates)
                {
                    try
                    {
                        if (string.IsNullOrEmpty(update.Url))
                        {
                            AvatarController.ClearAvatar(Request, update.Type, update.EntityID);
                        }
                        else
                        {
                            AvatarController.SaveAvatarUrl(Request, update.Type, update.EntityID, update.Url);
                        }
                    }
                    catch (Exception ex)
                    {
                        failures.Add(Tuple.Create(update, ex.Message));
                    }
                }

                if(failures.Count > 0)
                {
                    Logger.Warn("At least one avatar failed to update during a bulk request", new { failures });
                }
            });
            return StatusCode(HttpStatusCode.Accepted);
        }
    }
}