﻿using System;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Text;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Extensions;
using Curse.Friends.AvatarsWebService.Configuration;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.ImageManager;
using Curse.Friends.MicroService;
using Curse.Friends.MicroService.Attributes;
using Curse.Friends.MicroService.Exceptions;
using Curse.Friends.TwitchApi;
using Curse.Logging;
using Stream = System.IO.Stream;

namespace Curse.Friends.AvatarsWebService.Controllers
{
    /// <summary>
    /// Handles actions related to files.
    /// </summary>    
    [ExcludeFromGenerator]
    public class AvatarController : MicroServiceUploadController
    {
        protected override string[] FileExtensionsFilter
        {
            get { return ImageManagerConfiguration.Global.AllowedImageTypes.ToArray(); }
        }

        /// <summary>
        /// Gets the avatar for the specified entity.
        /// </summary>
        /// <param name="fileID">The ID of the file.</param>
        /// <param name="filename">The name of the file.</param>
        /// <returns>The file.</returns>
        [HttpGet]
        [Route("files/{fileID}/{filename}")]
        [ResponseType(typeof(Stream))]
        [CacheControl(1440)]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult Get(string fileID, string filename)
        {
            Guid fileGuid;
            if (!Guid.TryParse(fileID, out fileGuid))
            {
                return NotFound();
            }

            ImageMetadata imageMetadata;
            var stream = ImageManager.ImageManager.GetImageStream(fileID, filename, out imageMetadata);

            if (stream == null)
            {
                return NotFound();
            }

            var resp = new HttpResponseMessage()
            {
                Content = new StreamContent(stream),

            };

            // Find the MIME type            
            resp.Content.Headers.ContentType = new MediaTypeHeaderValue(imageMetadata.MimeType);

            return ResponseMessage(resp);
        }

        #region User Avatar

        /// <summary>
        /// Gets the avatar for the user
        /// </summary>
        /// <param name="userID">The ID of the user.</param>
        /// <returns>The file.</returns>
        [HttpGet]
        [Route("users/{userID}")]
        [ResponseType(typeof(Stream))]
        [CacheControl]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult Get(int userID, long? t = null)
        {
            if (userID < 1)
            {
                return NotFound();
            }

            var avatar = Avatar.GetByTypeAndID(AvatarType.User, userID.ToString());

            if (avatar == null || !avatar.IsMigrated)
            {
                avatar = MigrateUserAvatar(userID, avatar);
            }

            return RedirectToAvatar(avatar, AvatarServiceConfiguration.Current.DefaultUserAvatarUrl, t);
        }

        private Avatar MigrateUserAvatar(int userID, Avatar avatar)
        {
            var userRegion = UserRegion.GetLocal(userID);
            if (userRegion == null)
            {
                return avatar;
            }

            var user = userRegion.GetUser();
            if (user == null || !user.IsMerged)
            {
                return avatar;
            }

            var twitchAvatar = Avatar.GetByTypeAndID(AvatarType.SyncedAccount, ExternalAccount.GetAvatarEntityID(AccountType.Twitch, user.TwitchID));
            if(twitchAvatar == null || string.IsNullOrEmpty(twitchAvatar.Url))
            {
                var acct = ExternalAccount.GetByTwitchUserID(user.TwitchID);
                if(acct != null)
                {
                    twitchAvatar = Avatar.CreateOrUpdate(AvatarType.SyncedAccount, acct.GetAvatarEntityID(), acct.AvatarUrl);
                }
            }

            var syncToTwitch = twitchAvatar!=null && 
                string.IsNullOrEmpty(twitchAvatar.Url) && 
                !string.IsNullOrEmpty(avatar?.Url) && 
                !string.IsNullOrEmpty(avatar?.StorageKey);

            if (syncToTwitch)
            {
                // Sync user avatar to Twitch
                twitchAvatar.Url = avatar.Url;
                twitchAvatar.Timestamp = DateTime.UtcNow.ToEpochMilliseconds();
                twitchAvatar.Update(a => a.Url, a => a.Timestamp);
                SyncAvatarToTwitchWorker.Create(userID, user.TwitchID);
                avatar = twitchAvatar;
            }
            else
            {
                var url = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, "syncs/accounts", $"{AccountType.Twitch}/{user.TwitchID}");
                avatar = SaveAvatarUrl(AvatarType.User, userID.ToString(), url, true, true);
                avatar.IsMigrated = true;
                avatar.Update(a => a.IsMigrated);
            }

            return avatar;
        }

        /// <summary>
        /// Sets the avatar for the user, given an uploaded file.
        /// </summary>        
        /// <returns>The file.</returns>
        [HttpPost]
        [Route("users/{userID}/file")]
        [AuthenticationFilter]
        public IHttpActionResult File(int userID)
        {
            var me = GetCurrentUserAndRegion();

            if (userID != me.User.UserID)
            {
                return StatusCode(HttpStatusCode.Forbidden);
            }

            ImageMetadata imageMetadata;
            if (me.User.IsMerged)
            {
                var file = GetPostedFile();
                imageMetadata = SaveUserAvatarToTwitch(me.User, file.InputStream, file.FileName);
            }
            else
            {
                imageMetadata = SaveAvatar(AvatarType.User, userID.ToString());
            }

            // Stream Based
            return Ok(imageMetadata);
        }


        /// <summary>
        ///  Sets the avatar for the user, given a valid url.
        /// </summary>        
        /// <returns>The file.</returns>
        [HttpPost]
        [Route("users/{userID}/url")]
        [AuthenticationFilter]
        public IHttpActionResult Url(int userID, [FromBody] string avatarUrl)
        {
            var me = GetCurrentUserAndRegion();

            if (userID != me.User.UserID)
            {
                return StatusCode(HttpStatusCode.Forbidden);
            }

            if (me.User.IsMerged)
            {
                return StatusCode(HttpStatusCode.BadRequest);
            }

            // Avatar URL            
            SaveAvatarUrl(AvatarType.User, userID.ToString(), avatarUrl);
            return Ok();
        }


        /// <summary>
        ///  Sets the avatar for the user, given a valid Base64 string.
        /// </summary>        
        /// <returns>The file.</returns>
        [HttpPost]
        [Route("users/{userID}/base64")]
        [AuthenticationFilter]
        public IHttpActionResult Base64(int userID, [FromBody] string value)
        {
            var me = GetCurrentUserAndRegion();

            if (userID != me.User.UserID)
            {
                return StatusCode(HttpStatusCode.Forbidden);
            }

            ImageMetadata imageMetadata;

            if (me.User.IsMerged)
            {
                using (var stream = new MemoryStream(Convert.FromBase64String(value)))
                {
                    imageMetadata = SaveUserAvatarToTwitch(me.User, stream);
                }
            }
            else
            {
                imageMetadata = SaveAvatar(AvatarType.User, userID.ToString(), value);
            }

            return Ok(imageMetadata);
        }

        [HttpDelete]
        [Route("users/{userID}")]
        public IHttpActionResult ClearUserAvatar(int userID)
        {
            var me = GetCurrentUserAndRegion();

            if (userID != me.User.UserID)
            {
                return Forbidden();
            }

            if (me.User.IsMerged)
            {
                return StatusCode(HttpStatusCode.BadRequest);
            }

            ClearAvatar(AvatarType.User, userID.ToString());

            return StatusCode(HttpStatusCode.NoContent);
        }

        private ImageMetadata SaveUserAvatarToTwitch(Data.User user, Stream stream, string fileName = null)
        {
            var acct = ExternalAccount.GetByTwitchUserID(user.TwitchID);
            if(acct == null)
            {
                throw new FileUploadException(FileUploadFailureReason.Missing);
            }

            // Ensure we point to the file locally in our S3 since logo doesn't update right away
            var imageMetadata = SaveAvatar(AvatarType.SyncedAccount, acct.GetAvatarEntityID(), stream, fileName, false);
            stream.Seek(0, SeekOrigin.Begin);

            // Upload image to Twitch
            try
            {
                TwitchApiHelper.GetClient(acct.ClientID).UploadAvatar(acct.ExternalID, acct.AuthToken, stream);
            }
            catch(Exception ex)
            {
                Logger.Error(ex, "Failed to save avatar to Twitch");
                throw new FileUploadException(FileUploadFailureReason.Missing);
            }

            // Point user avatar to the twitch avatar
            var url = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, "syncs/accounts", $"{AccountType.Twitch}/{user.TwitchID}");
            var userAvatar = SaveAvatarUrl(AvatarType.User, user.UserID.ToString(), url, true);
            userAvatar.IsMigrated = true;
            userAvatar.Update(a => a.IsMigrated);

            return imageMetadata;
        }

        #endregion

        #region Group Avatar

        /// <summary>
        /// Updates the avatar for a group, given a posted stream and group ID.
        /// </summary>        
        /// <returns>The file.</returns>
        [HttpPost]
        [Route("groups/{groupID}/file")]
        [AuthenticationFilter]
        public IHttpActionResult File(string groupID)
        {
            var group = GetGroup(groupID);

            if (group == null)
            {
                return NotFound();
            }

            var imageMetadata = SaveAvatar(AvatarType.Group, groupID);
            return Ok(imageMetadata);
        }

        /// <summary>
        /// Updates the avatar for a group, given a posted stream and group ID.
        /// </summary>        
        /// <returns>The file.</returns>
        [HttpPost]
        [Route("groups/{groupID}/url")]
        [AuthenticationFilter]
        public IHttpActionResult Url(string groupID, [FromBody] string avatarUrl)
        {
            var group = GetGroup(groupID);

            if (group == null)
            {
                return NotFound();
            }

            SaveAvatarUrl(AvatarType.Group, groupID, avatarUrl);
            return Ok();
        }


        /// <summary>
        /// Sets the avatar for the user.
        /// </summary>        
        /// <returns>The file.</returns>
        [HttpPost]
        [Route("groups/{groupID}/base64")]
        [AuthenticationFilter]
        public IHttpActionResult Base64(string groupID, [FromBody] string value)
        {
            var group = GetGroup(groupID);

            if (group == null)
            {
                return NotFound();
            }

            // Stream Based
            var imageMetadata = SaveAvatar(AvatarType.Group, groupID, value);

            return Ok(imageMetadata);
        }

        private Avatar MigrateGroupAvatar(string groupID)
        {
            var group = Group.GetLocal(groupID);
            if (group == null)
            {
                return null;
            }

            var avatar = new Avatar
            {
                AvatarType = (int)AvatarType.Group,
                EntityID = groupID,
                Url = group.AvatarUrl,
                Timestamp = DateTime.UtcNow.ToEpochMilliseconds()
            };

            avatar.InsertLocal();
            return avatar;
        }

        /// <summary>
        /// Gets the avatar for the user
        /// </summary>
        /// <param name="groupID">The ID of the group.</param>
        /// <returns>The file.</returns>
        [HttpGet]
        [Route("groups/{groupID}")]
        [ResponseType(typeof(Stream))]
        [CacheControl]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult Get(string groupID)
        {
            var avatar = Avatar.GetByTypeAndID(AvatarType.Group, groupID);

            if (avatar == null) // if null, fallback their user record and migrate it!
            {
                avatar = MigrateGroupAvatar(groupID);
            }

            return RedirectToAvatar(avatar, AvatarServiceConfiguration.Current.DefaultGroupAvatarUrl);
        }

        [HttpDelete]
        [Route("groups/{groupID}")]
        public IHttpActionResult ClearGroupAvatar(string groupID)
        {
            var group = GetGroup(groupID);
            if (group == null)
            {
                return NotFound();
            }

            ClearAvatar(AvatarType.Group, groupID);

            return StatusCode(HttpStatusCode.NoContent);
        }

        #endregion

        #region Group Cover

        /// <summary>
        /// Gets the group's cover.
        /// </summary>
        /// <param name="groupID">The group's ID</param>
        /// <returns></returns>
        [HttpGet]
        [Route("groups/{groupID}/cover")]
        [ResponseType(typeof(Stream))]
        [CacheControl]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult GetGroupCover(string groupID)
        {
            var avatar = Avatar.GetByTypeAndID(AvatarType.GroupCover, groupID);

            if (avatar == null || string.IsNullOrWhiteSpace(avatar.Url)) // if null, fallback to their regular group avatar
            {
                return NotFound();
            }

            return RedirectToAvatar(avatar, null);
        }

        /// <summary>
        /// Sets the group's cover to the URL.
        /// </summary>
        /// <param name="groupID">The group's ID</param>
        /// <param name="avatarUrl">The avatar URL</param>
        [HttpPost]
        [Route("groups/{groupID}/cover/url")]
        [ResponseType(typeof(void))]
        [AuthenticationFilter]
        public IHttpActionResult SetGroupCoverUrl(string groupID, [FromBody] string avatarUrl)
        {
            var group = GetGroup(groupID);

            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.ManageServer, Token.UserID);

            SaveAvatarUrl(AvatarType.GroupCover, groupID, avatarUrl);
            return Ok();
        }

        /// <summary>
        /// Sets the group's cover to the uploaded file.
        /// </summary>
        /// <param name="groupID">The group's ID</param>
        [HttpPost]
        [Route("groups/{groupID}/cover/file")]
        [ResponseType(typeof(void))]
        [AuthenticationFilter]
        public IHttpActionResult SetGroupCoverFile(string groupID)
        {
            var group = GetGroup(groupID);

            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.ManageServer, Token.UserID);

            var imageMetadata = SaveAvatar(AvatarType.GroupCover, groupID);
            return Ok(imageMetadata);
        }

        /// <summary>
        /// Sets the group's cover to the base 64 string image data.
        /// </summary>
        /// <param name="groupID">The group's ID</param>
        /// <param name="value">The image data in base 64 string format</param>
        [HttpPost]
        [Route("groups/{groupID}/cover/base64")]
        [ResponseType(typeof(void))]
        [AuthenticationFilter]
        public IHttpActionResult SetGroupCoverBase64(string groupID, [FromBody] string value)
        {
            var group = GetGroup(groupID);

            if (group == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.ManageServer, Token.UserID);

            // Stream Based
            var imageMetadata = SaveAvatar(AvatarType.GroupCover, groupID, value);

            return Ok(imageMetadata);
        }

        [HttpDelete]
        [Route("groups/{groupID}/cover")]
        public IHttpActionResult ClearGroupCover(string groupID)
        {
            var group = GetGroup(groupID);
            if (group == null)
            {
                return NotFound();
            }

            ClearAvatar(AvatarType.GroupCover, groupID);

            return StatusCode(HttpStatusCode.NoContent);
        }

        #endregion

        #region Synced Account Avatar

        /// <summary>
        /// Updates the avatar for a synced account.
        /// </summary>        
        /// <returns>The file.</returns>
        [HttpPost]
        [Route("syncs/accounts/{type}/{syncID}/url")]
        [AuthenticationFilter(AuthenticationLevel.ApiKey)]
        [ResponseType(typeof(void))]
        public IHttpActionResult UpdateSyncedAccountUrl(AccountType type, string syncID)
        {
            if (!Token.IsApiRequest)
            {
                return StatusCode(HttpStatusCode.Forbidden);
            }

            MigrateSyncedAccountAvatar(type, syncID);
            return Ok();
        }

        /// <summary>
        /// Gets the avatar for a synced account.
        /// </summary>
        /// <param name="type">The type of account.</param>
        /// <param name="syncID">The ID of the account.</param>
        /// <returns></returns>
        [HttpGet]
        [Route("syncs/accounts/{type}/{syncID}")]
        [CacheControl]
        [ResponseType(typeof(Stream))]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult GetSyncedAccount(AccountType type, string syncID)
        {
            var entityID = ExternalAccount.GetAvatarEntityID(type, syncID);

            var avatar = Avatar.GetByTypeAndID(AvatarType.SyncedAccount, entityID)
                         ?? MigrateSyncedAccountAvatar(type, syncID, entityID);

            var defaultSyncUrls = AvatarServiceConfiguration.Current.DefaultSyncedAvatarUrls;
            var defaultUrlMapping = AvatarServiceConfiguration.Current.DefaultSyncedAvatarUrls.Mappings.FirstOrDefault(u => u.Type == type);
            var defaultUrl = defaultUrlMapping != null ? defaultUrlMapping.DefaultAvatarUrl : defaultSyncUrls.BackupAvatarUrl;

            return RedirectToAvatar(avatar, defaultUrl);
        }

        private Avatar MigrateSyncedAccountAvatar(AccountType type, string syncID, string entityID = null)
        {
            var account = ExternalAccount.GetLocal(syncID, type);
            if (account == null || string.IsNullOrEmpty(account.AvatarUrl))
            {
                return null;
            }

            entityID = entityID ?? ExternalAccount.GetAvatarEntityID(type, syncID);
            SaveAvatarUrl(AvatarType.SyncedAccount, entityID, account.AvatarUrl);
            return Avatar.GetByTypeAndID(AvatarType.SyncedAccount, entityID);
        }

        #endregion

        #region Synced Communities Avatar

        /// <summary>
        /// Updates the avatar for a synced community.
        /// </summary>        
        /// <returns>The file.</returns>
        [HttpPost]
        [Route("syncs/communities/{type}/{syncID}/url")]
        [AuthenticationFilter(AuthenticationLevel.ApiKey)]
        [ResponseType(typeof(void))]
        public IHttpActionResult UpdateSyncedCommunityUrl(AccountType type, string syncID)
        {
            if (!Token.IsApiRequest)
            {
                return StatusCode(HttpStatusCode.Forbidden);
            }

            MigrateSyncedCommunityAvatar(type, syncID);
            return Ok();
        }

        /// <summary>
        /// Gets the avatar for a synced community.
        /// </summary>
        /// <param name="type">The community type.</param>
        /// <param name="syncID">The ID of the community.</param>
        [HttpGet]
        [Route("syncs/communities/{type}/{syncID}")]
        [CacheControl]
        [ResponseType(typeof(Stream))]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult GetSyncedCommunity(AccountType type, string syncID)
        {
            var entityID = ExternalCommunity.GetAvatarEntityID(type, syncID);

            var avatar = Avatar.GetByTypeAndID(AvatarType.SyncedCommunity, entityID)
                         ?? MigrateSyncedCommunityAvatar(type, syncID, entityID);

            var defaultSyncUrls = AvatarServiceConfiguration.Current.DefaultSyncedAvatarUrls;
            var defaultUrlMapping = AvatarServiceConfiguration.Current.DefaultSyncedAvatarUrls.Mappings.FirstOrDefault(u => u.Type == type);
            var defaultUrl = defaultUrlMapping != null ? defaultUrlMapping.DefaultAvatarUrl : defaultSyncUrls.BackupAvatarUrl;

            return RedirectToAvatar(avatar, defaultUrl);
        }

        private Avatar MigrateSyncedCommunityAvatar(AccountType type, string syncID, string entityID = null)
        {
            var community = ExternalCommunity.GetLocal(syncID, type);
            if (community == null || string.IsNullOrWhiteSpace(community.AvatarUrl))
            {
                return null;
            }

            entityID = entityID ?? ExternalCommunity.GetAvatarEntityID(type, syncID);
            SaveAvatarUrl(AvatarType.SyncedCommunity, entityID, community.AvatarUrl);
            return Avatar.GetByTypeAndID(AvatarType.SyncedCommunity, entityID);
        }

        #endregion

        #region Group Emoticons

        private static string GetEmoticonEntityID(Guid groupID, string regex)
        {
            return string.Format("{0}/{1}", groupID, regex);
        }

        [HttpDelete]
        [ResponseType(typeof(void))]
        [Route("emoticons/{groupID}/{regex}")]
        [AuthenticationFilter(AuthenticationLevel.ApiKey)]
        public IHttpActionResult DeleteEmoticon(Guid groupID, string regex)
        {
            var entityID = GetEmoticonEntityID(groupID, regex);
            var avatar = Avatar.GetByTypeAndID(AvatarType.GroupEmoticon, entityID);
            if (avatar != null)
            {
                var formerKey = avatar.StorageKey;
                var formerRegion = avatar.FileRegionID;

                avatar.Url = string.Empty;
                avatar.FileRegionID = 0;
                avatar.StorageKey = string.Empty;
                avatar.Timestamp = DateTime.UtcNow.ToEpochMilliseconds();
                avatar.Update();

                DeleteFormerAvatar(formerKey, formerRegion);
            }

            return Ok();
        }

        [HttpGet]
        [ResponseType(typeof(Stream))]
        [CacheControl]
        [Route("emoticons/{groupID}/{regex}")]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult GetEmoticon(Guid groupID, string regex)
        {
            var entityID = GetEmoticonEntityID(groupID, regex);
            var avatar = Avatar.GetByTypeAndID(AvatarType.GroupEmoticon, entityID)
                         ?? MigrateGroupEmoticon(groupID, regex, entityID);

            if (avatar == null || string.IsNullOrWhiteSpace(avatar.Url))
            {
                return NotFound();
            }

            return RedirectToAvatar(avatar, "");
        }

        private Avatar MigrateGroupEmoticon(Guid groupID, string regex, string entityID)
        {
            var emoticon = GroupEmoticon.GetAllLocal(g => g.GroupID, groupID).FirstOrDefault(e => e.RegexString == regex && e.Status == EmoticonStatus.Active);
            if (emoticon == null || string.IsNullOrWhiteSpace(emoticon.Url))
            {
                return null;
            }

            entityID = entityID ?? GetEmoticonEntityID(groupID, regex);
            SaveAvatarUrl(AvatarType.GroupEmoticon, entityID, emoticon.Url);
            return Avatar.GetByTypeAndID(AvatarType.GroupEmoticon, entityID);
        }

        #endregion

        #region Synced Emoticons

        [HttpGet]
        [ResponseType(typeof(Stream))]
        [CacheControl]
        [Route("syncs/emoticons/{type}/{syncID}/{regex}")]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult GetSyncedEmoticon(AccountType type, string syncID, string regex)
        {
            var entityID = ExternalCommunityEmoticon.GetAvatarEntityID(type, syncID, regex);
            var avatar = Avatar.GetByTypeAndID(AvatarType.SyncedEmoticon, entityID)
                         ?? MigrateSyncedEmoticon(type, syncID, regex, entityID);
            if (avatar == null || string.IsNullOrWhiteSpace(avatar.Url))
            {
                return NotFound();
            }

            return RedirectToAvatar(avatar, "");
        }

        private Avatar MigrateSyncedEmoticon(AccountType type, string syncID, string regex, string entityID = null)
        {
            var emoticon = ExternalCommunityEmoticon.GetLocal(type, syncID, regex);
            if (emoticon == null || emoticon.IsDeleted || string.IsNullOrWhiteSpace(emoticon.Url))
            {
                return null;
            }

            entityID = entityID ?? ExternalCommunityEmoticon.GetAvatarEntityID(type, syncID, regex);
            SaveAvatarUrl(AvatarType.SyncedEmoticon, entityID, emoticon.Url);
            return Avatar.GetByTypeAndID(AvatarType.SyncedEmoticon, entityID);
        }

        #endregion

        #region Twitch Emoticons

        private static string GetTwitchEmoteIdentity(int emoteSet, string regex)
        {
            return string.Join(":", emoteSet, regex);
        }

        [HttpGet]
        [ResponseType(typeof (Stream))]
        [CacheControl]
        [Route("twitch-emotes/{emoteID}")]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult GetTwitchEmoticon(long emoteID)
        {
            var entityID = emoteID.ToString();
            var avatar = Avatar.GetByTypeAndID(AvatarType.TwitchEmote, entityID)
                         ?? MigrateTwitchEmote(emoteID);
            if (avatar == null || string.IsNullOrWhiteSpace(avatar.Url))
            {
                return NotFound();
            }

            return RedirectToAvatar(avatar, string.Empty);
        }

        private Avatar MigrateTwitchEmote(long emoteID, string entityID=null)
        {
            var id = entityID ?? emoteID.ToString();
            var twitchEmote = TwitchEmote.GetLocal(emoteID);
            if (twitchEmote == null || twitchEmote.IsDeleted || string.IsNullOrWhiteSpace(twitchEmote.Url))
            {
                return null;
            }

            SaveAvatarUrl(AvatarType.TwitchEmote, id, twitchEmote.Url);
            return Avatar.GetByTypeAndID(AvatarType.TwitchEmote, id);
        }

        #endregion

        #region Twitch Chat Badges

        [HttpGet]
        [Route("twitch-badges/{badgeSet}/{version}")]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        [CacheControl]
        [ResponseType(typeof(Stream))]
        public IHttpActionResult GetTwitchChatBadge(string badgeSet, string version, string externalID=null)
        {
            var entityID = TwitchChatBadge.GetAvatarEntityID(badgeSet, version, externalID);
            var avatar = Avatar.GetByTypeAndID(AvatarType.TwitchChatBadge, entityID) ?? MigrateTwitchChatBadge(badgeSet, version, externalID, entityID);

            if (avatar == null || string.IsNullOrEmpty(avatar.Url))
            {
                return NotFound();
            }

            return RedirectToAvatar(avatar, avatar.Url);
        }

        private Avatar MigrateTwitchChatBadge(string badgeSet, string version, string externalID, string entityID = null)
        {
            entityID = entityID ?? TwitchChatBadge.GetAvatarEntityID(badgeSet, version, externalID);

            TwitchChatBadge badge;
            if (externalID != null && (badgeSet == "subscriber" || badgeSet == "bits"))
            {
                badge = TwitchChatBadge.GetLocal(badgeSet, version, externalID) ?? TwitchChatBadge.GetLocal(badgeSet, version, TwitchChatBadge.GlobalExternalID);
            }
            else
            {
                badge = TwitchChatBadge.GetLocal(badgeSet, version, TwitchChatBadge.GlobalExternalID);
            }

            if (badge == null || string.IsNullOrEmpty(badge.ImageUrl))
            {
                return null;
            }

            SaveAvatarUrl(AvatarType.TwitchChatBadge, entityID, badge.ImageUrl);
            return Avatar.GetByTypeAndID(AvatarType.TwitchChatBadge, entityID);
        }

        #endregion

        #region Shared Methods

        public static void ClearAvatar(HttpRequestMessage request, AvatarType avatarType, string entityID)
        {
            var avatar = Avatar.GetByTypeAndID(avatarType, entityID);

            if (avatar == null)
            {
                return;
            }

            var previousAvatarKey = avatar.StorageKey;
            var previousAvatarRegion = avatar.FileRegionID;

            avatar.Filename = string.Empty;
            avatar.Url = string.Empty;
            avatar.FileRegionID = 0;
            avatar.StorageKey = string.Empty;
            avatar.Timestamp = DateTime.UtcNow.ToEpochMilliseconds();
            avatar.AddOrUpdateLocal();

            DeleteFormerAvatar(previousAvatarKey, previousAvatarRegion);

            UpdateAvatarTimestamp(entityID, avatarType);
        }

        private void ClearAvatar(AvatarType avatarType, string entityID)
        {
            ClearAvatar(Request, avatarType, entityID);
        }

        private Avatar SaveAvatarUrl(AvatarType avatarType, string entityID, string url, bool bypassAppropriateCheck = false, bool skipClearingOld = false)
        {
            return SaveAvatarUrl(Request, avatarType, entityID, url, bypassAppropriateCheck, skipClearingOld);
        }

        internal static Avatar SaveAvatarUrl(HttpRequestMessage request, AvatarType avatarType, string entityID, string url, bool resetAppropriate = false, bool skipClearingOld = false)
        {
            var scheme = GetRequestScheme(request);
            string sanitizedUrl;

            if (!AvatarManager.Whitelist.IsWhiteListed(url, scheme, out sanitizedUrl))
            {
                throw new FileUploadException(FileUploadFailureReason.InvalidUrl);
            }

            var userAvatar = Avatar.GetByTypeAndID(avatarType, entityID);
            string previousAvatarKey = null;
            var previousAvatarRegion = 0;

            if (userAvatar == null)
            {
                userAvatar = new Avatar
                {
                    AvatarType = (int)avatarType,
                    EntityID = entityID
                };
            }
            else if (!resetAppropriate && userAvatar.IsInappropriate)
            {
                throw new DataConflictException("This avatar has been flagged as inappropriate and cannot be modified");
            }
            else
            {
                previousAvatarKey = userAvatar.StorageKey;
                previousAvatarRegion = userAvatar.FileRegionID;
            }

            userAvatar.Filename = string.Empty;
            userAvatar.Url = url;
            userAvatar.FileRegionID = 0;
            userAvatar.StorageKey = string.Empty;
            userAvatar.Timestamp = DateTime.UtcNow.ToEpochMilliseconds();
            if (resetAppropriate)
            {
                userAvatar.IsInappropriate = false;
            }
            userAvatar.AddOrUpdateLocal();

            if (!skipClearingOld)
            {
                DeleteFormerAvatar(previousAvatarKey, previousAvatarRegion);
            }

            UpdateAvatarTimestamp(entityID, avatarType);

            return userAvatar;
        }

        internal static ImageMetadata SaveAvatar(AvatarType avatarType, string entityID, string base64String)
        {
            if (string.IsNullOrEmpty(base64String))
            {
                throw new FileUploadException(FileUploadFailureReason.Missing);
            }

            byte[] fileBytes;

            try
            {
                fileBytes = Convert.FromBase64String(base64String);
            }
            catch
            {
                throw new FileUploadException(FileUploadFailureReason.UnsupportedFormat);
            }


            using (var stream = new MemoryStream(fileBytes))
            {
                return SaveAvatar(avatarType, entityID, stream);
            }
        }

        private ImageMetadata SaveAvatar(AvatarType avatarType, string entityID)
        {
            // Try to save the uploaded file
            var file = GetPostedFile();

            return SaveAvatar(avatarType, entityID, file.InputStream, file.FileName);
        }

        private static ImageMetadata ValidateImage(AvatarType avatarType, Stream stream, string filename=null)
        {
            ImageMetadata imageMetadata;
            var validationStatus = ImageManager.ImageManager.IsValidImage(stream, filename ?? Guid.NewGuid().ToString(), out imageMetadata);
            if (validationStatus != ImageValidationStatus.Valid)
            {
                switch (validationStatus)
                {
                    case ImageValidationStatus.HeightTooLarge:
                    case ImageValidationStatus.WidthTooLarge:
                        throw new FileUploadException(FileUploadFailureReason.TooLarge);
                    case ImageValidationStatus.HeightTooSmall:
                    case ImageValidationStatus.WidthTooSmall:
                        throw new FileUploadException(FileUploadFailureReason.TooSmall);
                    default:
                        throw new FileUploadException(FileUploadFailureReason.UnsupportedFormat);
                }
            }

            var constraints = AvatarServiceConfiguration.Current.GetConstraints(avatarType);

            if (!constraints.AllowAnimation && imageMetadata.IsAnimated)
            {
                throw new FileUploadException(FileUploadFailureReason.UnsupportedFormat);
            }

            if (!constraints.IgnoreAspectRatio && imageMetadata.Height != imageMetadata.Width)
            {
                throw new FileUploadException(FileUploadFailureReason.IncorrectAspectRatio);
            }

            if (imageMetadata.Width > constraints.MaxWidth)
            {
                throw new FileUploadException(FileUploadFailureReason.TooLarge);
            }

            if (imageMetadata.Width < constraints.MinWidth)
            {
                throw new FileUploadException(FileUploadFailureReason.TooSmall);
            }

            if (imageMetadata.Height > constraints.MaxHeight)
            {
                throw new FileUploadException(FileUploadFailureReason.TooLarge);
            }

            if (imageMetadata.Height < constraints.MinHeight)
            {
                throw new FileUploadException(FileUploadFailureReason.TooSmall);
            }

            return imageMetadata;
        }

        private static ImageMetadata SaveAvatar(AvatarType avatarType, string entityID, Stream stream, string fileName = null, bool autoCloseStream = true)
        {
            var imageMetadata = ValidateImage(avatarType, stream, fileName);

            var userAvatar = Avatar.GetByTypeAndID(avatarType, entityID);

            var id = ImageManager.ImageManager.SaveImage(imageMetadata, stream, autoCloseStream);
            var url = ImageManager.ImageManager.GetImageUrl(id.ToString(), imageMetadata.Filename);

            string previousAvatarKey = null;
            var previousAvatarRegion = 0;

            if (userAvatar == null)
            {
                userAvatar = new Avatar
                {
                    AvatarType = (int)avatarType,
                    EntityID = entityID
                };
            }
            else if (userAvatar.IsInappropriate)
            {
                throw new DataConflictException("This avatar has been flagged as inappropriate and cannot be modified");
            }
            else
            {
                previousAvatarKey = userAvatar.StorageKey;
                previousAvatarRegion = userAvatar.FileRegionID;
            }

            userAvatar.Filename = imageMetadata.Filename;
            userAvatar.Url = url;
            userAvatar.FileRegionID = StorageConfiguration.CurrentRegion.ID;
            userAvatar.StorageKey = id.ToString();
            userAvatar.Timestamp = DateTime.UtcNow.ToEpochMilliseconds();
            userAvatar.AddOrUpdateLocal();

            DeleteFormerAvatar(previousAvatarKey, previousAvatarRegion);

            UpdateAvatarTimestamp(entityID, avatarType);

            return imageMetadata;
        }

        private static void UpdateAvatarTimestamp(string entityID, AvatarType avatarType)
        {
            switch (avatarType)
            {
                case AvatarType.User:
                    {
                        int userID;
                        if (!int.TryParse(entityID, out userID))
                        {
                            Logger.Warn("Failed to parse user ID from entityID: " + entityID);
                            return;
                        }
                        var region = UserRegion.GetLocal(userID);
                        if (region == null)
                        {
                            return;
                        }
                        var user = Data.User.Get(region.RegionID, userID);
                        if (user == null)
                        {
                            return;
                        }

                        user.UpdateAvatarTimestamp(DateTime.UtcNow);

                        break;
                    }
                case AvatarType.Group:
                    {
                        Guid groupID;
                        if (!Guid.TryParse(entityID, out groupID))
                        {
                            Logger.Warn("Failed to parse group ID from entityID: " + entityID);
                            return;
                        }
                        var group = Group.GetWritableByID(groupID);
                        if (group == null)
                        {
                            return;
                        }

                        group.AvatarTimestamp = DateTime.UtcNow.ToEpochMilliseconds();
                        group.Update(g => g.AvatarTimestamp);

                        break;
                    }
                case AvatarType.SyncedAccount:
                    {
                        var split = entityID.Split('/');
                        if (split.Length != 2)
                        {
                            return;
                        }

                        var account = ExternalAccount.GetLocal(split[1], AccountType.Twitch);
                        if (!account.IsMerged)
                        {
                            return;
                        }

                        var user = Data.User.FindByUserID(account.MergedUserID);
                        if(user == null || !user.IsMerged)
                        {
                            return;
                        }

                        user.UpdateAvatarTimestamp(DateTime.UtcNow);
                        break;
                    }
            }

        }

        private static void DeleteFormerAvatar(string avatarKey, int avatarRegion)
        {
            if (string.IsNullOrEmpty(avatarKey) || avatarRegion == 0) return;

            try
            {
                ImageManager.ImageManager.DeleteImage(avatarKey, avatarRegion);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to cleanup former avatar!");
            }
        }

        private string GetRequestScheme()
        {
            return GetRequestScheme(Request);
        }

        private static string GetRequestScheme(HttpRequestMessage request)
        {
            return request.Headers.Contains("X-Forwarded-Proto") ? request.Headers.GetValues("X-Forwarded-Proto").First() : request.RequestUri.Scheme;
        }

        private IHttpActionResult RedirectToAvatar(Avatar avatar, string defaultUrl, long? cacheBreaker = null)
        {
            if (avatar == null || string.IsNullOrWhiteSpace(avatar.Url) || avatar.IsInappropriate)
            {
                return Redirect(defaultUrl);
            }

            string sanitizedUrl;

            var scheme = GetRequestScheme();

            if (!AvatarManager.Whitelist.IsWhiteListed(avatar.Url, scheme, out sanitizedUrl))
            {
                return NotFound();
            }

            if (FriendsServiceConfiguration.Mode == ConfigurationMode.Staging)
            {
                // Remove .dev domains that were used in testing
                sanitizedUrl = sanitizedUrl.Replace(".dev/", ".tech/");

                // Remove any regional sub-domains that were used in testing
                sanitizedUrl = sanitizedUrl.Replace("-na.", ".").Replace("-eu.", ".").Replace("-ap.", ".");
            }

            if (cacheBreaker.HasValue)
            {
                var queryIndex = sanitizedUrl.IndexOf('?');
                if (queryIndex < 0)
                {
                    sanitizedUrl += "?t=" + cacheBreaker.Value;
                }
                else
                {
                    sanitizedUrl += "&t=" + cacheBreaker.Value;
                }
            }

            return Redirect(sanitizedUrl);
        }

        private Group GetGroup(string groupID)
        {
            var group = Group.GetLocal(groupID);

            if (group == null)
            {
                return null;
            }

            var userRegion = UserRegion.GetLocal(Token.UserID);

            if (userRegion == null)
            {
                return null;
            }

            group.CheckPermission(GroupPermissions.ManageServer, Token.UserID);

            return group;
        }

        #endregion
    }
}