﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Text;
using Curse.Friends.Enums;
using Newtonsoft.Json;

namespace Curse.Friends.BattleNet
{
    public class BattleNetApiHelper
    {
        private static readonly Dictionary<BattleNetRegion, BattleNetRegionInfo> _regions;

        static BattleNetApiHelper()
        {
            _regions = new[]
            {
                new BattleNetRegionInfo
                {
                    ID = BattleNetRegion.US,
                    OAuthBaseUrl = "https://us.battle.net",
                    ApiBaseUrl = "https://us.api.battle.net"
                },
                new BattleNetRegionInfo
                {
                    ID = BattleNetRegion.EU,
                    OAuthBaseUrl = "https://eu.battle.net",
                    ApiBaseUrl = "https://eu.api.battle.net"
                },
                new BattleNetRegionInfo
                {
                    ID = BattleNetRegion.KR,
                    OAuthBaseUrl = "https://kr.battle.net",
                    ApiBaseUrl = "https://kr.api.battle.net"
                },
                new BattleNetRegionInfo
                {
                    ID = BattleNetRegion.TW,
                    OAuthBaseUrl = "https://tw.battle.net",
                    ApiBaseUrl = "https://tw.api.battle.net"
                },
                new BattleNetRegionInfo
                {
                    ID = BattleNetRegion.CN,
                    OAuthBaseUrl = "https://www.battlenet.com.cn",
                    ApiBaseUrl = "https://api.battlenet.com.cn"
                },

            }.ToDictionary(r => r.ID);
        }

        private static BattleNetApplication _app;

        public static void Initialize(BattleNetApplication application)
        {
            _app = application;
        }

        public static BattleNetApplication GetApp()
        {
            if (_app == null)
            {
                throw new InvalidOperationException("BattleNet Api Helper not initialized");
            }

            return _app;
        }

        public static BattleNetApiResponse<TokenResponse> GetToken(BattleNetRegion region, string code, string clientID, string scope)
        {
            var app = GetApp();
            var regionInfo = GetRegionInfo(region);
            using (var client = new HttpClient())
            {
                client.DefaultRequestHeaders.Authorization = new AuthenticationHeaderValue("Basic", Convert.ToBase64String(Encoding.UTF8.GetBytes(string.Format("{0}:{1}", app.ClientID, app.ClientSecret))));

                var content = new FormUrlEncodedContent(new List<KeyValuePair<string, string>>
                {
                    new KeyValuePair<string, string>("redirect_uri", app.RedirectUri),
                    new KeyValuePair<string, string>("scope", scope),
                    new KeyValuePair<string, string>("grant_type", "authorization_code"),
                    new KeyValuePair<string, string>("code", code)
                });

                var result = client.PostAsync(string.Format("{0}/oauth/token", regionInfo.OAuthBaseUrl), content).Result;
                return CreateResult<TokenResponse>(regionInfo, result);
            }
        }

        public static BattleNetApiResponse<BattleNetUser> GetUser(BattleNetRegion region, string accessToken)
        {
            var regionInfo = GetRegionInfo(region);
            using (var client = new HttpClient())
            {
                var result = client.GetAsync(string.Format("{0}/account/user?access_token={1}", regionInfo.ApiBaseUrl, accessToken)).Result;
                return CreateResult<BattleNetUser>(regionInfo, result);
            }
        }

        public static BattleNetApiResponse<WowUserCharacters> GetWowCharacters(BattleNetRegion region, string accessToken)
        {
            var regionInfo = GetRegionInfo(region);
            using (var client = new HttpClient())
            {
                var result = client.GetAsync(string.Format("{0}/wow/user/characters?access_token={1}", regionInfo.ApiBaseUrl, accessToken)).Result;
                return CreateResult<WowUserCharacters>(regionInfo, result);
            }
        }

        public static BattleNetApiResponse<WowGuild> GetWowGuild(BattleNetRegion region, string server, string guildName)
        {
            var app = GetApp();
            var regionInfo = GetRegionInfo(region);
            using (var client = new HttpClient())
            {
                var result = client.GetAsync(string.Format("{0}/wow/guild/{1}/{2}?fields=members&locale=en_US&apiKey={3}", regionInfo.ApiBaseUrl, server, guildName, app.ClientID)).Result;
                return CreateResult<WowGuild>(regionInfo, result);
            }
        }

        private static BattleNetApiResponse<TResult> CreateResult<TResult>(BattleNetRegionInfo regionInfo, HttpResponseMessage response)
        {
            var result = new BattleNetApiResponse<TResult>
            {
                RegionInfo = regionInfo,
                StatusCode = response.StatusCode
            };

            var responseCode = (int)response.StatusCode;
            if (responseCode >= 500)
            {
                result.Status = BattleNetApiResponseStatus.ServerError;
                return result;
            }

            var content = response.Content.ReadAsStringAsync().Result;
            if (string.IsNullOrEmpty(content))
            {
                result.Status = BattleNetApiResponseStatus.InvalidResponseFormat;
                return result;
            }

            try
            {
                result.Content = JsonConvert.DeserializeObject<TResult>(content);
            }
            catch (JsonException)
            {
                result.Status = BattleNetApiResponseStatus.InvalidResponseFormat;
                return result;
            }

            if (responseCode >= 400)
            {
                result.Status = BattleNetApiResponseStatus.GeneralError;
                return result;
            }

            result.Status = BattleNetApiResponseStatus.Success;
            return result;
        }

        public static BattleNetRegionInfo GetRegionInfo(BattleNetRegion region)
        {
            BattleNetRegionInfo regionInfo;
            if (!_regions.TryGetValue(region, out regionInfo))
            {
                throw new InvalidOperationException("Unknown BattleNet region: " + region);
            }
            return regionInfo;
        }
    }
}
