﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Logging;

namespace Curse.Friends.BattleNet
{
    public static class BattleNetModelHelper
    {
        public static ExternalAccount CreateAccount(TokenResponse token, BattleNetUser user)
        {
            var account = ExternalAccount.GetLocal(user.ID, AccountType.WorldOfWarcraft);
            if (account == null)
            {
                account = new ExternalAccount
                {
                    Type = AccountType.WorldOfWarcraft,
                    ExternalID = user.ID,
                    ExternalUsername = user.BattleTag,
                    ExternalDisplayName = user.BattleTag,
                    MappedUsers = new HashSet<int>(),
                    AuthToken = token.AccessToken,
                    Scopes = new HashSet<string>(token.Scope.Split(' ')),
                };
                account.InsertLocal();
            }
            else
            {
                account.ExternalUsername = user.BattleTag;
                account.ExternalDisplayName = user.BattleTag;
                account.AuthToken = token.AccessToken;
                account.Update(a => a.ExternalDisplayName, a => a.ExternalUsername, a => a.AuthToken);
            }

            return account;
        }

        public static bool SyncAccount(ExternalAccount account, int region, int userID)
        {
            try
            {
                // Get all initial mappings
                var existingMembers = ExternalGuildMember.GetAllLocal(m => m.AccountID, account.ExternalID)
                    .ToDictionary(m => new ExternalGuildIdentifier(AccountType.WorldOfWarcraft, m.GuildGameRegion, m.GameServer, m.Name));

                var updated = false;

                var battleNetRegion = (BattleNetRegion) region;

                // Get all current characters in this region from BattleNet
                var charactersResult = BattleNetApiHelper.GetWowCharacters(battleNetRegion, account.AuthToken);
                if (charactersResult.Status != BattleNetApiResponseStatus.Success)
                {
                    throw new BattleNetResponseException("Failed to get WOW character results: " + charactersResult.StatusCode)
                    {
                        Content = charactersResult.Content,
                        RegionInfo = charactersResult.RegionInfo,
                        Status = charactersResult.Status,
                    };
                }

                // Get or create all guilds for all characters on the account
                var currentCharacters = charactersResult.Content.Characters
                    .Where(c=>!string.IsNullOrEmpty(c.Realm) && !string.IsNullOrEmpty(c.Name))
                    .GroupBy(c => new ExternalGuildIdentifier(AccountType.WorldOfWarcraft, region, c.Realm, c.Name))
                    .ToDictionary(c => c.Key, c => c.First());
                var guilds = new HashSet<ExternalGuildIdentifier>(
                    currentCharacters.Where(kvp => !string.IsNullOrEmpty(kvp.Value.Guild) && !string.IsNullOrEmpty(kvp.Value.GuildRealm))
                        .Select(kvp => new ExternalGuildIdentifier(AccountType.WorldOfWarcraft, region, kvp.Value.GuildRealm, kvp.Value.Guild)));
                var trackedGuilds = ExternalGuild.MultiGetLocal(guilds.Select(g => new KeyInfo(g.Type, g.GameRegion, g.GameServer, g.Name))).ToDictionary(g => g.GetGuildInfo());

                foreach (var guild in guilds)
                {
                    ExternalGuild trackedGuild;
                    if (trackedGuilds.TryGetValue(guild, out trackedGuild) && DateTime.UtcNow - trackedGuild.DateLastSynced < TimeSpan.FromHours(2))
                    {
                        // Don't request from bnet if we have the data and it is recent
                        continue;
                    }

                    // Sync guild and create
                    trackedGuild = CreateGuild(guild);
                    if (trackedGuild != null)
                    {
                        trackedGuilds[guild] = trackedGuild;
                    }
                }

                // Add or update current member mappings
                var currentMembers = ExternalGuildMember.MultiGetLocal(currentCharacters.Values.Select(c => new KeyInfo(AccountType.WorldOfWarcraft, region, c.Realm, c.Name)))
                    .ToDictionary(m => Tuple.Create(m.GameServer, m.Name));
                foreach (var character in currentCharacters.Values)
                {
                    var memberInfo = new ExternalGuildMemberInfo
                    {
                        AccountID = account.ExternalID,
                        GameRegion = region,
                        Name = character.Name,
                        GameServer = character.Realm,
                        Class = character.Class,
                        Gender = character.Gender,
                        Level = character.Level,
                        Race = character.Race
                    };

                    var guildInfo = new ExternalGuildIdentifier(AccountType.WorldOfWarcraft, region, character.GuildRealm, character.Guild);
                    ExternalGuild trackedGuild;
                    if (!string.IsNullOrEmpty(character.GuildRealm) && !string.IsNullOrEmpty(character.Guild) && trackedGuilds.TryGetValue(guildInfo, out trackedGuild))
                    {
                        memberInfo.GuildIdentifier = guildInfo;
                        memberInfo.GuildRole = trackedGuild.GetMemberRole(character.Realm, character.Name);
                    }

                    ExternalGuildMember member;
                    if (currentMembers.TryGetValue(Tuple.Create(character.Realm, character.Name), out member))
                    {
                        updated |= member.Update(memberInfo);
                    }
                    else
                    {
                        ExternalGuildMember.Create(AccountType.WorldOfWarcraft, memberInfo);
                        updated = true;
                    }
                }

                // Delete old member mappings
                var oldData = existingMembers.Where(m => m.Value.GameRegion == region && !currentCharacters.ContainsKey(m.Key));
                foreach (var oldMember in oldData)
                {
                    oldMember.Value.AccountID = string.Empty;
                    oldMember.Value.Update(m => m.AccountID);
                    updated = true;
                }
                return updated;
            }
            catch (BattleNetResponseException)
            {
                throw;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unexpected exception. Failed to sync wow account", new {account, userID});
                return false;
            }
        }

        public static bool UpdateGuild(ExternalGuild guild)
        {
            var guildInfoResult = BattleNetApiHelper.GetWowGuild((BattleNetRegion)guild.GameRegion, guild.GameServer, guild.Name);
            if (guildInfoResult.Status != BattleNetApiResponseStatus.Success)
            {
                Logger.Debug("Failed to obtain guild information", guild);
                return false;
            }

            var guildInfo = GetGuildInfo(guild.GetGuildInfo(), guild.GetGuildMasterInfo(), guildInfoResult.Content);
            guild.Update(guildInfo);
            return UpdateMembers(guild, guildInfoResult.Content);
        }

        private static ExternalGuildInfo GetGuildInfo(ExternalGuildIdentifier guildIdentifier, ExternalGuildIdentifier gmIdentifier, WowGuild wowGuild)
        {
            var guildInfo = new ExternalGuildInfo
            {
                GuildIdentifier = guildIdentifier,
                Faction = wowGuild.Side,
                Level = wowGuild.Level,
                AchievementPoints = wowGuild.AchievementPoints,
                GuildMasterIdentifier = gmIdentifier,
            };

            if (wowGuild.Members != null)
            {
                guildInfo.MemberCount = wowGuild.Members.Length;
            }

            if (wowGuild.Emblem != null)
            {
                guildInfo.Emblem = new ExternalGuildEmblem
                {
                    Icon = wowGuild.Emblem.Icon,
                    IconColor = wowGuild.Emblem.IconColor,
                    Border = wowGuild.Emblem.Border,
                    BorderColor = wowGuild.Emblem.BorderColor,
                    BackgroundColor = wowGuild.Emblem.BackgroundColor
                };
            }

            return guildInfo;
        }

        public static ExternalGuild CreateGuild(ExternalGuildIdentifier guildIdentifier)
        {
            try
            {
                var guildInfoResult = BattleNetApiHelper.GetWowGuild((BattleNetRegion) guildIdentifier.GameRegion, guildIdentifier.GameServer, guildIdentifier.Name);
                if (guildInfoResult.Status != BattleNetApiResponseStatus.Success)
                {
                    Logger.Debug("Failed to obtain Guild information", new {guildIdentifier, guildInfoResult});
                    return null;
                }

                var gm = guildInfoResult.Content.Members.FirstOrDefault(r => r.Rank == 0);
                var gmIdentifier = gm == null
                    ? new ExternalGuildIdentifier(AccountType.WorldOfWarcraft, guildIdentifier.GameRegion, string.Empty, string.Empty)
                    : new ExternalGuildIdentifier(AccountType.WorldOfWarcraft, guildIdentifier.GameRegion, gm.Character.Realm, gm.Character.Name);

                var guildInfo = GetGuildInfo(guildIdentifier, gmIdentifier, guildInfoResult.Content);

                var guild = ExternalGuild.Get(guildIdentifier);
                if (guild == null)
                {
                    guild = ExternalGuild.Create(guildInfo);
                }
                else
                {
                    guild.Update(guildInfo);
                }

                var tags = ExternalGuildRole.GetRoleTags(AccountType.WorldOfWarcraft);
                var roles = ExternalGuildRole.MultiGetLocal(tags.Select(tag => new KeyInfo(guildIdentifier.Type, guildIdentifier.GameRegion, guildIdentifier.GameServer, guildIdentifier.Name, tag)))
                    .ToDictionary(r => r.RoleTag);

                foreach (var tag in tags)
                {
                    ExternalGuildRole role;
                    if (roles.TryGetValue(tag, out role))
                    {
                        continue;
                    }

                    guild.CreateRole(tag);
                }

                guild.DateLastSynced = DateTime.UtcNow;
                guild.Update(g => g.DateLastSynced);
                BattleNetGuildWorker.SyncMembers(guild);

                return guild;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error while creating external wow guild: " + guildIdentifier);
                return null;
            }
        }

        public static void SyncGuildMembers(ExternalGuild guild)
        {
            var guildInfoResult = BattleNetApiHelper.GetWowGuild((BattleNetRegion)guild.GameRegion, guild.GameServer, guild.Name);
            if (guildInfoResult.Status != BattleNetApiResponseStatus.Success)
            {
                Logger.Debug("Failed to obtain Guild information", new { guild, guildInfoResult });
                return;
            }

            UpdateMembers(guild, guildInfoResult.Content);
        }

        public static ExternalGuildMember CreateMember(int gameRegion, ExternalGuildIdentifier guildIdentifier, WowGuildMember wowGuildMember)
        {
            var memberInfo = new ExternalGuildMemberInfo
            {
                Name = wowGuildMember.Character.Name,
                GuildRole = GetRoleTag(wowGuildMember.Rank),
                GameRegion = gameRegion,
                GameServer = wowGuildMember.Character.Realm,
                AccountID = string.Empty,
                GuildIdentifier = guildIdentifier,
                Class = wowGuildMember.Character.Class,
                Race = wowGuildMember.Character.Race,
                Gender = wowGuildMember.Character.Gender,
                Level = wowGuildMember.Character.Level
            };
            return ExternalGuildMember.CreateOrUpdate(AccountType.WorldOfWarcraft, memberInfo);
        }

        public static bool UpdateMembers(ExternalGuild guild, WowGuild wowGuild)
        {
            try
            {
                guild.DateLastSynced = DateTime.UtcNow;
                guild.MemberCount = wowGuild.Members.Length;
                guild.Update(g => g.DateLastSynced, g => g.MemberCount);

                var guildIndex = guild.GetGuildIndex();

                var updated = false;

                var existingMembers = ExternalGuildMember.GetAllLocal(g => g.GuildIndex, guildIndex).ToDictionary(m => Tuple.Create(m.GameServer, m.Name));

                var currentMembersDictionary = wowGuild.Members.Where(m => !string.IsNullOrEmpty(m.Character.Realm) && !string.IsNullOrEmpty(m.Character.Name))
                    .GroupBy(m => Tuple.Create(m.Character.Realm, m.Character.Name)).ToDictionary(m => m.Key, m => m.First());

                // Detect new members and update roles
                var missingMembers = new Dictionary<Tuple<string, string>, WowGuildMember>();
                foreach (var wowGuildMember in currentMembersDictionary)
                {
                    ExternalGuildMember existingMember;
                    if (!existingMembers.TryGetValue(wowGuildMember.Key, out existingMember))
                    {
                        updated = true;
                        missingMembers.Add(wowGuildMember.Key, wowGuildMember.Value);
                        continue;
                    }

                    var newTag = GetRoleTag(wowGuildMember.Value.Rank);
                    if (existingMember.GuildRole != newTag)
                    {
                        updated = true;
                        existingMember.GuildRole = newTag;
                        existingMember.Update(m => m.GuildRole);
                    }
                }

                // Add/update missing members and their roles
                var missings = ExternalGuildMember.MultiGetLocal(missingMembers.Select(m => new KeyInfo(AccountType.WorldOfWarcraft, guild.GameRegion, m.Key.Item1, m.Key.Item2)))
                    .ToDictionary(m => Tuple.Create(m.GameServer, m.Name));
                foreach (var missingMember in missingMembers)
                {
                    ExternalGuildMember missing;
                    if (missings.TryGetValue(missingMember.Key, out missing))
                    {
                        missing.GuildGameRegion = guild.GameRegion;
                        missing.GuildGameServer = wowGuild.Realm;
                        missing.GuildName = wowGuild.Name;
                        missing.GuildRole = GetRoleTag(missingMember.Value.Rank);
                        missing.GuildIndex = guildIndex;
                        missing.Update(m => m.GuildRole, m => m.GuildGameRegion, m => m.GuildGameServer, m => m.GuildIndex, m => m.GuildName);
                    }
                    else
                    {
                        CreateMember(guild.GameRegion, guild.GetGuildInfo(), missingMember.Value);
                    }
                }

                // Remove existing members that are no longer members
                foreach (var existingMember in existingMembers)
                {
                    WowGuildMember currentMember;
                    if (currentMembersDictionary.TryGetValue(existingMember.Key, out currentMember))
                    {
                        continue;
                    }

                    updated = true;
                    existingMember.Value.GuildRole = GroupRoleTag.None;
                    existingMember.Value.GuildGameRegion = 0;
                    existingMember.Value.GuildGameServer = string.Empty;
                    existingMember.Value.GuildName = string.Empty;
                    existingMember.Value.GuildIndex = string.Empty;
                    existingMember.Value.Update(m => m.GuildRole, m => m.GuildGameRegion, m => m.GuildGameServer, m => m.GuildIndex, m => m.GuildName);
                }
                return updated;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update members for guild: " + guild.GetGuildIndex(), new {guild});
                return false;
            }
        }

        private static GroupRoleTag GetRoleTag(int rank)
        {
            switch (rank)
            {
                case 0:
                    return GroupRoleTag.GuildMasterRank;
                case 1:
                    return GroupRoleTag.GuildRank1;
                case 2:
                    return GroupRoleTag.GuildRank2;
                case 3:
                    return GroupRoleTag.GuildRank3;
                case 4:
                    return GroupRoleTag.GuildRank4;
                case 5:
                    return GroupRoleTag.GuildRank5;
                case 6:
                    return GroupRoleTag.GuildRank6;
                case 7:
                    return GroupRoleTag.GuildRank7;
                case 8:
                    return GroupRoleTag.GuildRank8;
                case 9:
                    return GroupRoleTag.GuildRank9;
                default:
                    return GroupRoleTag.None;
            }
        }
    }
}
