﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Threading;
using System.Threading.Tasks;
using Aerospike.Client;
using Curse.Aerospike;
using Curse.Friends.BattleNet;
using Curse.Friends.BattleNetService.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.ServerHosting;
using Curse.Logging;

namespace Curse.Friends.BattleNetService
{
    public class BattleNetServer : ServerHost<BattleNetHost,ExternalGuild>
    {
        private static readonly BattleNetServer _instance = new BattleNetServer();

        private BattleNetServer()
        {
            
        }

        public static void StartServer()
        {
            _instance.Start();
        }

        public static void StopServer()
        {
            _instance.Stop();
        }

        public const int BucketCount = 30;
        private readonly ConcurrentDictionary<ExternalGuildIdentifier, WowGuildSession> _wowGuildSessions = new ConcurrentDictionary<ExternalGuildIdentifier, WowGuildSession>();

        private static int _bucketIncrementer = -1;
        private bool _isRunning;

        protected override void CustomStartup()
        {
            _isRunning = true;

            // battlenet oauth and api breaks if we allow tls 1.0
            ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls11 | SecurityProtocolType.Tls12;

            BattleNetApiHelper.Initialize(BattleNetServiceConfiguration.Instance.Application);

            ExternalGuildCoordinator.StartProcessor(ExternalGuildCoordinator_ProcessMessage);
            BattleNetGuildWorker.StartProcessor(BattleNetGuildWorker_ProcessMessage);

            AddTimer("Claim Orphaned Guilds", TimeSpan.FromSeconds(10), ClaimOrphanedGuilds);
            AddTimer("RebalanceGuilds", TimeSpan.FromMinutes(1), RebalanceGuilds);

            //var period = 120 / (double)BucketCount;
            var period = 1 / (double)BucketCount;
            var periodTimespan = TimeSpan.FromMinutes(period);
            AddTimer("Run Bucketed Tasks", periodTimespan, RunBucketedTasks, false);
        }

        protected override void CustomStop()
        {
            try
            {
                _isRunning = false;

                var sessions = _wowGuildSessions.Values.ToArray();
                Logger.Info("Shutting down " + sessions.Length + " sessions...");
                foreach (var session in sessions)
                {
                    session.Shutdown();
                }

                Logger.Info("All sessions have been shut down!");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while shutting down sessions.");

            }
        }

        private readonly ConcurrentDictionary<ExternalGuildIdentifier, object> _createSessionLocks = new ConcurrentDictionary<ExternalGuildIdentifier, object>();

        private WowGuildSession GetOrAddGuildSession(ExternalGuildIdentifier guild, out bool retrievedFromCache)
        {
            WowGuildSession session;
            if (_wowGuildSessions.TryGetValue(guild, out session))
            {
                retrievedFromCache = true;
                return session;
            }

            var createSession = _createSessionLocks.GetOrAdd(guild, new object());
            var lockAcquired = false;
            Monitor.TryEnter(createSession, TimeSpan.FromSeconds(20), ref lockAcquired);

            try
            {
                if (!lockAcquired)
                {
                    throw new InvalidOperationException("Failed to acquire create session lock for guild session: " + guild);
                }

                lock (createSession)
                {
                    if (_wowGuildSessions.TryGetValue(guild, out session))
                    {
                        retrievedFromCache = true;
                        return session;
                    }

                    retrievedFromCache = false;
                    return _wowGuildSessions.GetOrAdd(guild, id => new WowGuildSession(guild, GetBestBucket()));
                }
            }
            finally
            {
                if (lockAcquired)
                {
                    Monitor.Exit(createSession);
                }
            }

        }

        private int GetBestBucket()
        {
            var weights = new Dictionary<int, int>();
            var groupings = _wowGuildSessions.Values.GroupBy(s => s.Bucket).ToDictionary(s => s.Key, s => s.Count());
            for (var i = 1; i <= BucketCount; ++i)
            {
                int guildCount;
                if (!groupings.TryGetValue(i, out guildCount))
                {
                    guildCount = 0;
                }

                weights.Add(i, guildCount);
            }

            return weights.OrderBy(w => w.Value).First().Key;
        }

        private void RunBucketedTasks()
        {
            var bucketIncrement = Interlocked.Increment(ref _bucketIncrementer) % BucketCount;
            var currentBucket = (bucketIncrement < 0 ? bucketIncrement + BucketCount : bucketIncrement) + 1;
            var bucketedGuilds = _wowGuildSessions.Values.Where(v => v.Bucket == currentBucket).ToArray();

            if (bucketedGuilds.Length == 0)
            {
                Logger.Trace(string.Format("[Bucket {0}]: No guilds in bucket", currentBucket));
                return;
            }

            Logger.Trace(string.Format("[Bucket {0}]: Running bucket tasks for {1} guilds", currentBucket, bucketedGuilds.Length));
            Parallel.ForEach(bucketedGuilds, session =>
            {
                if (!_isRunning)
                {
                    return;
                }

                try
                {
                    session.RunBucketedTasks();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to run bucketed tasks!", new { session.GuildInfo });
                }
            });
            Logger.Trace(string.Format("[Bucket {0}]: Finished bucket tasks for {1} guilds", currentBucket, bucketedGuilds.Length));
        }

        private void ClaimOrphanedGuilds()
        {
            if(_host == null)
            {
                return; 
            }
            // Pick a random orphan            
            var hosts = BattleNetHost.GetAllLocal(p => p.IndexMode, IndexMode.Default).Where(p => p.Status == ServiceHostStatus.Online).ToArray();
            var hostIndex = Array.FindIndex(hosts, p => p.MachineName == _host.MachineName);

            var allLocalGuilds = ExternalGuild.GetAllLocal(s => s.RegionID, ExternalCommunity.LocalConfigID);
            var allOrphanedGuilds = allLocalGuilds.Where(m => string.IsNullOrWhiteSpace(m.MachineName) && m.MappedGroups != null && m.MappedGroups.Any()).ToArray();

            if (!allOrphanedGuilds.Any())
            {
                return;
            }

            var take = (int)Math.Ceiling(allOrphanedGuilds.Length / (double)hosts.Length);
            var skip = hostIndex * take;
            var orphans = allOrphanedGuilds.Skip(skip).Take(take).ToArray();

            Logger.Info("Found " + allOrphanedGuilds.Length + " orphaned guilds. Attempting to rehost...", new { hostIndex, skip, take });

            foreach (var orphan in orphans)
            {
                if (orphan == null)
                {
                    return;
                }

                orphan.Refresh();

                if (!string.IsNullOrEmpty(orphan.MachineName))
                {
                    Logger.Info("Skipping guild. It's been rehosted by: " + orphan.MachineName);
                    continue;
                }


                try
                {
                    orphan.MachineName = Environment.MachineName;
                    orphan.Update(UpdateMode.Concurrent, m => m.MachineName);
                }
                catch (AerospikeException ex)
                {
                    if (ex.Message.Contains("Generation error"))  // This means that another host has already claimed it
                    {
                        return;
                    }

                    Logger.Error(ex, "Error claiming an orphaned guild.");
                    continue;
                }


                if (_wowGuildSessions.ContainsKey(orphan.GetGuildInfo()))
                {
                    Logger.Info("Skipping guild '" + orphan.GetGuildIndex() + "'. It is already hosted here.");
                    continue;
                }

                try
                {
                    Logger.Info("Hosting orphaned guild: " + orphan.GetGuildIndex());
                    // Force create a session
                    bool retrievedFromCache;
                    GetOrAddGuildSession(orphan.GetGuildInfo(), out retrievedFromCache);
                }
                catch (Exception ex)
                {

                    Logger.Error(ex, "Failed to create WOW session for community", orphan);
                }

            }

            Logger.Info("Finished hosting orpaned guilds! We are now hosting: " + _wowGuildSessions.Count + " sessions");

        }

        private void RebalanceGuilds()
        {
            if(_host == null)
            {
                return; 
            }

            var guilds = ExternalGuild.GetAllLocal(c => c.RegionID, ExternalGuild.LocalConfigID)
                .Where(c => c.Type == AccountType.WorldOfWarcraft && !string.IsNullOrEmpty(c.MachineName)).ToArray();

            var guildsByMachine = guilds.GroupBy(c => c.MachineName).ToDictionary(g => g.Key, g => g.ToArray());
            var guildsByID = guilds.ToDictionary(g => g.GetGuildInfo());

            var allHosts = BattleNetHost.GetAllLocal(g => g.IndexMode, IndexMode.Default).Where(h => h.Status == ServiceHostStatus.Online);
            foreach (var host in allHosts.Where(h => !guildsByMachine.ContainsKey(h.MachineName)))
            {
                guildsByMachine.Add(host.MachineName, new ExternalGuild[0]);
            }

            ExternalGuild[] myGuilds;
            if (!guildsByMachine.TryGetValue(_host.MachineName, out myGuilds))
            {
                foreach (var session in _wowGuildSessions.ToArray())
                {
                    WowGuildSession throwaway;
                    session.Value.Shutdown();
                    _wowGuildSessions.TryRemove(session.Key, out throwaway);
                }
                return;
            }

            // Clear sessions that should not be hosted by this machine
            var keys = new HashSet<ExternalGuildIdentifier>(myGuilds.Select(c => c.GetGuildInfo()));
            var removedGuilds = _wowGuildSessions.Where(s => !keys.Contains(s.Key)).ToArray();

            if (removedGuilds.Any())
            {
                Logger.Info("Removing " + removedGuilds.Length + " guild sessions that should be hosted by other machines");
            }

            foreach (var session in removedGuilds)
            {
                string newHost = null;
                ExternalGuild guild;
                if (guildsByID.TryGetValue(session.Key, out guild))
                {
                    newHost = guild.MachineName;
                }

                session.Value.Shutdown(newHost);
                WowGuildSession throwaway;
                _wowGuildSessions.TryRemove(session.Key, out throwaway);
            }



            // Rehost sessions that should be hosted but are missing
            var addingSessions = myGuilds.Where(c => !_wowGuildSessions.ContainsKey(c.GetGuildInfo())).ToArray();

            if (addingSessions.Any())
            {
                Logger.Info("Adding " + addingSessions.Length + " sessions that should hosted by this machine");
            }

            foreach (var session in addingSessions)
            {
                bool retrievedFromCache;
                GetOrAddGuildSession(session.GetGuildInfo(), out retrievedFromCache);
            }

            // Figure out if this machine is hosting the most guilds
            var highestCount = guildsByMachine.Max(s => s.Value.Length);
            if (myGuilds.Length != highestCount)
            {
                return;
            }

            // Pass out guilds, favoring hosts with the least guilds hosted
            var total = guilds.Length;
            var target = total / guildsByMachine.Count;
            var totalGuildsToGive = myGuilds.Length - target;
            var hostsToBalance = guildsByMachine.Where(s => s.Key != _host.MachineName && myGuilds.Length - s.Value.Length > myGuilds.Length * 0.2).ToArray();

            var guildsLeft = totalGuildsToGive;
            foreach (var hostKvp in hostsToBalance.OrderBy(h => h.Value.Length))
            {
                if (guildsLeft == 0)
                {
                    break;
                }

                var guildsToGive = new[] { totalGuildsToGive, target - hostKvp.Value.Length, guildsLeft }.Min();
                if (guildsToGive == 0)
                {
                    continue;
                }

                Logger.Info(string.Format("Giving {0} guild sessions to {1}", guildsToGive, hostKvp.Key));
                var sessionsToGive = _wowGuildSessions.Take(guildsToGive).ToArray();
                foreach (var session in sessionsToGive)
                {
                    session.Value.Shutdown(hostKvp.Key);
                    WowGuildSession throwaway;
                    _wowGuildSessions.TryRemove(session.Key, out throwaway);
                }

                ExternalGuildCoordinator.TransferGuilds(ExternalGuild.LocalConfigID, hostKvp.Key, sessionsToGive.Select(s => s.Key).ToArray());
                guildsLeft -= guildsToGive;
            }
        }

        private void ExternalGuildCoordinator_ProcessMessage(ExternalGuildCoordinator coordinator)
        {
            try
            {
                switch (coordinator.Type)
                {
                    case ExternalGuildCoordinatorType.Commission:
                    case ExternalGuildCoordinatorType.Transfer:
                        foreach (var guild in coordinator.GuildIdentifiers)
                        {
                            bool retrievedFromCache;
                            GetOrAddGuildSession(guild, out retrievedFromCache);
                        }
                        break;
                    case ExternalGuildCoordinatorType.Decommission:
                        foreach (var guild in coordinator.GuildIdentifiers)
                        {
                            WowGuildSession session;
                            if (_wowGuildSessions.TryRemove(guild, out session))
                            {
                                session.Shutdown();
                            }
                        }
                        break;
                    default:
                        throw new InvalidOperationException("Unknown ExternalGuildCoordinatorType");
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error coordinating Guild", coordinator);
            }
        }

        private static void BattleNetGuildWorker_ProcessMessage(BattleNetGuildWorker worker)
        {
            try
            {
                switch (worker.Type)
                {
                    case BattleNetGuildWorkerType.SyncMembers:
                        Logger.Debug("Syncing members for " + worker.Guild.GetGuildIndex());
                        BattleNetModelHelper.SyncGuildMembers(worker.Guild);
                        break;
                    default:
                        throw new InvalidOperationException("Unsupported BattleNetGuildWorkerType: " + worker.Type);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error processing Battle.Net Guild Worker", worker);
            }
        }
    }
}
