﻿using System;
using System.Collections.Concurrent;
using System.Data.SqlClient;
using System.Net;
using System.Threading;
using Curse.Friends.BugsWebService.Contracts;
using Curse.Friends.BugsWebService.Models;
using Curse.Friends.Enums;
using Curse.Friends.BugsWebService.Configuration;
using Curse.Logging;
using System.Threading.Tasks;
using Curse.Friends.BugsWebService;

namespace Curse.ClientService.Managers
{
    public static class BugReportingManager
    {
        private static readonly ConcurrentDictionary<Int64, Int32> StackTraceToReportID = new ConcurrentDictionary<Int64, Int32>();
        private static readonly ConcurrentDictionary<string, Int32> VersionStringToID = new ConcurrentDictionary<string, Int32>();

        private static readonly ConcurrentQueue<BugReportQueueItem> CrashReportQueue = new ConcurrentQueue<BugReportQueueItem>();
        private static readonly ConcurrentQueue<FeedbackQueueItem> FeedbackQueue = new ConcurrentQueue<FeedbackQueueItem>();
        private static readonly ConcurrentQueue<BugReportQueueItem> BugReportQueue = new ConcurrentQueue<BugReportQueueItem>();
        private static readonly bool ProcessGameCrashes = BugsWebServiceConfiguration.Current.ProcessGameCrashes;
        private static bool _isRunning;

        private class BugReportQueueItem
        {
            public BugReport Report { get; private set; }
            public ReportingUserInfo ReportingUser { get; private set; }
            public IPAddress IPAddress { get; set; }

            public BugReportQueueItem(BugReport report, ReportingUserInfo reportingUser, IPAddress ipAddress)
            {
                Report = report;
                ReportingUser = reportingUser;
                IPAddress = ipAddress;
            }
        }

        private class FeedbackQueueItem
        {
            public Feedback Feedback { get; private set; }
            public ReportingUserInfo ReportingUser { get; private set; }

            public FeedbackQueueItem(Feedback feedback, ReportingUserInfo reportingUser)
            {
                Feedback = feedback;
                ReportingUser = reportingUser;
            }
        }


        public static void Start()
        {
            _isRunning = true;
            new Thread(ProcessQueueThread) { IsBackground = true }.Start();
        }

        public static void Stop()
        {
            _isRunning = false;
        }

        public static void QueueCrashReport(BugReport report, ReportingUserInfo reportingUser, IPAddress ipAddress)
        {
            if (CrashReportQueue.Count > BugsWebServiceConfiguration.Current.MaxQueueSize)
            {
                Logger.Warn("Max queue size reached for crash reports.");
                return;
            }

            CrashReportQueue.Enqueue(new BugReportQueueItem(report, reportingUser, ipAddress));
        }

        public static void QueueBugReport(BugReport report, ReportingUserInfo reportingUser, IPAddress ipAddress)
        {
            if (BugReportQueue.Count > BugsWebServiceConfiguration.Current.MaxQueueSize)
            {
                Logger.Warn("Max queue size reached for bug reports.");
                return;
            }

            BugReportQueue.Enqueue(new BugReportQueueItem(report, reportingUser, ipAddress));
        }

        public static void QueueGameCrash(BugReport report, ReportingUserInfo reportingUser, IPAddress ipAddress)
        {
            if (!ProcessGameCrashes)
            {
                return;
            }
            QueueBugReport(report, reportingUser, ipAddress);
        }

        public static void QueueFeedback(Feedback feedback, ReportingUserInfo reportingUser)
        {
            if (FeedbackQueue.Count > BugsWebServiceConfiguration.Current.MaxQueueSize)
            {
                Logger.Warn("Max queue size reached for bug reports.");
                return;
            }

            FeedbackQueue.Enqueue(new FeedbackQueueItem(feedback, reportingUser));
        }

        private static readonly LogCategory ThrottledLogger = new LogCategory("BugReporting") { Throttle = TimeSpan.FromMinutes(1) };

        private static void ProcessQueueThread()
        {
            while (_isRunning)
            {
                try
                {
                    Thread.Sleep(1000);

                    BugReportQueueItem report;

                    while (BugReportQueue.TryDequeue(out report))
                    {
                        try
                        {
                            SaveBugReport(report);
                        }
                        catch (SqlException ex)
                        {
                            ThrottledLogger.Error(ex, "Failed to save bug report due to SQL error.");
                        }
                        catch (Exception ex)
                        {
                            Logger.Warn(ex, "Bug report was not saved, due to invalid data or exception.", new { report.Report.Platform, report.Report.ClientVersion });
                        }

                        Thread.Sleep(10); // Caps this to 100 reports per second
                    }

                    while (CrashReportQueue.TryDequeue(out report))
                    {
                        try
                        {
                            SaveCrashReport(report);
                        }
                        catch (SqlException ex)
                        {
                            ThrottledLogger.Error(ex, "Failed to save crash report due to SQL error.");
                        }
                        catch (Exception ex)
                        {
                            Logger.Warn(ex, "Crash report was not saved to the database, due to invalid data or exception.", new { report.Report.Platform, report.Report.ClientVersion });
                        }
                        Thread.Sleep(10); // Caps this to 100 reports per second
                    }

                    FeedbackQueueItem feedback;
                    while (FeedbackQueue.TryDequeue(out feedback))
                    {
                        try
                        {
                            SaveFeedback(feedback);
                        }
                        catch (SqlException ex)
                        {
                            ThrottledLogger.Error(ex, "Failed to save feedback due to SQL error.");
                        }
                        catch (Exception ex)
                        {
                            Logger.Warn(ex, "Feedback was not saved, due to invalid data or exception.", new { feedback.Feedback.Platform, feedback.Feedback.ClientVersion });
                        }

                        Thread.Sleep(10); // Caps this to 100 reports per second
                    }


                }
                catch (ThreadAbortException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to process bug report queue");
                }
            }
        }


        private static void SaveCrashReport(BugReportQueueItem queueItem)
        {
            try
            {
                MongooseHelper.SubmitReport(queueItem.Report, queueItem.ReportingUser);
            }
            catch (Exception ex)
            {
                ThrottledLogger.Error(ex, "Error posting CrashReport to mongoose", queueItem);
            }

            if(BugsWebServiceConfiguration.Current.DatabaseShutoff)
            {
                return;
            }

            // Get the client version ID
            var clientVersionID = GetOrCreateVersion(queueItem.Report.ClientVersion, queueItem.Report.Platform);

            if (clientVersionID == null)
            {
                Logger.Warn("Unable to save crash report. The client version supplied could not be retrieved or created!", queueItem.Report.ClientVersion);
                return;
            }


            // For logging
            var retrievedFromCache = false;
            var retrievedByHash = false;
            var inserted = false;

            // See if a crash report already exists with the same stack trace
            var alreadyExists = false;
            var reportID = 0;

            try
            {
                if (queueItem.Report.StackTraceHash.HasValue)
                {
                    if (!StackTraceToReportID.TryGetValue(queueItem.Report.StackTraceHash.Value, out reportID))
                    {
                        reportID = ClientBugReport.GetIDByHash(queueItem.Report.StackTraceHash.Value);
                        if (reportID > 0)
                        {
                            StackTraceToReportID.TryAdd(queueItem.Report.StackTraceHash.Value, reportID);
                            retrievedByHash = true;
                        }
                    }
                    else
                    {
                        retrievedFromCache = true;
                    }
                }

                if (reportID > 0)
                {
                    // Update the existing report data                
                    ClientBugReport.Update(reportID, clientVersionID.Value, queueItem.Report.Message);
                    alreadyExists = true;
                }
                else // Save a new client report
                {
                    reportID = ClientBugReport.SaveToDatabase(queueItem.Report, queueItem.ReportingUser.CurseUserID, clientVersionID.Value,
                        BugReportPriority.Medium);
                    inserted = true;
                }

                // Save the incident
                ClientCrashIncident.SaveToDatabase(reportID, clientVersionID.Value, queueItem.ReportingUser.CurseUserID,
                    queueItem.IPAddress);
            }
            catch (SqlException ex)
            {
                if (queueItem.Report.StackTraceHash.HasValue && ex.Message.Contains("FOREIGN KEY"))
                {
                    int removedReportID;
                    StackTraceToReportID.TryRemove(queueItem.Report.StackTraceHash.Value, out removedReportID);
                    Logger.Info("Removing cached stack trace hash", new { StackTraceHach = queueItem.Report.StackTraceHash.Value, removedReportID });
                }
                else
                {
                    throw;
                }

            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Crash report was not saved to the database, due to invalid data or exception.", new
                {
                    queueItem.Report.Platform,
                    queueItem.Report.ClientVersion,
                    BugReportID = reportID,
                    BugReportRetrievedFromCache = retrievedFromCache,
                    BugReportRetrievedByHash = retrievedByHash,
                    BugReportInserted = inserted
                });
                return;
            }

            if (reportID == 0)
            {
                return;
            }

            try
            {
                var existingReport = ClientBugReport.GetByID(reportID);                
                if (alreadyExists)
                {
                    // Fill in data
                    existingReport.UpdateProperties(queueItem.Report);
                }
                existingReport.SaveToS3();
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Crash report was not saved to S3.", new { queueItem.Report.Platform, queueItem.Report.ClientVersion });
            }
        }

        private static void SaveBugReport(BugReportQueueItem queueItem)
        {
            try
            {
                MongooseHelper.SubmitReport(queueItem.Report, queueItem.ReportingUser);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error posting BugReport to mongoose", queueItem);
            }

            // ATL Migration - don't write to sql05 anymore now that new bug reports service is in place.
            if (BugsWebServiceConfiguration.Current.DatabaseShutoff)
            {
                return;
            }

            // Get the client version ID
            var clientVersionID = GetOrCreateVersion(queueItem.Report.ClientVersion, queueItem.Report.Platform);

            if (clientVersionID == null)
            {
                Logger.Warn("Unable to save bug report. The client version supplied could not be retrieved or created!", queueItem.Report.ClientVersion);
                return;
            }

            int reportID;
            try
            {
                reportID = ClientBugReport.SaveToDatabase(queueItem.Report, queueItem.ReportingUser.CurseUserID, clientVersionID.Value, BugReportPriority.Medium);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Bug report was not saved to the database, due to invalid data or exception.", new { queueItem.Report.Platform, queueItem.Report.ClientVersion });
                return;
            }

            if (reportID == 0)
            {
                return;
            }

            try
            {
                var bugReport = ClientBugReport.GetByID(reportID);
                bugReport.SaveToS3();
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Bug report was not saved to S3", new { queueItem.Report.Platform, queueItem.Report.ClientVersion });
            }
        }

        private static void SaveFeedback(FeedbackQueueItem feedbackQueueItem)
        {
            var feedback = feedbackQueueItem.Feedback;
            var bugReport = new BugReport
            {
                Type = feedback.FeedbackType == FeedbackType.Positive ? BugReportType.PostiveFeedback : BugReportType.NegativeFeedback,
                ClientVersion = feedback.ClientVersion,
                Message = feedback.Message,
                LogData = feedback.LogData,
                LastAttemptedVoiceHostID = feedback.LastAttemptedVoiceHostID,
                Platform = feedback.Platform,
                Title = feedback.FeedbackType == FeedbackType.Positive ? "Positive Feedback" : "Negative Feedback",
            };

            try
            {
                MongooseHelper.SubmitReport(bugReport, feedbackQueueItem.ReportingUser);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error posting CrashReport to mongoose", new
                {
                    bugReport,
                    feedbackQueueItem
                });
            }

            // ATL Migration - don't write to sql05 anymore now that new bug reports service is in place.
            if (BugsWebServiceConfiguration.Current.DatabaseShutoff)
            {
                return;
            }

            // Get the client version ID
            var clientVersionID = GetOrCreateVersion(feedback.ClientVersion, feedback.Platform);

            if (clientVersionID == null)
            {
                Logger.Warn("Unable to save bug report. The client version supplied could not be retrieved or created!", feedback.ClientVersion);
                return;
            }

            int reportID;
            try
            {
                reportID = ClientBugReport.SaveToDatabase(bugReport, feedbackQueueItem.ReportingUser.CurseUserID, clientVersionID.Value, BugReportPriority.Medium);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Feedback was not saved to the database, due to invalid data or exception.", new { feedbackQueueItem.Feedback.Platform, feedbackQueueItem.Feedback.ClientVersion });
                return;
            }

            if (reportID == 0)
            {
                return;
            }

            try
            {
                var feedbackReport = ClientBugReport.GetByID(reportID);
                feedbackReport.SaveToS3();
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Feedback was not saved to S3.", new { feedbackQueueItem.Feedback.Platform, feedbackQueueItem.Feedback.ClientVersion });
            }

        }


        private static int? GetOrCreateVersion(string version, DevicePlatform platform)
        {
            int versionID;

            var cacheKey = version + "-" + platform;

            // Check our cache of hash code mappings
            if (VersionStringToID.TryGetValue(cacheKey, out versionID))
            {
                return versionID;
            }

            var versionIDValue = ClientVersion.GetOrCreateVersion(version, platform);
            if (!versionIDValue.HasValue)
            {
                return null;
            }

            VersionStringToID.TryAdd(cacheKey, versionIDValue.Value);
            return versionIDValue.Value;
        }


    }
}