﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.ClientService.Managers;
using Curse.Extensions;
using Curse.Friends.BugsWebService.Contracts;
using Curse.Friends.MicroService;
using Curse.Friends.MicroService.Exceptions;
using Curse.Logging;

namespace Curse.Friends.BugsWebService.Controllers
{
    [RoutePrefix("bugs")]
    public class BugsController : MicroServiceController
    {
        private static DateTime? _lastCorruptInstallReport = null;

        [Route("")]
        [HttpPost]
        [ResponseType(typeof (void))]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult Report(BugReport report)
        {
            try
            {
                report.Validate();
            }
            catch (RequestValidationException ex)
            {
                Logger.Warn("Failed to submit a bug report", new {FailureReason = ex.ToSerializableObject()});
                throw;
            }

            if (report.Type == BugReportType.AutoReport && report.Title.Equals("CorruptInstallException"))
            {
                if (_lastCorruptInstallReport.HasValue &&
                    DateTime.UtcNow - _lastCorruptInstallReport.Value < TimeSpan.FromMinutes(30))
                {
                    return Ok();
                }

                _lastCorruptInstallReport = DateTime.UtcNow;
            }

            PreprocessBugReport(report);            

            var reportingUserInfo = GetReportingUserInfo();               
                
            switch (report.Type)
            {
                case BugReportType.CrashReport:
                    BugReportingManager.QueueCrashReport(report, reportingUserInfo, GetCurrentIpAddress());
                    break;
                case BugReportType.GameCrash:
                    BugReportingManager.QueueGameCrash(report, reportingUserInfo, GetCurrentIpAddress());
                    break;
                default:
                    BugReportingManager.QueueBugReport(report, reportingUserInfo, GetCurrentIpAddress());
                    break;
            }

            return Ok();
        }

        private ReportingUserInfo GetReportingUserInfo()
        {            
            if (Token.IsAnonymous)
            {
                return ReportingUserInfo.Default;                    
            }
            
            UserRegionInfo loggedInUser = null;
            try
            {
                loggedInUser = GetCurrentUserAndRegion();
            }
            catch { }

            if (loggedInUser != null && loggedInUser.User != null)
            {
                return new ReportingUserInfo
                {
                    CurseUserID = loggedInUser.User.UserID,
                    Username = loggedInUser.User.Username,
                    TwitchUserID = loggedInUser.User.TwitchID,
                    Email = loggedInUser.User.EmailAddress
                };
            }
            return ReportingUserInfo.Default;
        }

        private void PreprocessBugReport(BugReport report)
        {
            var preprocessMessages = new List<string>();

            if (report.Title != null && report.Title.Length > 64)
            {
                preprocessMessages.Add("Bug report title too long: " + report.Title + ", truncating");
                report.Title = report.Title.Substring(0, 64);
            }

            if (report.Message != null && report.Message.Length > 3000)
            {
                preprocessMessages.Add("Bug report message too long: " + report.Message.Length + ", truncating");
                report.Message = report.Message.Substring(0, 3000);
            }

            if (report.ExceptionDetails != null && report.ExceptionDetails.Length > 3000)
            {
                preprocessMessages.Add("Bug report exception details too long: " + report.ExceptionDetails.Length + ", truncating");
                report.ExceptionDetails = report.ExceptionDetails.Substring(0, 3000);
            }

            if (report.UserSettings != null && report.UserSettings.Length > 50000)
            {
                preprocessMessages.Add("Bug report user settings too long: " + report.UserSettings.Length + ", truncating");
                report.UserSettings = report.UserSettings.Substring(0, 50000);
            }

            if (report.SystemInformation != null && report.SystemInformation.Length > 10000)
            {
                preprocessMessages.Add("Bug report system information too long: " + report.SystemInformation.Length + ", truncating");
                report.SystemInformation = report.SystemInformation.Substring(0, 10000);
            }

            if (report.IncompatibleProcesses != null && report.IncompatibleProcesses.Length > 10000)
            {
                preprocessMessages.Add("Bug report incompatible processes too long: " + report.IncompatibleProcesses.Length + ", truncating");
                report.IncompatibleProcesses = report.IncompatibleProcesses.Substring(0, 10000);
            }

            if (report.GameConfiguration != null && report.GameConfiguration.Length > 10000)
            {
                preprocessMessages.Add("Bug report game configuration too long: " + report.GameConfiguration.Length + ", truncating");
                report.GameConfiguration = report.GameConfiguration.Substring(0, 10000);
            }

            if (report.TimeConfiguration != null && report.TimeConfiguration.Length > 10000)
            {
                preprocessMessages.Add("Bug report time configuration too long: " + report.TimeConfiguration.Length + ", truncating");
                report.TimeConfiguration = report.TimeConfiguration.Substring(0, 10000);
            }

            if (report.RunningProcesses != null)
            {
                if (report.RunningProcesses.Length > 1000)
                {
                    preprocessMessages.Add("Bug report has too many running processes: " + report.RunningProcesses.Length + ", truncating");
                    report.RunningProcesses = report.RunningProcesses.Take(1000).ToArray();
                }

                var longestName = 0;
                var longestFullName = 0;
                foreach (var runningProcess in report.RunningProcesses)
                {
                    if (runningProcess.Name != null && runningProcess.Name.Length > 128)
                    {
                        longestName = Math.Max(longestName, runningProcess.Name.Length);
                        runningProcess.Name = runningProcess.Name.Substring(0, 128);
                    }

                    if (runningProcess.FullName != null && runningProcess.FullName.Length > 256)
                    {
                        longestFullName = Math.Max(longestFullName, runningProcess.FullName.Length);
                        runningProcess.FullName = runningProcess.FullName.Substring(0, 256);
                    }
                }

                if (longestName > 0)
                {
                    preprocessMessages.Add("Bug report had at least one running process with a long name that was truncated. Longest length was " + longestName);
                }

                if (longestFullName > 0)
                {
                    preprocessMessages.Add("Bug report had at least one running process with a long full name that was truncated. Longest length was " + longestFullName);
                }

            }

            if (preprocessMessages.Count > 0)
            {
                Logger.Warn("A bug report had to be preprocessed prior to submitting", new {PreprocessMessages = preprocessMessages});
            }
        }
    }
}