﻿using System;
using System.Data.SqlClient;
using System.IO;
using System.Text;
using Curse.Extensions;
using Curse.Friends.BugsWebService.Contracts;
using Curse.Friends.BugsWebService.Extensions;
using Curse.Friends.S3;
using Curse.Friends.BugsWebService.Configuration;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.Friends.BugsWebService.Models
{
    public class ClientBugReport
    {
        #region Column Ordinals

        private const int _idOrdinal = 0;
        private const int _userIDOrdinal = 1;
        private const int _clientVersionIDOrdinal = 2;
        private const int _dateCreatedOrdinal = 3;
        private const int _dateLastReportedOrdinal = 5;
        private const int _titleOrdinal = 7;
        private const int _typeOrdinal = 8;
        private const int _priorityOrdinal = 9;
        private const int _stackTraceHashOrdinal = 10;
        private const int _userSettingsOrdinal = 11;
        private const int _systemInformationOrdinal = 12;
        private const int _incompatibleProcessesOrdinal = 13;
        private const int _gameConfigurationOrdinal = 14;
        private const int _timeConfigurationOrdinal = 15;
        private const int _logDataOrdinal = 16;
        private const int _exceptionDetailsOrdinal = 17;
        private const int _applicationLocationOrdinal = 18;
        private const int _stepsToReproduceOrdinal = 19;
        private const int _lastAttemptedVoiceHostIDOrdinal = 21;
        private const int _overlayExitCodeOrdinal = 22;
        private const int _runningProcessesOrdinal = 23;
        private const int _overlayGameStateOrdinal = 29;
        private const int _callQualityOrdinal = 35;

        #endregion

        public int ID { get; set; }
        public int UserID { get; set; }
        public int VersionID { get; set; }
        public DateTime DateCreated { get; set; }
        public DateTime DateLastReported { get; set; }
        public string Title { get; set; }
        public BugReportType Type { get; set; }
        public BugReportPriority Priority { get; set; }
        public long? StackTraceHash { get; set; }
        public string UserSettings { get; set; }
        public string SystemInformation { get; set; }
        public string IncompatibleProcesses { get; set; }
        public string GameConfiguration { get; set; }
        public string TimeConfiguration { get; set; }
        public string LogData { get; set; }
        public string ExceptionDetails { get; set; }
        public string ApplicationLocation { get; set; }
        public string StepsToReproduce { get; set; }
        public int? LastAttemptedVoiceHostID { get; set; }
        public int? OverlayExitCode { get; set; }
        public byte? OverlayGameState { get; set; }
        public int? CallQuality { get; set; }
        public string RunningProcesses { get; set; }

        private ClientBugReport(int id)
        {
            ID = id;
        }

        public ClientBugReport(BugReport report, int userID, int clientVersionID, BugReportPriority priority)
        {
            UserID = userID;
            VersionID = clientVersionID;
            Priority = priority;
            UpdateProperties(report);
        }

        public void UpdateProperties(BugReport report)
        {
            Title = report.Title;
            Type = report.Type;
            StackTraceHash = report.StackTraceHash;
            UserSettings = report.UserSettings;
            SystemInformation = report.SystemInformation;
            IncompatibleProcesses = report.IncompatibleProcesses;
            GameConfiguration = report.GameConfiguration;
            LogData = report.LogData;
            ExceptionDetails = report.ExceptionDetails;
            ApplicationLocation = report.ApplicationLocation;
            StepsToReproduce = report.Message;
            LastAttemptedVoiceHostID = report.LastAttemptedVoiceHostID.HasValue && report.LastAttemptedVoiceHostID != 0 ? report.LastAttemptedVoiceHostID : null;
            OverlayExitCode = report.OverlayExitCode;
            OverlayGameState = report.OverlayGameState;
            CallQuality = report.CallQuality;
            TimeConfiguration = report.TimeConfiguration;

            try
            {
                RunningProcesses = report.RunningProcesses == null || report.RunningProcesses.Length == 0 ? null : JsonConvert.SerializeObject(report.RunningProcesses);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to convert running process list to JSON. ");
            }

        }

        private void UpdateProperties(SqlDataReader reader)
        {
            ID = reader.SafeGet<int>(_idOrdinal);
            UserID = reader.SafeGet<int>(_userIDOrdinal);
            VersionID = reader.SafeGet<int>(_clientVersionIDOrdinal);
            DateCreated = reader.SafeGet<DateTime>(_dateCreatedOrdinal).NormalizeToUtc();
            DateLastReported = reader.SafeGet<DateTime>(_dateLastReportedOrdinal).NormalizeToUtc();
            Title = reader.SafeGet<string>(_titleOrdinal);
            Type = (BugReportType)reader.SafeGet<byte>(_typeOrdinal);
            Priority = (BugReportPriority)reader.SafeGet<byte>(_priorityOrdinal);
            StackTraceHash = reader.SafeGet<long?>(_stackTraceHashOrdinal);
            UserSettings = reader.SafeGet<string>(_userSettingsOrdinal);
            SystemInformation = reader.SafeGet<string>(_systemInformationOrdinal);
            IncompatibleProcesses = reader.SafeGet<string>(_incompatibleProcessesOrdinal);
            GameConfiguration = reader.SafeGet<string>(_gameConfigurationOrdinal);
            TimeConfiguration = reader.SafeGet<string>(_timeConfigurationOrdinal);
            LogData = reader.SafeGet<string>(_logDataOrdinal);
            ExceptionDetails = reader.SafeGet<string>(_exceptionDetailsOrdinal);
            ApplicationLocation = reader.SafeGet<string>(_applicationLocationOrdinal);
            StepsToReproduce = reader.SafeGet<string>(_stepsToReproduceOrdinal);
            LastAttemptedVoiceHostID = reader.SafeGet<int?>(_lastAttemptedVoiceHostIDOrdinal);
            OverlayExitCode = reader.SafeGet<int?>(_overlayExitCodeOrdinal);
            RunningProcesses = reader.SafeGet<string>(_runningProcessesOrdinal);
            OverlayGameState = reader.SafeGet<byte?>(_overlayGameStateOrdinal);
            CallQuality = reader.SafeGet<int?>(_callQualityOrdinal);
        }

        private int Save(SqlConnection conn)
        {
            using (var cmd = conn.CreateCommand())
            {
                cmd.CommandText = string.Join(" ",
                    "INSERT INTO [ClientBugReport] (UserID, VersionID, Title, Type, Priority, StackTraceHash, UserSettings,",
                    "SystemInformation, IncompatibleProcesses, GameConfiguration, TimeConfiguration, LogData, ExceptionDetails,",
                    "ApplicationLocation, StepsToReproduce, LastAttemptedVoiceHostID, OverlayExitCode, RunningProcesses,",
                    "OverlayGameState, CallQuality)",
                    "OUTPUT INSERTED.ID",
                    "VALUES(@UserID, @VersionID, @Title, @Type, @Priority, @StackTraceHash, @UserSettings,",
                    "@SystemInformation, @IncompatibleProcesses, @GameConfiguration, @TimeConfiguration, @LogData, @ExceptionDetails,",
                    "@ApplicationLocation, @StepsToReproduce, @LastAttemptedVoiceHostID, @OverlayExitCode, @RunningProcesses,",
                    "@OverlayGameState, @CallQuality)");

                cmd.Parameters.AddWithValue("@UserID", UserID);
                cmd.Parameters.AddWithValue("@VersionID", VersionID);
                cmd.Parameters.AddWithValue("@Title", Title);
                cmd.Parameters.AddWithValue("@Type", Type);
                cmd.Parameters.AddWithValue("@Priority", Priority);
                cmd.Parameters.AddWithValue("@StackTraceHash", StackTraceHash.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@UserSettings", UserSettings.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@SystemInformation", SystemInformation.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@IncompatibleProcesses", IncompatibleProcesses.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@GameConfiguration", GameConfiguration.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@TimeConfiguration", TimeConfiguration.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@LogData", LogData.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@ExceptionDetails", ExceptionDetails.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@ApplicationLocation", ApplicationLocation.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@StepsToReproduce", StepsToReproduce.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@LastAttemptedVoiceHostID", LastAttemptedVoiceHostID.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@OverlayExitCode", OverlayExitCode.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@OverlayGameState", OverlayGameState.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@CallQuality", CallQuality.CoalesceToDBNull());
                cmd.Parameters.AddWithValue("@RunningProcesses", RunningProcesses.CoalesceToDBNull());
                ID = (int) cmd.ExecuteScalar();
            }

            return ID;
        }

        public static int SaveToDatabase(BugReport report, int userID, int clientVersionID, BugReportPriority priority)
        {
            var clientBugReport = new ClientBugReport(report, userID, clientVersionID, priority);

            using (var connection = new SqlConnection(BugsWebServiceConfiguration.Current.DatabaseConnectionString))
            {
                connection.Open();

                clientBugReport.Save(connection);

                if (report.Attachments == null)
                {
                    return clientBugReport.ID;
                }

                foreach (var attachment in report.Attachments)
                {
                    ClientBugReportAttachment.SaveToDatabase(connection, clientBugReport.ID, attachment);
                }
            }

            return clientBugReport.ID;
        }

        public static void Update(int id, int clientVersionID, string stepsToReproduce)
        {
            using (var conn = new SqlConnection(BugsWebServiceConfiguration.Current.DatabaseConnectionString))
            {
                conn.Open();
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText =
                        "UPDATE [ClientBugReport] SET VersionID = @VersionID, StepsToReproduce = COALESCE(StepsToReproduce, @StepsToReproduce),  DateLastReported = GETUTCDATE(), ReportedCount = ReportedCount + 1 WHERE ID = @ID";
                    cmd.Parameters.AddWithValue("@VersionID", clientVersionID);
                    cmd.Parameters.AddWithValue("@StepsToReproduce", stepsToReproduce.CoalesceToDBNull());
                    cmd.Parameters.AddWithValue("@ID", id);
                    cmd.ExecuteNonQuery();
                }
            }
        }

        public bool Refresh()
        {
            try
            {
                using (var conn = new SqlConnection(BugsWebServiceConfiguration.Current.DatabaseConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "SELECT * FROM [ClientBugReport] WHERE [ID] = @ID";
                        cmd.Parameters.AddWithValue("@ID", ID);

                        using (var reader = cmd.ExecuteReader())
                        {
                            if (reader.Read())
                            {
                                UpdateProperties(reader);
                                return true;
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "BugReporting: Unable to get report from ID!");
            }

            return false;
        }

        public static ClientBugReport GetByID(int id)
        {
            var report = new ClientBugReport(id);
            return report.Refresh() ? report : null;
        }

        public static int GetIDByHash(Int64 hash)
        {
            try
            {
                // Check the database
                using (var conn = new SqlConnection(BugsWebServiceConfiguration.Current.DatabaseConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "SELECT TOP 1 [ID] from [ClientBugReport] where [StackTraceHash] = @StackTraceHash";
                        cmd.Parameters.AddWithValue("@StackTraceHash", hash);

                        using (var reader = cmd.ExecuteReader())
                        {
                            if (reader.Read())
                            {
                                return reader.SafeGet<int>(0);
                            }
                            return 0;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "BugReporting: Unable to get report from hash!");
                return 0;
            }
        }

        public void SaveToS3()
        {
#if CONFIG_RELEASE
            const string mode = "release";
#else
            const string mode = "staging";
#endif
            var key = string.Format("{0}/{1}/{2}/{3}", mode, Type.ToString().ToLowerInvariant(), DateLastReported.Date.ToString("yyyy-MM-dd"), Guid.NewGuid());

            var filename = string.Format("report-{0}.json", ID);
            var bytes = Encoding.UTF8.GetBytes(JsonConvert.SerializeObject(this));
            using (var stream = new MemoryStream(bytes))
            {
                var metadata = new FileMetadata(UserID, filename, bytes.Length, DateLastReported, "application/json");
                S3Manager.SaveFile(key, BugsWebServiceConfiguration.Current.Bucket, metadata, stream);
            }
        }
    }
}