﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using Curse.Extensions;
using Curse.Friends.BugsWebService.Contracts;
using Curse.Friends.BugsWebService.Extensions;
using Curse.Friends.BugsWebService.Configuration;
using Curse.Logging;

namespace Curse.Friends.BugsWebService.Models
{
    public class ClientBugReportAttachment
    {
        public int AttachmentID { get; set; }

        public int ReportID { get; set; }

        public string Filename { get; set; }

        public long FileLength { get; set; }

        public byte[] FileContents { get; set; }

        public DateTime DateCreated { get; set; }

        public ClientBugReportAttachment(int reportID, BugReportAttachment attachment)
        {
            ReportID = reportID;
            Filename = attachment.FileName;
            FileLength = attachment.FileContents.Length;
            FileContents = attachment.FileContents;
        }

        private ClientBugReportAttachment(SqlDataReader reader)
        {
            AttachmentID = reader.GetInt32(0);
            ReportID = reader.GetInt32(1);
            Filename = reader.GetString(2);
            FileLength = reader.GetInt64(3);
            DateCreated = reader.GetDateTime(4).NormalizeToUtc();
            FileContents = reader.SafeGet<byte[]>(5);
        }

        private void Save(SqlConnection conn)
        {
            using (var attachmentCmd = conn.CreateCommand())
            {
                // Now, save the record to the model
                attachmentCmd.CommandText =
                    "INSERT INTO [ClientBugReportAttachment] (ReportID, FileName, FileLength, FileContents) VALUES(@ReportID, @FileName, @FileLength, @FileContents)";
                attachmentCmd.Parameters.AddWithValue("@ReportID", ReportID);
                attachmentCmd.Parameters.AddWithValue("@FileName", Filename);
                attachmentCmd.Parameters.AddWithValue("@FileLength", FileLength);
                attachmentCmd.Parameters.AddWithValue("@FileContents", FileContents);
                attachmentCmd.ExecuteNonQuery();
            }
        }

        public static void SaveToDatabase(SqlConnection connection, int reportID, BugReportAttachment attachment)
        {
            if (!IsValidAttachment(attachment))
            {
                return;
            }

            var clientBugReportAttachment = new ClientBugReportAttachment(reportID, attachment);
            clientBugReportAttachment.Save(connection);

        }

        public static bool IsValidAttachment(BugReportAttachment attachment)
        {
            if (string.IsNullOrWhiteSpace(attachment.FileName))
            {
                Logger.Warn("Skipping bug report attachment: Missing filename.");
                return false;
            }

            if (attachment.FileName.Length > 64)
            {
                Logger.Warn("Skipping bug report attachment: Filename '" + attachment.FileName + "' too long");
                return false;
            }

            if (attachment.FileContents == null)
            {
                Logger.Warn("Skipping bug report attachment: File contents are null.");
                return false;
            }

            if (attachment.FileContents.Length == 0)
            {
                Logger.Warn("Skipping bug report attachment: File contents are 0 bytes.");
                return false;
            }

            return true;
        }

        public static ClientBugReportAttachment[] GetAttachmentsForReport(int bugReportID)
        {
            var attachments = new List<ClientBugReportAttachment>();
            using (var connection = new SqlConnection(BugsWebServiceConfiguration.Current.DatabaseConnectionString))
            {
                connection.Open();
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [ClientBugReportAttachment] WHERE [ReportID] = @reportID";
                    command.Parameters.AddWithValue("@reportID", bugReportID);

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            attachments.Add(new ClientBugReportAttachment(reader));
                        }
                    }
                }
            }
            return attachments.ToArray();
        }

        public static ClientBugReportAttachment GetByID(int attachmentId)
        {
            using (var connection = new SqlConnection(BugsWebServiceConfiguration.Current.DatabaseConnectionString))
            {
                connection.Open();
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [ClientBugReportAttachment] WHERE [ID] = @attachmentID";
                    command.Parameters.AddWithValue("@attachmentID", attachmentId);

                    using (var reader = command.ExecuteReader())
                    {
                        if (!reader.Read())
                        {
                            return null;
                        }

                        return new ClientBugReportAttachment(reader);
                    }
                }
            }
        }
    }
}