﻿using System;
using System.Net;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Friends.CallsWebService.Contracts;
using Curse.Friends.Configuration.CurseVoiceService;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Logging;

namespace Curse.Friends.CallsWebService.Controllers
{
    [RoutePrefix("calls/api")]
    [ApiExplorerSettings(IgnoreApi = true)]
    public class CallsAdminController : MicroServiceController
    {
        private static readonly LogCategory Logger = new LogCategory("CallsAdminController") { AlphaLevel = LogLevel.Trace };

        [Route("host-callback")]
        [AuthenticationFilter((AuthenticationLevel.ApiKey))]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult VoiceHostCallback(VoiceHostCallbackRequest request)
        {

            // No need to return errors
            var group = Group.GetWritable(request.GroupID);

            if (group == null)
            {
                return Ok();
            }

            GroupChangeType changeType;
            switch (request.ChangeType)
            {
                case VoiceHostCallbackType.CallStarted:
                    changeType = GroupChangeType.VoiceSessionStarted;
                    break;
                case VoiceHostCallbackType.CallEnded:
                    changeType = GroupChangeType.VoiceSessionEnded;
                    break;
                case VoiceHostCallbackType.UsersJoined:
                    changeType = GroupChangeType.VoiceSessionUserJoined;
                    break;
                case VoiceHostCallbackType.UsersLeft:
                    changeType = GroupChangeType.VoiceSessionUserLeft;
                    break;
                default:
                    return BadRequest("Unsupported voice host callback type");
            }

            group.VoiceSessionChanged(request.InviteCode, request.UserIDs, changeType);

            return Ok();
        }

        [Route("kick")]
        [ResponseType(typeof(void))]
        [HttpPost]
        [AuthenticationFilter(AuthenticationLevel.ApiKey)]
        public IHttpActionResult KickUsersFromCall(KickUsersFromCallRequest request)
        {

            var userDisconnectReason = request.Reason == KickReason.UserLeft
                ? UserDisconnectReason.LeftGroup
                : UserDisconnectReason.Kicked;

            var removeUsersResult = CurseVoiceServiceClient.TryServiceCall("RemoveUsersFromGroupCall",
                svc => svc.RemoveUsersFromGroupCall(Token.ApiKey, null, request.GroupID, request.RequestorUserID, request.UserIDs, userDisconnectReason));
            if (removeUsersResult.Status != RemoveUsersFromGroupCallStatus.Successful && removeUsersResult.Status != RemoveUsersFromGroupCallStatus.NotFound)
            {
                Logger.Warn("[RemoveUsersFromGroup] Failed to remove users from group call: " + removeUsersResult);
            }

            return StatusCode(HttpStatusCode.NoContent);
        }

        [Route("check-permissions/mute")]
        [HttpPost]
        [AuthenticationFilter(AuthenticationLevel.ApiKey)]
        [Obsolete("Permissions should be requested through GetModerationStatus initially and are updated automatically")]
        public IHttpActionResult CheckMuteUserPermission([FromBody] CheckVoicePermissionsRequest request)
        {
            return DoCheckPermission(request.GroupID, request.RequestingUserID, request.AffectedUserID, GroupPermissions.VoiceMuteUser);
        }


        [Route("check-permissions/deafen")]
        [HttpPost]
        [AuthenticationFilter(AuthenticationLevel.ApiKey)]
        [Obsolete("Permissions should be requested through GetModerationStatus initially and are updated automatically")]
        public IHttpActionResult CheckDeafenUserPermission([FromBody] CheckVoicePermissionsRequest request)
        {
            return DoCheckPermission(request.GroupID, request.RequestingUserID, request.AffectedUserID, GroupPermissions.VoiceDeafenUser);
        }

        [Route("check-permissions/kick")]
        [HttpPost]
        [AuthenticationFilter(AuthenticationLevel.ApiKey)]
        [Obsolete("Permissions should be requested through GetModerationStatus initially and are updated automatically")]
        public IHttpActionResult CheckKickUserPermission([FromBody] CheckVoicePermissionsRequest request)
        {
            return DoCheckPermission(request.GroupID, request.RequestingUserID, request.AffectedUserID, GroupPermissions.VoiceKickUser);
        }

        private IHttpActionResult DoCheckPermission(Guid groupID, int requestorUserID, int affectedUserID, GroupPermissions permission)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            if (group.RootGroup.CanModerateUser(requestorUserID, affectedUserID) &&
                group.HasPermission(permission, requestorUserID))
            {
                return StatusCode(HttpStatusCode.NoContent);
            }

            return Forbidden();
        }

        [Route("moderation-status/{groupID}/{userID}")]
        [HttpGet]
        [AuthenticationFilter(AuthenticationLevel.ApiKey)]
        [ResponseType(typeof (GetModerationStatusResponse))]
        public IHttpActionResult GetModerationStatus(Guid groupID, int userID)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }
            
            var rootGroup = group.RootGroup;
            if (rootGroup == null || rootGroup.IsDeleted)
            {
                return NotFound();
            }            

            GetModerationStatusResponse resp = null;

            if (rootGroup.Type == GroupType.Normal)
            {
                resp = new GetModerationStatusResponse
                {
                    Permissions = new VoicePermissions
                    {
                        BestRoleRank = 1,
                        CanModDeafen = false,
                        CanModMute = false,
                        CanSpeak = true,
                        CanModKick = false
                    },
                    IsMuted = false,
                    IsDeafened = false,
                };
            }
            else
            {
                var rootMember = rootGroup.GetMember(userID);
                if (rootMember == null)
                {
                    return NotFound();
                }

                resp = new GetModerationStatusResponse
                {
                    Permissions = new VoicePermissions
                    {
                        BestRoleRank = rootMember.BestRoleRank,                                                
                        CanModDeafen = group.HasPermission(GroupPermissions.VoiceDeafenUser, userID),
                        CanModMute = group.HasPermission(GroupPermissions.VoiceMuteUser, userID),
                        CanModKick = group.HasPermission(GroupPermissions.VoiceKickUser, userID),
                        // This is weird, but basically we assume if a user doesn't have access that they were dragged in
                        CanSpeak = !group.HasPermission(GroupPermissions.Access, userID) || group.HasPermission(GroupPermissions.VoiceSpeak, userID),
                    },
                    IsMuted = rootMember.IsVoiceMuted,
                    IsDeafened = rootMember.IsVoiceDeafened,
                };

                Logger.Trace("Retrieved moderation status for user: " + userID, resp);
            }            

            return Ok(resp);
        }
    }
}
