﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Extensions;
using Curse.Friends.CallsWebService.Contracts;
using Curse.Friends.Configuration;
using Curse.Friends.Configuration.CurseVoiceService;
using Curse.Friends.Data;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.Statistics;
using Curse.Logging;
using Curse.Friends.NotificationContracts;
using Curse.Voice.Helpers;
using Curse.Friends.MicroService.Filters;

namespace Curse.Friends.CallsWebService.Controllers
{
    [RoutePrefix("calls")]
    public class CallsController : MicroServiceController
    {
        private static readonly LogCategory Logger = new LogCategory("Calls");
        private static readonly LogCategory DiagLogger = new LogCategory("CallsDiagnostics") { Throttle = TimeSpan.FromSeconds(30), ReleaseLevel = LogLevel.Debug };

        /// <summary>
        /// Start or join call for the requested conversation.
        /// </summary>
        [Route("conversations/{conversationID}")]
        [HttpPost]
        [ResponseType(typeof(CallNotification))]
        [SocialBanFilter]
        public IHttpActionResult Conversation(string conversationID, ConversationCallRequest request)
        {
            request.Validate();

            var conversation = ConversationManager.GetConversationContainer(Token.UserID, conversationID, true);

            if (conversation == null)
            {
                DiagLogger.Warn("Attempt to make a call to a non-existing conversation. ", new { Token.UserID, conversationID, request });
                return NotFound();
            }

            if (!conversation.CanCall(Token.UserID))
            {
                var privateConversation = conversation as PrivateConversation;
                // This could be a data issue.
                if (privateConversation != null && privateConversation.RepairFriendship())
                {
                    Logger.Warn("Discovered a data integrity issue with a private conversation friendship state. The conversation data has been repaired.", new { privateConversation.ConversationID, privateConversation.DateCreated });                                        
                }
                else
                {
                    DiagLogger.Warn("Attempt to make a call to a stranger. ", new { Token.UserID, conversationID, request });
                    return Forbidden();
                }
                
            }

            if (conversation.ConversationType == ConversationType.Friendship)
            {
                return CallFriend((PrivateConversation)conversation, request);
            }

            if (conversation.ConversationType == ConversationType.Group)
            {
                return CallGroup((Group)conversation, request);
            }

            return BadRequest();
        }

        /// <summary>
        /// Start an ad hoc call.
        /// </summary>
        [Route("adhoc")]
        [HttpPost]
        [ResponseType(typeof(CallNotification))]
        [SocialBanFilter]
        public IHttpActionResult AdHoc(AdHocCallRequest request)
        {

            Trace("Making ad hoc call...", new { IPAddress = GetCurrentIpAddress() });

            var userAndRegion = GetCurrentUserAndRegion();
            var displayName = request.UserDisplayName ?? userAndRegion.User.GetTitleName();

            // Make an API call to the central voice server
            var resp = CurseVoiceServiceClient.TryServiceCall("CreateVoiceSession",
                c => c.AdHocVoiceSessionV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, displayName, request.GameID, Token.UserID, false, Version.Parse(request.ClientVersion), GetCurrentIpAddress()));

            if (resp.Status != CreateVoiceSessionStatus.Successful)
            {
                switch (resp.Status)
                {
                    case CreateVoiceSessionStatus.AlreadyExists:
                    case CreateVoiceSessionStatus.IncompatibleClient:
                        return BadRequest();
                    case CreateVoiceSessionStatus.Throttled:
                        return TooManyRequests();
                    default:
                        return ServiceUnavailable();
                }
            }

            var callDetails = FromVoiceInstanceDetails(resp.VoiceInstance, null);
            return CallNotification(callDetails);            

        }

        /// <summary>
        /// Opts out of the current AutoMatch Handshake
        /// </summary>
        /// <param name="autoMatchKey">The AutoMatch key</param>
        [HttpDelete]
        [Route("automatch/handshake/{autoMatchKey}")]
        [ResponseType(typeof(void))]
        public IHttpActionResult DeclineAutoMatch(long autoMatchKey)
        {
            var resp = CurseVoiceServiceClient.TryServiceCall("Something", svc => svc.DeclineAutoMatchV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                Token.UserID, autoMatchKey));
            if (resp.Status == DeclineAutoMatchStatus.Error)
            {
                return ErrorResponse(HttpStatusCode.InternalServerError, "service_call_error");
            }

            return Ok();
        }

        /// <summary>
        /// Performs AutoMatch Handshaking
        /// </summary>
        /// <param name="autoMatchKey">The AutoMatch key</param>
        /// <param name="request">Handshake details</param>
        [HttpPost]
        [Route("automatch/handshake/{autoMatchKey}")]
        [ResponseType(typeof(HandshakeAutoMatchResponse))]
        public IHttpActionResult AutoMatchHandshake(long autoMatchKey, HandshakeAutoMatchRequest request)
        {
            var resp2 = CurseVoiceServiceClient.TryServiceCall("Something1", svc => svc.AutoMatchHandshakeByDisplayNameV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                Token.UserID, request.Attempt, autoMatchKey, request.DisplayName, request.GameID));
            if (resp2.Status == AutoMatchHandshakeStatus.Error)
            {
                return ErrorResponse(HttpStatusCode.InternalServerError, "service_call_error");
            }

            return Ok(HandshakeAutoMatchResponse.FromServiceContract(resp2));
        }

        /// <summary>
        /// Start an automatch voice session.
        /// </summary>
        [Route("automatch")]
        [HttpPost]
        [ResponseType(typeof(CallNotification))]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult AutoMatch(AutoMatchCallRequest request)
        {

            Trace("Making an automatch call...", new { IPAddress = GetCurrentIpAddress() });
            
            // Make an API call to the central voice server
            var resp = CurseVoiceServiceClient.TryServiceCall("AutoMatchVoiceSessionV2", c => c.AutoMatchVoiceSessionV2(request.UserDisplayName, request.GameID, GetCurrentIpAddress(), Version.Parse(request.ClientVersion), request.AutoMatchKey));

           if (resp.Status != CreateVoiceSessionStatus.Successful)
            {
                switch (resp.Status)
                {
                    case CreateVoiceSessionStatus.AlreadyExists:
                    case CreateVoiceSessionStatus.IncompatibleClient:
                        return BadRequest();
                    case CreateVoiceSessionStatus.Throttled:
                        return TooManyRequests();
                    default:
                        return ServiceUnavailable();
                }
            }

            var callDetails = FromVoiceInstanceDetails(resp.VoiceInstance, null);
            return CallNotification(callDetails);

        }

        /// <summary>
        /// Gets the call details needed to join an ad hoc call
        /// </summary>
        /// <param name="inviteCode"></param>
        /// <returns></returns>
        [Route("adhoc/{inviteCode}")]
        [HttpGet]
        [ResponseType(typeof(CallNotification))]
        [SocialBanFilter]
        public IHttpActionResult AdHocDetails(string inviteCode)
        {
            if (string.IsNullOrWhiteSpace(inviteCode))
            {
                return BadRequest();
            }

            var resp = CurseVoiceServiceClient.TryServiceCall("FindVoiceSessionV2",
               c => c.FindVoiceSessionV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, inviteCode));

            if (resp == null || resp.Status != VoiceInstanceStatus.Active || resp.Type==VoiceInstanceType.Friend || resp.Type==VoiceInstanceType.Group)
            {
                return NotFound();
            }

            var callDetails = FromVoiceInstanceDetails(resp, null);
            return CallNotification(callDetails);            
        }

        /// <summary>
        /// Gets the publicly displayed call details
        /// </summary>
        /// <param name="inviteCode"></param>
        [Route("adhoc/{inviteCode}/display")]
        [HttpGet]
        [ResponseType(typeof(AdHocCallDisplayDetailsResponse))]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult AdHocDisplayDetails(string inviteCode)
        {
            if (string.IsNullOrWhiteSpace(inviteCode))
            {
                return BadRequest();
            }

            var resp = CurseVoiceServiceClient.TryServiceCall("FindVoiceSessionV2",
               c => c.FindVoiceSessionV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, inviteCode));

            if (resp == null || resp.Status != VoiceInstanceStatus.Active || (resp.Type!=VoiceInstanceType.AdHoc && resp.Type!=VoiceInstanceType.MultiFriend))
            {
                return NotFound();
            }

            return Ok(new AdHocCallDisplayDetailsResponse
            {
                UserID = resp.UserID,
                Username = resp.CreatorName,
                Timestamp = resp.DateCreated.ToEpochMilliseconds(),
                InviteCode = resp.InviteCode
            });
        }

        private IHttpActionResult CallNotification(CallDetails callDetails)
        {
            var currentUser = GetCurrentUserAndRegion();
            var accessToken = AccessTokenHelper.CreateAccessToken(callDetails.CallID, Token.UserID);
            return Ok(callDetails.ToNotification(accessToken, Token.UserID, currentUser.User.GetTitleName()));
        }
        
        private void Trace(string message, object data)
        {
            if (!FriendsServiceConfiguration.Instance.TraceUsers.Contains(Token.UserID))
            {
                return;
            }

            Logger.Info(message, data);
        }

        private IHttpActionResult CallFriend(PrivateConversation privateConversation, ConversationCallRequest request)
        {

            var callerUserAndRegion = GetCurrentUserAndRegion();
            
            // If this call includes an invite, add the pending user to the call
            PendingUserRequest pendingUser = null;
            if (request.SendInvitation)
            {
                pendingUser = new PendingUserRequest
                {
                    UserID = privateConversation.OtherUserID,
                    DisplayName = privateConversation.Title,
                    AvatarUrl = Avatar.GetUserAvatarUrl(privateConversation.OtherUserID)
                };
            }

            Trace("Calling friend...", new { IPAddress = GetCurrentIpAddress(), FriendID = privateConversation.OtherUserID });            

            // Make an API call to the central voice server
            var resp = CurseVoiceServiceClient.TryServiceCall("FriendVoiceSession",
                c => c.FriendVoiceSessionV3(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                    GetCurrentIpAddress(),
                    Version.Parse(request.ClientVersion),
                    callerUserAndRegion.User.UserID,
                    callerUserAndRegion.User.GetTitleName(),
                    privateConversation.OtherUserID,
                    request.Force,
                    pendingUser, request.Mode == CallMode.Audio ? VoiceInstanceMode.Audio : VoiceInstanceMode.Video)
                );

            switch (resp.Status)
            {
                case CreateVoiceSessionStatus.IncompatibleClient:
                    return BadRequest("Unsupported client version: " + request.ClientVersion);
                case CreateVoiceSessionStatus.Throttled:
                    return TooManyRequests();
                case CreateVoiceSessionStatus.Error:
                case CreateVoiceSessionStatus.NoHostsAvailable:
                    return ServiceUnavailable();
            }

            var callDetails = FromVoiceInstanceDetails(resp.VoiceInstance, privateConversation.ConversationID);

            if (request.SendInvitation)
            {                
                try
                {
                    FriendsStatsManager.Current.VoiceInvitationsSent.Track();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "[CallFriend] Failed to track stat!");
                }

                CallResolver.Create(callerUserAndRegion.User.UserID, callerUserAndRegion.User.GetTitleName(), privateConversation.OtherUserID, callDetails);
            }

            // Update privateConversation recency
            privateConversation.DateMessaged = callDetails.Timestamp;
            privateConversation.Update(f => f.DateMessaged);

            return CallNotification(callDetails);            
        }

        private IHttpActionResult CallGroup(Group group, ConversationCallRequest request)
        {
            Trace("Calling group...", new { IPAddress = GetCurrentIpAddress(), group.GroupID });

            // Only allow invites to be sent to large groups
            request.SendInvitation = request.SendInvitation && group.Type == GroupType.Normal && group.MemberCount < Group.MaxUsersForCallNotification;

            var resp = CurseVoiceServiceClient.TryServiceCall("GroupVoiceSession", c => c.GroupVoiceSessionV3(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                GetCurrentIpAddress(),
                Version.Parse(request.ClientVersion),
                Token.UserID,
                group.Title,
                group.GroupID,
                request.Force,
                group.RootGroup.VoiceRegionID, request.Mode == CallMode.Audio ? VoiceInstanceMode.Audio : VoiceInstanceMode.Video));


            switch (resp.Status)
            {
                case CreateVoiceSessionStatus.IncompatibleClient:
                    return BadRequest("Unsupported client version: " + request.ClientVersion);
                case CreateVoiceSessionStatus.Throttled:
                    return TooManyRequests();
                case CreateVoiceSessionStatus.Error:
                case CreateVoiceSessionStatus.NoHostsAvailable:
                    return ServiceUnavailable();
            }

            if (group.VoiceSessionCode != resp.VoiceInstance.InviteCode)
            {
                // Check the home region record, to ensure this is consistent
                var writableGroup = group.EnsureWritable();
                if (writableGroup.VoiceSessionCode != resp.VoiceInstance.InviteCode)
                {                 
                    writableGroup.VoiceSessionCode = resp.VoiceInstance.InviteCode;
                    writableGroup.Update(p => p.VoiceSessionCode);
                }                
            }

            group.CheckPushToTalkThreshold();

            var callDetails = FromVoiceInstanceDetails(resp.VoiceInstance, group.ConversationID);
            
            if (request.SendInvitation)
            {
                try
                {
                    FriendsStatsManager.Current.GroupVoiceInvitationsSent.Track();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "[CallGroup] Failed to track stat!");
                }
                
                GroupCallCoordinator.CallStarted(Token.UserID, group, callDetails);                
            }

            return CallNotification(callDetails);
        }

        /// <summary>
        /// Invite a friend to the requested call.
        /// </summary>
        /// <param name="callID"></param>
        /// <param name="friendID"></param>
        /// <returns></returns>
        [Route("{callID}/invite")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult AddFriendToCall(string callID, [FromBody] int friendID)
        {

            Guid callGuid;
            if (!Guid.TryParse(callID, out callGuid))
            {
                return BadRequest("The supplied callID is invalid: " + callGuid);
            }

            var me = GetUserAndRegion(Token.UserID);
            var them = Friendship.Get(me.Region.RegionID, me.User.UserID, friendID);

            // Confirm that the target recipient is a confirmed friend
            if (them == null || them.Status != FriendshipStatus.Confirmed)
            {
                return Forbidden();
            }

            // Unlock the session (may already be unlocked, but we don't care)
            var unlockResp = UnlockVoiceSession(callID);
            
            switch (unlockResp.Status)
            {
                case UnlockVoiceSessionStatus.Forbidden:
                    return Forbidden();
                case UnlockVoiceSessionStatus.NotFound:
                    return NotFound();
                case UnlockVoiceSessionStatus.Error:
                    return ServiceUnavailable();
            }


            FriendsStatsManager.Current.VoiceInvitationsSent.Track();

            // Now let our voice host know that we have a pending user (if this fails, log it, but still send the invite)
            try
            {
                var pendingUser = new PendingUserRequest
                {
                    UserID = them.OtherUserID,
                    DisplayName = them.OtherUsername,
                    AvatarUrl = Avatar.GetUserAvatarUrl(them.OtherUserID)
                };


                // Add pending user to the call
                var pendingResp = CurseVoiceServiceClient.TryServiceCall("AddPendingVoiceUsers",
                    c => c.AddPendingVoiceUsersV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, GetCurrentIpAddress(), callID, Token.UserID, new[] { pendingUser }));
                
                if (pendingResp.Status != BasicVoiceServiceStatus.Successful)
                {
                    Logger.Warn("[AddFriendToCall] Unsuccessful attempt to add pending users to voice host: " + pendingResp.Status, new { pendingResp, callID });
                    return BadRequest();
                }

                var callDetails = FromVoiceInstanceDetails(pendingResp.VoiceDetails, pendingResp.VoiceDetails.ExternalID);
                CallResolver.Create(Token.UserID, me.User.GetTitleName(), friendID, callDetails);
                return Ok();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[AddFriendToCall] Failed to add pending users to voice host.");
                return ServiceUnavailable();
            }            
            
        }

        /// <summary>
        /// Decline a request to join a call.
        /// </summary>
        /// <param name="callID"></param>
        /// <returns></returns>
        [Route("{callID}/decline")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult Decline(string callID)
        {
            Guid callGuid;
            if (!Guid.TryParse(callID, out callGuid))
            {
                return BadRequest("The supplied callID is invalid: " + callGuid);
            }
            
            var callDetails = GetWrappedCallDetails(callID);
            if (callDetails == null)
            {
                return NotFound();
            }

            if (!callDetails.Conversation.CanCall(Token.UserID))
            {
                return Forbidden();
            }
            
            // Track the stats
            if (callDetails.Conversation is Group)
            {
                FriendsStatsManager.Current.GroupVoiceInvitationsDeclined.Track();                
            }
            else if (callDetails.Conversation is Friendship)
            {
                FriendsStatsManager.Current.VoiceInvitationsDeclined.Track();                
            }

            RespondToCall(callDetails, CallResponseReason.Declined);

            // Remove pending user from voice host
            try
            {
                var removeResponse = CurseVoiceServiceClient.TryServiceCall("RemovePendingVoiceUser",
                        c => c.RemovePendingVoiceUser(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                                                      GetCurrentIpAddress(),
                                                      callDetails.CallDetails.InviteCode,
                                                      Token.UserID));

                if (removeResponse.Status == BasicVoiceServiceStatus.Error || removeResponse.Status == BasicVoiceServiceStatus.Failed || removeResponse.Status == BasicVoiceServiceStatus.Forbidden)
                {
                    Logger.Warn("[DeclineCall] Unsuccessful attempt to remove pending user from voice host: " + removeResponse.Status, new { removeResponse });
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[DeclineCall] Failed to remove pending user from voice host.");
            }

            return Ok();
        }

       
        /// <summary>
        /// Accept a request to join a call.
        /// </summary>
        /// <param name="callID"></param>
        /// <returns></returns>
        [Route("{callID}/accept")]
        [HttpPost]
        [ResponseType(typeof(void))]
        [SocialBanFilter]
        public IHttpActionResult Accept(string callID)
        {
            Guid callGuid;
            if (!Guid.TryParse(callID, out callGuid))
            {
                return BadRequest("The supplied callID is invalid: " + callGuid);
            }

            var callParent = GetWrappedCallDetails(callID);
            if (callParent == null)
            {
                return NotFound();
            }

            if (!callParent.Conversation.CanCall(Token.UserID))
            {
                return Forbidden();
            }
           
            RespondToCall(callParent, CallResponseReason.Accepted);

            return Ok();
        }

        /// <summary>
        /// Unlocks a call so that anyone with the URL can join it.
        /// Note: Automatch and Server Channel calls cannot be unlocked.
        /// </summary>
        /// <param name="callID"></param>
        /// <returns></returns>
        [Route("{callID}/unlock")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult Unlock(string callID)
        {
            Guid callGuid;
            if (!Guid.TryParse(callID, out callGuid))
            {
                return BadRequest("The supplied call is invalid: " + callGuid);
            }

            var callDetails = GetWrappedCallDetails(callID);
            if (callDetails == null)
            {
                return NotFound();
            }

            if (callDetails.CallDetails.Type == CallType.AutoMatch)
            {
                return Forbidden("AutoMatch calls cannot be unlocked.");
            }

            if (callDetails.CallDetails.Type == CallType.AdHoc)
            {
                return Ok("Call is already unlocked");
            }

            if (!callDetails.Conversation.CanUnlockCall(Token.UserID))
            {
                return Forbidden();
            }

            var unlockResp = UnlockVoiceSession(callID);

            switch (unlockResp.Status)
            {
                case UnlockVoiceSessionStatus.NotFound:
                    return NotFound();
                case UnlockVoiceSessionStatus.Forbidden:
                    return Forbidden();
                case UnlockVoiceSessionStatus.Successful:
                    return Ok();
                case UnlockVoiceSessionStatus.Error:
                default:
                    return ServiceUnavailable();
            }
        }

        /// <summary>
        /// Enables video in the call (failing it over to another host)        
        /// </summary>
        /// <param name="callID"></param>
        /// <returns></returns>
        [Route("{callID}/enable-video")]
        [HttpPost]
        [ResponseType(typeof(void))]
        public IHttpActionResult EnableVideo(string callID)
        {
            Guid callGuid;
            if (!Guid.TryParse(callID, out callGuid))
            {
                return BadRequest("The supplied call is invalid: " + callGuid);
            }

            var callDetails = GetWrappedCallDetails(callID);
            if (callDetails == null)
            {
                return NotFound();
            }

            if (callDetails.CallDetails.Mode == CallMode.Video)
            {
                return BadRequest("Call mode is already video.");
            }

            // Ensure the requesting user has access
            if (!callDetails.Conversation.CanCall(Token.UserID))
            {
                return Forbidden();
            }

            var unlockResp = ChangeVoiceSessionMode(callID, VoiceInstanceMode.Video);

            switch (unlockResp.Status)
            {
                case ChangeVoiceSessionModeStatus.NotFound:
                    return NotFound();
                case ChangeVoiceSessionModeStatus.Forbidden:
                    return Forbidden();
                case ChangeVoiceSessionModeStatus.Successful:
                    return Ok();
                case ChangeVoiceSessionModeStatus.Error:
                default:
                    return ServiceUnavailable();
            }
        }

        private void RespondToCall(WrappedCallDetails wrappedCall, CallResponseReason reason)
        {
            var currentUser = GetCurrentUserAndRegion();

            var callRespondedNotification = new CallRespondedNotification
            {
                CallID = wrappedCall.CallDetails.CallID,
                Reason = reason,
                Timestamp = DateTime.UtcNow,
                UserID = currentUser.User.UserID,
                Username = currentUser.User.GetTitleName()
            };

            if (wrappedCall.Conversation is Group)
            {
                GroupCallCoordinator.CallResponded((Group)wrappedCall.Conversation, wrappedCall.CallDetails, callRespondedNotification);
            }
            else if (wrappedCall.Conversation is Friendship)
            {
                CallRespondedResolver.Create(Token.UserID, callRespondedNotification);
            }
        }

        private WrappedCallDetails GetWrappedCallDetails(string callID)
        {
            Guid callGuid;
            if (!Guid.TryParse(callID, out callGuid))
            {
                return null;
            }

            // Get the call details
            var voiceInstance = CurseVoiceServiceClient.TryServiceCall("GetVoiceInstanceDetails",
                    c => c.GetVoiceInstanceDetails(FriendsServiceConfiguration.Instance.CentralServiceApiKey, callID));


            if (voiceInstance == null)
            {
                Logger.Warn("Unable to retrieve voice instance details for call.", new { CallID = callID });
                return null;
            }

            if (voiceInstance.Status == VoiceInstanceStatus.Shutdown)
            {
                return null;
            }

            if (voiceInstance.Type == VoiceInstanceType.AutoMatch || voiceInstance.Type == VoiceInstanceType.AdHoc || voiceInstance.Type == VoiceInstanceType.MultiFriend)
            {
                return new WrappedCallDetails(voiceInstance, new AdHocConversation(Token.UserID));
            }

            if(voiceInstance.ExternalID == null)
            {
                Logger.Warn("Unable to get conversation for call. The voice instance's external ID is null.", new { CallID = callID, voiceInstance.Type, voiceInstance.UserID, voiceInstance.Status });
                return null;
            }

            var conversation = ConversationManager.GetConversationContainer(Token.UserID, voiceInstance.ExternalID, true);

            if (conversation == null)
            {
                Logger.Warn("Unable to get conversation for call.", new { CallID = callID, voiceInstance.Type, voiceInstance.UserID, voiceInstance.Status, ExternalID = voiceInstance.ExternalID ?? "NULL" });
                return null;
            }

            return new WrappedCallDetails(voiceInstance, conversation);
        }

        private class WrappedCallDetails
        {
            public CallDetails CallDetails { get; private set; }
            public IConversationContainer Conversation { get; private set; }

            public WrappedCallDetails(VoiceInstanceDetailsResponse response, IConversationContainer conversation)
            {
                CallDetails = FromVoiceInstanceDetails(response, conversation.ConversationID);
                Conversation = conversation;
            }
        }

        private static CallDetails FromVoiceInstanceDetails(VoiceInstanceDetailsResponse response, string conversationID, bool forceJoin = false)
        {
            conversationID = conversationID ?? string.Empty;
            
            return new CallDetails
            {
                HostName = response.HostName,
                InviteUrl = response.InviteUrl,
                CallID = response.CallID,
                Type = FromVoiceInstanceType(response.Type),
                Mode = FromVoiceInstanceMode(response.Mode),
                ConversationID = conversationID,
                Timestamp = DateTime.UtcNow,
                HostID = response.HostID,
                RegionName = response.RegionName,
                GameID = response.GameID,
                CreatorName = response.CreatorName,
                AutoMatchKey = response.AutoMatchKey,
                IpAddress = response.IpAddress,
                InviteCode = response.InviteCode,
                CreatorID = response.UserID,
                ForceJoin = forceJoin
            };
        }

        private static CallType FromVoiceInstanceType(VoiceInstanceType type)
        {
            switch (type)
            {
                case VoiceInstanceType.AdHoc:
                    return CallType.AdHoc;
                case VoiceInstanceType.AutoMatch:
                    return CallType.AutoMatch;
                case VoiceInstanceType.Friend:
                    return CallType.Friend;
                case VoiceInstanceType.Group:
                    return CallType.Group;
                case VoiceInstanceType.MultiFriend:
                    return CallType.MultiFriend;
                default:
                    Logger.Warn("Unknown voice instance type, assuming ad hoc.", new { VoiceInstanceType = (int)type });
                    return CallType.AdHoc;
            }
        }

        private static CallMode FromVoiceInstanceMode(VoiceInstanceMode mode)
        {
            switch (mode)
            {
                case VoiceInstanceMode.Audio:
                    return CallMode.Audio;
                case VoiceInstanceMode.Video:
                    return CallMode.Video;                
                default:
                    Logger.Warn("Unknown voice instance mode, assuming audio.", new { mode });
                    return CallMode.Audio;
            }
        }

        private UnlockVoiceSessionResponse UnlockVoiceSession(string sessionGuid)
        {
            try
            {

                return CurseVoiceServiceClient.TryServiceCall("UnlockVoiceSession",
                        c => c.UnlockVoiceSession(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                        GetCurrentIpAddress(),
                        sessionGuid,
                        Token.UserID));

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to unlock voice session.");
                return new UnlockVoiceSessionResponse { Status = UnlockVoiceSessionStatus.Error };
            }
        }

        private ChangeVoiceSessionModeResponse ChangeVoiceSessionMode(string sessionGuid, VoiceInstanceMode mode)
        {
            try
            {

                return CurseVoiceServiceClient.TryServiceCall("ChangeVoiceSessionMode",
                        c => c.ChangeVoiceSessionMode(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                        GetCurrentIpAddress(),
                        sessionGuid,
                        Token.UserID, 
                        mode), 100, 1);

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to change voice session mode.");
                return new ChangeVoiceSessionModeResponse { Status = ChangeVoiceSessionModeStatus.Error };
            }
        }

        [Route("{conversationID}/moderation/muted-users")]
        [ResponseType(typeof(void))]
        [HttpPost]
        public IHttpActionResult MuteUser(string conversationID, [FromBody] int userID)
        {
            return DoMuteUser(conversationID, userID, true);
        }

        [Route("{conversationID}/moderation/muted-users/{userID}")]
        [ResponseType(typeof(void))]
        [HttpDelete]
        public IHttpActionResult UnmuteUser(string conversationID, int userID)
        {
            return DoMuteUser(conversationID, userID, false);
        }

        private IHttpActionResult DoMuteUser(string conversationID, int userID, bool mute)
        {
            var conversationContainer = ConversationManager.GetConversationContainer(Token.UserID, conversationID);
            if (conversationContainer.ConversationType != ConversationType.Group)
            {
                return Forbidden();
            }

            var group = conversationContainer as Group;
            if (group == null)
            {
                return NotFound();
            }

            var rootGroup = group.RootGroup;
            var member = rootGroup.GetMember(userID);
            if (member == null)
            {
                return NotFound();
            }

            var details = GetGroupVoiceInstanceDetails(group);
            if (details == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.VoiceMuteUser, Token.UserID, m => rootGroup.CanModerateUser(Token.UserID, member.UserID));

            var result = CurseVoiceServiceClient.TryServiceCall("KickUser",
                svc => svc.MuteUserV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, details.CallID, Token.UserID, member.UserID, mute));

            if (result.Status != BasicVoiceServiceStatus.Successful)
            {
                return Conflict();
            }

            member.IsVoiceMuted = mute;
            member.Update(u => u.IsVoiceMuted);

            GroupChangeCoordinator.UpdateUsers(group, Token.UserID, new HashSet<int> { userID });

            return StatusCode(HttpStatusCode.NoContent);
        }


        [Route("{conversationID}/moderation/deafened-users")]
        [ResponseType(typeof(void))]
        [HttpPost]
        public IHttpActionResult DeafenUser(string conversationID, [FromBody] DeafenUserRequest request)
        {
            var res = DoDeafenUser(conversationID, request.UserID, true);
            if (res == HttpStatusCode.NoContent && request.Mute)
            {
                DoMuteUser(conversationID, request.UserID, true);
            }
            return StatusCode(res);
        }

        [Route("{conversationID}/moderation/deafened-users/{userID}")]
        [ResponseType(typeof (void))]
        [HttpDelete]
        public IHttpActionResult UndeafenUser(string conversationID, int userID, [FromUri] bool unmute=false)
        {
            var res = DoDeafenUser(conversationID, userID, false);
            if (res == HttpStatusCode.NoContent && unmute)
            {
                DoMuteUser(conversationID, userID, false);
            }
            return StatusCode(res);
        }

        private HttpStatusCode DoDeafenUser(string conversationID, int userID, bool deafen)
        {
            var conversationContainer = ConversationManager.GetConversationContainer(Token.UserID, conversationID);
            if (conversationContainer.ConversationType != ConversationType.Group)
            {
                return HttpStatusCode.Forbidden;
            }

            var group = conversationContainer as Group;
            if (group == null)
            {
                return HttpStatusCode.NotFound;
            }

            var rootGroup = group.RootGroup;

            if (rootGroup == null)
            {
                return HttpStatusCode.NotFound;
            }

            var member = rootGroup.GetMember(userID);
            if (member == null)
            {
                return HttpStatusCode.NotFound;
            }

            var details = GetGroupVoiceInstanceDetails(group);
            if (details == null)
            {
                return HttpStatusCode.NotFound;
            }

            group.CheckPermission(GroupPermissions.VoiceDeafenUser, Token.UserID, m => rootGroup.CanModerateUser(Token.UserID, userID));

            var result = CurseVoiceServiceClient.TryServiceCall("DeafenUser",
                svc => svc.DeafenUserV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, details.CallID, Token.UserID, userID, deafen));

            if (result.Status != BasicVoiceServiceStatus.Successful)
            {
                return HttpStatusCode.Conflict;
            }

            member.IsVoiceDeafened = deafen;
            member.Update(u => u.IsVoiceDeafened);

            GroupChangeCoordinator.UpdateUsers(group, Token.UserID, new HashSet<int> {userID});

            return HttpStatusCode.NoContent;
        }

        [Route("{conversationID}/moderation/{userID}")]
        [ResponseType(typeof (void))]
        [HttpDelete]
        public IHttpActionResult KickUser(string conversationID, int userID)
        {
            var conversationContainer = ConversationManager.GetConversationContainer(Token.UserID, conversationID);
            if (conversationContainer.ConversationType != ConversationType.Group)
            {
                return Forbidden();
            }

            var group = conversationContainer as Group;
            if (group == null)
            {
                return NotFound();
            }

            var details = GetGroupVoiceInstanceDetails(group);
            if (details == null)
            {
                return NotFound();
            }

            var rootGroup = group.RootGroup;

            if (rootGroup == null)
            {
                return NotFound();
            }

            group.CheckPermission(GroupPermissions.VoiceKickUser, Token.UserID, member => rootGroup.CanModerateUser(Token.UserID, userID));

            var result = CurseVoiceServiceClient.TryServiceCall("KickUser",
                svc => svc.KickUserV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, details.CallID, Token.UserID, userID));
            if (result.Status != BasicVoiceServiceStatus.Successful)
            {
                return Conflict();
            }

            return StatusCode(HttpStatusCode.NoContent);
        }

        private VoiceInstanceDetailsResponse GetGroupVoiceInstanceDetails(Group group)
        {
            var inviteCode = group.VoiceSessionCode;
            if (string.IsNullOrEmpty(inviteCode))
            {
                return null;
            }

            var details = CurseVoiceServiceClient.TryServiceCall("FindSession", svc => svc.FindVoiceSessionV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, inviteCode));
            return details;
        }

        [HttpPost]
        [Route("{groupID}/moderation/{userID}/move")]
        [ResponseType(typeof(void))]
        public IHttpActionResult MoveUser(Guid groupID, int userID, [FromBody] MoveUserRequest request)
        {
            var group = Group.GetByID(groupID);
            if (group == null)
            {
                return NotFound();
            }

            var targetGroup = Group.GetByID(request.TargetGroupID);
            if (targetGroup == null)
            {
                return NotFound();
            }

            if (group.RootGroupID != targetGroup.RootGroupID)
            {
                return Forbidden();
            }

            var rootGroup = group.RootGroup;

            // Ensure the user has permission to move voice users in both channels
            group.CheckPermission(GroupPermissions.VoiceMoveUser, Token.UserID, member => rootGroup.CanModerateUser(member.UserID, userID));
            targetGroup.CheckPermission(GroupPermissions.VoiceMoveUser, Token.UserID, member => rootGroup.CanModerateUser(member.UserID, userID));

            // Make sure the user is in the source call
            var memberList = GroupCallMemberList.GetLocal(groupID);
            if (memberList == null || !memberList.Members.Contains(userID))
            {
                return Forbidden();
            }
            
            // Ensure the target voice session exists
            var resp = CurseVoiceServiceClient.TryServiceCall("GroupVoiceSession", c => c.GroupVoiceSessionV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                GetCurrentIpAddress(),
                ClientEndpoint.GetMostRecentClientVersion(Token.UserID),
                Token.UserID,
                targetGroup.Title,
                targetGroup.GroupID,
                false,
                targetGroup.RootGroup.VoiceRegionID));

            if (resp.Status != CreateVoiceSessionStatus.Successful)
            {
                return Conflict();
            }

            
            var callDetails = FromVoiceInstanceDetails(resp.VoiceInstance, targetGroup.ConversationID, true);
            var currentUser = GetCurrentUserAndRegion();

            CallResolver.Create(Token.UserID, currentUser.User.GetTitleName(), userID, callDetails);
            return StatusCode(HttpStatusCode.NoContent);
        }

        /// <summary>
        /// Reports game session metrics.
        /// </summary>
        /// <param name="request">Game Session Metrics information</param>
        [HttpPost]
        [Route("games/metrics")]
        [ResponseType(typeof(void))]
        public IHttpActionResult ReportGameSessions(ReportGameSessionsRequest request)
        {
            var resp3 = CurseVoiceServiceClient.TryServiceCall("Something2", svc => svc.ReportGameSessionsV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, 
                Token.UserID, request.GameSessionMetrics.Select(GameSessionMetricsContract.ToServiceContract).ToArray()));
            if (!resp3.Successful)
            {
                return ErrorResponse(HttpStatusCode.InternalServerError, "service_call_error");
            }

            return Ok();
        }

        /// <summary>
        /// Sets the game associated with the specified voice session.
        /// </summary>
        /// <param name="sessionGuid">ID of the voice session</param>
        /// <param name="gameID">ID of the game</param>
        [HttpPost]
        [Route("sessions/{sessionGuid}/games/{gameID}")]
        [ResponseType(typeof(void))]
        public IHttpActionResult SetSessionGameState(string sessionGuid, int gameID)
        {
            var resp = CurseVoiceServiceClient.TryServiceCall("Something3", svc => svc.ReportGameStateV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, 
                sessionGuid, gameID));

            switch (resp.Status)
            {
                case BasicVoiceServiceStatus.Successful:
                    return Ok();
                case BasicVoiceServiceStatus.Forbidden:
                    return ErrorResponse(HttpStatusCode.Forbidden, "user_not_allowed");
                case BasicVoiceServiceStatus.NotFound:
                    return ErrorResponse(HttpStatusCode.NotFound, "session_not_found");
                case BasicVoiceServiceStatus.Failed:
                case BasicVoiceServiceStatus.Error:
                default:
                    return ErrorResponse(HttpStatusCode.InternalServerError, "service_call_error");

            }
        }
    }
}