﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Text.RegularExpressions;

namespace Curse.Friends.ChatHistoryTests
{
    /// <summary>
    /// Examines the body of a message, and determines the nature of the content
    /// </summary>
    public static class ConversationParser
    {

        
        private static readonly Regex HostedImageEmbedRegex = new Regex(@"\bhttp(s)?://images.*?\.curseapp\.net/.*?\.(png|jpg|jpeg|gif)\b", RegexOptions.Compiled | RegexOptions.IgnoreCase);
        private static readonly Regex HostedFileRegex = new Regex(@"\bhttp(s)?://files.*?\.curseapp\.net/.*?\b", RegexOptions.Compiled);
        private static readonly Regex ImgurRegex = new Regex(@"\bhttp(s)?://((m\.)|((www)\.)|((i)\.))?imgur\.com/(a/)?[a-zA-Z0-9&]+((\.jpg)|(\.gif)|(\.gifv)|(\.png))?\b", RegexOptions.Compiled | RegexOptions.IgnoreCase);
        private static readonly Regex YouTubeRegex = new Regex(@"(^|\s)http(s)?\://(www\.)?(?:youtube\.com\/\S*(?:(?:\/e(?:mbed))?\/|watch\?(?:\S*?&?v\=))|youtu\.be\/)([a-zA-Z0-9_-]{6,11})($|\s)", RegexOptions.Compiled | RegexOptions.IgnoreCase);
        private static readonly Regex UrlRegex = new Regex(@"\b(www\.|(https|http|curse)+\:\/\/)[\w\-_]+(?:\.[\w\-_]+)+(?:[\w\-\.,@?^=%&;:/~\+#'!()]*[\w\-\@?^=%&;/~\+#'()])\b", RegexOptions.Compiled | RegexOptions.IgnoreCase);

        public static bool IsImageEmbed(string body)
        {
            return HostedImageEmbedRegex.IsMatch(body) || ImgurRegex.IsMatch(body);
        }

        public static bool IsVideoEmbed(string body)
        {
            return YouTubeRegex.IsMatch(body);
        }

        public static bool IsFileEmbed(string body)
        {
            return HostedFileRegex.IsMatch(body);
        }

        public static bool HasHyperlink(string body)
        {
            return UrlRegex.IsMatch(body);
        }

        public static HashSet<ContentTag> GetContentTags(string body)
        {
            var tags = new HashSet<ContentTag>();

            if(IsImageEmbed(body))
            {
                tags.Add(ContentTag.Image);
            }

            if (IsVideoEmbed(body))
            {
                tags.Add(ContentTag.Video);
            }

            if (IsFileEmbed(body))
            {
                tags.Add(ContentTag.File);
            }

            if (HasHyperlink(body))
            {
                tags.Add(ContentTag.Hyperlink);
            }

            return tags;
        }

        private static readonly Regex MentionsRegex = new Regex("(?<mentionStart>\\s@|^@)(?<userID>\\d+):(?<username>\\w+)(?:\\b|$)", RegexOptions.Compiled);

        public static int[] GetMentions(string body)
        {
            var matches = MentionsRegex.Matches(body);
            var userIDs = new HashSet<int>();
            foreach (Match match in matches)
            {                
                int userID;
                if (!int.TryParse(match.Groups["userID"].Value, out userID))
                {
                    continue;
                }

                userIDs.Add(userID);
            }

            return userIDs.ToArray();
        }

        
    }
}
