﻿using Elasticsearch.Net.ConnectionPool;
using Nest;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.Extensions;
using System.Diagnostics;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using System.Globalization;
using Aerospike.Client;
using System.Threading;
using System.Collections.Concurrent;
using Curse.CloudSearch;
using Elasticsearch.Net;
using Curse.Friends.Enums;
using Curse.Friends.Data.Messaging;

namespace Curse.Friends.ChatHistoryTests
{
    class Program
    {        
        private static string TemplateName = "conversation-history-template";
        private static string TestIndexName = "conversations-2016m01";
        private const string IndexTypeName = "conversations";
        private const int zusernameID = 942516;
        private const string DefaultGroupID = "c7a0c394-921a-4288-a5c1-386b3b455874";

        static Tuple<ConfigurationMode, ConfigurationRegion> PromptRegion()
        {
           
            Console.Write("Type the configuration name you would like to use: ");
            Console.WriteLine();
            ConfigurationMode mode;
            var configs = Enum.GetNames(typeof(ConfigurationMode));
            foreach (var c in configs)
            {
                Console.WriteLine("-" + c);
            }
            
            var enteredConfig = Console.ReadLine();

            if (!Enum.TryParse(enteredConfig, true, out mode))
            {
                Console.WriteLine("Invalid mode: {0}", enteredConfig);
                Console.ReadKey(true);
                return null;
            }

            Console.Write("Type the region name you would like to use: ");
            Console.WriteLine();

            foreach (var r in ConfigurationRegion.AllRegions)
            {
                Console.WriteLine("-" + r.Key);
            }

            var regionVal = Console.ReadLine();

            var region = ConfigurationRegion.FindRegionByKey(regionVal);

            if (region == null)
            {
                Console.WriteLine("Unknown region: " + regionVal);
                return null;
            }

            return new Tuple<ConfigurationMode, ConfigurationRegion>(mode, region);
        }

        static void Main(string[] args)
        {

            Console.CancelKeyPress += Console_CancelKeyPress;

            Tuple<ConfigurationMode, ConfigurationRegion> config;

            while (true)
            {
                config = PromptRegion();
                if (config != null)
                {
                    break;
                }
            }

            Console.WriteLine("Using configuration mode: " + config.Item1 + ", region: " + config.Item2.Key);
            Console.WriteLine("Press the Y key to confirm...");
            if (Console.ReadKey(true).Key != ConsoleKey.Y)
            {
                return;
            }

            Console.Write("Initializing configuration...");
            StorageConfiguration.Initialize("DataMigration", config.Item2.Key, config.Item1, ConfigurationServices.Search | ConfigurationServices.Database | ConfigurationServices.LocalOnly);
            
            Console.WriteLine("Inialized! Region: " + StorageConfiguration.CurrentRegion.Key + ", Mode: " + StorageConfiguration.CurrentMode);
            Console.WriteLine("Press any key to continue...");
            Console.ReadKey(true);
            try
            {
                while (true)
                {
                    Console.Clear();
                    Console.WriteLine("Welcome to the Elasticsearch test app. This app is designed to quickly prototype out framework-level improvements to our Aerospike wrapper.");
                    Console.WriteLine("0: Import all conversations");
                    Console.WriteLine("1. Test index creation and population.");
                    Console.WriteLine("2. Test getting a document via its id.");
                    Console.WriteLine("3. Test querying by conversation and date range.");
                    Console.WriteLine("4. Test querying by conversation and search text.");
                    Console.WriteLine("5. Import user data");
                    Console.WriteLine("6. Delete all test indexes.");
                    Console.WriteLine("7. Query by conversation and mentioned user.");
                    Console.WriteLine("8. Query by conversation and content tag.");
                    Console.WriteLine("9. Import group data");

                    var selection = Console.ReadKey(true).Key;
                    Console.Clear();

                    switch (selection)
                    {
                        case ConsoleKey.D0:
                            ImportAllConversations();
                            return;
                        case ConsoleKey.Escape:
                            return;
                        case ConsoleKey.D1:
                            TestIndexCreation();
                            break;
                        case ConsoleKey.D2:
                            {
                                var docID = Console.ReadLine();
                                TestGetByID(docID);
                            }
                            break;
                        case ConsoleKey.D3:
                            Console.Write("Enter a conversation ID: ");
                            {
                                var conversationID = Console.ReadLine();
                                conversationID = !string.IsNullOrEmpty(conversationID) ? conversationID : DefaultGroupID;
                                TestQueryingByConversationAndDateRange(conversationID, DateTime.UtcNow.AddDays(-60), DateTime.UtcNow);
                            }
                            break;
                        case ConsoleKey.D4:
                            {
                                Console.Write("Enter a conversation ID: ");
                                var conversationID = Console.ReadLine();
                                Console.Write("Enter some search text: ");
                                TestQueryingByConversationAndText(conversationID, Console.ReadLine());
                            }
                            break;
                        case ConsoleKey.D5:
                            {
                                Console.Write("Enter a user ID: ");
                                var userID = int.Parse(Console.ReadLine());
                                ImportUserData(userID);
                            }
                            break;
                        case ConsoleKey.D6:
                            DeleteIndexes();
                            break;
                        case ConsoleKey.D7:
                            {
                                Console.Write("Enter a conversation ID: ");
                                var conversationID = Console.ReadLine();
                                Console.Write("Enter a user ID: ");
                                var userID = int.Parse(Console.ReadLine());
                                TestQueryingByConversationAndMentionedUser(conversationID, userID);
                            }
                            break;
                        case ConsoleKey.D8:
                            {
                                Console.Write("Enter a conversation ID: ");
                                var conversationID = Console.ReadLine();
                                Console.Write("Enter a user ID: ");
                                var contentTag = int.Parse(Console.ReadLine());
                                TestQueryingByConversationAndContentType(conversationID, (ContentTag)contentTag);
                            }
                            break;
                        case ConsoleKey.D9:
                            {
                                Console.Write("Enter a group ID: ");
                                var groupID = Console.ReadLine();
                                if (!string.IsNullOrEmpty(groupID))
                                {
                                    ImportGroupData(groupID);
                                }
                                else
                                {
                                    ImportGroupData();
                                }
                            }
                            break;
                        case ConsoleKey.F1:
                            {
                                Console.Write("Enter a conversation ID: ");
                                var groupID = Console.ReadLine();
                                TestQueryingByConversationSortedByLikeCount(groupID);
                            }
                            break;

                        case ConsoleKey.F2:
                            {
                                Console.Write("Enter a conversation ID: ");
                                var groupID = Console.ReadLine();
                                Console.Write("Enter a user ID: ");
                                var userID = int.Parse(Console.ReadLine());
                                TestQueryingByConversationAndLikedByUser(groupID, userID);
                            }
                            break;

                        case ConsoleKey.F3:
                            {
                                Console.Write("Enter a user ID: ");
                                var userID = int.Parse(Console.ReadLine());
                                TestGetAllUnread(userID);
                            }
                            break;

                        case ConsoleKey.F4:
                            {
                                TestRandomQuery();
                            }
                            break;
                        default:
                            Console.WriteLine("Invalidation selection! Press any key to continue...");
                            Console.ReadKey(true);
                            continue;

                    }
                    //TestGenericDictionaries();

                    //TestLargeListConfig();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("General error: " + ex.Message);
                Console.ReadLine();
            }

        }

        static void DeleteIndexes()
        {
            var setupClient = ConversationManager.GetClient();
            for (var year = 2014; year <= DateTime.UtcNow.Year; year++)
            {
                for (var month = 1; month <= 12; month++)
                {
                    setupClient.DeleteIndex(TimeSeriesIndexing.GetIndexName(IndexTypeName, year, month));
                }
            }

            Console.WriteLine("Done!");
            Console.ReadLine();

        }

        //// This presumes that weeks start with Monday.
        //// Week 1 is the 1st week of the year with a Thursday in it.
        //public static int GetIso8601WeekOfYear(DateTime time)
        //{
        //    // Seriously cheat.  If its Monday, Tuesday or Wednesday, then it'll 
        //    // be the same week# as whatever Thursday, Friday or Saturday are,
        //    // and we always get those right
        //    DayOfWeek day = CultureInfo.InvariantCulture.Calendar.GetDayOfWeek(time);
        //    if (day >= DayOfWeek.Monday && day <= DayOfWeek.Wednesday)
        //    {
        //        time = time.AddDays(3);
        //    }

        //    // Return the week of our adjusted day
        //    return CultureInfo.InvariantCulture.Calendar.GetWeekOfYear(time, CalendarWeekRule.FirstFourDayWeek, DayOfWeek.Monday);
        //}


        static void ImportAllConversations()
        {           
            KeepImporting = true;
            try
            {
                var convoCounter = 0;
                var messageCounter = 0;
                var dict = new ConcurrentDictionary<string, ConcurrentQueue<ConversationMessage>>();

                Conversation.BatchOperateLocal(100, (conversations) =>
                {

                    foreach (var c in conversations)
                    {
                        if (!KeepImporting)
                        {
                            throw new AerospikeException.ScanTerminated();
                        }

                        ++convoCounter;
                        
                        IEnumerable<ConversationEntry> messages;
                        try
                        {
                            messages = c.History.GetValues();
                        }
                        catch (Exception ex)
                        {
                            Console.WriteLine("Failed to get messages for conversation: " + c.ID);
                            continue;
                        }

                        var localCounter = 0;
                        var localTotalCount = messages.Count();
                        var writeLock = new object();

                        Parallel.ForEach(messages, m =>
                        {
                            var newMessageCounter = Interlocked.Increment(ref messageCounter);
                            var newLocalCounter = Interlocked.Increment(ref localCounter);
                            if (newLocalCounter % 100 == 0)
                            {
                                Console.Title = "Processing conversation " + convoCounter + "  (Message " + newLocalCounter + " of " + localTotalCount + ", Total Messages: " + newMessageCounter + ")";
                            }

                            ConversationMessage model;

                            try
                            {
                                model = GetModelFromEntry(c.ID, c.Type == ConversationType.Group ? c.ID : null, m, c.Type);
                            }
                            catch (Exception ex)
                            {
                                Console.WriteLine("Failed to create model from entity: " + ex.Message);
                                return;
                            }

                            var indexName = TimeSeriesIndexing.GetIndexName(IndexTypeName, m.Timestamp.Year, m.Timestamp.Month);
                            var queue = dict.GetOrAdd(indexName, new ConcurrentQueue<ConversationMessage>());
                            queue.Enqueue(model);

                            lock (writeLock)
                            {
                                if (queue.Count <= 5000)
                                {
                                    return;
                                }
                                ProcessQueue(queue, indexName);
                            }

                        });
                    }
                });

                Console.WriteLine("Clearing final queues...");
                foreach (var kvp in dict)
                {
                    ProcessQueue(kvp.Value, kvp.Key);
                }
                Console.WriteLine("Import complete! Press enter to continue...");
                Console.ReadLine();
            }
            catch (Exception ex)
            {                
                Console.WriteLine("Failed to import! Error: " + ex.Message);
            }
        }

        private static void ProcessQueue(ConcurrentQueue<ConversationMessage> queue, string indexName)
        {
            ConversationMessage conversationMessage;
            var operations = new List<IBulkOperation>();

            while (queue.TryDequeue(out conversationMessage))
            {
                operations.Add(new BulkIndexOperation<ConversationMessage>(conversationMessage) { Id = conversationMessage.ID, Index = indexName });
            }

            var request = new BulkRequest()
            {
                Refresh = true,
                Consistency = Consistency.One,
                Operations = operations
            };

            var client = ConversationManager.GetClient();
            var response = client.Bulk(request);
            if (!response.ConnectionStatus.Success)
            {
                Console.WriteLine("Failed to bulk index!");
            }
        }

        private static bool KeepImporting = false;

        static void Console_CancelKeyPress(object sender, ConsoleCancelEventArgs e)
        {
            e.Cancel = true;
            KeepImporting = false;
        }

        static IEnumerable<DateTime> MonthsBetween(DateTime d0, DateTime d1)
        {
            return Enumerable.Range(0, (d1.Year - d0.Year) * 12 + (d1.Month - d0.Month + 1))
                             .Select(m => new DateTime(d0.Year, d0.Month, 1).AddMonths(m));
        }

        static void TestGetAllUnread(int userID)
        {            
            try
            {
                
                var friendships = Friendship.GetAllLocal(p => p.UserID, userID)
                                    .Where(p => p.Status == FriendshipStatus.Confirmed && p.DateMessaged > p.DateRead 
                                        //&& p.DateRead.Subtract(p.DateMessaged).TotalMilliseconds >= 1
                                    )
                                    .Take(30)
                                    .ToArray();

                Console.WriteLine("Searching...");

                var client = ConversationManager.GetClient();

                var search = client.MultiSearch(ms => {

                    var baseSearch = ms;

                    foreach(var friend in friendships)
                    {
                        var startDate = friend.DateConfirmed > friend.DateRead ? friend.DateConfirmed : friend.DateRead;
                        startDate = DateTime.UtcNow.AddDays(-60);
                        var endDate = DateTime.UtcNow;

                        var monthsBetween = MonthsBetween(startDate, endDate);

                        var indexNames = monthsBetween.Select(p => TimeSeriesIndexing.GetIndexName(IndexTypeName, p.Year, p.Month)).ToArray();

                        baseSearch = ms.Search<ConversationMessage>(friend.ConversationID, s => s.Filter(f => 
                                f.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, friend.ConversationID)))
                                && f.Bool(b => b.Must(qd => qd.Range(r => r.OnField(t => t.Timestamp).GreaterOrEquals(startDate.ToEpochMilliseconds()).LowerOrEquals(endDate.ToEpochMilliseconds()))))
                                )
                                .Indices(indexNames)
                                .Size(100)
                                .Routing(friend.ConversationID)                              
                                .SortDescending(p => p.Timestamp)
                            );                            
                    }

                    return baseSearch;

                });

                var responses = search.GetResponses<ConversationMessage>();
                var totalTime = responses.Max(r => r.ElapsedMilliseconds);
                
                var totalDocs = responses.Sum(r => r.Documents.Count());
                
                Console.WriteLine("Request Url: " + search.RequestInformation.RequestUrl);
                Console.WriteLine("Request Parms:" + Encoding.Default.GetString(search.RequestInformation.Request));
                Console.WriteLine("Returned " + totalDocs + " documents in " + totalTime + "ms");
                Console.ReadLine();

                foreach (var resp in responses.Where(p => p.Total > 0))
                {
                    foreach (var doc in resp.Documents.Take(10))
                    {
                        Console.WriteLine(doc.SenderName + " (Tags: " + string.Join(", ", doc.ContentTags) + ") at " + doc.Timestamp.FromEpochMilliconds().ToLocalTime().ToString() + ": " + doc.Body);
                    }
                }
                Console.ReadLine();
               
            }
            finally
            {
                
            }



        }

        static void ImportUserData(int userID = zusernameID)
        {            
            var friendships = Friendship.GetAllLocal(p => p.UserID, userID);
            var dict = new Dictionary<string, List<ConversationMessage>>();

            foreach (var friendship in friendships)
            {
                Console.WriteLine("Process conversation for " + friendship.OtherUsername + "...");
                var conversationID = Math.Min(friendship.UserID, friendship.OtherUserID)
                    + ":" + Math.Max(friendship.UserID, friendship.OtherUserID);

                var conversation = Conversation.GetByTypeAndID(ConversationType.Friendship, conversationID);
                if (conversation == null)
                {
                    continue;
                }

                var allEntries = conversation.History.GetValues();
                foreach (var entry in allEntries)
                {
                    var model = GetModelFromEntry(conversation.ID, null, entry, ConversationType.Friendship);
                    var indexName = TimeSeriesIndexing.GetIndexName(IndexTypeName, entry.Timestamp.Year, entry.Timestamp.Month);
                    if (!dict.ContainsKey(indexName))
                    {
                        dict.Add(indexName, new List<ConversationMessage>());
                    }

                    dict[indexName].Add(model);
                }
            }
            

            var operations = new List<IBulkOperation>();
            foreach (var kvp in dict)
            {
                foreach (var val in kvp.Value)
                {
                    operations.Add(new BulkIndexOperation<ConversationMessage>(val) { Id = val.ID, Index = kvp.Key });
                }
            }

            var request = new BulkRequest()
            {
                Refresh = true,
                Consistency = Consistency.One,
                Operations = operations
            };

            var client = ConversationManager.GetClient();
            var response = client.Bulk(request);
            if (!response.ConnectionStatus.Success)
            {
                Console.WriteLine("Failed to bulk index!");
            }


        }

        static void ImportGroupData(string groupID = DefaultGroupID)
        {            
            try
            {
                var conversation = Conversation.GetByTypeAndID(ConversationType.Group, groupID);
                if (conversation == null)
                {
                    return;
                }

                Console.WriteLine("Getting conversation history...");
                var allEntries = conversation.History.GetValues();
                var total = allEntries.Count();
                int counter = 0;
                foreach (var entry in allEntries)
                {
                    if (++counter % 10 == 0)
                    {
                        Console.Title = "Importing " + counter + " of " + total;
                    }
                    ImportEntry(groupID, entry, ConversationType.Group);
                }
            }
            finally
            {                
            }
        }

        static void ImportEntry(string conversationID, ConversationEntry entry, ConversationType type)
        {
            var indexName = TimeSeriesIndexing.GetIndexName(IndexTypeName, entry.Timestamp.Year, entry.Timestamp.Month);
            var client = ConversationManager.GetClient();
            var testMessage = GetModelFromEntry(conversationID, conversationID, entry, type);
            
            var resp = client.Index<ConversationMessage>(testMessage, idx => idx.Index(indexName));

            if (!resp.ConnectionStatus.Success)
            {
                Console.WriteLine("Failed to index document!");
            }

        }

        static ConversationMessage GetModelFromEntry(string conversationID, string rootConversationID, ConversationEntry entry, ConversationType type)
        {

            return new ConversationMessage
            {
                ID = entry.ID.ToString(),
                ConversationID = conversationID,
                RootConversationID = rootConversationID,
                Body = entry.Body,
                SenderID = entry.SenderID,
                RecipientID = entry.RecipientID,
                SenderName = entry.SenderName,
                Timestamp = entry.Timestamp.ToEpochMilliseconds(),
                ContentTags = ConversationParser.GetContentTags(entry.Body).Cast<int>().ToArray(),
                LikeUserIDs = new int[0],
                LikeUsernames = new string[0],
                LikeCount = 0,
                Mentions = ConversationParser.GetMentions(entry.Body),
                Attachments = new ConversationAttachment[0],
                SenderPermissions = 0,
                SenderRoles = new int[0], 
                IsDeleted = false, 
                ConversationType = (int)type, 
            };

        }

        static void TestIndexCreation()
        {
            var setupClient = ConversationManager.GetClient();
            Console.WriteLine("Would you like to delete the index and start over? (Y|N)");
            if (Console.ReadKey(true).Key == ConsoleKey.Y)
            {
                setupClient.DeleteIndex(c => c.Index(TestIndexName));
                setupClient.DeleteTemplate(TemplateName);

                var putResponse = setupClient.PutTemplate("", t => t
                    .Name(TemplateName)
                    .Template("conversations-*")
                    .AddMapping<ConversationMessage>(s => s
                        .MapFromAttributes()
                        .SourceField(sf => sf.Enabled(true))
                        .AllField(af => af.Enabled(false))
                    )
                );

                Console.WriteLine("Template created!");
                Console.WriteLine("Request Url: " + putResponse.RequestInformation.RequestUrl);
                Console.WriteLine("Request Parms:" + Encoding.Default.GetString(putResponse.RequestInformation.Request));

            }

            Console.WriteLine("How many conversations would you like to create?");
            var conversationCount = int.Parse(Console.ReadLine());

            Console.WriteLine("How many documents would you like to index, per conversation?");
            var documentCount = int.Parse(Console.ReadLine());

            Console.WriteLine("How many weeks ago to start?");
            var weekCount = int.Parse(Console.ReadLine());

            var totalCount = conversationCount * documentCount;

            Console.WriteLine("Inserting docs...");
            var sw = Stopwatch.StartNew();
            var rnd = new Random();

            for (var i = 0; i < conversationCount; i++)
            {
                var conversationID = Guid.NewGuid().ToString();
                var startingTimestamp = DateTime.UtcNow.AddDays(-7 * weekCount);

                Parallel.For(0, documentCount, j =>
                {
                    var client = ConversationManager.GetClient();
                    var timestamp = startingTimestamp.AddSeconds(j + rnd.Next(1, 30));

                    if (timestamp > DateTime.UtcNow)
                    {
                        timestamp = DateTime.UtcNow;
                    }

                    var simulatedLike = new SimulatedLike(0, 1000);

                    var attachmentID = Guid.NewGuid().ToString();
                    var testAttachment = new ConversationAttachment
                    {
                        ID = attachmentID,
                        FileType = "image/jpg",
                        Height = 1000,
                        Width = 1000,
                        IsAnimated = false,                        
                        Name = "test.jpg",
                        Title = "My Awesome Image",
                        Size = 625539,
                        Url = "http://images.curseapp.net/" + attachmentID + "/test.jpg",
                        IsEmbed = true, AuthorID = 1, AuthorName = "adamar", Timestamp = DateTime.UtcNow.ToEpochMilliseconds()
                    };

                    var testMessage = new ConversationMessage
                    {
                        ID = Guid.NewGuid().ToString(),
                        ConversationID = conversationID,
                        Body = "Hello from the test app! The time now " + timestamp.ToLocalTime(),
                        SenderID = 1,
                        RecipientID = 4,
                        SenderName = "Adamar",
                        Timestamp = timestamp.ToEpochMilliseconds(),
                        LikeCount = simulatedLike.LikeCount,
                        LikeUserIDs = simulatedLike.LikeUserIDs,
                        LikeUsernames = simulatedLike.LikeUsernames.Take(10).ToArray(),
                        SenderPermissions = 10010101000,
                        SenderRoles = new int[] { 1, 2, 3 },
                        ContentTags = new int[0],
                        Mentions = new int[0],
                        Attachments = new[] { testAttachment }
                    };
                    var resp = client.Index<ConversationMessage>(testMessage, idx => idx.Index(TestIndexName));
                    if (!resp.ConnectionStatus.Success)
                    {
                        Console.WriteLine("Request Url: " + resp.RequestInformation.RequestUrl);
                        Console.WriteLine("Request Parms:" + Encoding.Default.GetString(resp.RequestInformation.Request));
                        Console.WriteLine("Failed to index document!");
                    }
                });
            }

            sw.Stop();
            var timePer = sw.ElapsedMilliseconds / (double)totalCount;
            Console.WriteLine("Inserted " + totalCount + " documents in " + sw.ElapsedMilliseconds + "ms (" + timePer + " ms/doc)");
            Console.ReadLine();
        }

        private class SimulatedLike
        {

            private static readonly Random Random = new Random();

            public int LikeCount
            {
                get;
                set;
            }

            public int[] LikeUserIDs
            {
                get;
                set;
            }

            public string[] LikeUsernames
            {
                get;
                set;
            }

            public SimulatedLike(int minCount, int maxCount)
            {
                LikeCount = Random.Next(minCount, maxCount);

                var userIDs = new List<int>();
                var usernames = new List<string>();

                for (var i = 0; i < LikeCount; i++)
                {
                    userIDs.Add(i + 1);
                    usernames.Add("user-" + i);
                }

                LikeUserIDs = userIDs.ToArray();
                LikeUsernames = usernames.ToArray();
            }
        }

        static void TestGetByID(string id)
        {
            var client = ConversationManager.GetClient();
            var sw = Stopwatch.StartNew();
            var resp = client.Get<ConversationMessage>(p => p.Index(TestIndexName).Id(id));
            sw.Stop();
            Console.WriteLine("Retrieved item " + resp.Source.ID + " in " + sw.ElapsedMilliseconds + "ms");
            Console.ReadLine();
        }

        static void TestQueryingByConversationAndDateRange(string conversationID, DateTime startDate, DateTime endDate)
        {

            var client = ConversationManager.GetClient();
            var pageSize = 100;
            var currentPage = 0;
            while (true)
            {
                var resp = client.Search<ConversationMessage>(s => s.Query(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, conversationID)))
                && q.Bool(b => b.Must(qd => qd.Range(r => r.OnField(f => f.Timestamp).GreaterOrEquals(startDate.ToEpochMilliseconds()).LowerOrEquals(endDate.ToEpochMilliseconds()))))
                )
                .Routing(conversationID)
                .Size(pageSize)
                .Skip(currentPage * pageSize)
                .SortDescending(p => p.Timestamp));
                Console.WriteLine("Request Url: " + resp.RequestInformation.RequestUrl);
                Console.WriteLine("Request Parms:" + Encoding.Default.GetString(resp.RequestInformation.Request));

                var docs = resp.Documents;
                Console.WriteLine("Returned " + docs.Count() + " documents in " + resp.ElapsedMilliseconds + "ms");
                if (docs.Count() < pageSize)
                {
                    break;
                }
                Console.WriteLine("Press enter to continue...");
                Console.ReadLine();
                ++currentPage;
            }

            Console.ReadLine();

        }

        static void TestQueryingByConversationAndMentionedUser(string conversationID, int mentionedUser)
        {
            var client = ConversationManager.GetClient();
            var resp = client.Search<ConversationMessage>(s => s.Query(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, conversationID)))
                && q.Bool(b => b.Must(qd => qd.Term(t => t.Mentions, mentionedUser))))
                .Routing(conversationID)
                .Sort(sort => sort.OnField(f => f.Timestamp).Descending())
                .Size(20));

            Console.WriteLine("Request Url: " + resp.RequestInformation.RequestUrl);
            Console.WriteLine("Request Parms:" + Encoding.Default.GetString(resp.RequestInformation.Request));
            Console.ReadKey(true);
            var docs = resp.Documents;
            foreach (var doc in docs)
            {
                Console.WriteLine(doc.ID + ": " + string.Join(", ", doc.Mentions));
            }
            Console.WriteLine("Returned " + docs.Count() + " documents in " + resp.ElapsedMilliseconds + "ms");
            Console.ReadLine();
        }

        static void TestQueryingByConversationAndText(string conversationID, string text)
        {
            conversationID = string.IsNullOrEmpty(conversationID) ? DefaultGroupID : conversationID;

            var client = ConversationManager.GetClient();

            var resp = client.Search<ConversationMessage>(s => s.Query(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, conversationID)))
                && q.Bool(b => b.Must(qd => qd.Match(t => t.OnField(f => f.Body).Query(text)))))
                .Routing(conversationID)                
                .Sort(sort => sort.OnField(f => f.Timestamp).Descending())
                .Size(50)
                .Highlight(h => h
                    .PreTags("<b>")
                    .PostTags("</b>")
                    .OnFields(f => f
                        .OnField(e => e.Body)
                        .PreTags("<em>")
                        .PostTags("</em>")
                    )
                ));

            Console.WriteLine("Request Url: " + resp.RequestInformation.RequestUrl);
            Console.WriteLine("Request Parms:" + Encoding.Default.GetString(resp.RequestInformation.Request));
            Console.ReadKey(true);
            var docs = resp.Documents;
            Console.WriteLine("Returned " + docs.Count() + " documents in " + resp.ElapsedMilliseconds + "ms");

            foreach (var h in resp.Highlights)
            {
                var doc = docs.First(p => p.ID == h.Key);
                var hit = resp.Hits.FirstOrDefault(p => p.Id == h.Key);
                if (doc == null)
                {
                    Console.WriteLine("Missing doc: " + h.Key);
                    continue;
                }
                foreach (var h2 in h.Value)
                {
                    Console.WriteLine("--------------------------------------");
                    if (doc.RecipientID == 0)
                    {
                        Console.WriteLine("From: " + (doc.SenderName ?? doc.SenderID.ToString()) + ", To: " + doc.ConversationID);
                    }
                    else
                    {
                        Console.WriteLine("From: " + (doc.SenderName ?? doc.SenderID.ToString()) + ", To: " + doc.RecipientID);
                    }

                    Console.WriteLine("Date: " + doc.Timestamp.FromEpochMilliconds().ToLocalTime().ToString());
                    Console.WriteLine("Score: " + hit.Score);
                    Console.WriteLine(string.Join(" ", h2.Value.Highlights));
                }
            }
            Console.ReadLine();

        }

        static void TestQueryingByConversationAndContentType(string conversationID, ContentTag tag)
        {
            conversationID = string.IsNullOrEmpty(conversationID) ? DefaultGroupID : conversationID;

            var client = ConversationManager.GetClient();
            var resp = client.Search<ConversationMessage>(s => s.Query(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, conversationID)))
                && q.Bool(b => b.Must(qd => qd.Term(t => t.ContentTags, tag))))
                .Routing(conversationID)
                .Sort(sort => sort.OnField(f => f.Timestamp).Descending())
                .Size(20));

            Console.WriteLine("Request Url: " + resp.RequestInformation.RequestUrl);
            Console.WriteLine("Request Parms:" + Encoding.Default.GetString(resp.RequestInformation.Request));
            Console.ReadKey(true);
            var docs = resp.Documents;
            foreach (var doc in docs)
            {
                Console.WriteLine(doc.ID + "(Tags: " + string.Join(", ", doc.ContentTags) + ") " + doc.Body);
            }
            Console.WriteLine("Returned " + docs.Count() + " documents in " + resp.ElapsedMilliseconds + "ms");
            Console.ReadLine();
        }

        static void TestQueryingByConversationSortedByLikeCount(string conversationID)
        {
            conversationID = string.IsNullOrEmpty(conversationID) ? DefaultGroupID : conversationID;

            var client = ConversationManager.GetClient();
            var resp = client.Search<ConversationMessage>(s => s.Query(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, conversationID))))
                .Routing(conversationID)
                .Sort(sort => sort.OnField(f => f.LikeCount).Descending())
                .Size(20));

            Console.WriteLine("Request Url: " + resp.RequestInformation.RequestUrl);
            Console.WriteLine("Request Parms:" + Encoding.Default.GetString(resp.RequestInformation.Request));
            Console.ReadKey(true);
            var docs = resp.Documents;
            foreach (var doc in docs)
            {
                Console.WriteLine(doc.ID + "(Tags: " + string.Join(", ", doc.ContentTags) + ") " + doc.Body);
            }
            Console.WriteLine("Returned " + docs.Count() + " documents in " + resp.ElapsedMilliseconds + "ms");
            Console.ReadLine();
        }

        static void TestQueryingByConversationAndLikedByUser(string conversationID, int userID)
        {
            var client = ConversationManager.GetClient();
            var resp = client.Search<ConversationMessage>(s => s.Query(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, conversationID)))
                && q.Bool(b => b.Must(qd => qd.Term(t => t.LikeUserIDs, userID))))
                .Routing(conversationID)
                .Sort(sort => sort.OnField(f => f.Timestamp).Descending())
                .Size(20));

            Console.WriteLine("Request Url: " + resp.RequestInformation.RequestUrl);
            Console.WriteLine("Request Parms:" + Encoding.Default.GetString(resp.RequestInformation.Request));
            Console.ReadKey(true);
            var docs = resp.Documents;
            foreach (var doc in docs)
            {
                Console.WriteLine(doc.ID + ": " + string.Join(", ", doc.LikeUserIDs));
            }
            Console.WriteLine("Returned " + docs.Count() + " documents in " + resp.ElapsedMilliseconds + "ms");
            Console.ReadLine();
        }

        static void TestRandomQuery()
        {
            var search = new ConversationSearch( "1:128")
            {                
                StartDate = DateTime.UtcNow.AddDays(-300),
                EndDate = DateTime.UtcNow,
                PerPage = 10, 
                Keyword = "test",
                HighlightTagName = "searchHighlight"

            };

            var resp = ConversationManager.Search(search);

            Console.WriteLine(resp.Documents.Count() + " in " + resp.ElapsedMilliseconds);
            Console.ReadLine();
            
            foreach(var hit in resp.Hits)
            {
                var bodyHighlight = hit.Highlights["body"];
                var bodyHighlightValue = string.Join("", bodyHighlight.Highlights);
                //hit.Highlights.First(p => p. string.Join(" ", h2.Value.Highlights)
                Console.WriteLine(hit.Source.SenderID + " at " + hit.Source.Timestamp.FromEpochMilliconds().ToLocalTime().ToString() + ": " + bodyHighlightValue);
            }

            Console.ReadLine();
        }

    }


}
