﻿using System;
using System.IO;
using System.Reflection;
using System.Xml.Serialization;
using Curse.Logging;

namespace Curse.Friends.Configuration
{
    public static class ConfigurationHelper
    {
        public static T[] LoadConfigurations<T>(ConfigurationMode mode)
        {            
            // Determine the filename
            var xmlTypeAttr = typeof(T).GetCustomAttribute<XmlTypeAttribute>();

            if (xmlTypeAttr == null)
            {
                throw new ArgumentException("The supplied type must have an XmlType attribute");
            }

            var filename = xmlTypeAttr.TypeName + "." + mode + ".config";

            // Look for the appropriate configuration file, based on the configuration mode
            var configPath = FindConfigPath(filename);

            Logger.Info("Loading service configuration from: " + configPath);

            using (var reader = new StreamReader(configPath))
            {
                var dcs = new XmlSerializer(typeof(T));
                return (T[])dcs.Deserialize(reader);
            }
        }

        public static T LoadConfiguration<T>(ConfigurationMode mode, string name)
        {   
            var filename = name + "." + mode + ".config";

            // Look for the appropriate configuration file, based on the configuration mode
            var configPath = FindConfigPath(filename);

            Logger.Info("Loading service configuration from: " + configPath);

            using (var reader = new StreamReader(configPath))
            {
                var dcs = new XmlSerializer(typeof(T));
                return (T)dcs.Deserialize(reader);
            }
        
        }

        private static string FindConfigPath(string filename)
        {
            var configPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, filename);

            if (!File.Exists(configPath))
            {
                configPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Configuration", filename);
            }

            if (!File.Exists(configPath))
            {
                configPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "bin", filename);
            }

            if (!File.Exists(configPath))
            {
                configPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "bin", "Configuration", filename);
            }

            if (!File.Exists(configPath))
            {
                throw new Exception("Configuration could not be loaded from '" + configPath + "'");
            }

            return configPath;
        }
    }
}
