﻿using System;
using System.Collections.Generic;
using Curse.Logging;

namespace Curse.Friends.Configuration
{
    public class ConfigurationRegion
    {
        static readonly Dictionary<string, ConfigurationRegion> RegionsByKey = new Dictionary<string, ConfigurationRegion>(StringComparer.InvariantCultureIgnoreCase);
        static readonly Dictionary<int, ConfigurationRegion> RegionsByID = new Dictionary<int, ConfigurationRegion>();
        public static readonly HashSet<int> AllRegionIDs = new HashSet<int>();

        public static IEnumerable<ConfigurationRegion> AllRegions
        {
            get { return RegionsByKey.Values; }
        }

        public static ConfigurationRegion FindRegionByKey(string key)
        {
            ConfigurationRegion value = null;
            return RegionsByKey.TryGetValue(key, out value) ? value : null;
        }

        public static ConfigurationRegion FindRegionByID(int id)
        {
            ConfigurationRegion value = null;
            return RegionsByID.TryGetValue(id, out value) ? value : null;
        }

        public static ConfigurationRegion DiscoverCurrent()
        {
            try
            {
                var currentKey = RegionDiscovery.GetCurrentRegionKey();
                return FindRegionByKey(currentKey);
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }

            return null;
        }

        public ConfigurationRegion(int id, string key, string dnsPrefix, bool isDefault = false)
        {
            ID = id;
            Key = key;
            DnsPrefix = dnsPrefix;
            IsDefault = isDefault;
            RegionsByKey.Add(key, this);
            RegionsByID.Add(id, this);
            AllRegionIDs.Add(id);
        }

        /// <summary>
        ///  The numeric ID for this region, eg: 1
        /// </summary>
        public int ID
        {
            get;
            private set;
        }

        // The textual key for this region, eg: NA
        public string Key
        {
            get;
            private set;
        }

        public string DnsPrefix { get; private set; }
        
        public bool IsDefault { get; private set; }

        // Active Regions
        public static ConfigurationRegion USEast = new ConfigurationRegion(1, "US-East", "na", true);
        public static ConfigurationRegion EuropeWest = new ConfigurationRegion(3, "EU-West", "eu");
        public static ConfigurationRegion AsiaPacific = new ConfigurationRegion(5, "AP-Southeast", "ap");

        // Placeholder Regions
        //public static ConfigurationRegion USWest = new ConfigurationRegion(2, "US-West");
        //public static ConfigurationRegion EuropeNorth = new ConfigurationRegion(4, "EU-North");
        //public static ConfigurationRegion AsiaEast = new ConfigurationRegion(6, "AP-North");
        //public static ConfigurationRegion SouthAmericaSouth = new ConfigurationRegion(7, "SA-East");
     
    }
}
