﻿using System;
using System.IO;
using System.Xml.Serialization;
using Curse.Logging;
using System.Collections.Generic;
using System.Configuration;

namespace Curse.Friends.Configuration
{
    

    [XmlType("Service")]
    public class FriendsServiceConfiguration
    {
        public static ConfigurationMode Mode { get; private set; }

        public static readonly FriendsServiceConfiguration Instance = null; 
        
        static FriendsServiceConfiguration()
        {
            
#if CONFIG_DEBUG
            Mode = ConfigurationMode.Debug;
#elif CONFIG_STAGING
            Mode = ConfigurationMode.Staging;
#elif CONFIG_BETA
            Mode = ConfigurationMode.Beta;
#elif CONFIG_RELEASE
            Mode = ConfigurationMode.Release;
#elif CONFIG_LOADTESTING
            Mode = ConfigurationMode.LoadTesting;
#endif
            // Try to load the config, based on the compilation mode

            Logger.Info("Current configuration is: " + Mode);
            
            var configPath = FindConfigPath();

            Logger.Info("Loading service configuration from: " + configPath);

            using (var reader = new StreamReader(configPath))
            {
                var dcs = new XmlSerializer(typeof(FriendsServiceConfiguration));
                FriendsServiceConfiguration deserialized;
                try
                {
                    deserialized = (FriendsServiceConfiguration)dcs.Deserialize(reader);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to deserialize configuration");                    
                    throw;
                }
                
                Instance = deserialized;
            }
            var clientKeys = Instance.TwitchLoginClientKeys.Split(';');
            Instance.TwitchLoginClientMap = new Dictionary<string, string>();
            foreach (var keyPair in clientKeys)
            {
                var splits = keyPair.Split(':');
                if (splits.Length != 2)
                {
                    throw new ConfigurationErrorsException("LoginsWebServiceConfiguration contains an invalid configuration value for TwitchLoginClientKeys");
                }

                Instance.TwitchLoginClientMap[splits[0]] = splits[1];
            }

        }

        private static string FindConfigPath()
        {
            var configPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Service." + Mode.ToString() + ".config");

            if (!File.Exists(configPath))
            {
                configPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "bin", "Service." + Mode.ToString() + ".config");
            }

            if (!File.Exists(configPath))
            {
                throw new Exception("Configuration could not be loaded from '" + configPath + "'");
            }

            return configPath;
        }

        public void Save()
        {
            var configPath = FindConfigPath();

            using (var writer = new StreamWriter(configPath))
            {
                var dcs = new XmlSerializer(typeof(FriendsServiceConfiguration));
                dcs.Serialize(writer, this);
            }
        }

        [XmlElement()]
        public string NotificationServiceApiKey { get; set; }

        [XmlArray()]
        [XmlArrayItem("Port")]
        public int[] NotificationServicePorts { get; set; }        

        [XmlElement()]
        public string LastKnownRegion { get; set; }

        [XmlElement()]
        public string CentralServiceUrl { get; set; }

        [XmlElement()]
        public string CentralServiceApiKey { get; set; }

        [XmlElement()]
        public int NotificationServiceMaxConnections { get; set; }

        [XmlElement()]
        public string EncryptionKey { get; set; }

        [XmlElement()]
        public int EncryptionIterations { get; set; }
        
        [XmlElement()]
        public bool AlwaysIndexUserSearch { get; set; }        

        [XmlElement()]
        public string CentralLogServiceUrl { get; set; }

        [XmlElement()]
        public int AuthTokenLifespanSeconds { get; set; }

        [XmlElement]
        public string PromoApiKey { get; set; }

        [XmlArray]
        [XmlArrayItem("Domain")]
        public HashSet<string> CrossDomainWhitelist { get; set; }

        public string AuthTokenCookieDomain { get; set; }

        [XmlElement()]
        public string StatsServiceUrl { get; set; }

        [XmlElement()]
        public string AvatarUrlFormat { get; set; }

        [XmlElement]
        public int VanityUrlFollowerRequirement { get; set; }

        [XmlElement]
        public string GroupsRootUrl { get; set; }

        [XmlElement]
        public string SyncedEmoticonUrlFormat { get; set; }

        [XmlArray()]
        [XmlArrayItem("User")]
        public HashSet<int> TraceUsers { get; set; }

        [XmlElement]
        public string ServiceShutdownKey { get; set; }

        [XmlElement]
        public string PublicPollUrlFormat { get; set; }

        [XmlElement]
        public string TwitchChannelUrlFormat { get; set; }

        [XmlArray]
        [XmlArrayItem("Slug")]
        public HashSet<string> ReservedSlugs { get; set; }

        [XmlElement]
        public string TwitchClientID { get; set; }

        [XmlArray]
        [XmlArrayItem("Command")]
        public string[] UnsupportedChatCommands { get; set; }

        [XmlArray]
        [XmlArrayItem("EmoteID")]
        public long[] AdditionalEmoteIDs { get; set; }

        [XmlElement]
        public string TwitchShimUrl { get; set; }

        [XmlElement]
        public string TwitchShimApiKey { get; set; }

        [XmlElement("TwitchLoginClientKeys")]
        public string TwitchLoginClientKeys { get; set; }

        [XmlIgnore]
        public Dictionary<string, string> TwitchLoginClientMap { get; set; }

        [XmlElement]
        public int MaxWhispersPerSecond { get; set; }

        [XmlElement]
        public int MaxWhispersPerMinute { get; set; }

        [XmlElement]
        public string LoggingServiceApiKey { get; set; }
    }
}
