﻿using Curse.Aerospike;
using Curse.CloudQueue;
using Curse.CloudSearch;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.TwitchApi;

namespace Curse.Friends.Configuration
{

    [Flags]
    public enum ConfigurationServices
    {
        None = 0,
        Database = 1,
        Queue = 2,
        Search = 4,
        LocalOnly = 8,
        TwitchApi = 16,

        All = Database | Queue | Search | TwitchApi
    }

    public static class StorageConfiguration
    {        
        private static readonly LogCategory Logger = new LogCategory("Storage Configuration");        
        private static readonly object SyncRoot = new object();

        public static ConfigurationRegion CurrentRegion
        {
            get;
            private set;
        }

        public static void Shutdown()
        {
            // Stop processing any queues. It is important that this is done before closing Aerospike connections.
            CloudQueueInstanceManager.CloseConnections();

            // Shutdown Aerospike
            AerospikeConfiguration.Shutdown();            
        }

        public static ConfigurationMode CurrentMode { get; private set; }

        public static ConfigurationRegion GetCurrentRegion(string regionKey = null)
        {
            var currentRegionKey = regionKey ?? RegionDiscovery.GetCurrentRegionKey();
            Logger.Info("Current region discovered: " + currentRegionKey);

            var configRegion = ConfigurationRegion.FindRegionByKey(currentRegionKey);

            if (configRegion == null)
            {
                Logger.Fatal("Failed to get configuration region for key: " + currentRegionKey);
            }

            return configRegion;
        }

        public static void Initialize(string applicationIdentity, string regionKey = null, ConfigurationMode? configMode = null, ConfigurationServices options = ConfigurationServices.All)
        {            

            if (!configMode.HasValue)
            {
#if CONFIG_DEBUG
                CurrentMode = ConfigurationMode.Debug;
#elif CONFIG_STAGING
                CurrentMode = ConfigurationMode.Debug;
#elif CONFIG_LOADTESTING
                CurrentMode = ConfigurationMode.LoadTesting;
#elif CONFIG_RELEASE
                CurrentMode = ConfigurationMode.Release;
#endif
            }
            else
            {
                CurrentMode = configMode.Value;
            }

            var currentRegion = GetCurrentRegion(regionKey);
            Logger.Info("Initializing");

            if (currentRegion != null)
            {
                if (FriendsServiceConfiguration.Instance.LastKnownRegion != currentRegion.Key)
                {
                    FriendsServiceConfiguration.Instance.LastKnownRegion = currentRegion.Key;
                    FriendsServiceConfiguration.Instance.Save();
                }
            }
            else
            {
                currentRegion = ConfigurationRegion.FindRegionByKey(FriendsServiceConfiguration.Instance.LastKnownRegion);
            }

            CurrentRegion = currentRegion;

            if (CurrentRegion == null)
            {                
                throw new InvalidOperationException("Unable to find configuration region!");
            }

            var localRegionKey = CurrentRegion.Key;
            
            Logger.Info("Region key: " + localRegionKey);
            
            if (options.HasFlag(ConfigurationServices.Database))
            {
                Logger.Info("Initialzing Database Configuration");
                var databaseConfigurations = ConfigurationHelper.LoadConfiguration<AerospikeConfigurationCollection>(CurrentMode, "Aerospike");
                AerospikeConfiguration.Initialize(localRegionKey, "CurseVoice", databaseConfigurations, true, CurrentMode == ConfigurationMode.Debug, options.HasFlag(ConfigurationServices.LocalOnly));
                Logger.Info("Database Configuration Initialized!");
            }

            if (options.HasFlag(ConfigurationServices.Queue))
            {
                Logger.Info("Initialzing Queue Configuration...");
                var queueConfigurations = ConfigurationHelper.LoadConfiguration<QueueConfigurationCollection>(CurrentMode, "Queue");
                QueueConfigurationState.Initialize(applicationIdentity);
                
                if (CurrentMode == ConfigurationMode.Debug)
                {
                    QueueConfiguration.FastCreateQueues = true;
                }
                else
                {
                    QueueConfiguration.FastCreateQueues = false;
                }

                QueueConfiguration.Initialize(localRegionKey, queueConfigurations, QueueVersionCheck, QueueVersionUpdate);
                Logger.Info("Queue Configuration Initialized!");
            }

            if (options.HasFlag(ConfigurationServices.Search))
            {
                Logger.Info("Initialzing Search Configuration...");
                var searchConfigurations = ConfigurationHelper.LoadConfiguration<SearchConfigurationCollection>(CurrentMode, "Search");
                SearchConfiguration.Initialize(localRegionKey, searchConfigurations.Configurations);
                Logger.Info("Search Configuration Initialized!");
            }

            if (options.HasFlag(ConfigurationServices.TwitchApi))
            {
                Logger.Info("Initialzing Twitch API Client...");

                TwitchApiHelper.Initialize(FriendsServiceConfiguration.Instance.TwitchClientID, null, FriendsServiceConfiguration.Instance.TwitchLoginClientMap);
                Logger.Info("Twitch API Client Initialized!", new { ClientIDs = TwitchApiHelper.ClientIDs });
            }
            
        }

        private static bool QueueVersionCheck(string queueName)
        {
            lock(SyncRoot)
            {
                if (QueueConfigurationState.Instance.QueueVersions == null)
                {
                    return false;
                }

                var kvp = QueueConfigurationState.Instance.QueueVersions.FirstOrDefault(p => p.Key == queueName);

                if (kvp == null)
                {
                    return false;
                }

                var upToDate = kvp.Value >= QueueConfiguration.Version;
                return upToDate;
            }
        }


        private static void QueueVersionUpdate(string queueName)
        {
            lock(SyncRoot)
            {
                if (QueueConfigurationState.Instance.QueueVersions == null)
                {
                    QueueConfigurationState.Instance.QueueVersions = new List<QueueVersion>();
                }

                var list = QueueConfigurationState.Instance.QueueVersions;
                QueueConfigurationState.Instance.QueueVersions.RemoveAll(p => p.Key == queueName);
                list.Add(new QueueVersion(queueName, QueueConfiguration.Version));

                QueueConfigurationState.Instance.Save();
            }
        }
    }
}
