﻿using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.ContactsWebService.Contracts;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using System;
using Curse.Aerospike;
using Curse.Friends.NotificationContracts;
using Curse.Logging;

namespace Curse.Friends.ContactsWebService.Controllers
{
    [RoutePrefix("contacts")]
    public class ContactsController : MicroServiceController
    {

        private static readonly LogCategory ThrottledLogger = new LogCategory("Contacts") { Throttle = TimeSpan.FromSeconds(30) };
        private Dictionary<int, PrivateConversation> GetPrivateConversations(UserRegionInfo userAndRegion, DateTime? minDate = null)
        {
            var conversations = PrivateConversation.GetAllLocal(p => p.UserID, userAndRegion.User.UserID)
                                                   .ToDictionary(p => p.OtherUserID);

            if (conversations.Count > Friendship.MaxFriendCount)
            {
                ThrottledLogger.Warn("A user has more private conversations than should be possible!", new { user = userAndRegion.User.GetLogData(), conversationCount = conversations.Count });
            }

            return conversations;
        }


        private BlockedUserContract[] GetBlockedUsers(UserRegionInfo userAndRegion, DateTime? minDate = null)
        {
            var myBlockedUsers = UserBlock.GetAllForUser(userAndRegion.User.UserID);
            if (minDate.HasValue)
            {
                myBlockedUsers = myBlockedUsers.Where(p => p.Recency >= minDate.Value).ToArray();
            }

            if (myBlockedUsers.Length > Friendship.MaxFriendCount)
            {
                ThrottledLogger.Warn("A user has more blocked users than should be possible!", new { user = userAndRegion.User.GetLogData(), count = myBlockedUsers.Length });
            }

            var userStats = UserStatistics.GetDictionaryByUserIDs(myBlockedUsers.Select(p => p.OtherUserID));

            var missingUserIDs = myBlockedUsers.Select(u => u.OtherUserID).Except(userStats.Where(u => !string.IsNullOrEmpty(u.Value.Username)).Select(u => u.Key)).ToArray();
            if (missingUserIDs.Any())
            {
                UserStatistics.UpdateUserStatistics(missingUserIDs);
                userStats = UserStatistics.GetDictionaryByUserIDs(myBlockedUsers.Select(p => p.OtherUserID));
            }

            return myBlockedUsers.Select(p => p.ToContract(userStats.GetValueOrDefault(p.OtherUserID))).ToArray();
        }


        private Dictionary<int, ExternalCommunity> GetWatchingDictionary(Dictionary<int, UserStatistics> friendStats)
        {
            try
            {
                // For any connected friends, get the list of channel's being watched
                var watchingIDs = new HashSet<string>(friendStats.Where(p => p.Value.ConnectionStatus != UserConnectionStatus.Offline && !string.IsNullOrEmpty(p.Value.WatchingChannelID))
                                                                 .Select(p => p.Value.WatchingChannelID));

                // Get the external communities
                var communitiesByID = ExternalCommunity.GetDictionaryByTwitchIDs(watchingIDs);

                // Create a dictionary mapping user ID to external community
                return friendStats.Where(p => !string.IsNullOrEmpty(p.Value.WatchingChannelID))
                                                    .Select(p => new { UserID = p.Key, Watching = communitiesByID.GetValueOrDefault(p.Value.WatchingChannelID) })
                                                    .ToDictionary(p => p.UserID, p => p.Watching);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to retrieve watching dictionary.");
                return new Dictionary<int, ExternalCommunity>();
            }

        }

        private FriendshipContract[] GetFriendContracts(UserRegionInfo userAndRegion, Dictionary<int, PrivateConversation> privateConversations, DateTime? minDate = null)
        {
            var myFriends = Friendship.GetAll(userAndRegion.Region.RegionID, p => p.UserID, Token.UserID)
               .Where(p => p.IsSurfaced)
               .ToArray();

            if (myFriends.Length > Friendship.MaxFriendCount)
            {
                ThrottledLogger.Warn("A user has more friends than should be possible!", new { user = userAndRegion.User.GetLogData(), count = myFriends.Length });
            }

            // Get the stats for all friends (regardless of relationship status)

            var friendStats = UserStatistics.GetDictionaryByUserIDs(myFriends.Select(p => p.OtherUserID));

            // Get a dictionary mapping user ID to external community
            var watchingDictionary = GetWatchingDictionary(friendStats);


            // Create the friend contracts
            var friendNotifications = myFriends.Select(p =>
                p.ToNotification(friendStats.GetValueOrDefault(p.OtherUserID),
                    privateConversations.GetValueOrDefault(p.OtherUserID),
                    watchingDictionary.GetValueOrDefault(p.OtherUserID)))
                .ToArray();

            if (minDate.HasValue)
            {
                var epoch = minDate.Value.ToEpochMilliseconds();

                friendNotifications = friendNotifications.Where(p => p.DateMessaged >= minDate.Value
                                                                || p.DateRead >= minDate.Value
                                                                || p.DateConfirmed >= minDate.Value
                                                                || p.OtherUserConnectionStatusTimestamp >= epoch
                                                                || p.RequestedTimestamp >= epoch)
                                                          .ToArray();
            }

            return friendNotifications;
        }

        private const int GroupLimit = 5000;

        private GroupNotification[] GetGroupContracts(UserRegionInfo userAndRegion, DateTime? minDate = null)
        {
            // Group memberships
            var groupMemberships = GroupMember.GetDictionaryByUserID(userAndRegion.User.UserID, false, GroupLimit);

            if (groupMemberships.Count >= GroupLimit)
            {
                ThrottledLogger.Warn("A user has a ton of group memberships!", new { user = userAndRegion.User.GetLogData(), limit = GroupLimit });
            }

            // Memberships for each root group
            var membershipsByRoot = groupMemberships.ToLookup(g => g.Value.RootGroupID);

            // Group dictionary
            var groups = Group.GetDictionaryFromIDs(groupMemberships.Keys);

            var groupNotifications = groups.Values.Where(p => p.IsRootGroup)
                                                  .Select(p => p.ToContactNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, groups, groupMemberships));


            if (minDate.HasValue)
            {
                groupNotifications = groupNotifications.Where(p =>
                {
                    if (p.Membership == null)
                    {
                        return false;
                    }

                    var isRootRecent = p.Membership.DateJoined >= minDate.Value || p.Membership.DateMessaged >= minDate.Value || p.Membership.DateRead >= minDate.Value;
                    if (isRootRecent)
                    {
                        return true;
                    }

                    return membershipsByRoot[p.GroupID].Any(m => m.Value.DateMessaged >= minDate.Value || m.Value.DateRead >= minDate.Value);
                });
            }

            return groupNotifications.ToArray();

        }


        [Route("")]
        [HttpGet]
        public ContactsResponse GetAll()
        {

            var userAndRegion = GetCurrentUserAndRegion();
            var privateConversations = GetPrivateConversations(userAndRegion);
            var friendNotifications = GetFriendContracts(userAndRegion, privateConversations);
            var friendIDs = new HashSet<int>(friendNotifications.Select(p => p.OtherUserID));
            var privateConversationContracts = privateConversations.Values.Where(p => !friendIDs.Contains(p.OtherUserID) && !p.IsHidden && p.HasMessaged).Select(p => p.ToContract()).ToArray();
            var groupNotifications = GetGroupContracts(userAndRegion);
            var blockedUser = GetBlockedUsers(userAndRegion);

            return new ContactsResponse
            {
                Friends = friendNotifications,
                Groups = groupNotifications,
                PrivateConversations = privateConversationContracts,
                BlockedUsers = blockedUser,
            };
        }

        [Route("recent/{timestamp}")]
        [HttpGet]
        public ContactsResponse GetRecent(long timestamp)
        {
            var minDate = DateTime.UtcNow.AddDays(-7);
            var defaultDate = DateTime.UtcNow.AddDays(-1);

            if (timestamp == 0)
            {
                timestamp = DateTime.UtcNow.AddDays(-1).ToEpochMilliseconds();
            }

            var date = timestamp.FromEpochMilliconds();
            if (date < minDate)
            {
                date = minDate;
            }
            else if (date > DateTime.UtcNow)
            {
                date = defaultDate;
            }

            var userAndRegion = GetCurrentUserAndRegion();
            var privateConversations = GetPrivateConversations(userAndRegion, date);
            var friendNotifications = GetFriendContracts(userAndRegion, privateConversations, date);
            var groupNotifications = GetGroupContracts(userAndRegion, date).ToList();
            var friendIDs = new HashSet<int>(friendNotifications.Select(p => p.OtherUserID));
            var privateConversationContracts = privateConversations.Values.Where(p => !friendIDs.Contains(p.OtherUserID) && p.Recency >= date && !p.IsHidden && p.HasMessaged).Select(p => p.ToContract()).ToArray();
            var blockedUsers = GetBlockedUsers(userAndRegion, date);

            return new ContactsResponse
            {
                Friends = friendNotifications,
                Groups = groupNotifications.ToArray(),
                PrivateConversations = privateConversationContracts,
                BlockedUsers = blockedUsers
            };
        }

        [Route("url")]
        [HttpPost]
        [ResponseType(typeof(ContactUrlResponse))]
        [AuthenticationFilter(AuthenticationLevel.Anonymous)]
        public IHttpActionResult GetByUrl([FromBody] string url)
        {
            if (string.IsNullOrEmpty(url))
            {
                return NotFound();
            }

            if (!url.StartsWith("/"))
            {
                url = '/' + url;
            }

            var vanityUrl = GetVanityUrl(url);
            if (vanityUrl == null)
            {
                return NotFound();
            }

            switch (vanityUrl.Type)
            {
                case VanityUrlType.Group: // Try to get the related group

                    var group = Group.GetByID(vanityUrl.MappedID);
                    if (group == null)
                    {
                        return NotFound();
                    }

                    var groupSearchSettings = group.GetServerSearchSettingsOrDefault();

                    return Ok(new ContactUrlResponse
                    {
                        Url = vanityUrl.DisplayUrl,
                        Title = group.Title,
                        ContactID = group.GroupID.ToString(),
                        ConversationID = group.ConversationID,
                        Type = ContactUrlType.Server,
                        GroupContact = new GroupContactUrlResponse
                        {
                            Description = groupSearchSettings.Description,
                            IsPublic = group.IsPublic,
                            Subtype = group.Subtype,
                            MemberCount = group.MemberCount,
                        }
                    });

                default:
                    return NotFound();
            }


        }

        private static VanityUrl GetVanityUrl(string url, int redirects = 0)
        {
            if (++redirects > 10)
            {
                return null;
            }

            var vanityUrl = VanityUrl.GetByUrl(url);
            if (vanityUrl == null)
            {
                return null;
            }

            if (!string.IsNullOrWhiteSpace(vanityUrl.MappedUrl))
            {
                return GetVanityUrl(vanityUrl.MappedUrl);
            }

            return vanityUrl;
        }
    }
}