﻿using System;
using System.Linq;
using System.Net;
using System.Web.Http;
using Curse.Aerospike;
using Curse.Friends.Configuration;
using Curse.Friends.ContactsWebService.Contracts;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.MicroService.Exceptions;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Statistics;
using Curse.Logging;
using Curse.Friends.MicroService.Filters;

namespace Curse.Friends.ContactsWebService.Controllers
{
    [RoutePrefix("friend-sync")]
    public class FriendSyncController : MicroServiceController
    {

        [HttpPost]
        [Route("")]
        [SocialBanFilter]
        public IHttpActionResult Upload(FriendHintContract[] identities)
        {            
            if (identities == null)
            {
                return BadRequest("No idenitites in request");
            }

            if (identities.Length > FriendHint.MaxIdentitiesPerUser)
            {
                return BadRequest("Too many identities in request.");
            }
            
            var user = GetCurrentUserAndRegion();

            if (FriendsServiceConfiguration.Mode == ConfigurationMode.Release)
            {
                if (DateTime.UtcNow - user.User.LastFriendSyncUpload < TimeSpan.FromSeconds(5))
                {
                    throw new RequestThrottledException();
                }
            }

            user.User.LastFriendSyncUpload = DateTime.UtcNow;
            user.User.Update(p => p.LastFriendSyncUpload);

            var hints = identities.Select(FriendHint.FromNotification).ToArray();

            if (!ValidateFriendHints(hints))
            {
                return BadRequest("One or more friend hints has failed validation.");
            }

            var hasChanges = ProcessFriendHints(hints);

            if (hasChanges)
            {                
                new FriendHintSearchIndexer { UserID = Token.UserID }.Enqueue();                
            }

            return Ok();
        }

        private bool ValidateFriendHints(FriendHint[] hints)
        {
            foreach (var hint in hints)
            {
                string validationReason;

                if (!hint.Validate(out validationReason))
                {
                    Logger.Warn("Uploaded friend hint failed validation: " + validationReason, hint);
                    return false;

                }

                if (hint.Type != FriendHintType.Game && hint.Type != FriendHintType.Platform)
                {
                    Logger.Warn("Uploaded friend hint is not a suitable type: " + hint.Type, hint);
                    return false;
                }
            }

            return true;
        }

        private bool ProcessFriendHints(FriendHint[] uploadedHints)
        {
            var hasChanges = false;
            // Update the user ID on all friend hints
            foreach (var hint in uploadedHints)
            {
                hint.UserID = Token.UserID;
            }

            // Create a set of hint keys so that we can do a multi-get
            var hintKeys = uploadedHints.Select(p => p.GetGeneratedKey());

            // Get all existing keys and create a dictionary from them
            var existingHints = FriendHint.MultiGetLocal(hintKeys.Select(p => new KeyInfo(p))).ToDictionary(p => p.GetGeneratedKey());

            foreach (var uploadedHint in uploadedHints)
            {
                var trackStat = false;

                FriendHint friendHint;
                if (!existingHints.TryGetValue(uploadedHint.GetGeneratedKey(), out friendHint))
                {
                    uploadedHint.InsertLocal();
                    trackStat = true;
                    hasChanges = true;
                }
                else if(friendHint.HasChanges(uploadedHint))
                {
                    hasChanges = true;

                    friendHint.DisplayName = uploadedHint.DisplayName;

                    if (friendHint.Status != uploadedHint.Status)
                    {
                        trackStat = true;
                    }

                    friendHint.Status = uploadedHint.Status;
                    friendHint.Visibility = uploadedHint.Visibility;
                    friendHint.AvatarUrl = uploadedHint.AvatarUrl;
                    if (uploadedHint.Verification == FriendHintVerification.ClientObserved &&
                        friendHint.Verification < FriendHintVerification.ClientObserved)
                    {
                        friendHint.Verification = FriendHintVerification.ClientObserved;
                    }

                    friendHint.Update();
                }

                if (trackStat)
                {
                    if (uploadedHint.Status == FriendHintStatus.Normal)
                    {
                        FriendsStatsManager.Current.IdentitiesCreated.Track(Token.UserID);
                        if (uploadedHint.Type == FriendHintType.Platform)
                        {
                            FriendsStatsManager.Current.PlatformIdentitiesCreatedByType.Track((int)uploadedHint.Platform);
                        }
                    }
                    else
                    {
                        FriendsStatsManager.Current.IdentitiesDeclined.Track(Token.UserID);
                        if (uploadedHint.Type == FriendHintType.Platform)
                        {
                            FriendsStatsManager.Current.PlatformIdentitiesDeclinedByType.Track((int)uploadedHint.Platform);
                        }
                    }
                }
            }

            return hasChanges;
        }

        [HttpGet]
        [Route("")]
        public FriendHintContract[] Get()
        {
            var hints = FriendHint.GetAllLocal(p => p.UserID, Token.UserID);
            return
                hints.Where(p => p.Type == FriendHintType.Game || p.Type == FriendHintType.Platform)
                    .Select(p => p.ToNotification(false))
                    .ToArray();
        }

        [HttpDelete]
        [Route("{id}")]
        public IHttpActionResult Delete(long id)
        {
            var hint = FriendHint.GetAllLocal(p => p.UserID, Token.UserID)
                                 .FirstOrDefault(p => p.GetUniqueID() == id);
            if (hint == null || hint.Status == FriendHintStatus.Deleted)
            {
                return NotFound();
            }

            hint.Status = FriendHintStatus.Deleted;
            hint.Update(p => p.Status);
            new FriendHintSearchIndexer { UserID = Token.UserID }.Enqueue();
            return Ok();
        }

        [HttpPost]
        [Route("search")]
        public IHttpActionResult Search(FriendListSearchRequest request)
        {

            request.Validate();

            var user = GetCurrentUserAndRegion();

            if (DateTime.UtcNow - user.User.LastFriendSyncSearch < TimeSpan.FromMinutes(5))
            {
                throw new RequestThrottledException();
            }

            new FriendListSearchWorker
            {
                UserID = Token.UserID,
                Identity = FriendHint.FromNotification(request.Identity),
                FriendsList = request.FriendsList.Where(p=>p!=null).Select(FriendHint.FromNotification).ToArray()
            }.Enqueue();

            user.User.LastFriendSyncSearch = DateTime.UtcNow;
            user.User.Update(p => p.LastFriendSyncSearch);

            FriendsStatsManager.Current.FriendSyncsByResult.Track(4); // 4 is the legacy code for Success

            return StatusCode(HttpStatusCode.Accepted);
        }

        [HttpPost]
        [Route("multi-search")]
        public IHttpActionResult MultiSearch(FriendListMultiSearchRequest request)
        {

            request.Validate();

            var user = GetCurrentUserAndRegion();

            if (DateTime.UtcNow - user.User.LastFriendSyncSearch < TimeSpan.FromMinutes(5))
            {
                throw new RequestThrottledException();
            }

            user.User.LastFriendSyncSearch = DateTime.UtcNow;
            user.User.Update(p => p.LastFriendSyncSearch);

            FriendsStatsManager.Current.FriendSyncsByResult.Track(4); // 4 is the legacy code for Success

            foreach (var list in request.FriendLists)
            {
                new FriendListSearchWorker
                {
                    UserID = Token.UserID,
                    Identity = FriendHint.FromNotification(list.Identity),
                    FriendsList = list.FriendsList.Where(p => p != null).Select(FriendHint.FromNotification).ToArray()
                }.Enqueue();    
            }
           
            return StatusCode(HttpStatusCode.Accepted);
        }

    }
}
