﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Aerospike;
using Curse.CloudSearch;
using Curse.Extensions;
using Curse.Friends.ContactsWebService.Contracts;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Search;
using Curse.Friends.Data.Search.FriendSearch;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.MicroService.Exceptions;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Statistics;
using Curse.Friends.UserEvents;
using Curse.Logging;
using Nest;
using Curse.Friends.TwitchIdentityMerge;
using Curse.Friends.MicroService.Filters;

namespace Curse.Friends.ContactsWebService.Controllers
{
    [RoutePrefix("friendship")]
    public class FriendshipsController : MicroServiceController
    {
        [Route("{friendID}")]
        [HttpGet]
        [ResponseType(typeof(FriendshipContract))]
        public IHttpActionResult Get(int friendID)
        {
            var friendshipContext = new FriendshipContext(Token.UserID, friendID);
            var userStats = UserStatistics.GetByUserOrDefault(friendID);

            var conversation = PrivateConversation.GetByUserIDAndOtherUserID(Token.UserID, friendID);
            var watching = userStats != null ? userStats.GetWatchingCommunity() : null;
            var friendship = friendshipContext.GetMyFriendship(true).ToNotification(userStats, conversation, watching);
            return Ok(friendship);
        }

        [Route("{friendID}/confirm")]
        [HttpPost]
        [SocialBanFilter]
        public IHttpActionResult Confirm(int friendID)
        {

            var friendshipContext = new FriendshipContext(Token.UserID, friendID);

            // Try to get the friendship requested                
            var myFriendship = friendshipContext.GetMyFriendship();
            var theirFriendship = friendshipContext.GetTheirFriendship();

            if (myFriendship == null && theirFriendship == null)
            {
                Logger.Debug("[ConfirmFriendship] Not found", friendID);
                return NotFound();
            }

            if (myFriendship == null)
            {
                Logger.Debug("[ConfirmFriendship] Data issue detected, myFriendship is null", friendID);
                theirFriendship.Status = FriendshipStatus.Deleted;
                theirFriendship.Update(p => p.Status);
                return NotFound();
            }

            if (theirFriendship == null)
            {
                Logger.Debug("[ConfirmFriendship] Data issue detected, theirFriendship is null", friendID);
                myFriendship.Status = FriendshipStatus.Deleted;
                myFriendship.Update(p => p.Status);
                return NotFound();
            }

            // Ensure that the far end friendship is in the 'AwaitingConfirmation' state.
            if (theirFriendship.Status != FriendshipStatus.AwaitingThem)
            {
                Logger.Debug("[ConfirmFriendship] Other user's status is not AwaitingThem", friendID);
                return NotFound();
            }

            // Finalize the friendship
            Friendship.Confirm(friendshipContext.Me, friendshipContext.MyRegion, myFriendship, friendshipContext.Them, friendshipContext.TheirRegion, theirFriendship);

            // Track this, for stats
            FriendsStatsManager.Current.FriendRequestsConfirmed.Track();

            new ConfirmFriendRequestEvent { UserID = Token.UserID, OtherUserID = friendID }.Enqueue();

            return Ok();
        }

        [Route("{friendID}/decline")]
        [HttpDelete]
        public IHttpActionResult Decline(int friendID)
        {
            var friendshipContext = new FriendshipContext(Token.UserID, friendID);

            var myFriendship = friendshipContext.GetMyFriendship();
            var theirFriendship = friendshipContext.GetTheirFriendship();

            if (myFriendship == null && theirFriendship == null)
            {
                return NotFound();
            }

            if (myFriendship == null)
            {
                theirFriendship.Delete();
                return NotFound();
            }

            if (theirFriendship == null)
            {
                myFriendship.Delete();
                return NotFound();
            }

            // If this friendship isn't awaiting me, consider it to be not found
            if (myFriendship.Status != FriendshipStatus.AwaitingMe)
            {
                return NotFound();
            }

            // Create a deny and decline status, and then notify the client
            Friendship.Remove(false, friendshipContext.Me, myFriendship,
                friendshipContext.MyRegion, friendshipContext.Them, theirFriendship, friendshipContext.TheirRegion);

            new FriendshipRemovedResolver()
            {
                UserID = friendshipContext.Me.UserID,
                FriendID = myFriendship.OtherUserID
            }.Enqueue();

            new FriendshipRemovedResolver()
            {
                UserID = friendshipContext.Them.UserID,
                FriendID = theirFriendship.OtherUserID
            }.Enqueue();

            // Track this, for stats
            FriendsStatsManager.Current.FriendRequestsDeclined.Track();

            new RejectFriendRequestEvent { UserID = Token.UserID, OtherUserID = friendID }.Enqueue();
            return Ok();
        }

        [Route("twitch/{twitchID}/create")]
        [HttpPost]
        [SocialBanFilter]
        public IHttpActionResult CreateTwitchFriend(string twitchID, FriendshipRequest request)
        {
            var me = GetCurrentUserAndRegion();
            if (!me.User.IsMerged)
            {
                return Forbidden();
            }

            string conversationID;
            var otherAccount = ExternalAccount.GetLocal(twitchID, AccountType.Twitch);
            if (otherAccount != null && otherAccount.MergedUserID > 0)
            {
                // treat it like a normal message
                conversationID = (me.User.UserID < otherAccount.MergedUserID) ? string.Join(":", me.User.UserID, otherAccount.MergedUserID) : string.Join(":", otherAccount.MergedUserID, me.User.UserID); ;
                return Create(otherAccount.MergedUserID, request);
            }

            var mergeState = IdentityMergeServiceHelper.AutoProvisionAccount(twitchID);
            return Create(mergeState.CurseUserID, request);
        }

        [Route("{friendID}/create")]
        [HttpPost]
        [SocialBanFilter]
        public IHttpActionResult Create(int friendID, FriendshipRequest request)
        {
            request.Validate();

            if(Token.UserID == friendID)
            {
                Logger.Debug("Can't create a friendship with self", new { Token.UserID, friendID });
                return BadRequest();
            }

            // Ensure that the target user hasn't blocked the requesting user
            if (UserBlock.IsBlocked(Token.UserID, friendID))
            {
                Logger.Debug("Preventing blocked user from requesting friendship", new { Token.UserID, friendID });
                return Forbidden();
            }

            var friendshipContext = new FriendshipContext(Token.UserID, friendID);

            var myFriendship = friendshipContext.GetMyFriendship();
            var theirFriendship = friendshipContext.GetTheirFriendship();


            // Already requested
            if (myFriendship != null && theirFriendship != null)
            {
                if (request.IsFromSuggestion)
                {
                    var suggestion = FriendSuggestion.Get(friendshipContext.MyRegion.RegionID,
                        friendshipContext.Me.UserID, friendID);
                    if (suggestion != null && suggestion.Status == FriendSuggestionStatus.Pending)
                    {
                        suggestion.Status = FriendSuggestionStatus.Accepted;
                        suggestion.Update();
                    }
                }

                if (myFriendship.Status == FriendshipStatus.Confirmed ||
                    myFriendship.Status == FriendshipStatus.AwaitingThem ||
                    myFriendship.Status == FriendshipStatus.DeclinedByThem)
                {
                    return Forbidden();
                }
            }

            // Ensure this user doesn't have a ton of friendship records already. We check the actual friendship set, instead of user statistics, to ensure consistency.
            var myFriendCount = Friendship.GetConfirmedOrRequestedCount(friendshipContext.MyRegion.RegionID, friendshipContext.Me.UserID);

            if (myFriendCount > Friendship.MaxFriendCount)
            {
                return BadRequest("You have exceeded the maximum number of friends: " + myFriendCount);
            }
            
            var theirFriendCount = Friendship.GetConfirmedOrRequestedCount(friendshipContext.TheirRegion.RegionID, friendshipContext.Them.UserID);

            if (theirFriendCount > Friendship.MaxFriendCount)
            {
                return BadRequest("The person you are trying to add has exceeded the maximim number of friends: " + theirFriendCount);
            }

            if (theirFriendship != null && myFriendship != null && myFriendship.Status == FriendshipStatus.AwaitingMe)
            {
                // Confirm it
                Friendship.Confirm(friendshipContext.Me, friendshipContext.MyRegion, myFriendship, friendshipContext.Them, friendshipContext.TheirRegion, theirFriendship);

                new ConfirmFriendRequestEvent { UserID = Token.UserID, OtherUserID = friendID }.Enqueue();
            }
            else
            {
                if(!Curse.Friends.Data.User.CheckFriendshipPrivacy(friendshipContext.Me, friendshipContext.Them))
                {
                    return Forbidden();
                }
                
                // Create the request on both sides
                var result = Friendship.CreateRequest(friendshipContext.Me, friendshipContext.MyRegion,
                    friendshipContext.Them, friendshipContext.TheirRegion, request.KnownIdentity,
                    request.InvitationMessage, request.IsFromSuggestion, myFriendship, theirFriendship);

                new RequestFriendshipEvent { IsFromSuggestion = request.IsFromSuggestion, UserID = Token.UserID, OtherUserID = friendID }.Enqueue();
            }

            // Track this, for stats
            if (request.IsFromSuggestion)
            {
                FriendsStatsManager.Current.SuggestionsAccepted.Track();
            }
            else
            {
                FriendsStatsManager.Current.FriendRequestsSent.Track();
            }

            return Ok();
        }

        [Route("{friendID}/unfriend")]
        [HttpDelete]
        public IHttpActionResult Unfriend(int friendID)
        {
            // Ensure that the friendship exists            
            var friendshipContext = new FriendshipContext(Token.UserID, friendID);

            var myFriendship = friendshipContext.GetMyFriendship();
            var theirFriendship = friendshipContext.GetTheirFriendship();

            // If neither side of the friendship exists, this is an invalid request
            if (myFriendship == null && theirFriendship == null)
            {
                return NotFound();
            }

            // If only their side exists, remove it (this is a data issue)
            if (myFriendship == null)
            {
                Logger.Debug("[RemoveFriendship] Detected data issue, myFriendshup is null");
                theirFriendship.Status = FriendshipStatus.Deleted;
                theirFriendship.Update(p => p.Status);
                return Ok();
            }

            // If only my side exists, remove it (this is a data issue)
            if (theirFriendship == null)
            {
                Logger.Debug("[RemoveFriendship] Detected data issue, theirFriendship is null");
                myFriendship.Status = FriendshipStatus.Deleted;
                myFriendship.Update(p => p.Status);
                return Ok();
            }

            // If this was an outgoing friend request, consider it a cancellation
            var isCancellation = myFriendship.Status == FriendshipStatus.AwaitingThem;

            if (!myFriendship.IsSurfaced)
            {
                Logger.Debug("[RemoveFriendship] Unable to remove friendship", new { friendID, myFriendship, theirFriendship });
                return Forbidden();
            }

            Friendship.Remove(false, friendshipContext.Me, myFriendship, friendshipContext.MyRegion, friendshipContext.Them, theirFriendship, friendshipContext.TheirRegion);


            // Notify Myself        
            new FriendshipRemovedResolver { UserID = friendshipContext.Me.UserID, FriendID = friendshipContext.Them.UserID }.Enqueue();

            // Notify Them
            new FriendshipRemovedResolver { UserID = friendshipContext.Them.UserID, FriendID = friendshipContext.Me.UserID }.Enqueue();

            // Track this, for stats
            FriendsStatsManager.Current.FriendRequestsRemoved.Track();

            if (isCancellation)
            {
                new CancelFriendRequestEvent { UserID = Token.UserID, OtherUserID = friendID }.Enqueue();
            }
            else
            {
                new RemoveFriendshipEvent { UserID = Token.UserID, OtherUserID = friendID }.Enqueue();
            }
            

            return Ok();

        }

        /// <summary>
        /// Unblocks a user from sending future friend requests
        /// </summary>
        [Route("{friendID}/favorite")]
        [HttpPost]
        public IHttpActionResult Favorite(int friendID)
        {
            return ToggleFavorite(friendID, true);
        }

        /// <summary>
        /// Unblocks a user from sending future friend requests
        /// </summary>
        [Route("{friendID}/unfavorite")]
        [HttpPost]
        public IHttpActionResult Unfavorite(int friendID)
        {
            return ToggleFavorite(friendID, false);
        }

        private IHttpActionResult ToggleFavorite(int friendID, bool isFavorite)
        {
            var friendshipContext = new FriendshipContext(Token.UserID, friendID);
            var myFriendship = friendshipContext.GetMyFriendship();
            if (myFriendship == null)
            {
                return NotFound();
            }

            if (myFriendship.Status != FriendshipStatus.Confirmed)
            {
                return Forbidden();
            }

            Friendship.UpdateIsFavorite(friendshipContext.MyRegion.RegionID, Token.UserID, friendID, isFavorite);

            RefreshFriendshipAndNotify(Token.UserID, friendshipContext);
            return Ok();
        }


        [Route("{friendID}/friends")]
        [HttpGet]
        [ResponseType(typeof(FriendOfFriendDetails[]))]
        [SocialBanFilter]
        public IHttpActionResult GetFriendsOfFriend(int friendID)
        {
            var me = GetCurrentUserAndRegion();
            var myFriendship = Friendship.Get(me.Region.RegionID, me.User.UserID, friendID);

            if (myFriendship == null || myFriendship.Status != FriendshipStatus.Confirmed)
            {
                return Forbidden();
            }

            var result = Friendship.GetAllConfirmed(myFriendship.OtherUserRegionID, friendID)
                .Select(p => new FriendOfFriendDetails { UserID = p.OtherUserID, Username = p.OtherUsername })
                .ToArray();

            return Ok(result);
        }

        [Route("{friendID}/rename")]
        [HttpPost]
        public IHttpActionResult RenameFriend(int friendID, [FromBody] string nickname)
        {
            if (nickname.SafeLength() > Friendship.OtherUserNicknameMaxLength)
            {
                return BadRequest();
            }

            var friendshipContext = new FriendshipContext(Token.UserID, friendID);
            var myFriendship = friendshipContext.GetMyFriendship();
            if (myFriendship == null)
            {
                return NotFound();
            }

            Friendship.UpdateNickname(friendshipContext.MyRegion.RegionID, Token.UserID, friendID, nickname);

            RefreshFriendshipAndNotify(Token.UserID, friendshipContext);

            return Ok();
        }

        private FriendSearchResponse GetEmailSearchResults(string query)
        {                     
            CloudSearchResult<EmailFriendSearchModel>[] searchResult;
            if (!EmailFriendSearchManager.TrySearch(query, out searchResult))
            {
                return new FriendSearchResponse { Status = FriendSearchStatus.Invalid };
            }

            if (!searchResult.Any())
            {
                return new FriendSearchResponse()
                {
                    Status = FriendSearchStatus.Successful,
                    EmailMatches = new EmailSearchContract[0],
                    CharacterMatches = new CharacterSearchContract[0],
                    PlatformMatches = new PlatformSearchContract[0],
                    UserMatches = new UsernameSearchContract[0]
                };
            }

            var result = searchResult.First();
            var stats = UserStatistics.GetDictionaryByUserIDs(searchResult.Select(s => s.Value.UserID));
            var friendshipAvailability = GetFriendRequestAvailability(Token.UserID, searchResult.Select(s => s.Value.UserID).ToArray(), stats);
            return new FriendSearchResponse()
            {
                Status = FriendSearchStatus.Successful,
                EmailMatches = new[] { result.Value.ToNotification(result.Score, friendshipAvailability.GetValueOrDefault(result.Value.UserID)) },
                CharacterMatches = new CharacterSearchContract[0],
                PlatformMatches = new PlatformSearchContract[0],
                UserMatches = new UsernameSearchContract[0]
            };
            
        }

        [HttpGet]
        [Route("search")]
        public FriendSearchResponse Search(string query)
        {
            if (string.IsNullOrEmpty(query) || query.Length < 2 || query.Length > 128)
            {
                throw new RequestValidationException("Query string must be between 2 and 128 characters long");
            }
            
            query = query.ToLowerInvariant();

            // If the input contains an @ symbol, always perform an email search
            if (query.Contains("@"))
            {
                return GetEmailSearchResults(query);
            }
            
            // User Results
            CloudSearchResult<UsernameFriendSearchModel>[] usernameSearchResults;
            int usernameSearchTime;
            if (!UsernameFriendSearchManager.TrySearch(query, out usernameSearchResults, out usernameSearchTime))
            {
                return new FriendSearchResponse {Status = FriendSearchStatus.Invalid};
            }

            // Platform searches
            CloudSearchResult<PlatformFriendSearchModel>[] platformSearchResults;
            int platformSearchTime;
            if (!PlatformFriendSearchManager.TrySearch(query, out platformSearchResults, out platformSearchTime))
            {
                return new FriendSearchResponse() { Status = FriendSearchStatus.Invalid };
            }

            // Character searches
            CloudSearchResult<CharacterFriendSearchModel>[] characterSearchResults;
            int characterSearchTime;
            if (!CharacterFriendSearchManager.TrySearch(query, FriendHint.GetAllLocal(h => h.UserID, Token.UserID), out characterSearchResults, out characterSearchTime))
            {
                return new FriendSearchResponse() { Status = FriendSearchStatus.Invalid };
            }


            var allUserIDs = new HashSet<int>(usernameSearchResults
                .Select(u => u.Value.UserID)
                .Concat(platformSearchResults.Select(p => p.Value.UserID))
                .Concat(characterSearchResults.Select(c => c.Value.UserID)));

            var stats = UserStatistics.GetDictionaryByUserIDs(usernameSearchResults.Select(u => u.Value.UserID));
            var friendshipAvailability = GetFriendRequestAvailability(Token.UserID, allUserIDs.ToArray(), stats);

            var resp = new FriendSearchResponse
            {
                Status = FriendSearchStatus.Successful,
                CharacterMatches = characterSearchResults.Select(r => r.Value.ToNotification(r.Score, friendshipAvailability.GetValueOrDefault(r.Value.UserID))).ToArray(),
                UserMatches = usernameSearchResults.Select(r => r.Value.ToNotification(r.Score, friendshipAvailability.GetValueOrDefault(r.Value.UserID), stats.GetValueOrDefault(r.Value.UserID))).ToArray(),
                PlatformMatches = platformSearchResults.Select(r => r.Value.ToNotification(r.Score, friendshipAvailability.GetValueOrDefault(r.Value.UserID))).ToArray(),
                EmailMatches = new EmailSearchContract[0],
            };

            resp.AddSearchTime(usernameSearchTime);
            resp.AddSearchTime(platformSearchTime);
            resp.AddSearchTime(characterSearchTime);

            FriendsStatsManager.Current.SearchesPerformed.Track();

            return resp;
        }

        [Route("{friendID}/preferences")]
        [HttpPost]
        public IHttpActionResult ChangeFriendNotificationPreferences(int friendID, ChangeFriendshipPreferencesRequest request)
        {
            request.Validate();
            var friendshipContext = new FriendshipContext(Token.UserID, friendID);
            var myFriendship = friendshipContext.GetMyFriendship(true);
            myFriendship.UpdateNotificationPreferences(request.Preference, request.FilterSet);
            RefreshFriendshipAndNotify(Token.UserID, friendshipContext);
            return Ok();
        }

        private static Dictionary<int, FriendshipRequestAvailability> GetFriendRequestAvailability(int myID, int[] otherUserIDs, Dictionary<int, UserStatistics> stats)
        {
            var allPrivacySettings = UserPrivacySettings.MultiGetLocal(otherUserIDs.Select(id => new KeyInfo(id)))
                .DistinctByProperty(p => p.UserID).ToDictionary(p => p.UserID);

            var myStats = UserStatistics.GetByUserOrDefault(myID);
            
            var availabilities = new Dictionary<int, FriendshipRequestAvailability>();
            foreach (var id in otherUserIDs)
            {
                var availability = FriendshipRequestAvailability.Allowed;

                if (id == myID)
                {
                    continue;
                }

                UserPrivacySettings privacySettings;
                if (allPrivacySettings.TryGetValue(id, out privacySettings))
                {
                    UserStatistics them;
                    if (privacySettings.FriendRequestPrivacy == FriendRequestPrivacy.NoOne)
                    {
                        availability = FriendshipRequestAvailability.NotAllowed;
                    }
                    else if (privacySettings.FriendRequestPrivacy == FriendRequestPrivacy.FriendsOfFriends &&
                             (!stats.TryGetValue(id, out them) || !myStats.FriendIDs.Overlaps(them.FriendIDs)))
                    {
                        availability = FriendshipRequestAvailability.NoMutualFriends;
                    }
                }

                availabilities.Add(id, availability);

            }
            return availabilities;
        }

        /// <summary>
        /// Notify a user about a change to one of their friendships
        /// </summary>
        /// <param name="userID"></param>
        /// <param name="friendshipContext"></param>
        private void RefreshFriendshipAndNotify(int userID, FriendshipContext friendshipContext)
        {
            var myFriendship = friendshipContext.GetMyFriendship(); // Refresh
            var myContract = myFriendship.ToNotification(); // Create the contract
            
            // Dispatch the notification
            ClientEndpoint.DispatchNotification(userID, endpoint =>
            {
                FriendshipChangeNotifier.Create(endpoint, myContract);
            });           
        }

    }
}
