﻿using System.Linq;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Statistics;
using System.Collections.Generic;
using System;
using Curse.Friends.Data.Models;
using Curse.Friends.UserEvents;

namespace Curse.Friends.ContactsWebService.Controllers
{
    [RoutePrefix("users")]
    public class UsersController : MicroServiceController
    {
        [HttpGet]
        [Route("{id}")]
        [ResponseType(typeof(UserProfileNotification))]
        public IHttpActionResult Get(int id)
        {
            // Get the user and region for both sides            
            var other = GetUserAndRegion(id);

            // For API requests, return the profile immediately
            if (Token.IsApiRequest)
            {
                return Ok(other.User.GetUserProfile(Token.UserID, true));
            }

            // Get the requesting user's info
            var me = GetCurrentUserAndRegion();

            // Build the relationship info
            var relationship = new UserRelationship(other, me);

            // Ensure that the requesting user is friends with the target user, or is a pending suggestion
            if (!relationship.CanViewProfile)
            {
                return Forbidden();
            }

            FriendsStatsManager.Current.ProfilesViewed.Track(id);

            return Ok(other.User.GetUserProfile(Token.UserID, relationship.IncludePlatformIdentities));

        }                

        private class UserRelationship
        {
            public bool IsSelf { get; private set; }
            public bool IsSuggestion { get; private set; }
            public bool IsFriend { get; private set; }
            public bool IsFriendRequest { get; private set; }
            public bool IsGroupMember { get; private set; }
            public bool IsFamiliar { get; private set; }
            public bool HasBeenWhispered { get; private set; }

            public UserRelationship(UserRegionInfo other, UserRegionInfo me)
            {                
                if (me.User.UserID == other.User.UserID)
                {
                    IsSelf = true;
                }

                var suggestion = FriendSuggestion.Get(me.Region.RegionID, me.User.UserID, other.User.UserID);

                if (suggestion != null && suggestion.Status != FriendSuggestionStatus.Declined)
                {
                    IsSuggestion = true;
                }

                var theirGroupMemberships = new HashSet<Guid>(GroupMember.GetAllByUserID(other.User.UserID).Where(p => p.IsRootGroup).Select(p => p.GroupID));
                var myGroupMemberships = new HashSet<Guid>(GroupMember.GetAllByUserID(me.User.UserID).Where(p => p.IsRootGroup).Select(p => p.GroupID));

                if (myGroupMemberships.Any(p => theirGroupMemberships.Contains(p)))
                {
                    IsGroupMember = true;
                }

                
                var friendship = Friendship.Get(me.Region.RegionID, me.User.UserID, other.User.UserID);

                if (friendship != null)
                {
                    IsFriend = friendship.Status == FriendshipStatus.Confirmed;
                    IsFriendRequest = friendship.Status == FriendshipStatus.AwaitingMe;
                }

                var conversation = PrivateConversation.GetLocal(me.User.UserID, other.User.UserID);

                if(conversation != null)
                {
                    IsFamiliar = conversation.IsFamiliar;
                }

                var otherConversation = PrivateConversation.GetLocal(other.User.UserID, me.User.UserID);

                if(otherConversation != null)
                {
                    HasBeenWhispered = otherConversation.HasSent;
                }
            }

            public bool CanViewProfile
            {
                get
                {
                    return IsSelf || IsSuggestion || IsGroupMember || IsFriend || IsFriendRequest || IsFamiliar || HasBeenWhispered;
                }
            }

            public bool IncludePlatformIdentities
            {
                get { return IsSelf || IsSuggestion || IsFriend || IsFriendRequest; }
            
            }
        }

        [HttpPost]
        [Route("{id}/block")]
        [ResponseType(typeof (void))]
        public IHttpActionResult Block(int id)
        {
            // Get the user and region for both sides            
            var other = GetUserAndRegion(id);

            if (Token.UserID == id)
            {
                return BadRequest("You cannot block yourself");
            }

            var current = GetCurrentUserAndRegion();

            UserBlock.ToggleBlock(current.User, other.User, UserBlockStatus.Active);
            new BlockUserEvent { UserID = current.User.UserID, OtherUserID = other.User.UserID }.Enqueue();
            return Ok();
        }

        [HttpPost]
        [Route("{id}/unblock")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Unblock(int id)
        {
            // Get the user and region for both sides            
            var other = GetUserAndRegion(id);

            if (Token.UserID == id)
            {
                return BadRequest("You cannot unblock yourself");
            }

            var current = GetCurrentUserAndRegion();

            UserBlock.ToggleBlock(current.User, other.User, UserBlockStatus.Inactive);
            new UnblockUserEvent { UserID = current.User.UserID, OtherUserID = other.User.UserID }.Enqueue();
            return Ok();
        }

        
        
    }
}