﻿using System;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;

namespace Curse.Friends.ConversationsWebService.Contracts
{    
    public class ConversationSearchRequest : BaseRequestContract
    {
        public string ConversationID
        {
            get;
            set;
        }
        
        public string Keyword
        {
            get;
            set;
        }

        /// <summary>
        /// Only include conversation entries that were authored by the requesting user
        /// </summary>
        public bool SelfAuthored
        {
            get;
            set;
        }


        /// <summary>
        /// Only include conversation entries that were liked by the requesting user
        /// </summary>
        public bool SelfLiked
        {
            get;
            set;
        }

        /// <summary>
        /// Only include conversation entries that include mentions of the requesting user
        /// </summary>
        public bool SelfMentioned
        {
            get;
            set;
        }


        /// <summary>
        /// Only include message with one of the supplied content tags
        /// </summary>
        public ContentTag[] ContentTags
        {
            get;
            set;
        }

        public DateTime? StartDate 
        { 
            get; 
            set; 
        }

        public DateTime? EndDate
        { 
            get; 
            set; 
        }

        public int PerPage
        {
            get;
            set;
        }

        public int Page
        {
            get;
            set;
        }

        public string HighlightToken { get; set; }
        
        protected override bool DoValidate()
        {
            if(PerPage < 1 || PerPage > 100)
            {
                _validationMessage = "You may only have between 1 and 100 results per page.";
                return false;
            }

            if(Page < 1 || Page > 10)
            {
                _validationMessage = "You may only search up to 10 pages";
                return false;
            }

            if (string.IsNullOrEmpty(ConversationID))
            {
                _validationMessage = "A group or friend ID must be included.";
                return false;
            }

            if (Keyword.SafeLength() > 32)
            {
                _validationMessage = "Search keywords can only be up to 32 characters.";
                return false;
            }

            if (ContentTags != null && ContentTags.Length > Enum.GetValues(typeof(ContentTag)).Length)
            {
                _validationMessage = "Invalid content tags";
                return false;
            }

            return true;
        }
    }
}