﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web.Http;
using System.Web.Http.Description;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.ConversationsWebService.Contracts;
using Curse.Friends.Data;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.MicroService;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Statistics;
using Curse.Friends.Data.Models;
using Curse.Friends.UserEvents;
using Curse.Logging;
using Curse.Friends.TwitchIdentityMerge;
using Curse.Friends.MicroService.Filters;
using System.Net;

namespace Curse.Friends.ConversationsWebService.Controllers
{
    [RoutePrefix("conversations")]
    public class ConversationsController : MicroServiceController
    {
        private static readonly LogCategory MessagingLogger = new LogCategory("Messaging") { Throttle = TimeSpan.FromMinutes(10), ReleaseLevel = LogLevel.Debug };

        [HttpGet]
        [Route("{id}")]
        [ResponseType(typeof(ConversationMessageNotification[]))]
        public IHttpActionResult History(string id, long endTimestamp = 0, int pageSize = 100, long startTimestamp = 0)
        {
            var conversationContainer = ConversationManager.GetConversationContainer(Token.UserID, id, true);

            if (conversationContainer == null)
            {
                return NotFound();
            }

            if (startTimestamp > 0)
            {
                return GetAscendingConversationHistory(conversationContainer, pageSize, startTimestamp);                
            }
            else
            {
                return GetDescendingConversationHistory(conversationContainer, pageSize, endTimestamp);
            }
        }

        [HttpGet]
        [Route("{id}/jump")]
        [ResponseType(typeof(ConversationMessageNotification[]))]
        public IHttpActionResult Jump(string id, long timestamp, int pageSizeBefore, int pageSizeAfter)
        {
            if (pageSizeBefore > 100 || pageSizeBefore < 1 )
            {
                pageSizeBefore = 10;
            }

            if (pageSizeAfter > 100 || pageSizeAfter < 1)
            {
                pageSizeAfter = 10;
            }
            
            var conversationContainer = ConversationManager.GetConversationContainer(Token.UserID, id);

            if (conversationContainer == null)
            {
                return NotFound();
            }

            DateTime earliestDate;
            DateTime latestDate;

            if (!conversationContainer.CanView(Token.UserID, out latestDate, out earliestDate))
            {
                return Forbidden();
            }

            var messageDate = timestamp.FromEpochMilliconds();

            // Do not permit queries earlier than the user has permission to view.
            if (messageDate < earliestDate)
            {
                return BadRequest("Start date cannot be earlier than " + earliestDate.ToEpochMilliseconds());
            }

            var messages = ConversationManager.GetJumpMessages(id, messageDate, pageSizeBefore, pageSizeAfter);

            if (messages == null)
            {
                return Ok(new ConversationMessageNotification[0]);
            }

            messages = messages.Where(p => p.Timestamp >= earliestDate.ToEpochMilliseconds()).ToArray();

            return Ok(messages.Select(p => p.ToNotification(Token.UserID, null, conversationContainer.ConversationType, ConversationNotificationType.Normal)));
        }

        /// <summary>
        /// Gets a set number of messages after a specific point in time.
        /// </summary>
        /// <param name="conversationContainer"></param>
        /// <param name="pageSize"></param>
        /// <param name="startTimestamp"></param>
        /// <returns></returns>
        private IHttpActionResult GetAscendingConversationHistory(IConversationContainer conversationContainer, int pageSize, long startTimestamp)
        {
            if (pageSize > 100)
            {
                pageSize = 100;
            }

            if (pageSize < 10)
            {
                pageSize = 10;
            }

            DateTime earliestDate;
            DateTime latestDate;

            if (!conversationContainer.CanView(Token.UserID, out latestDate, out earliestDate))
            {
                return Forbidden();
            }

            
            var startDate = startTimestamp.FromEpochMilliconds();

            // Do not permit queries earlier than the user has permission to view.
            if (startDate < earliestDate)
            {
                return BadRequest("Start date cannot be earlier than " + earliestDate.ToEpochMilliseconds());
            }

            var endDate = startDate.AddDays(30);
            if (endDate > latestDate)
            {
                endDate = latestDate;                
            }

            var messages = ConversationManager.GetMessagesByConversationAndDateRangeAscending(conversationContainer.ConversationID, startDate, endDate, pageSize);

            if (messages != null && !messages.Any())
            {
                messages = ConversationManager.GetMessagesByConversationAndDateRangeAscending(conversationContainer.ConversationID, startDate, DateTime.UtcNow, pageSize);
            }
            
            if (messages == null)
            {
                return Ok(new ConversationMessageNotification[0]);
            }


            return Ok(GetMessageContracts(conversationContainer.ConversationType, messages));

        }

        private ConversationMessageNotification[] GetMessageContracts(ConversationType type, ConversationMessage[] messages)
        {
            var userIds = new HashSet<int>(messages.Select(p => p.SenderID));
            var stats = UserStatistics.GetDictionaryByUserIDs(userIds);
            return
                messages.Select(
                    p =>
                        p.ToNotification(Token.UserID, null, type, ConversationNotificationType.Normal,
                            SpamConfidence.Unknown, stats.GetValueOrDefault(p.SenderID))).ToArray();
        }

        private IHttpActionResult GetDescendingConversationHistory(IConversationContainer conversationContainer, int pageSize, long endTimestamp)
        {
            if (pageSize > 100)
            {
                pageSize = 100;
            }

            if (pageSize < 10)
            {
                pageSize = 10;
            }
            

            DateTime earliestDate;
            DateTime latestDate;

            if (!conversationContainer.CanView(Token.UserID, out latestDate, out earliestDate))
            {
                return Forbidden();
            }


            // No messages, so return an empty array
            if (latestDate == DateTime.MinValue)
            {
                return Ok(new ConversationMessageNotification[0]);
            }

            DateTime endDate;
            DateTime startDate;

            if (endTimestamp == 0) // We need to figure out an appropriate tail
            {
                startDate = earliestDate;

                if (latestDate < DateTime.UtcNow)
                {
                    latestDate = DateTime.UtcNow;                    
                }

                endDate = latestDate.AddMinutes(1); // Ensure the last message is included

            }
            else
            {
                startDate = earliestDate;
                endDate = endTimestamp.FromEpochMilliconds();
                if (endDate < earliestDate)
                {
                    return Forbidden();
                }
            }

            ConversationMessage[] messages;


            // If more than 60 days have elapsed between the earliest and most recent, try to get the last 30 days first
            // This is done to prevent relatively slow multi-index range queries.
            if ((endDate - startDate).TotalDays > 30)
            {
                messages = ConversationManager.GetMessagesByConversationAndDateRangeDescending(conversationContainer.ConversationID, endDate.AddDays(-30), endDate, pageSize);

                if (messages != null && !messages.Any())
                {
                    messages = ConversationManager.GetMessagesByConversationAndDateRangeDescending(conversationContainer.ConversationID, startDate, endDate, pageSize);
                }                
            }
            else
            {
                messages = ConversationManager.GetMessagesByConversationAndDateRangeDescending(conversationContainer.ConversationID, startDate, endDate, pageSize);
            }

            if (messages == null)
            {
                return Ok(new ConversationMessageNotification[0]);
            }

            return Ok(GetMessageContracts(conversationContainer.ConversationType, messages));
        }

        [SocialBanFilter]
        [HttpPost]
        [Route("{conversationID}/attachments/{attachmentID}")]
        public IHttpActionResult EditAttachment(string conversationID, string attachmentID, ConversationAttachmentEditRequest request)
        {

            request.Validate();

            Attachment attachment;
            ConversationAttachment conversationAttachment;
            ConversationMessage conversationMessage;
            IConversationContainer conversationContainer;

            if (!TryEditAttachment(Token.UserID, conversationID, request.MessageID,
                request.MessageTimestamp, attachmentID, out attachment, out conversationMessage,
                out conversationAttachment, out conversationContainer))
            {
                return NotFound();
            }

            // Update the attachment title in Aerospike
            attachment.Title = request.Title;
            attachment.Update(p => p.Title);

            var user = GetCurrentUserAndRegion();

            // Queue off a worker item to edit the attachment
            conversationAttachment.Title = request.Title;
            ConversationMessageWorker.CreateEditAttachment(conversationMessage.ConversationID, conversationMessage.ID, request.MessageTimestamp.FromEpochMilliconds(), user.User.UserID, user.User.GetTitleName(), conversationAttachment);

            // Change the message in-memory
            conversationMessage.EditedTimestamp = DateTime.UtcNow.ToEpochMilliseconds();
            conversationMessage.EditedUsername = user.User.GetTitleName();
            conversationMessage.EditedUserID = user.User.UserID;

            // Notify of this change
            conversationContainer.OnChatMessageChanged(conversationMessage, ConversationNotificationType.Edited);

            return Ok();
        }

        private bool TryEditAttachment(int userID, string conversationID, string messageID, long messageTimestamp, string attachmentID, out Attachment attachment, out ConversationMessage conversationMessage, out ConversationAttachment conversationAttachment, out IConversationContainer conversationContainer)
        {
            conversationAttachment = null;
            conversationMessage = null;
            attachment = Attachment.GetLocal(attachmentID);
            conversationContainer = ConversationManager.GetConversationContainer(userID, conversationID);

            if (attachment == null || conversationContainer == null)
            {
                return false;
            }

            if (!conversationContainer.CanEditAttachment(userID, attachment))
            {
                throw new GroupPermissionException(GroupPermissions.ChatAttachFiles);
            }

            // Then update the message
            conversationMessage = ConversationManager.GetMessageByID(conversationID, messageID, messageTimestamp.FromEpochMilliconds());

            if (conversationMessage == null || conversationMessage.Attachments == null)
            {
                return false;
            }

            conversationAttachment = conversationMessage.Attachments.FirstOrDefault(p => p.ID == attachmentID);

            if (conversationAttachment == null)
            {
                return false;
            }

            return true;
        }

        [SocialBanFilter]
        [HttpDelete]
        [Route("{conversationID}/attachments/{attachmentID}")]
        public IHttpActionResult DeleteAttachment(string conversationID, string attachmentID, ConversationDeleteAttachmentRequest request)
        {
            request.Validate();

            Attachment attachment;
            ConversationAttachment conversationAttachment;
            ConversationMessage conversationMessage;
            IConversationContainer conversationContainer;

            if (!TryEditAttachment(Token.UserID, conversationID, request.MessageID,
                request.MessageTimestamp, attachmentID, out attachment, out conversationMessage,
                out conversationAttachment, out conversationContainer))
            {
                return NotFound();
            }


            var currentUser = GetCurrentUserAndRegion();

            // Update the attachment to the deleted status in Aerospike
            attachment.Status = AttachmentStatus.Deleted;
            attachment.Update(p => p.Status);

            // Queue off a worker item to delete the attachment
            ConversationMessageWorker.CreateDeleteAttachment(conversationMessage.ConversationID, conversationMessage.ID, request.MessageTimestamp.FromEpochMilliconds(), currentUser.User.UserID, currentUser.User.GetTitleName());

            // Send the notification of the change
            conversationMessage.IsDeleted = true;
            conversationContainer.OnChatMessageChanged(conversationMessage, ConversationNotificationType.Deleted);

            return Ok();
        }

        [HttpPost]
        [Route("{conversationID}/search")]
        [ResponseType(typeof(ConversationSearchItem[]))]
        public IHttpActionResult Search(string conversationID, [FromBody] ConversationSearchRequest searchFilters)
        {

            if (searchFilters == null)
            {
                return BadRequest("You must supply at least one search filter");
            }

            searchFilters.Validate();

            var conversationContainer = ConversationManager.GetConversationContainer(Token.UserID, searchFilters.ConversationID);

            if (conversationContainer == null)
            {
                return NotFound();
            }

            DateTime? minSearchDate;
            if (!conversationContainer.CanSearch(Token.UserID, out minSearchDate))
            {
                return Forbidden();
            }

            var search = new ConversationSearch(conversationContainer.ConversationID);
            search.StartDate = minSearchDate;

            if (searchFilters.StartDate.HasValue)
            {
                if (searchFilters.StartDate > search.StartDate)
                {
                    search.StartDate = searchFilters.StartDate;
                }
                search.EndDate = searchFilters.EndDate;
            }

            if (searchFilters.SelfMentioned)
            {
                search.MentionedUserID = Token.UserID;
            }

            if (searchFilters.SelfLiked)
            {
                search.LikedUserID = Token.UserID;
            }

            if (searchFilters.SelfAuthored)
            {
                search.SenderUserID = Token.UserID;
            }

            if (!string.IsNullOrWhiteSpace(searchFilters.Keyword))
            {
                search.Keyword = searchFilters.Keyword;
            }

            if (searchFilters.ContentTags != null && searchFilters.ContentTags.Any())
            {
                search.ContentTags = searchFilters.ContentTags;
            }

            search.Page = searchFilters.Page;
            search.PerPage = searchFilters.PerPage;
            search.HighlightTagName = searchFilters.HighlightToken ?? "searchHighlight";

            var results = ConversationManager.Search(search);

            if (!results.ConnectionStatus.Success)
            {
                return BadRequest();
            }

            // If a keyword was supplied, we must iterate over the hits, otherwise just return the documents
            if (search.Keyword != null)
            {
                return Ok(results.Hits.Select(p => new ConversationSearchItem
                {
                    Message = p.Source.ToNotification(Token.UserID, null, conversationContainer.ConversationType, ConversationNotificationType.Normal),
                    Highlight = string.Join("", p.Highlights.FirstOrDefault().Value.Highlights)
                }).ToArray());
            }

            return Ok(results.Documents.Select(
                p => new ConversationSearchItem { Message = p.ToNotification(Token.UserID, null, conversationContainer.ConversationType, ConversationNotificationType.Normal) }).ToArray());            
        }


        [HttpPost]
        [Route("{conversationID}/hide")]        
        [ResponseType(typeof(void))]
        public IHttpActionResult Hide(string conversationID)
        {
            return ToggleHidden(conversationID, Token.UserID, true);
        }

        [HttpPost]
        [Route("{conversationID}/unhide")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Unhide(string conversationID)
        {
            return ToggleHidden(conversationID, Token.UserID, false);
        }

        private IHttpActionResult ToggleHidden(string conversationID, int userID, bool isHidden)
        {

            var conversation = ConversationManager.GetConversationContainer(userID, conversationID);

            if (conversation == null)
            {
                return NotFound();                
            }

            if (!conversation.CanHide())
            {
                return BadRequest("This conversation type cannot be hidden.");
            }

            var conversationParent = conversation.GetConversationParent(userID);

            if (conversationParent == null)
            {
                return NotFound();
            }

            
            conversationParent.ToggleHidden(isHidden);

            new HideConversationEvent { UserID = userID, ConversationID = conversationID, IsHidden = isHidden }.Enqueue();
            
            //TODO: Notify user's endpoints that this conversation is now hidden
            return Ok();
        }

        [HttpPost]
        [Route("{conversationID}/mute")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Mute(string conversationID)
        {
            return ToggleMuted(conversationID, Token.UserID, true);
        }

        [HttpPost]
        [Route("{conversationID}/unmute")]
        [ResponseType(typeof(void))]
        public IHttpActionResult Unmute(string conversationID)
        {
            return ToggleMuted(conversationID, Token.UserID, false);
        }

        private IHttpActionResult ToggleMuted(string conversationID, int userID, bool isMuted)
        {

            var conversation = ConversationManager.GetConversationContainer(userID, conversationID);

            if (conversation == null)
            {
                return NotFound();
            }

            if (!conversation.CanHide())
            {
                return BadRequest("This conversation type cannot be hidden.");
            }

            var conversationParent = conversation.GetConversationParent(userID);

            if (conversationParent == null)
            {
                return NotFound();
            }


            conversationParent.ToggleMuted(isMuted);

            if (isMuted)
            {
                new MuteConversationEvent { ConversationID = conversationID, UserID = userID }.Enqueue();   
            }
            else
            {
                new UnmuteConversationEvent { ConversationID = conversationID, UserID = userID }.Enqueue();    
            }
            

            //TODO: Notify user's endpoints that this conversation is now muted
            return Ok();
        }


        /// <summary>
        /// Creates a new message in the conversation specified.
        /// </summary>
        /// <param name="conversationID"></param>
        /// <param name="id"></param>
        /// <param name="timestamp"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [SocialBanFilter]
        [HttpPost]
        [Route("{conversationID}")]
        public IHttpActionResult CreateMessage(string conversationID, ConversationCreateMessageRequest request)
        {
            request.Validate();

            var response = new ConversationMessageResponse
            {
                ClientID = request.ClientID,
                ConversationID = conversationID,
                Status = DeliveryStatus.Successful
            };

            var clientEndpoint = ClientEndpoint.GetLocal(Token.UserID, request.MachineKey);

            if (clientEndpoint == null)
            {
                MessagingLogger.Warn("Failed to retrieve a client endpoint. This message will be discarded.", new { Token.UserID, request.MachineKey });
                return BadRequest("Unknown client endpoint!");
            }

            
            try
            {
                // Process any included attachments
                Attachment attachment = null;

                if (request.AttachmentID != Guid.Empty)
                {
                    // First try to get the attachment
                    var regionID = ConfigurationRegion.AllRegionIDs.Contains(request.AttachmentRegionID) ? request.AttachmentRegionID : Attachment.LocalConfigID;
                    attachment = Attachment.Get(regionID, request.AttachmentID);

                    // If the attachment is missing, or was not created by the sender, forbidden!
                    if (attachment == null || attachment.UploaderUserID != Token.UserID)
                    {
                        MessagingLogger.Warn("Failed to retrieve a valid attachment or an attachment owned by the requesting user. This message will be discarded. ", new { Token.UserID, request.AttachmentID });
                        response.Status = DeliveryStatus.Forbidden;
                        response.ForbiddenReason = MessageForbiddenReason.NotAttachmentOwner;
                        return Forbidden();
                    }
                }

                // Get the conversation container (Friendship, Group, etc)
                var conversationContainer = ConversationManager.GetConversationContainer(Token.UserID, conversationID, true);
                if (conversationContainer == null)
                {
                    MessagingLogger.Warn("Failed to retrieve a conversation. This message will be discarded. ", new { Token.UserID, ConversationID = conversationID });
                    response.Status = DeliveryStatus.UnknownUser;                    
                    return NotFound();
                }


                var messageRequest = new ConversationMessageRequest
                {
                    ConversationID = conversationID,
                    Message = request.Body,
                    AttachmentID = request.AttachmentID,
                    AttachmentRegionID = request.AttachmentRegionID,
                    ClientID = request.ClientID
                };
                                
                switch (conversationContainer.ConversationType)
                {
                    case ConversationType.Friendship:                    
                        {
                            MessagingLogger.Debug("Queueing private message worker", new { conversationID, conversationContainer.Title, conversationContainer.ConversationType });
                            PrivateMessageWorker.Create(Token.UserID, request.MachineKey, conversationID, attachment != null ? attachment.Url : request.Body, request.ClientID, attachment);                            
                            break;
                        }
                    case ConversationType.Group:
                        {
                            // Note: We do not check permissons here. This is left up to the group server

                            var group = conversationContainer as Group;

                            if (group.Type == GroupType.Large && group.IsRootGroup)
                            {
                                response.Status = DeliveryStatus.Forbidden;                                
                                response.ForbiddenReason = MessageForbiddenReason.InvalidChannel;
                                return Forbidden();
                            }

                            FriendsStatsManager.Current.GroupMessagesSent.Track(group.GroupID);
                            FriendsStatsManager.Current.GroupMessagesSentByPlatform.Track((int)clientEndpoint.Platform);
                            GroupMessageCordinator.Create(group, messageRequest, Token.UserID, clientEndpoint, attachment);
                            break;
                        }
                    case ConversationType.GroupPrivateConversation:
                        {
                            if (!conversationContainer.CanSendMessage(Token.UserID))
                            {
                                // TODO: Proper Reason
                                response.Status = DeliveryStatus.Forbidden;
                                response.ForbiddenReason = MessageForbiddenReason.NotFamiliar;                                
                                return Forbidden();
                            }

                            var conversation = conversationContainer as GroupPrivateConversation;
                            if (conversation == null || conversation.Group == null || !conversation.Group.IsRootGroup)
                            {
                                // TODO: Proper Reason
                                response.Status = DeliveryStatus.Forbidden;                                
                                response.ForbiddenReason = MessageForbiddenReason.NotFamiliar;
                                return Forbidden();
                            }

                            FriendsStatsManager.Current.PrivateGroupMessagesSent.Track(conversationID);
                            GroupPrivateMessageCoordinator.Create(conversation.Group, messageRequest, Token.UserID, conversation.OtherUserID, clientEndpoint, attachment);
                            break;
                        }
                }

                return Ok();
            }
            finally 
            {
                if (response.Status != DeliveryStatus.Successful)
                {
                    if (clientEndpoint.IsRoutable)
                    {
                        ChatMessageResponseNotifier.Create(clientEndpoint, response);
                    }
                    else
                    {
                        MessagingLogger.Warn("Message sent from an un-routable endpoint.", new { clientEndpoint, conversationID });
                    }
                }
            }            
        }

        [HttpPost]
        [Route("twitch/{twitchID}")]
        [SocialBanFilter]
        public IHttpActionResult SendTwitchWhisper(string twitchID, ConversationCreateMessageRequest request)
        {
            var me = GetCurrentUserAndRegion();
            if (!me.User.IsMerged)
            {
                return Forbidden();
            }

            string conversationID;
            var otherAccount = ExternalAccount.GetLocal(twitchID, AccountType.Twitch);
            if(otherAccount != null && otherAccount.MergedUserID > 0)
            {
                // treat it like a normal message
                conversationID = (me.User.UserID < otherAccount.MergedUserID) ? string.Join(":", me.User.UserID, otherAccount.MergedUserID) : string.Join(":", otherAccount.MergedUserID, me.User.UserID); ;
                return CreateMessage(conversationID, request);
            }

            var mergeState = IdentityMergeServiceHelper.AutoProvisionAccount(twitchID);

            if(mergeState.Status == IdentityMergeStatus.Failed)
            {
                return StatusCode(HttpStatusCode.InternalServerError, new { mergeState.FailureReason });
            }

            // Treat it like a normal message with the new provisioned user
            conversationID = (me.User.UserID < mergeState.CurseUserID) ? string.Join(":", me.User.UserID, mergeState.CurseUserID) : string.Join(":", mergeState.CurseUserID, me.User.UserID);
            return CreateMessage(conversationID, request);
        }

        /// <summary>
        /// Edits the body of the requested message
        /// </summary>
        /// <param name="conversationID"></param>
        /// <param name="id"></param>
        /// <param name="timestamp"></param>
        /// <param name="request"></param>
        /// <returns></returns>
        [SocialBanFilter]
        [HttpPost]
        [Route("{conversationID}/{id}-{timestamp}")]
        public IHttpActionResult EditMessage(string conversationID, string id, long timestamp, ConversationEditMessageRequest request)
        {
            request.Validate();

            // Get the container for the conversation (Group, Friend, etc)
            var conversationContainer = ConversationManager.GetConversationContainer(Token.UserID, conversationID);

            if (conversationContainer == null)
            {
                return NotFound();
            }

            // Get the message itself
            var conversationMessage = ConversationManager.GetMessageByID(conversationID, id, timestamp.FromEpochMilliconds());

            if (conversationMessage == null || conversationMessage.IsDeleted)
            {
                return NotFound();
            }

            if (conversationMessage.Attachments != null && conversationMessage.Attachments.Any())
            {
                return NotFound();
            }

            if (!conversationContainer.CanEditMessage(Token.UserID, conversationMessage))
            {
                return NotFound();
            }

            if (request.Mentions != null)
            {
                if (conversationMessage.Mentions.Contains(0) && !conversationContainer.CanMentionEveryone(Token.UserID, conversationMessage))
                {
                    return Forbidden("You do not have permission to mention everyone.");
                }
                
                if (conversationMessage.Mentions.Length > 0 && !conversationContainer.CanMention(Token.UserID, conversationMessage))
                {
                    return Forbidden("You do not have permission to mention users.");
                }
            }

            var editTimestamp = DateTime.UtcNow;
            var currentUser = GetCurrentUserAndRegion();

            // Make the change in memory, and dispatch it
            conversationMessage.EditMessage(Token.UserID, currentUser.User.GetTitleName(), request.Body.Trim(), editTimestamp.ToEpochMilliseconds(), request.Mentions);

            conversationContainer.OnChatMessageChanged(conversationMessage, ConversationNotificationType.Edited);

            // Queue off a worker item to persist the change
            ConversationMessageWorker.CreateEditMessage(conversationMessage.ConversationID, conversationMessage.ID, timestamp.FromEpochMilliconds(), Token.UserID, currentUser.User.GetTitleName(), request.Body, editTimestamp, conversationMessage.Mentions, conversationMessage.EmoteSubstitutions);

            new EditConversationMessageEvent
            {
                UserID = Token.UserID,
                ConversationID = conversationID,
                MessageID = id,
                MessageBody = conversationMessage.Body
            }.Enqueue();

            return Ok();

        }

        /// <summary>
        /// Deletes the requested message.
        /// </summary>
        /// <param name="conversationID"></param>
        /// <param name="id"></param>
        /// <param name="timestamp"></param>
        /// <returns></returns>
        [SocialBanFilter]
        [HttpDelete]
        [Route("{conversationID}/{id}-{timestamp}")]
        public IHttpActionResult DeleteMessage(string conversationID, string id, long timestamp)
        {
            // Get the container for the conversation (Group, Friend, etc)
            var conversationContainer = ConversationManager.GetConversationContainer(Token.UserID, conversationID);

            if (conversationContainer == null)
            {
                return NotFound();
            }

            // Get the message itself
            var conversationMessage = ConversationManager.GetMessageByID(conversationID, id, timestamp.FromEpochMilliconds());

            if (conversationMessage == null)
            {
                return NotFound();
            }

            if (!conversationContainer.CanDeleteMessage(Token.UserID, conversationMessage))
            {
                return NotFound();
            }

            var editTimestamp = DateTime.UtcNow;
            var currentUser = GetCurrentUserAndRegion();

            // Make the change in memory, and dispatch it
            conversationMessage.DeletedTimestamp = editTimestamp.ToEpochMilliseconds();
            conversationMessage.DeletedUsername = currentUser.User.GetTitleName();
            conversationMessage.DeletedUserID = currentUser.User.UserID;
            conversationMessage.IsDeleted = true;
            conversationContainer.OnChatMessageChanged(conversationMessage, ConversationNotificationType.Deleted);

            // Queue off a worker item to persist the change
            ConversationMessageWorker.CreateDeleteMessage(conversationMessage.ConversationID, conversationMessage.ID, timestamp.FromEpochMilliconds(), Token.UserID, currentUser.User.GetTitleName(), editTimestamp);

            new DeleteConversationMessageEvent
            {
                UserID = Token.UserID,
                ConversationID = conversationID,
                MessageID = id                
            }.Enqueue();

            return Ok();

        }

        [SocialBanFilter]
        [HttpPost]
        [Route("{conversationID}/{id}-{timestamp}/like")]
        public IHttpActionResult LikeMessage(string conversationID, string id, long timestamp)
        {
            return LikeOrUnlikeMessage(conversationID, id, timestamp, false);
        }

        [SocialBanFilter]
        [HttpPost]
        [Route("{conversationID}/{id}-{timestamp}/unlike")]
        public IHttpActionResult UnlikeMessage(string conversationID, string id, long timestamp)
        {
            return LikeOrUnlikeMessage(conversationID, id, timestamp, false);
        }

     
        private IHttpActionResult LikeOrUnlikeMessage(string conversationID, string id, long timestamp, bool unlike)
        {
            // Get the container for the conversation (Group, Friend, etc)
            var conversationContainer = ConversationManager.GetConversationContainer(Token.UserID, conversationID);

            if (conversationContainer == null)
            {
                return NotFound();
            }

            // Get the message itself
            var conversationMessage = ConversationManager.GetMessageByID(conversationID, id, timestamp.FromEpochMilliconds());

            if (conversationMessage == null || conversationMessage.IsDeleted)
            {
                return NotFound();
            }

            if (!conversationContainer.CanLikeMessage(Token.UserID, conversationMessage))
            {
                return Forbidden();
            }

            // Try to get the existing 
            var like = Like.GetByEntityTypeAndEntityAndUser((int)LikeType.ConversationMessage, conversationMessage.ID, Token.UserID);


            if (unlike && like == null)
            {
                return Forbidden();
            }

            if (like != null)
            {
                like.Unlike = !like.Unlike;
                like.Update(UpdateMode.Concurrent);
            }
            else
            {
                like = Like.Create(LikeType.ConversationMessage, conversationMessage.ID, Token.UserID);
            }

            var currentUser = GetCurrentUserAndRegion();
            conversationContainer.OnChatMessageLike(Token.UserID, currentUser.User.GetTitleName(), conversationMessage, like);

            return Ok();
        }


        [HttpPost]
        [Route("{conversationID}/mark-as-read")]
        public IHttpActionResult MarkAsRead(string conversationID, ConversationMarkReadRequest request)
        {
            request.Validate();

            // Ensure time is UTC
            var dateRead = request.Timestamp.FromEpochMilliconds().NormalizeToUtc();
            ConversationReadWorker.Create(Token.UserID, conversationID, dateRead, request.MachineKey, request.MessageID, false);            
            return Ok();
        }

        [HttpPost]
        [Route("mark-all-as-read")]
        [ResponseType(typeof(void))]
        public IHttpActionResult MarkAllAsRead()
        {
            // Get all group memberships
            var groupMemberships = GroupMember.GetAllByUserID(Token.UserID);
            
            foreach (var membership in groupMemberships)
            {
                membership.MarkAsRead(membership.DateMessaged);
            }

            // Get all conversations for the current user
            var conversations = PrivateConversation.GetAllByUserID(Token.UserID);

            foreach (var conversation in conversations)
            {               
                conversation.MarkAsRead(conversation.DateMessaged);
            }

            // Get the most recent date messages for any conversation
            var mostRecentDate = Math.Max(conversations.DefaultIfEmpty().Max(p => p.DateMessaged).SafeToEpochMilliseconds(), groupMemberships.DefaultIfEmpty().Max(p => p.DateMessaged).ToEpochMilliseconds());

            var notification = new ConversationReadNotification
            {
                MarkAllAsRead = true
            };

            // Dispatch a notification to the user's endpoints, indicating everything should be marked as read            
            ClientEndpoint.DispatchNotification(Token.UserID, ep =>
            {
                ConversationReadNotifier.Create(ep, notification);
            });

            new MarkAllConversationReadEvent { UserID = Token.UserID, Timestamp = mostRecentDate }.Enqueue();

            return Ok();
        }

        [HttpPost]
        [Route("{conversationID}/flag-as-spam")]        
        public IHttpActionResult FlagAsSpam(string conversationID, [FromBody] ConversationFlagSpamRequest request)
        {
            request.Validate();
                            
            var otherID = ConversationManager.GetFriendID(Token.UserID, conversationID);           
            new FlagConversationSpamEvent { UserID = Token.UserID, OtherUserID = otherID, IsSpam = request.IsSpam}.Enqueue();
                                       
            return Ok();
        }
    }
}