﻿using Curse.CloudSearch;
using Nest;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Curse.Extensions;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;
using Curse.Logging;
using Curse.Friends.Enums;

namespace Curse.Friends.Data.Messaging
{
    public class ConversationManager : CloudSearchManager<ConversationMessage>
    {
        private static string IndexNameWildcard = "conversations-*";

        /// <summary>
        /// Gets the messages for a conversation, given a conversation ID and a date range
        /// </summary>
        /// <param name="conversationID">The conversation ID for the messages (friend pair, group ID, etc)</param>
        /// <param name="startDate"></param>
        /// <param name="endDate"></param>
        /// <param name="pageSize">How many documents to return per page</param>
        /// <param name="currentPage">The current page</param>
        /// <returns></returns>
        public static ConversationMessage[] GetMessagesByConversationAndDateRangeDescending(string conversationID, DateTime startDate, DateTime endDate, int pageSize = 100)
        {

            if (startDate < ConversationConstants.DateOfInception)
            {
                startDate = ConversationConstants.DateOfInception;
            }

            if (endDate < startDate)
            {
                endDate = startDate.AddDays(1);
            }

            var monthsBetween = TimeSeriesIndexing.GetMonthsBetween(startDate, endDate);
            var indexNames = monthsBetween.Select(p => TimeSeriesIndexing.GetIndexName(IndexTypeName, p.Year, p.Month)).ToArray();

            var client = GetClient();

            var resp = client.Search<ConversationMessage>(s => s.Filter(q =>
                    q.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, conversationID)))                 
                 && q.Bool(b => b.Must(qd => qd.Range(r => r.OnField(f => f.Timestamp).GreaterOrEquals(startDate.ToEpochMilliseconds()).Lower(endDate.ToEpochMilliseconds()))))
                )
                .Indices(indexNames)
                .Routing(conversationID)
                .Size(pageSize)
                .SortDescending(p => p.Timestamp));                

            if (!resp.ConnectionStatus.Success)
            {
                Logger.Warn("Unable to retrieve conversation messages: " + resp.ConnectionStatus.HttpStatusCode,
                    new { conversationID, startDate, endDate, indexNames });
                return null;
            }

            var requestParams = Encoding.Default.GetString(resp.RequestInformation.Request);
            Logger.Trace("Request Url: " + resp.RequestInformation.RequestUrl, new { Params = requestParams });
            return resp.Documents.ToArray();
        }

        /// <summary>
        /// Gets the messages for a conversation, given a conversation ID and a date range
        /// </summary>
        /// <param name="conversationID">The conversation ID for the messages (friend pair, group ID, etc)</param>
        /// <param name="startDate"></param>
        /// <param name="endDate"></param>
        /// <param name="pageSize">How many documents to return per page</param>
        /// <param name="currentPage">The current page</param>
        /// <returns></returns>
        public static ConversationMessage[] GetMessagesByConversationAndDateRangeAscending(string conversationID, DateTime startDate, DateTime endDate, int pageSize = 100)
        {

            if (startDate < ConversationConstants.DateOfInception)
            {
                startDate = ConversationConstants.DateOfInception;
            }

            if (endDate < startDate)
            {
                endDate = startDate.AddDays(1);
            }

            var monthsBetween = TimeSeriesIndexing.GetMonthsBetween(startDate, endDate);
            var indexNames = monthsBetween.Select(p => TimeSeriesIndexing.GetIndexName(IndexTypeName, p.Year, p.Month)).ToArray();

            var client = GetClient();

            var resp = client.Search<ConversationMessage>(s => s.Filter(q =>
                    q.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, conversationID)))
                 && q.Bool(b => b.Must(qd => qd.Range(r => r.OnField(f => f.Timestamp).Greater(startDate.ToEpochMilliseconds()).LowerOrEquals(endDate.ToEpochMilliseconds()))))
                )
                .Indices(indexNames)
                .Routing(conversationID)
                .Size(pageSize)
                .SortAscending(p => p.Timestamp));

            if (!resp.ConnectionStatus.Success)
            {
                Logger.Warn("Unable to retrieve conversation messages: " + resp.ConnectionStatus.HttpStatusCode,
                    new { conversationID, startDate, endDate, indexNames });
                return null;
            }

            var requestParams = Encoding.Default.GetString(resp.RequestInformation.Request);
            Logger.Trace("Request Url: " + resp.RequestInformation.RequestUrl, new { Params = requestParams });
            return resp.Documents.ToArray();
        }

        private static void DumpElasticError(ISearchResponse<ConversationMessage> resp)
        {
            var requestParams = Encoding.Default.GetString(resp.RequestInformation.Request);
            var responseString = Encoding.Default.GetString(resp.RequestInformation.ResponseRaw);
            Logger.Trace("Request Url: " + resp.RequestInformation.RequestUrl, new { Params = requestParams });
        }

        /// <summary>
        /// Gets the messages for a conversation, given a conversation ID and a date range
        /// </summary>        
        /// <returns></returns>
        public static ConversationMessage[] GetJumpMessages(string conversationID, DateTime referenceMessageTimestamp, int pageSizeBefore, int pageSizeAfter)
        {

            if (referenceMessageTimestamp < ConversationConstants.DateOfInception)
            {
                throw new InvalidOperationException();
            }
            
            var client = GetClient();

            var beforeResp = client.Search<ConversationMessage>(s => s.Filter(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, conversationID)))
                && q.Bool(b => b.Must(qd => qd.Range(r => r.OnField(f => f.Timestamp).Lower(referenceMessageTimestamp.ToEpochMilliseconds()))))
                )
                .AllIndices()
                .Routing(conversationID)
                .Size(pageSizeBefore)
                .SortDescending(p => p.Timestamp));

            if (!beforeResp.ConnectionStatus.Success)
            {
                DumpElasticError(beforeResp);
                Logger.Warn("Unable to retrieve conversation messages: " + beforeResp.ConnectionStatus.HttpStatusCode);             
                return null;
            }

             var afterResp = client.Search<ConversationMessage>(s => s.Filter(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, conversationID)))
                && q.Bool(b => b.Must(qd => qd.Range(r => r.OnField(f => f.Timestamp).GreaterOrEquals(referenceMessageTimestamp.ToEpochMilliseconds()))))
                )
                .AllIndices()
                .Routing(conversationID)
                .Size(pageSizeAfter + 1)
                .SortAscending(p => p.Timestamp));

            if (!afterResp.ConnectionStatus.Success)
            {
                Logger.Warn("Unable to retrieve conversation messages: " + afterResp.ConnectionStatus.HttpStatusCode);
                return null;
            }

            return beforeResp.Documents.OrderBy(p => p.Timestamp).Concat(afterResp.Documents).ToArray();
        }


        public static ISearchResponse<ConversationMessage> Search(ConversationSearch search)
        {
            if (search.EndDate.HasValue && !search.StartDate.HasValue)
            {
                throw new InvalidOperationException("An start date must be specified if an end date is specified.");
            }

            string[] indexNames = null;

            if (search.StartDate.HasValue)
            {
                // Ensure the start date is not before the date of inception
                if (search.StartDate.Value < ConversationConstants.DateOfInception)
                {
                    search.StartDate = ConversationConstants.DateOfInception;
                }

                var now = DateTime.UtcNow.AddMinutes(5);
                var endDate = search.EndDate ?? now;

                // Ensure the end date is after the start date
                if (endDate < search.StartDate.Value)
                {
                    throw new InvalidOperationException("The end date must be after the start date. ");
                }

                if (endDate > now)
                {
                    endDate = now;
                }

                var monthsBetween = TimeSeriesIndexing.GetMonthsBetween(search.StartDate.Value, endDate);
                indexNames = monthsBetween.Select(p => TimeSeriesIndexing.GetIndexName(IndexTypeName, p.Year, p.Month)).ToArray();
            }

            var queries = new List<QueryContainer>();
            var filters = new List<FilterContainer>();
            var shouldFilters = new List<FilterContainer>();

            // Always exclude deleted content
            filters.Add(new FilterDescriptor<ConversationMessage>().Term(t => t.IsDeleted, false));

            // Keyword
            if (!string.IsNullOrWhiteSpace(search.Keyword))
            {
                queries.Add(new QueryDescriptor<ConversationMessage>().Match(t => t.OnField(f => f.Body).Query(search.Keyword)));
            }

            // ConversationID or RootConversationID
            if (!string.IsNullOrWhiteSpace(search.ConversationID))
            {
                filters.Add(new FilterDescriptor<ConversationMessage>().Term(t => t.ConversationID, search.ConversationID));
            }
            else if (!string.IsNullOrEmpty(search.RootConversationID))
            {
                filters.Add(new FilterDescriptor<ConversationMessage>().Term(t => t.RootConversationID, search.RootConversationID));
            }

            // Date Range
            if (search.StartDate.HasValue)
            {
                if (!search.EndDate.HasValue)
                {
                    search.EndDate = DateTime.UtcNow.AddDays(1);
                }

                filters.Add(new FilterDescriptor<ConversationMessage>().Range(r => r
                    .OnField(t => t.Timestamp)
                        .GreaterOrEquals(search.StartDate.Value.ToEpochMilliseconds())
                        .LowerOrEquals(search.EndDate.Value.ToEpochMilliseconds())
                    )
                );
            }

            // Mentioned Users
            if (search.MentionedUserID > 0)
            {
                filters.Add(new FilterDescriptor<ConversationMessage>().Term(t => t.Mentions, search.MentionedUserID));
            }

            // Liked Users
            if (search.LikedUserID > 0)
            {
                filters.Add(new FilterDescriptor<ConversationMessage>().Term(t => t.LikeUserIDs, search.LikedUserID));
            }

            // Sender User
            if (search.SenderUserID > 0)
            {
                filters.Add(new FilterDescriptor<ConversationMessage>().Term(t => t.SenderID, search.SenderUserID));
            }

            if (search.Page == 0)
            {
                search.Page = 1;
            }

            var skip = (search.Page - 1) * search.PerPage;


            // Content Tags
            if (search.ContentTags != null && search.ContentTags.Any())
            {
                foreach (var contentTag in search.ContentTags)
                {
                    shouldFilters.Add(new FilterDescriptor<ConversationMessage>().Term(t => t.ContentTags, contentTag));
                }
            }

            if (search.PerPage < 1 || search.PerPage > 100)
            {
                search.PerPage = 10;
            }

            if (string.IsNullOrWhiteSpace(search.HighlightTagName))
            {
                search.HighlightTagName = "em";
            }

            // If no index name is specified, use the wildcard
            if (indexNames == null)
            {
                indexNames = new[] { IndexNameWildcard };
            }

            var client = GetClient();
            var resp = client.Search<ConversationMessage>(s => s
                .Query(q => q
                    .Bool(b => b
                        .Must(queries.ToArray())))
                .Filter(f => f
                    .Bool(b =>
                        b.Must(filters.ToArray()))
                    && f.Bool(b =>
                        b.Should(shouldFilters.ToArray())))

                .Indices(indexNames)
                .Skip(skip)
                .Size(search.PerPage)
                .Routing(search.ConversationID)
                .SortDescending(p => p.Timestamp)
                .Highlight(h => h
                    .OnFields(f => f
                        .OnField(e => e.Body)
                        .PreTags("<" + search.HighlightTagName + ">")
                        .PostTags("</" + search.HighlightTagName + ">")
                    )
                )
            );

            var requestParams = Encoding.Default.GetString(resp.RequestInformation.Request);

            if (!resp.ConnectionStatus.Success)
            {
                Logger.Warn("Unable to perform search: " + resp.ConnectionStatus.HttpStatusCode, new {requestParams, resp.RequestInformation.RequestUrl});
            }

            return resp;
        }

        
        public static Dictionary<string, ConversationMessage[]> MultiGetUnreadMessages(GroupMember[] groups, int pageSize = 100)
        {            
            var client = GetClient();

            var search = client.MultiSearch(ms =>
            {
                var baseSearch = ms;

                foreach (var groupMember in groups)
                {
                    // Skip anything without messages
                    if (groupMember.DateMessaged.Year < 2000)
                    {
                        continue;
                    }

                    // Skip anything that is off purely by ticks
                    if (groupMember.DateMessaged.Subtract(groupMember.DateRead).TotalMilliseconds < 1)
                    {
                        continue;
                    }

                    var startDate = groupMember.DateJoined > groupMember.DateRead ? groupMember.DateJoined : groupMember.DateRead;
                    var endDate = DateTime.UtcNow.AddSeconds(10);
                    var monthsBetween = TimeSeriesIndexing.GetMonthsBetween(startDate, endDate);
                    var indexNames = monthsBetween.Select(p => TimeSeriesIndexing.GetIndexName(IndexTypeName, p.Year, p.Month)).ToArray();

                    baseSearch = ms.Search<ConversationMessage>(groupMember.GroupID.ToString(), s => s.Filter(f =>
                            f.Bool(b => b.Must(qd => qd.Term(t => t.ConversationID, groupMember.GroupID.ToString())))
                            && f.Bool(b => b.Must(qd => qd.Range(r => r.OnField(t => t.Timestamp).GreaterOrEquals(startDate.ToEpochMilliseconds()).LowerOrEquals(endDate.ToEpochMilliseconds()))))
                            )
                            .Indices(indexNames)
                            .Size(pageSize)
                            .Routing(groupMember.GroupID.ToString())
                            .SortDescending(p => p.Timestamp)
                        );
                }

                return baseSearch;

            });

            var dict = new Dictionary<string, ConversationMessage[]>();
            var errorCount = 0;

            foreach (var group in groups)
            {
                var resp = search.GetResponse<ConversationMessage>(group.GroupID.ToString());
                if (resp == null)
                {
                    continue;
                }

                if (!resp.ConnectionStatus.Success)
                {
                    ++errorCount;
                    continue;
                }

                dict.Add(group.GroupID.ToString(), resp.Documents.ToArray());
            }

            if (errorCount > 0)
            {
                Logger.Warn("MultiGetUnreadMessages resulted in " + errorCount + " errors of " + groups.Length);
            }

            return dict;
        }

        public static void Index(ConversationMessage message)
        {
            var client = GetClient();
            var timestamp = message.Timestamp.FromEpochMilliconds();
            var indexName = TimeSeriesIndexing.GetIndexName(IndexTypeName, timestamp.Year, timestamp.Month);
            var resp = client.Index(message, idx => idx.Index(indexName).Routing(message.ConversationID));
            if (!resp.ConnectionStatus.Success)
            {
                throw new Exception("Failed to index message: " + resp.ConnectionStatus.HttpStatusCode);
            }
        }

        public static int GetFriendID(int userID, string conversationID)
        {
            var splits = conversationID.Split(':');
            if (splits.Length != 2)
            {
                return 0;
            }

            try
            {
                var ids = Array.ConvertAll(splits, int.Parse);

                if (ids[0] == userID)
                {
                    return ids[1];
                }
                else if (ids[1] == userID)
                {
                    return ids[0];
                }
                else
                {
                    return 0;
                }                
            }
            catch
            {
                return 0;
            }
        }
        
        public static Tuple<Guid, int, int> GetPrivateMessageIDs(string conversationID, int senderID)
        {
            var splits = conversationID.Split(':');
            if (splits.Length != 3)
            {
                return null;
            }

            try
            {
                Guid groupID;

                if (!Guid.TryParse(splits[0], out groupID))
                {
                    return null;
                }

                int userA;
                if (!int.TryParse(splits[1], out userA))
                {
                    return null;
                }


                int userB;
                if (!int.TryParse(splits[2], out userB))
                {
                    return null;
                }


                return new Tuple<Guid, int, int>(groupID, senderID, senderID == userA ? userB : userA);
            }
            catch
            {
                return null;
            }
        }

       
        public static ConversationMessage GetMessageByID(string conversationID, string messageID, DateTime messageTimestamp)
        {
            var indexName = TimeSeriesIndexing.GetIndexName(IndexTypeName, messageTimestamp.Year, messageTimestamp.Month);
            var client = GetClient();
            var resp = client.Get<ConversationMessage>(g => g
                                                       .Id(messageID.ToString())
                                                       .Index(indexName)
                                                       .Routing(conversationID)
                                                       );

            if (!resp.Found || resp.Source == null)
            {
                return null;
            }

            if (resp.Source.ConversationID != conversationID)
            {
                return null;
            }

            return resp.Source;
        }

        public static ConversationMessage FindMessageByID(string conversationID, string messageID)
        {            
            var client = GetClient();
            var resp = client.Get<ConversationMessage>(g => g
                                                       .Id(messageID.ToString())
                                                       .Index<ConversationMessage>()
                                                       .Routing(conversationID)
                                                       );

            if (!resp.Found || resp.Source == null)
            {
                return null;
            }

            if (resp.Source.ConversationID != conversationID)
            {
                return null;
            }

            return resp.Source;
        }

        public static IConversationContainer GetConversationContainer(int requestingUserID, string conversationID, bool createIfNull = false)
        {
            if (string.IsNullOrWhiteSpace(conversationID))
            {
                return null;
            }

            // This is a group conversation
            Guid groupID;
            if (Guid.TryParse(conversationID, out groupID))
            {
                return Group.GetByID(Guid.Parse(conversationID));
            }

            // Otherwise, it is a private conversation
            var otherUserID = GetFriendID(requestingUserID, conversationID);

            if (otherUserID > 0)
            { 
                return createIfNull ? PrivateConversation.GetOrCreateByUserIDAndOtherUserID(DateTime.UtcNow, requestingUserID, otherUserID) : PrivateConversation.GetByUserIDAndOtherUserID(requestingUserID, otherUserID);
            }

            var privateMessageIDs = GetPrivateMessageIDs(conversationID, requestingUserID);
            if (privateMessageIDs != null)
            {
                return GroupPrivateConversation.GetOrCreateBySenderIDAndRecipientIDAndGroupID(privateMessageIDs.Item2, privateMessageIDs.Item3, privateMessageIDs.Item1);
            }

            return null;
        }

        public static void EditAttachment(string conversationID, string messageID, DateTime messageTimestamp, int editUserID, DateTime editTimestamp, string editUsername, ConversationAttachment attachment)
        {
            var client = GetClient();

            var resp = client.Update<ConversationMessage>(u => u
                .Id(messageID.ToString())
                .Index(TimeSeriesIndexing.GetIndexName(IndexTypeName, messageTimestamp.Year, messageTimestamp.Month))
                .Routing(conversationID)
                .Script("ctx._source.attachments[0] = attachment;" +
                        "ctx._source.editedUserId = editedUserId;" +
                        "ctx._source.editedTimestamp = editedTimestamp;" +
                        "ctx._source.editedUsername = editedUsername;"
                )
                .Params(p => p
                    .Add("attachment", attachment)
                    .Add("editedUserId", editUserID)
                    .Add("editedTimestamp", editTimestamp.ToEpochMilliseconds())
                    .Add("editedUsername", editUsername)
                )
                .RetryOnConflict(3)
                .Refresh()
            );

            if (!resp.IsValid)
            {
                if (resp.ConnectionStatus.HttpStatusCode == 404)
                {
                    Logger.Warn("Unable to edit attachment. It was not found.", resp.ConnectionStatus);
                }
                else
                {
                    Logger.Error("Failed to edit attachment", resp.ConnectionStatus);
                }
            }


        }

        public static void DeleteMessage(string conversationID, string messageID, DateTime messageTimestamp, int userID, DateTime timestamp, string username)
        {
            var client = GetClient();

            var resp = client.Update<ConversationMessage>(u => u
                .Id(messageID.ToString())
                .Index(TimeSeriesIndexing.GetIndexName(IndexTypeName, messageTimestamp.Year, messageTimestamp.Month))
                .Routing(conversationID)
                .Script("ctx._source.isDeleted = isDeleted;" +
                        "ctx._source.deletedUserId = deletedUserId;" +
                        "ctx._source.deletedTimestamp = deletedTimestamp;" +
                        "ctx._source.deletedUsername = deletedUsername;"
                )
                .Params(p => p
                    .Add("isDeleted", true)
                    .Add("deletedUserId", userID)
                    .Add("deletedTimestamp", timestamp.ToEpochMilliseconds())
                    .Add("deletedUsername", username)
                )
                .RetryOnConflict(3)
                .Refresh()
            );

            if (!resp.IsValid)
            {
                if (resp.ConnectionStatus.HttpStatusCode == 404)
                {
                    Logger.Warn("Unable to delete attachment. It was not found.", resp.ConnectionStatus);
                }
                else
                {
                    Logger.Error("Failed to delete attachment", resp.ConnectionStatus);
                }

            }

        }

        public static IEnumerable<IEnumerable<ConversationMessage>> ScrollByRootConversationAndUser(string conversationID, int userID, DateTime startDate, DateTime endDate)
        {
            var client = GetClient();

            if (startDate < ConversationConstants.DateOfInception)
            {
                startDate = ConversationConstants.DateOfInception;
            }

            var now = DateTime.UtcNow;
            if (endDate > now)
            {
                endDate = now;
            }

            var months = TimeSeriesIndexing.GetMonthsBetween(startDate, endDate);
            var startTime = startDate.ToEpochMilliseconds();
            var endTime = endDate.ToEpochMilliseconds();

            var filters = new List<FilterContainer>
            {
                new FilterDescriptor<ConversationMessage>().Term(t => t.RootConversationID, conversationID),
                new FilterDescriptor<ConversationMessage>().Term(t => t.SenderID, userID),
                new FilterDescriptor<ConversationMessage>().Range(t => t.OnField(o => o.Timestamp).GreaterOrEquals(startTime).LowerOrEquals(endTime)),
                new FilterDescriptor<ConversationMessage>().Term(t => t.IsDeleted, false)
            };

            var searchResult = client.Search<ConversationMessage>(s => s
                .Indices(months.Select(m => TimeSeriesIndexing.GetIndexName(IndexTypeName, m.Year, m.Month)))
                .Size(1000)
                .Scroll("10s")
                .Sort(p=>p.OnField(f=>f.Timestamp).Order(SortOrder.Descending))
                .Filter(f => f.Bool(b => b.Must(filters.ToArray()))));
            while (searchResult.Documents.Any())
            {
                var scrollID = searchResult.ScrollId;
                yield return searchResult.Documents;
                searchResult = client.Scroll<ConversationMessage>(s => s.Scroll("10s").ScrollId(scrollID));
            }
        }
        
        public static void BulkDelete(ConversationMessage[] messagesToDelete, int requestorUserID, string requestorUsername, DateTime requestDate)
        {
            var client = GetClient();

            // Delete the found messages
            var deleteTimestamp = requestDate.ToEpochMilliseconds();
            
            var script = "ctx._source.isDeleted = isDeleted;" +
                         "ctx._source.deletedUserId = deletedUserId;" +
                         "ctx._source.deletedTimestamp = deletedTimestamp;" +
                         "ctx._source.deletedUsername = deletedUsername;";
            
            var parameters = new Dictionary<string, object>
            {
                {"isDeleted", true},
                {"deletedUserId", requestorUserID},
                {"deletedTimestamp", deleteTimestamp},
                {"deletedUsername", requestorUsername}
            };

            foreach (var set in messagesToDelete.InSetsOf(1000))
            {
                var bulkOperations = new List<IBulkOperation>();
                foreach (var message in set)
                {
                    message.IsDeleted = true;
                    message.DeletedUserID = requestorUserID;
                    message.DeletedUsername = requestorUsername;
                    message.DeletedTimestamp = deleteTimestamp;

                    bulkOperations.Add(new BulkUpdateOperation<ConversationMessage, ConversationMessage>(message.ID)
                    {
                        Routing = message.ConversationID,
                        Index = TimeSeriesIndexing.GetIndexName(IndexTypeName, message.Timestamp.FromEpochMilliconds()),
                        Script = script,
                        Params = parameters
                    });
                }

                var bulkRequest = new BulkRequest
                {
                    Operations = bulkOperations.ToArray(),
                    Refresh = true
                };
                client.Bulk(bulkRequest);
            }
        }

        public static void EditMessage(string conversationID, string messageID, DateTime messageTimestamp, int editUserID, DateTime editTimestamp, string editUsername, string body, int[] mentions, ConversationMessageEmoteSubstitution[] emoteSubstitutions)
        {
            var client = GetClient();

            var resp = client.Update<ConversationMessage>(u => u
                   .Id(messageID.ToString())
                   .Index(TimeSeriesIndexing.GetIndexName(IndexTypeName, messageTimestamp.Year, messageTimestamp.Month))
                   .Routing(conversationID)
                   .Script("ctx._source.body = body;" +
                           "ctx._source.editedUserId = editedUserId;" +
                            "ctx._source.editedTimestamp = editedTimestamp;" +
                            "ctx._source.editedUsername = editedUsername;" +
                            "ctx._source.mentions = mentions;" +
                            "ctx._source.emoteSubstitutions = emoteSubstitutions")
                   .Params(p => p
                        .Add("body", body)
                        .Add("editedUserId", editUserID)
                        .Add("editedTimestamp", editTimestamp.ToEpochMilliseconds())
                        .Add("editedUsername", editUsername)
                        .Add("mentions", mentions)
                        .Add("emoteSubstitutions", emoteSubstitutions)
                  )
                  .RetryOnConflict(3)
                  .Refresh()
            );

            if (!resp.IsValid)
            {
                if (resp.ConnectionStatus.HttpStatusCode == 404)
                {
                    Logger.Warn("Unable to edit message It was not found.", resp.ConnectionStatus);
                }
                else
                {
                    Logger.Error("Failed to edit message", resp.ConnectionStatus);
                }

            }
        }

        public static void LikeMessage(string conversationID, string messageID, DateTime messageTimestamp, int userID, string username)
        {
            var client = GetClient();
            var resp = client.Update<ConversationMessage>(u => u
                .Id(messageID.ToString())
                .Index(TimeSeriesIndexing.GetIndexName(IndexTypeName, messageTimestamp.Year, messageTimestamp.Month))
                .Routing(conversationID)
                .Script("ctx._source.likeUserIds += userID;" +
                        "ctx._source.likeUsernames += username;" +
                        "ctx._source.likeCount += likeCount;")
                .Params(p => p
                    .Add("userID", userID)
                    .Add("username", username)
                    .Add("likeCount", 1)
                )
                .RetryOnConflict(3)
                .Refresh()
            );

            if (!resp.IsValid)
            {
                if (resp.ConnectionStatus.HttpStatusCode == 404)
                {
                    if (messageTimestamp >= ConversationConstants.DateOfNewBackend)
                    {
                        Logger.Warn("Unable to like message It was not found.", resp.ConnectionStatus);    
                    }
                    
                }
                else
                {
                    Logger.Error("Failed to like message", resp.ConnectionStatus);
                }
            }
        }

        public static void UnlikeMessage(string conversationID, string messageID, DateTime messageTimestamp, int userID, string username)
        {
            var client = GetClient();
            var resp = client.Update<ConversationMessage>(u => u
                .Id(messageID.ToString())
                .Index(TimeSeriesIndexing.GetIndexName(IndexTypeName, messageTimestamp.Year, messageTimestamp.Month))
                .Routing(conversationID)
                .Script("ctx._source.likeUserIds -= userID;" +
                        "ctx._source.likeUsernames -= username;" +
                        "ctx._source.likeCount -= likeCount;")
                .Params(p => p
                    .Add("userID", userID)
                    .Add("username", username)
                    .Add("likeCount", 1)
                )
                .RetryOnConflict(3)
                .Refresh()
            );

            if (!resp.IsValid)
            {
                if (resp.ConnectionStatus.HttpStatusCode == 404)
                {
                    if (messageTimestamp >= ConversationConstants.DateOfNewBackend)
                    {
                        Logger.Warn("Unable to unlike message It was not found.", resp.ConnectionStatus);
                    }                    
                }
                else
                {
                    Logger.Error("Failed to unlike message", resp.ConnectionStatus);
                }
            }

        }

        public static void UpdateMessageLikes(string conversationID, string messageID, DateTime messageTimestamp, int likeCount, IEnumerable<int> likeUserIDs, IEnumerable<string> likeUsernames)
        {
            var client = GetClient();
            var resp = client.Update<ConversationMessage>(u => u
                .Id(messageID.ToString())
                .Index(TimeSeriesIndexing.GetIndexName(IndexTypeName, messageTimestamp.Year, messageTimestamp.Month))
                .Routing(conversationID)
                .Script("ctx._source.likeUserIds = likeUserIds;" +
                        "ctx._source.likeUsernames = likeUsernames;" +
                        "ctx._source.likeCount = likeCount;")
                .Params(p => p
                    .Add("likeUserIds", likeUserIDs.ToArray())
                    .Add("likeUsernames", likeUsernames.ToArray())
                    .Add("likeCount", likeCount)
                )
                .RetryOnConflict(3)
                .Refresh()
            );

            if (!resp.IsValid)
            {
                if (resp.ConnectionStatus.HttpStatusCode == 404)
                {
                    Logger.Warn("Unable to update message likes. It was not found.", resp.ConnectionStatus);
                }
                else
                {
                    Logger.Error("Failed to update message likes.", resp.ConnectionStatus);
                }
            }
        }        
    }
}
