﻿using Curse.CloudSearch;
using Nest;
using System;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;

namespace Curse.Friends.Data.Messaging
{
    [CloudSearchModel(UseDefaultIndex = false, IndexTypeName = "conversations")]
    [ElasticType(Name = "conversationMessage", IdProperty = "id")]
    public class ConversationMessage
    {
        public const int MaxLikeUsernames = 10;

        /// A unique GUID formatted string to uniquely identify this message        
        [ElasticProperty(Name = "id", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public string ID
        {
            get;
            set;
        }

        /// A GUID formatted string for which conversation this belongs to
        [ElasticProperty(Name = "conversationId", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public string ConversationID
        {
            get;
            set;
        }

        /// The type of conversation (maps to ConversationType enum)
        [ElasticProperty(Name = "conversationType", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int ConversationType
        {
            get;
            set;
        }

        /// A GUID formatted string for which root conversation this belongs to. This permits searching for all messages within a server, for example.
        [ElasticProperty(Name = "rootConversationId", Type = FieldType.String, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public string RootConversationID
        {
            get;
            set;
        }

        /// <summary>
        /// The epoch formatted timestamp, in millisconds.
        /// </summary>
        [ElasticProperty(Name = "timestamp", Type = FieldType.Long, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public long Timestamp
        {
            get;
            set;
        }

        /// <summary>
        /// The user ID of the sender
        /// </summary>        
        [ElasticProperty(Name = "senderId", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int SenderID { get; set; }

        /// <summary>
        /// The title of the sender (in case the key gets orphaned on the client side)
        /// </summary>
        [ElasticProperty(Name = "senderName", Type = FieldType.String, Store = true, Index = FieldIndexOption.No)]
        public string SenderName { get; set; }

        /// <summary>
        /// The username of the sender (in case the key gets orphaned on the client side)
        /// </summary>
        [ElasticProperty(Name = "senderUsername", Type = FieldType.String, Store = true, Index = FieldIndexOption.No)]
        public string SenderUsername { get; set; }

        /// <summary>
        /// The displayName of the sender (in case the key gets orphaned on the client side)
        /// </summary>
        [ElasticProperty(Name = "senderDisplayName", Type = FieldType.String, Store = true, Index = FieldIndexOption.No)]
        public string SenderDisplayName { get; set; }

        /// <summary>
        /// The effective permissions of the sender  (at the time of the post)
        /// </summary>
        [ElasticProperty(Name = "senderPermissions", Type = FieldType.Long, Store = true, Index = FieldIndexOption.No)]
        public long SenderPermissions { get; set; }

        /// <summary>
        /// The roles of the sender (at the time of the post)
        /// </summary>
        [ElasticProperty(Name = "senderRoles", Store = true, Index = FieldIndexOption.No)]
        public int[] SenderRoles { get; set; }

        /// <summary>
        /// The roles of the sender (at the time of the post)
        /// </summary>
        [ElasticProperty(Name = "senderVanityRole", Store = true, Index = FieldIndexOption.No)]
        public int SenderVanityRole { get; set; }

        /// <summary>
        /// The userIDs of anyone mentioned in this message
        /// </summary>
        [ElasticProperty(Name = "mentions", Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int[] Mentions { get; set; }

        /// <summary>
        /// The user ID of the recipient (this can be 0, depending on the conversation type)
        /// </summary>
        [ElasticProperty(Name = "recipientId", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int RecipientID { get; set; }

        /// <summary>
        ///  The body of the message (serialized to JSON)
        /// </summary>
        [ElasticProperty(Name = "body", Type = FieldType.String, Store = true, Index = FieldIndexOption.Analyzed)]
        public string Body { get; set; }

        /// <summary>
        /// Whether or not the message is deleted
        /// </summary>
        [ElasticProperty(Name = "isDeleted", Type = FieldType.Boolean, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public bool IsDeleted { get; set; }

        /// <summary>
        /// The date the message was deleted
        /// </summary>
        [ElasticProperty(Name = "deletedTimestamp", Type = FieldType.Long, Store = true, Index = FieldIndexOption.No)]
        public long DeletedTimestamp { get; set; }

        /// <summary>
        /// The user ID of the person that deleted the message
        /// </summary>
        [ElasticProperty(Name = "deletedUserId", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.No)]
        public int DeletedUserID { get; set; }

        /// <summary>
        /// The username of the person that deleted the message
        /// </summary>
        [ElasticProperty(Name = "deletedUsername", Type = FieldType.String, Store = true, Index = FieldIndexOption.No)]
        public string DeletedUsername { get; set; }

        /// <summary>
        /// The user ID of the person that edited the message
        /// </summary>
        [ElasticProperty(Name = "editedUserId", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.No)]
        public int EditedUserID { get; set; }

        /// <summary>
        /// The date the message was edit
        /// </summary>
        [ElasticProperty(Name = "editedTimestamp", Type = FieldType.Long, Store = true, Index = FieldIndexOption.No)]
        public long EditedTimestamp { get; set; }

        /// <summary>
        /// The username of the person that deleted the message
        /// </summary>
        [ElasticProperty(Name = "editedUsername", Type = FieldType.String, Store = true, Index = FieldIndexOption.No)]
        public string EditedUsername { get; set; }

        /// <summary>
        /// The number of likes the mssage has
        /// </summary>
        [ElasticProperty(Name = "likeCount", Type = FieldType.Integer, Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int LikeCount { get; set; }

        /// <summary>
        /// The IDs of all users that have liked this message
        /// </summary>
        [ElasticProperty(Name = "likeUserIds", Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int[] LikeUserIDs { get; set; }

        ///// <summary>
        ///// The names of all users that have liked this message
        ///// </summary>
        [ElasticProperty(Name = "likeUsernames", Store = true, Index = FieldIndexOption.No)]
        public string[] LikeUsernames { get; set; }

        /// <summary>
        /// The content type tags associates with this message
        /// </summary>
        [ElasticProperty(Name = "contentTags", Store = true, Index = FieldIndexOption.NotAnalyzed)]
        public int[] ContentTags { get; set; }


        /// <summary>
        /// The content type tags associates with this message
        /// </summary>
        [ElasticProperty(Name = "attachments", Type = FieldType.Object, Store = true, Index = FieldIndexOption.No)]
        public ConversationAttachment[] Attachments
        {
            get;
            set;
        }

        [ElasticProperty(Name="emoteSubstitutions", Type = FieldType.Object)]
        public ConversationMessageEmoteSubstitution[] EmoteSubstitutions { get; set; }

        #region Twitch Chat Integration Shim (NOT PERSISTED)

        [ElasticProperty(OptOut = true)]
        public string ExternalChannelID { get; set; }

        [ElasticProperty(OptOut = true)]
        public string ExternalUserID { get; set; }

        [ElasticProperty(OptOut = true)]
        public string ExternalUsername { get; set; }

        [ElasticProperty(OptOut = true)]
        public string ExternalUserDisplayName { get; set; }

        [ElasticProperty(OptOut = true)]
        public string ExternalUserColor { get; set; }

        [ElasticProperty(OptOut = true)]
        public ConversationMessageBadge[] Badges { get; set; }

        [ElasticProperty(OptOut = true)]
        public int BitsUsed { get; set; }

        #endregion

        public static ConversationMessage Create(string conversationID, ConversationType conversationType, Guid messageID, string messageBody, DateTime timestamp, IUserIdentity sender, 
            int[] senderRoles = null, int senderVanityRole = 0, long senderPermissions = 0, int recipientID = 0, Guid? rootConversationID = null, Attachment attachment = null, 
            ConversationMessageEmoteSubstitution[] emoteOverrides = null)
        {
            var conversationMessage = new ConversationMessage
            {
                Body = messageBody,
                ConversationType = (int) conversationType,
                Mentions = ConversationParser.GetMentions(messageBody),
                ConversationID = conversationID,
                ContentTags = ConversationParser.GetContentTags(messageBody, attachment),
                ID = messageID.ToString(),
                SenderID = sender.UserID,
                RootConversationID = rootConversationID != null ? rootConversationID.ToString() : null,
                RecipientID = recipientID,
                LikeCount = 0,
                LikeUserIDs = new int[0],
                LikeUsernames = new string[0],
                Timestamp = timestamp.ToEpochMilliseconds(),
                SenderName = sender.GetTitleName(),
                SenderUsername = sender.Username,
                SenderDisplayName = sender.DisplayName,
                IsDeleted = false,
                SenderRoles = senderRoles ?? new int[0],
                SenderPermissions = senderPermissions,
                SenderVanityRole = senderVanityRole,
                EmoteSubstitutions = emoteOverrides ?? ConversationParser.ParseEmotes(sender.UserID, messageBody),
            };

            if (attachment != null)
            {                
                conversationMessage.Attachments = new[] { ConversationAttachment.FromAttachment(attachment, conversationID, attachment.UploaderUserID == sender.UserID ? sender.GetTitleName() : attachment.GetAuthorName()) };
            }
            else
            {
                conversationMessage.Attachments = new ConversationAttachment[0];
            }

            return conversationMessage;
        }

        public ConversationMessageNotification ToNotification(int currentUserID, Guid? clientID, ConversationType conversationType, ConversationNotificationType changeType, SpamConfidence spamConfidence = SpamConfidence.Unknown, UserStatistics senderUserStatistics = null)
        {           
            var contactID = ConversationID;

            if (conversationType == Enums.ConversationType.Friendship)
            {
                if (currentUserID <= 0)
                {
                    throw new ArgumentException("The current user ID must be supplied for friend to friend chat messages.");
                }

                contactID = ConversationManager.GetFriendID(currentUserID, ConversationID).ToString();
            }
            else if (conversationType == Enums.ConversationType.GroupPrivateConversation)
            {
                if (currentUserID <= 0)
                {
                    throw new ArgumentException("The current user ID must be supplied for group private messages.");
                }

                contactID = ConversationManager.GetPrivateMessageIDs(ConversationID, currentUserID).Item3.ToString();
            }

            return new ConversationMessageNotification
            {
                ConversationID = ConversationID,
                ConversationType = conversationType,
                NotificationType = changeType,
                ContactID = contactID,
                Attachments = IsDeleted || Attachments == null ? new AttachmentNotification[0] : Attachments.Select(p => p.ToNotification(ConversationID, ID)).ToArray(),
                SenderID = SenderID,
                Timestamp = Timestamp,
                Body = IsDeleted ? string.Empty : Body,
                RecipientID = RecipientID,
                ClientID = clientID.HasValue ? clientID.ToString() : null,
                IsDeleted = IsDeleted,
                RootConversationID = RootConversationID,
                LikeUsernames = IsDeleted ? new string[0] : LikeUsernames,
                ContentTags = ContentTags,
                Mentions = IsDeleted ? new int[0] : Mentions,
                SenderRoles = SenderRoles,
                SenderName = SenderName ?? senderUserStatistics?.GetTitleName(),
                SenderUsername = senderUserStatistics?.Username ?? SenderUsername,
                SenderDisplayName = senderUserStatistics?.DisplayName ?? SenderDisplayName,
                SenderVanityRole = SenderVanityRole,
                LikeCount = IsDeleted ? 0 : LikeCount,
                SenderPermissions = SenderPermissions,
                LikeUserIDs = IsDeleted ? new int[0] : LikeUserIDs,
                DeletedTimestamp = DeletedTimestamp,
                DeletedUserID = DeletedUserID,
                DeletedUsername = DeletedUsername,
                EditedTimestamp = EditedTimestamp,
                EditedUserID = EditedUserID,
                EditedUsername = EditedUsername,
                ServerID = ID,
                EmoteSubstitutions = EmoteSubstitutions == null ? new ConversationMessageEmoteSubstitutionNotification[0] : EmoteSubstitutions.Select(e => e.ToNotification()).ToArray(),
                SpamConfidence = spamConfidence,

                // Shim
                ExternalChannelID = ExternalChannelID,
                ExternalUserID = ExternalUserID,
                ExternalUsername = ExternalUsername,
                ExternalUserDisplayName = ExternalUserDisplayName,
                ExternalUserColor = ExternalUserColor,
                Badges = Badges == null ? new ConversationMessageBadgeContract[0] : Badges.Select(b => b.ToContract()).ToArray(),
                BitsUsed = BitsUsed,
            };
        }

        public void EditMessage(int editorUserID, string editorUsername, string body, long editTimestamp, int[] mentions, ConversationMessageEmoteSubstitution[] emoteSubstitutions = null)
        {
            EditedTimestamp = editTimestamp;
            EditedUsername = editorUsername;
            EditedUserID = editorUserID;
            Body = body;
            if (mentions != null)
            {
                Mentions = mentions;
            }

            // Use the original sender ID for emotes, not the editor
            EmoteSubstitutions = emoteSubstitutions ?? ConversationParser.ParseEmotes(SenderID, body);
        }

        public object GetLogData()
        {
            return new
            {
                ID,
                SenderID,
                SenderName,
                SenderUsername,
                SenderDisplayName,
                ConversationID,
                
            };
        }
    }
}
