﻿using System;
using Curse.Friends.Enums;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using Curse.Aerospike;

namespace Curse.Friends.Data.Messaging
{
    /// <summary>
    /// Examines the body of a message, and determines the nature of the content
    /// </summary>
    public static class ConversationParser
    {
        private static readonly Regex HostedImageEmbedRegex = new Regex(@"\bhttp(s)?://images.*?\.curseapp\.net/.*?\.(png|jpg|jpeg|gif)\b", RegexOptions.Compiled | RegexOptions.IgnoreCase);
        private static readonly Regex HostedFileRegex = new Regex(@"\bhttp(s)?://files.*?\.curseapp\.net/.*?\b", RegexOptions.Compiled);
        private static readonly Regex ImgurRegex = new Regex(@"\bhttp(s)?://((m\.)|((www)\.)|((i)\.))?imgur\.com/(a/)?[a-zA-Z0-9&]+((\.jpg)|(\.gif)|(\.gifv)|(\.png))?\b", RegexOptions.Compiled | RegexOptions.IgnoreCase);
        private static readonly Regex YouTubeRegex = new Regex(@"(^|\s)http(s)?\://(www\.)?(?:youtube\.com\/\S*(?:(?:\/e(?:mbed))?\/|watch\?(?:\S*?&?v\=))|youtu\.be\/)([a-zA-Z0-9_-]{6,11})($|\s)", RegexOptions.Compiled | RegexOptions.IgnoreCase);
        private static readonly Regex UrlRegex = new Regex(@"\b(www\.|(https|http|curse)+\:\/\/)[\w\-_]+(?:\.[\w\-_]+)+(?:[\w\-\.,@?^=%&;:/~\+#'!()]*[\w\-\@?^=%&;/~\+#'()])\b", RegexOptions.Compiled | RegexOptions.IgnoreCase);
        private static readonly Regex GiphyImages = new Regex(@"\bhttp\://giphy\.com/gifs/.*?\b", RegexOptions.Compiled | RegexOptions.IgnoreCase);
        
        public static bool IsImageEmbed(string body)
        {
            return HostedImageEmbedRegex.IsMatch(body) 
                || ImgurRegex.IsMatch(body)
                || GiphyImages.IsMatch(body);
        }

        public static bool IsVideoEmbed(string body)
        {
            return YouTubeRegex.IsMatch(body);
        }

        public static bool IsFileEmbed(string body)
        {
            return HostedFileRegex.IsMatch(body);
        }

        public static bool HasHyperlink(string body)
        {
            return UrlRegex.IsMatch(body);
        }

        public static int[] GetContentTags(string body, Attachment attachment = null)
        {
            var tags = new HashSet<ContentTag>();

            if (attachment != null && attachment.IsEmbed && attachment.IsImage)
            {               
                tags.Add(ContentTag.Image);
            }
            else if (IsImageEmbed(body))
            {
                tags.Add(ContentTag.Image);
            }
            else if (IsVideoEmbed(body))
            {
                tags.Add(ContentTag.Video);
            }
            else if (IsFileEmbed(body))
            {
                tags.Add(ContentTag.File);
            }
            else if (HasHyperlink(body))
            {
                tags.Add(ContentTag.Hyperlink);
            }

            return tags.Cast<int>().ToArray();
        }

        private static readonly Regex MentionsRegex = new Regex("(?<mentionStart>\\s@|^@)(?<userID>\\d+):(?<username>\\w+)(?:\\b|$)", RegexOptions.Compiled);

        public static int[] GetMentions(string body)
        {
            var matches = MentionsRegex.Matches(body);
            var userIDs = new HashSet<int>();
            foreach (Match match in matches)
            {
                int userID;
                if (!int.TryParse(match.Groups["userID"].Value, out userID))
                {
                    continue;
                }

                userIDs.Add(userID);
            }

            return userIDs.ToArray();
        }

        public static ConversationMessageEmoteSubstitution[] ParseEmotes(int senderID, string messageBody)
        {
            if (senderID <= 0)
            {
                return new ConversationMessageEmoteSubstitution[0];
            }

            var substitutions = new List<ConversationMessageEmoteSubstitution>();
            var userStats = UserStatistics.GetByUserOrDefault(senderID, true);
            userStats.EnsureEmotes(TimeSpan.FromHours(1));
            if (userStats.EmoteMap == null || userStats.EmoteMap.Count == 0)
            {
                return new ConversationMessageEmoteSubstitution[0];
            }

            var emotes = TwitchEmote.MultiGetLocal(new HashSet<int>(userStats.EmoteMap.Values).Select(id => new KeyInfo(id))).Where(e=>e.IsAlphanumeric).ToDictionary(e => e.Regex);

            int startIndex = 0;
            foreach (var part in messageBody.Split())
            {
                TwitchEmote emote;
                if (emotes.TryGetValue(part, out emote))
                {
                    substitutions.Add(new ConversationMessageEmoteSubstitution
                    {
                        MessageText = emote.Regex,
                        EmoteID = emote.EmoteID,
                        EmoteSet = emote.EmoticonSet,
                        EmoteHeight = emote.Height,
                        EmoteWidth = emote.Width,
                        StartIndex = startIndex,
                        EndIndex = startIndex + part.Length - 1,
                    });
                }
                startIndex += part.Length + 1;
            }
            return substitutions.ToArray();
        }
    }
}
