﻿using System.Collections.Generic;
using System.Net;
using Aerospike.Client;
using Curse.Aerospike;
using System;
using System.Linq;
using System.Linq.Expressions;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Newtonsoft.Json;
using Curse.Extensions;

namespace Curse.Friends.Data
{

    [TableDefinition(TableName = "ClientEndpoint", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class ClientEndpoint : BaseTable<ClientEndpoint>, IEndpointModel
    {
        public static readonly Guid TwitchMachineKey = new Guid("fda79511-6dee-4b31-bc56-1e2983a9850a");

        public const int MaxEndpoints = 50;

        public const int PruneThreshold = 10;

        [Column("UserID", KeyOrdinal = 1, IsIndexed = true)]
        public int UserID { get; set; }

        [Column("MachineKey", KeyOrdinal = 2)]
        public string MachineKey { get; set; }

        [Column("RegionID")]
        public int RegionID { get; set; }

        [Column("ServerName", IsIndexed = true)]
        public string ServerName { get; set; }

        [Column("SessionID")]
        public string SessionID { get; set; }

        [Column("SessionDate")]
        public DateTime SessionDate { get; set; }

        [Column("IsConnected")]
        public bool IsConnected { get; set; }

        [Column("ConnectionID")]
        public string ConnectionID { get; set; }

        [Column("ConnDate")]
        public DateTime ConnectedDate { get; set; }

        [Column("DiscDate")]
        public DateTime DisconnectedDate { get; set; }

        [Column("IsIdle")]
        public bool IsIdle { get; set; }

        [Column("HeartbeatDate")]
        public DateTime HeartbeatDate { get; set; }

        [Column("ClientVersion")]
        public string ClientVersion { get; set; }

        [Column("Platform")]
        public DevicePlatform Platform { get; set; }

        [Column("DeviceID", IsIndexed = true)]
        public string DeviceID { get; set; }

        [Column("LastSyncDate")]
        public DateTime LastSyncDate { get; set; }

        [Column("IPAddress")]
        public string IPAddress { get; set; }

        /// <summary>
        /// Apple PushKit token, used for mobile call notifications.
        /// </summary>
        [Column("PushKitToken", IsIndexed = true)]
        public string PushKitToken { get; set; }

        [Column("IsDeleted")]
        public bool IsDeleted { get; set; }

        [Column("CurrentGroup")]
        public Guid CurrentGroup { get; set; }

        [Column("CurrentGroupTS")]
        public long CurrentGroupTimestamp { get; set; }

        #region Presence

        [Column("PlayGameID")]
        public int CurrentlyPlayingGameID { get; set; }

        [Column("PlayGameState")]
        public int CurrentlyPlayingGameState { get; set; }

        [Column("PlayGameMsg")]
        public string CurrentlyPlayingGameStatusMessage { get; set; }

        [Column("PlayGameTime")]
        public DateTime CurrentlyPlayingGameTimestamp { get; set; }

        [Column("WatchingID")]
        public string CurrentlyWatchingChannelID { get; set; }

        [Column("IsCasting")]
        public bool IsBroadcasting { get; set; }

        [Column("CastingGameID")]
        public int CurrentlyBroadcastingGameID { get; set; }

        [Column("ActivityTime")]
        public object ActivityTimestamp { get; set; }

        #endregion

        public bool IsMobile
        {
            get
            {
                return Platform == DevicePlatform.iOS
                    || Platform == DevicePlatform.Android
                    || Platform == DevicePlatform.WindowsPhone;
            }
        }

        #region Client Capabilities

        /// <summary>
        /// All endpoints support whispering
        /// </summary>
        [JsonIgnore]
        public bool SupportsWhispers
        {
            get
            {
                return true;
            }
        }

        /// <summary>
        /// Only non-Twitch endpoints that are connected or mobile
        /// </summary>
        [JsonIgnore]
        public bool SupportsAudioCalls
        {
            get
            {
                if (Platform == DevicePlatform.Twitch)
                {
                    return false;
                }

                return IsDeliverable;
            }
        }

        /// <summary>
        /// Only non-Twitch endpoints that are connected or mobile
        /// </summary>
        [JsonIgnore]
        public bool SupportsVideoCalls
        {
            get
            {
                if (Platform == DevicePlatform.Twitch)
                {
                    return false;
                }

                return IsDeliverable;
            }
        }

        /// <summary>
        /// Only non-Twitch endpoints that are connected or mobile
        /// </summary>
        [JsonIgnore]
        public bool SupportsGroups
        {
            get
            {
                return true;
            }
        }

        /// <summary>
        /// Only non-Twitch endpoints that are connected or mobile
        /// </summary>
        [JsonIgnore]
        public bool SupportsServers
        {
            get
            {
                return Platform != DevicePlatform.Twitch;
            }
        }

        /// <summary>
        /// Only non-Twitch endpoints that are connected or mobile
        /// </summary>
        [JsonIgnore]
        public bool SupportsFileSharing
        {
            get
            {
                return Platform != DevicePlatform.Twitch;
            }
        }

        /// <summary>
        /// Only non-Twitch endpoints that are connected or mobile
        /// </summary>
        [JsonIgnore]
        public bool SupportsImageSharing
        {
            get { return SupportsWhispers; }
        }

        #endregion

        public static void UpdateSession(string sessionID, int userID, string machineKey)
        {
            var model = GetLocal(userID, machineKey);
            model.SessionID = sessionID;
            model.SessionDate = DateTime.UtcNow;
            model.Update(p => p.SessionID, p => p.SessionDate);
        }

        public void UpdateSession(string sessionID)
        {
            SessionID = sessionID;
            SessionDate = DateTime.UtcNow;
            Update(p => p.SessionID, p => p.SessionDate);
        }

        /// <summary>
        /// Updates the connection status and regionID in the ClientEndPoint.
        /// Also queueus notifications to workerserver to update the status to all active groups user belongs to
        /// </summary>
        /// 
        public static void SetConnected(int regionID, string serverName, int userID, int userRegionID, string machineKey,
            string connectionID, DateTime connectionTimestamp, Version clientVersion, IPAddress ipAddress)
        {
            var model = GetLocal(userID, machineKey);
            if (model == null)
            {
                return;
            }

            model.ServerName = serverName;
            model.RegionID = regionID;
            model.ConnectedDate = connectionTimestamp;
            model.ConnectionID = connectionID;
            model.DisconnectedDate = default(DateTime);
            model.IsConnected = true;
            model.IsIdle = false;
            model.HeartbeatDate = default(DateTime);
            model.ClientVersion = clientVersion.ToString();
            model.Update();

            ClientEndpointResolver.Create(userID, userRegionID);
        }

        /// <summary>
        /// Updates the connection status and regionID in user's ClientEndPoint
        /// Also queueus notifications to workerserver to update the status to all active groups user belongs to
        /// </summary>
        public static bool SetDisconnected(int userID, int userRegionID, string machineKey, string sessionID, string connectionID, out ClientEndpoint model, out string reason)
        {
            if (userID <= 0)
            {
                model = null;
                reason = "Invalid user ID supplied: " + userID;
                return false;
            }

            if (machineKey == null)
            {
                model = null;
                reason = "Invalid machine key supplied!";
                return false;
            }

            model = GetLocal(userID, machineKey);
            reason = null;

            if (model == null)
            {
                reason = "Endpoint could not be retrieved from the database!";
                return false;
            }

            return model.Disconnect(sessionID, connectionID, userRegionID, out reason);            
        }

        /// <summary>
        /// Updates the connection status and regionID in user's ClientEndPoint
        /// Also queueus notifications to workerserver to update the status to all active groups user belongs to
        /// </summary>
        public bool Disconnect(string sessionID, string connectionID, int userRegionID, out string reason)
        {
            reason = null;

            if (connectionID != null && ConnectionID != connectionID)
            {
                reason = "Connection IDs do not match! Database Value: " + ConnectionID + ", Supplied Value: " + connectionID;
                return false;
            }

            if (sessionID != null && SessionID != sessionID)
            {
                reason = "Session IDs do not match! Database Value: " + SessionID + ", Supplied Value: " + sessionID;
                return false;
            }

            // Reset any twitch session rich presence
            ResetTwitchRichPresence();

            DisconnectedDate = DateTime.UtcNow;
            IsConnected = false;
            ServerName = string.Empty;
            CurrentlyPlayingGameID = 0;
            CurrentlyWatchingChannelID = string.Empty;
            IsBroadcasting = false;
            CurrentlyBroadcastingGameID = 0;
            Update(p => p.DisconnectedDate, p => p.IsConnected, p => p.ServerName, p => p.CurrentlyPlayingGameID, p => p.CurrentlyWatchingChannelID, p => p.CurrentlyBroadcastingGameID, p => p.IsBroadcasting);

            ClientEndpointResolver.Create(UserID, userRegionID);
            return true;
        }

        private static readonly Version DefaultVersion = new Version(7, 0, 0, 0);

        public static Version GetMostRecentClientVersion(int userID, bool returnDefault = true)
        {

            // Get the most recently connected endpoint client version
            var mostRecentEndpoint = GetAllConnected(userID).OrderByDescending(p => p.ConnectedDate)
                                                            .FirstOrDefault();

            if (mostRecentEndpoint == null)
            {
                return DefaultVersion;
            }

            Version clientVersion;
            return !Version.TryParse(mostRecentEndpoint.ClientVersion, out clientVersion) ? DefaultVersion : clientVersion;
        }

        public static IReadOnlyCollection<ClientEndpoint> GetAllConnected(int userID, bool repairEndpoints = false)
        {
            var allEndpoints = GetAllForUser(userID);

            if (allEndpoints.Count == 0)
            {
                if (repairEndpoints)
                {
                    return RepairEndpoints(userID);
                }
                return allEndpoints;
            }

            return allEndpoints.Where(p => p.IsConnected && !string.IsNullOrEmpty(p.ServerName)).ToArray();
        }

        public static DevicePlatform[] GetAllPlatforms(int userID)
        {

            var map = UserClientEndpointMap.GetLocal(userID);
            if (map != null && map.Platforms != null)
            {
                return map.Platforms.Cast<DevicePlatform>().ToArray();
            }

            return new DevicePlatform[0];
        }

        private static readonly ClientEndpoint[] EmptyList = new ClientEndpoint[0];

        public static ClientEndpoint[] RepairEndpoints(int userID)
        {
            var userRegion = UserRegion.GetByUserID(userID);

            if (userRegion == null)
            {
                Logger.Warn("No endpoints found for user. We could also not locate the user region!", new { userID });
                return EmptyList;
            }

            if (userRegion.RegionID != LocalConfigID)
            {
                var endpoints = GetAll(userRegion.RegionID, p => p.UserID, userID).Where(p => p.IsConnected && !string.IsNullOrEmpty(p.ServerName)).ToArray();

                if (endpoints.Any())
                {
                    foreach (var endpoint in endpoints)
                    {
                        endpoint.InsertLocal();
                    }

                    Logger.Info("No endpoints found for user locally, but we were able to retrieve them from their home region.", new { userID, userRegion });
                }
                else
                {
                    Logger.Warn("No endpoints found for user. We could not retrieve them from their home region.", new { userID, userRegion });
                }

                return endpoints;
            }

            Logger.Warn("No endpoints found for a local user (this should be impossible)", new { userRegion });
            return EmptyList;
        }

        public static IReadOnlyCollection<ClientEndpoint> GetAllDeliverable(int userID)
        {
            var allEndpoints = GetAllForUser(userID);

            if (allEndpoints.Count == 0)
            {
                return allEndpoints;
            }

            return allEndpoints.Where(p => p.IsDeliverable).ToArray();
        }

        public static IReadOnlyCollection<ClientEndpoint> GetAllPresenceEndpoints(int userID)
        {
            var allEndpoints = GetAllForUser(userID);

            if (allEndpoints.Count == 0)
            {
                return allEndpoints;
            }

            return allEndpoints.Where(p => p.IsPresence).ToArray();
        }

        public bool IsPresence
        {
            get { return IsDeliverable || (Platform == DevicePlatform.Twitch && IsConnected); }
        }

        public bool IsDeliverable
        {
            get
            {
                return IsConnected && !string.IsNullOrEmpty(ServerName) // Endpoint is acively connected
                       || (!string.IsNullOrEmpty(DeviceID) && DateTime.UtcNow.Subtract(ConnectedDate) < TimeSpan.FromDays(7));

            }
        }

        public bool IsRoutable
        {
            get
            {
                return IsConnected && !string.IsNullOrEmpty(ServerName); // Endpoint is acively connected 
            }
        }

        public static IReadOnlyCollection<ClientEndpoint> GetAllForUser(int userID)
        {
            // First look for a mapping
            var map = UserClientEndpointMap.GetLocal(userID);

            // If the map is missing, try the index
            if (map == null)
            {
                return UpdateUserMap(userID);
            }

            if (map.MachineKeys == null || !map.MachineKeys.Any() || map.Platforms == null || !map.Platforms.Any())
            {
                return UpdateUserMap(userID);
            }

            var keys = map.MachineKeys.Select(p => new KeyInfo(userID, p)).ToArray();
            return MultiGetLocal(keys);
        }

        private static ClientEndpoint[] PruneEndpoints(ClientEndpoint[] endpoints)
        {
            var recentEndpoints = endpoints.Where(p => p.IsConnected
                                                       || p.ConnectedDate >= DateTime.UtcNow.AddDays(-30)
                                                       || p.SessionDate >= DateTime.UtcNow.AddDays(-30))
                                    .OrderByDescending(p => p.ConnectedDate)
                                    .Take(MaxEndpoints)
                                    .ToArray();

            var pruneEndpoints = endpoints.Where(p => !recentEndpoints.Contains(p));
            foreach (var pruneEndpoint in pruneEndpoints)
            {
                pruneEndpoint.IsDeleted = true; // Just in case this record decided to re-appear
                pruneEndpoint.Update(p => p.IsDeleted);
                pruneEndpoint.DurableDelete();
            }

            return recentEndpoints;
        }

        public static ClientEndpoint[] UpdateUserMap(int userID)
        {
            // Check this user's region
            var userRegion = UserRegion.GetLocal(userID);
            if (userRegion == null)
            {
                return EmptyList;
            }


            return UpdateUserMap(userRegion);
        }

        public static ClientEndpoint[] UpdateUserMap(UserRegion userRegion, bool alwaysPrune = false)
        {
            // Get all of the user's endpoints
            var endpoints = GetAllLocal(p => p.UserID, userRegion.UserID).Where(p => !p.IsDeleted).ToArray();
            if (!endpoints.Any())
            {
                return endpoints;
            }

            // If this is not a local region, do not persist the data (doing so could result in inconsistencies)
            if (userRegion.RegionID != LocalConfigID)
            {
                return endpoints;
            }

            // If the number of endpoints has exceeded 50, get the most recent
            if (alwaysPrune || endpoints.Length > MaxEndpoints)
            {
                endpoints = PruneEndpoints(endpoints);
            }

            RecalculateUserMap(userRegion.UserID, endpoints);
            
            return endpoints;
        }

        public static void RecalculateUserMap(int userID, ClientEndpoint[] endpoints)
        {
            var map = UserClientEndpointMap.GetLocal(userID);
            var machineKeys = endpoints.Select(p => p.MachineKey).ToArray();
            var platforms = endpoints.Select(p => (int)p.Platform).Distinct().ToArray();

            if (map == null)
            {
                map = new UserClientEndpointMap()
                {
                    UserID = userID,
                    MachineKeys = new HashSet<string>(machineKeys),
                    Platforms = new HashSet<int>(platforms)
                };

                TrySaveMap(map, true);
            }
            else
            {
                map.MachineKeys = new HashSet<string>(machineKeys);
                map.Platforms = new HashSet<int>(platforms);
                TrySaveMap(map, false);
            }
        }

        private static void TrySaveMap(UserClientEndpointMap map, bool insertNew, ClientEndpoint[] endpoints = null)
        {
            try
            {
                if (insertNew)
                {
                    map.InsertLocal();
                }
                else
                {
                    map.Update();
                }
            }
            catch (AerospikeException ex)
            {
                if (ex.Result == 13 || ex.Message.ToLower().Contains("record too big")) // If the record is too big, get a pruned list
                {
                    if (endpoints == null)
                    {
                        endpoints = GetAllLocal(p => p.UserID, map.UserID);
                    }
                    var relevantEndpoints = endpoints.Where(p => p.IsConnected || p.ConnectedDate >= DateTime.UtcNow.AddDays(-10) || p.SessionDate >= DateTime.UtcNow.AddDays(-10)).Select(p => p.MachineKey).ToArray();
                    Logger.Warn(ex, "Unable to update a user's client endoint map. They have " + endpoints.Length + " endpoints total, and " + relevantEndpoints.Length + " which are recent.", new { map.UserID });
                    map.MachineKeys = new HashSet<string>(relevantEndpoints);
                    map.Update();
                }
                else
                {
                    throw;
                }
            }
        }

        public static void DispatchNotification(int userID, Action<ClientEndpoint> connectedEndpointAction, Action<ClientEndpoint> mobileEndpointAction = null)
        {
            DispatchNotification(GetAllDeliverable(userID), connectedEndpointAction, mobileEndpointAction);
        }

        public static void DispatchNotification(IReadOnlyCollection<ClientEndpoint> endpoints, Action<ClientEndpoint> connectedEndpointAction, Action<ClientEndpoint> mobileEndpointAction, bool sendMobile = true)
        {
            // Default push notifications to empty
            var pushEndpoints = new ClientEndpoint[0];

            // Only deliver non-push notifications to the receiver's connected endpoints            
            var connectedEndpoints = endpoints.Where(p => p.IsConnected).ToArray();

            // If the receiver has no connected endpoints, or all of them are idle, we will use push notifications
            if (!connectedEndpoints.Any() || connectedEndpoints.All(p => p.IsIdle))
            {
                pushEndpoints = endpoints.Where(p => !string.IsNullOrEmpty(p.DeviceID)).ToArray();
            }

            if (connectedEndpointAction != null)
            {
                foreach (var endpoint in connectedEndpoints)
                {
                    connectedEndpointAction(endpoint);
                }
            }

            if (mobileEndpointAction == null || !sendMobile)
            {
                return;
            }

            var processedTokens = new HashSet<string>();

            foreach (var endpoint in pushEndpoints)
            {
                if (processedTokens.Contains(endpoint.DeviceID))
                {
                    continue;
                }

                mobileEndpointAction(endpoint);
                processedTokens.Add(endpoint.DeviceID);
            }

        }

        public static void UpdateDeviceID(string deviceID, int userID, string machineKey)
        {
            var model = GetLocal(userID, machineKey);
            model.UpdateDeviceID(deviceID);
        }

        public void UpdateDeviceID(string deviceID)
        {
            if (DeviceID == deviceID)
            {
                return;
            }
            DeviceID = deviceID;
            Update(p => p.DeviceID);

            try
            {
                var region = UserRegion.GetLocal(UserID);
                ClientEndpointResolver.Create(UserID, region.RegionID);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to queue ClientEndpointResolver");
            }

        }

        public static void UpdatePushKitToken(string token, int userID, string machineKey)
        {
            var model = GetLocal(userID, machineKey);
            model.UpdatePushKitToken(token);
        }

        public void UpdatePushKitToken(string token)
        {
            if (PushKitToken == token)
            {
                return;
            }

            PushKitToken = token;
            Update(p => p.PushKitToken);

            try
            {
                var region = UserRegion.GetLocal(UserID);
                ClientEndpointResolver.Create(UserID, region.RegionID);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to queue ClientEndpointResolver");
            }
        }

        public static bool UpdateDeviceTokens(string deviceID, string pushkitToken, int userID, Guid machineKey)
        {
            var model = GetLocal(userID, machineKey);
            if (model == null)
            {
                return false;
            }

            var changes = new List<Expression<Func<ClientEndpoint, object>>>();
            if (model.DeviceID != deviceID)
            {
                model.DeviceID = deviceID;
                changes.Add(ep => ep.DeviceID);
            }

            if (pushkitToken != null && model.PushKitToken != pushkitToken)
            {
                model.PushKitToken = pushkitToken;
                changes.Add(ep => ep.PushKitToken);
            }

            if (changes.Any())
            {
                model.Update(changes.ToArray());

                try
                {
                    var region = UserRegion.GetLocal(model.UserID);
                    ClientEndpointResolver.Create(model.UserID, region.RegionID);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to queue ClientEndpointResolver");
                }
            }
            return true;
        }

        public static void UpdatePlatform(DevicePlatform platform, int userID, string machineKey)
        {
            var model = GetLocal(userID, machineKey);
            model.Platform = platform;
            model.Update(p => p.Platform);
        }

        public void ToggleIdle(bool isIdle, int userRegionID)
        {
            if (IsIdle == isIdle)
            {
                return;
            }

            IsIdle = isIdle;
            Update(p => p.IsIdle);
            ClientEndpointResolver.Create(UserID, userRegionID);
        }

        public static ClientEndpoint Create(int userID, string machineKey, string sessionID, DevicePlatform platform, string deviceID)
        {
            var endpoint = new ClientEndpoint
            {
                UserID = userID,
                MachineKey = machineKey,
                SessionDate = DateTime.UtcNow,
                IsConnected = false,
                RegionID = ClientEndpoint.LocalConfigID,
                SessionID = sessionID,
                Platform = platform,
                DeviceID = deviceID
            };

            endpoint.InsertLocal();

            return endpoint;
        }


        protected override void OnInserted()
        {
            ValidateMap();
        }

        /// <summary>
        /// Validates the UserClientEndpointMap to ensure that it exists for all endpoints
        /// </summary>        
        public void ValidateMap()
        {
            var map = UserClientEndpointMap.GetLocal(UserID);

            if (map == null)
            {
                map = new UserClientEndpointMap { UserID = UserID, MachineKeys = new HashSet<string>() { MachineKey } };
                TrySaveMap(map, true);
            }
            else
            {
                if (map.MachineKeys == null)
                {
                    map.MachineKeys = new HashSet<string>();
                }

                if (map.Platforms == null)
                {
                    map.Platforms = new HashSet<int>();
                }

                var hasChanges = false;
                if (!map.MachineKeys.Contains(MachineKey))
                {
                    map.MachineKeys.Add(MachineKey);
                    hasChanges = true;
                }

                if (!map.Platforms.Contains((int)Platform))
                {
                    map.Platforms.Add((int)Platform);
                    hasChanges = true;
                }

                if (hasChanges)
                {
                    TrySaveMap(map, false);
                }

            }
        }

        private static readonly Version EmptyVersion = new Version(1, 0, 0, 0);

        public Version GetVersion()
        {

            if (string.IsNullOrEmpty(ClientVersion))
            {
                return EmptyVersion;
            }

            try
            {
                return new Version(ClientVersion);
            }
            catch
            {
                return EmptyVersion;
            }
        }

        [JsonIgnore]
        public bool IsLegacyClient
        {
            get
            {
                if (string.IsNullOrEmpty(ClientVersion))
                {
                    return true;
                }

                try
                {
                    var version = new Version(ClientVersion);
                    return version.Major < 7;
                }
                catch
                {
                    return true;
                }
            }

        }

        public static Dictionary<int, ClientEndpoint> MultiGetMostRecentForUserIDs(int[] userIDs)
        {
            var allEndpointKeys = new List<KeyInfo>(userIDs.Length);
            var endpointsByUser = new Dictionary<int, ClientEndpoint>(userIDs.Length);

            foreach (var set in userIDs.InSetsOf(100))
            {
                var endpointMaps = UserClientEndpointMap.MultiGetLocal(set.Select(p => new KeyInfo(p)));
                foreach (var map in endpointMaps)
                {
                    allEndpointKeys.AddRange(map.MachineKeys.Select(p => new KeyInfo(map.UserID, p)));
                }
            }

            foreach (var endpointKeys in allEndpointKeys.InSetsOf(100))
            {
                var setEndpoints = MultiGetLocal(endpointKeys);
                foreach (var endpoint in setEndpoints)
                {
                    if (!endpointsByUser.ContainsKey(endpoint.UserID))
                    {
                        endpointsByUser.Add(endpoint.UserID, endpoint);
                    }
                    else if (endpointsByUser[endpoint.UserID].ConnectedDate < endpoint.ConnectedDate)
                    {
                        endpointsByUser[endpoint.UserID] = endpoint;
                    }
                }
            }

            return endpointsByUser;
        }

        public static Dictionary<int, List<ClientEndpoint>> MultiGetAllForUserIDs(int[] userIDs)
        {
            var allEndpointKeys = new List<KeyInfo>(userIDs.Length);
            var endpointsByUser = new Dictionary<int, List<ClientEndpoint>>(userIDs.Length);
            foreach (var set in userIDs.InSetsOf(100))
            {
                var endpointMaps = UserClientEndpointMap.MultiGetLocal(set.Select(p => new KeyInfo(p)));
                foreach (var map in endpointMaps)
                {
                    allEndpointKeys.AddRange(map.MachineKeys.Select(p => new KeyInfo(map.UserID, p)));
                }
            }

            foreach (var endpointKeys in allEndpointKeys.InSetsOf(100))
            {
                var setEndpoints = MultiGetLocal(endpointKeys);
                foreach (var endpoint in setEndpoints)
                {
                    if (!endpointsByUser.ContainsKey(endpoint.UserID))
                    {
                        endpointsByUser.Add(endpoint.UserID, new List<ClientEndpoint>());
                    }

                    if (endpoint.IsDeliverable)
                    {
                        endpointsByUser[endpoint.UserID].Add(endpoint);
                    }

                }
            }

            return endpointsByUser;
        }

        public void UpdateCurrentGroup(UserRegion region, Guid groupID)
        {
            CurrentGroup = groupID;
            CurrentGroupTimestamp = DateTime.UtcNow.ToEpochMilliseconds();
            Update(e => e.CurrentGroup, e => e.CurrentGroupTimestamp);

            ClientEndpointResolver.Create(region.UserID, region.RegionID);
        }

        public void ChangePlaying(int regionID, int gameID, int gameState, string statusMessage)
        {
            if (gameID == 0)
            {
                ResetTwitchRichPresence(PresenceActivityType.Playing);
            }

            CurrentlyPlayingGameID = gameID;
            CurrentlyPlayingGameState = gameState;
            CurrentlyPlayingGameStatusMessage = statusMessage;
            CurrentlyPlayingGameTimestamp = DateTime.UtcNow;
            ActivityTimestamp = CurrentlyPlayingGameTimestamp;
            Update(e => e.CurrentlyPlayingGameID, e => e.CurrentlyPlayingGameState, e => e.CurrentlyPlayingGameStatusMessage, e => e.CurrentlyPlayingGameTimestamp, e => ActivityTimestamp);
            UserActivityResolver.Create(regionID, UserID, MachineKey);
        }

        public void ChangeWatching(int regionID, string channelID)
        {
            if (string.IsNullOrEmpty(channelID))
            {
                ResetTwitchRichPresence(PresenceActivityType.Watching);
            }

            CurrentlyWatchingChannelID = channelID;
            ActivityTimestamp = DateTime.UtcNow;
            Update(e => e.CurrentlyWatchingChannelID, e => e.ActivityTimestamp);
            UserActivityResolver.Create(regionID, UserID, MachineKey);
        }

        private void ResetTwitchRichPresence(PresenceActivityType? type = null)
        {
            // See if the user has a Twitch endpoint as well
            var twitchEndpoint = GetTwitchEndpoint();

            if (twitchEndpoint == null)
            {
                return;
            }

            if (!type.HasValue || type.Value == PresenceActivityType.Playing)
            {
                if (twitchEndpoint.CurrentlyPlayingGameID == CurrentlyPlayingGameID)
                {
                    twitchEndpoint.CurrentlyPlayingGameID = 0;
                    twitchEndpoint.CurrentlyPlayingGameState = 0;
                    twitchEndpoint.CurrentlyPlayingGameStatusMessage = string.Empty;
                }
            }

            if (!type.HasValue || type.Value == PresenceActivityType.Watching)
            {
                if (twitchEndpoint.CurrentlyWatchingChannelID == CurrentlyWatchingChannelID)
                {
                    twitchEndpoint.CurrentlyWatchingChannelID = string.Empty;
                }
            }
            
            twitchEndpoint.Update(p => p.CurrentlyPlayingGameID, p => p.CurrentlyPlayingGameState, p => p.CurrentlyPlayingGameStatusMessage, p => p.CurrentlyWatchingChannelID);
        }

        public ClientEndpoint GetTwitchEndpoint()
        {
            return GetLocal(UserID, TwitchMachineKey);
        }
    }
}
