﻿using System;
using System.Collections.Generic;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Enums;

namespace Curse.Friends.Data
{    

    /// <summary>
    /// The serialized representation of a chat entry, stored in a LargeList within the Conversation
    /// </summary>
    public class ConversationEntry : ILargeListElement
    {        
        /// <summary>
        /// The globally unique ID of this entry
        /// </summary>
        public Guid ID { get; set; }

        /// <summary>
        /// The user ID of the sender
        /// </summary>
        public int SenderID { get; set; }

        /// <summary>
        /// The user ID of the recipient (this can be 0)
        /// </summary>
        public int RecipientID { get; set; }

        /// <summary>
        /// The username of the sender (in case the key gets orphaned on the client side)
        /// </summary>
        public string SenderName { get; set; }

        /// <summary>
        ///  The type of entry (Notification, message, etc)
        /// </summary>
        public ConversationEntryType Type { get; set; }

        /// <summary>
        ///  The body of the message (serialized to JSON)
        /// </summary>
        public DateTime Timestamp { get; set; }

        /// <summary>
        ///  The body of the message (serialized to JSON)
        /// </summary>
        public string Body { get; set; }

        public void Hydrate(Dictionary<object, object> values)
        {
            ID = (Guid) values["id"];
            Body = (string)values["body"];
            Timestamp = (DateTime)values["timestamp"];
            SenderID = Convert.ToInt32(values["senderID"]);
            SenderName = (string)values["senderName"];
            if (values.ContainsKey("recipientID"))
            {
                RecipientID = Convert.ToInt32(values["recipientID"]);
            }
            Type = (ConversationEntryType)Convert.ToInt32(values["type"]);
        }

        public Dictionary<object, object> ToHash()
        {
            var dict = new Dictionary<object, object>
            {                
                { "id", ID },
                { "body", Body },
                { "timestamp", Timestamp },
                { "senderID", SenderID },
                { "senderName", SenderName },
                { "type", (int)Type },
            };

            if (RecipientID > 0)
            {
                dict.Add("recipientID", RecipientID);                
            }

            return dict;
        }

        public long GetKey()
        {
            return Timestamp.ToEpochMilliseconds();
        }
        

    }
}
