﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using System.Diagnostics;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "ExternalAccount", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    [DebuggerDisplay("External: {ExternalID} {ExternalUsername} - Merged: {MergedUserID}")]
    public class ExternalAccount : BaseTable<ExternalAccount>, IHostable
    {
        private static readonly TimeSpan FullFollowSyncThrottle = TimeSpan.FromMinutes(30);

        [Column("ExternalID", KeyOrdinal = 1)]
        public string ExternalID { get; set; }

        [Column("Type", KeyOrdinal = 2)]
        public AccountType Type { get; set; }

        [Column("LastSyncTime")]
        public DateTime LastSyncTime { get; set; }

        [Column("IsPartnered")]
        public bool IsPartnered { get; set; }

        [Column("AvatarUrl")]
        public string AvatarUrl { get; set; }

        [Column("AuthToken")]
        public string AuthToken { get; set; }

        [Column("RefreshToken")]
        public string RefreshToken { get; set; }

        [Column("ExtUsername")]
        public string ExternalUsername { get; set; }

        [Column("ExtDisplayName")]
        public string ExternalDisplayName { get; set; }

        [Column("Scopes")]
        public HashSet<string> Scopes { get; set; }

        [Column("MappedUsers")]
        public HashSet<int> MappedUsers { get; set; }

        [Column("NeedsReauth")]
        public bool NeedsReauthentication { get; set; }

        [Column("MergedUserID")]
        public int MergedUserID { get; set; }

        [Column("ClientID")]
        public string ClientID { get; set; }

        [Column("AutoProvision")]
        public bool IsAutoProvisioned { get; set; }

        [Column("DateFollowSync")]
        public DateTime DateFullFollowSync { get; set; }

        [Column("DateRecentFol")]
        public DateTime DateRecentFollow { get; set; }

        public bool IsMerged
        {
            get { return MergedUserID > 0; }
        }


        #region IHostable

        [Column("HostName")]
        public string MachineName { get; set; }

        [Column("HostRegion")]
        public int RegionID { get; set; }

        public object[] KeyObjects
        {
            get { return new object[] {ExternalID, Type}; }
        }

        public bool IsHostable { get { return MappedUsers != null && MappedUsers.Count > 0; } }

        #endregion

        public ExternalAccountMapping MapUser(int userID, int userRegion, bool forceSync = false, bool resolveAccount = false)
        {
            if (MergedUserID > 0 && userID != MergedUserID)
            {
                Logger.Warn("User attempted to sync to an account that has previously been merged with another account.", new { userID, MergedUserID, ExternalUsername, ExternalID });
                throw new DataConflictException("A different user has already merged this account");
            }

            var link = ExternalAccountMapping.GetLocal(userID, Type, ExternalID);
            if (link == null)
            {
                link = new ExternalAccountMapping
                {
                    UserID = userID,
                    ExternalID = ExternalID,
                    ExternalName = ExternalUsername,
                    Type = Type,
                    RegionID = userRegion,
                    DateMapped = DateTime.UtcNow
                };
                link.InsertLocal();
            }
            else if (link.IsDeleted)
            {
                link.IsDeleted = false;
                link.DateMapped = DateTime.UtcNow;
                link.Update(l => l.IsDeleted, l => l.DateMapped);
            }
            else
            {
                link.DateMapped = DateTime.UtcNow;
                link.Update(l => l.DateMapped);
            }

            var added = MappedUsers.Add(userID);
            if (added)
            {
                Update(a => a.MappedUsers);
                NotifyLinkChanged(ExternalAccountChangeType.Linked);
            }

            if (added || forceSync)
            {
                switch (Type)
                {
                    case AccountType.WorldOfWarcraft:
                        ExternalGuildUserSyncWorker.Create(link.UserID, link.Type);
                        break;
                    case AccountType.Twitch:
                        if (resolveAccount)
                        {
                            TwitchAccountResolver.UserLinkChanged(link.UserID, this);
                        }
                        ExternalCommunityMemberIndexWorker.CreateAccountLinked(link);
                        ExternalUserSyncWorker.Create(link.UserID, link.ExternalID, link.Type);
                        break;
                }
            }

            return link;
        }

        public void DeleteMapping(int userID, bool resolveAccount = false)
        {
            if (MergedUserID == userID)
            {
                MergedUserID = 0;
                Update(a => a.MergedUserID);
            }

            var link = ExternalAccountMapping.GetLocal(userID, Type, ExternalID);
            if (link == null || link.IsDeleted)
            {
                return;
            }

            link.IsDeleted = true;
            link.Update(l => l.IsDeleted);

            MappedUsers.Remove(userID);
            Update(a => a.MappedUsers);
            NotifyLinkChanged(ExternalAccountChangeType.Unlinked);

            switch (Type)
            {
                case AccountType.WorldOfWarcraft:
                    ExternalGuildUserSyncWorker.Create(link.UserID, link.Type);
                    break;
                case AccountType.Twitch:
                    if (resolveAccount)
                    {
                        TwitchAccountResolver.UserLinkChanged(link.UserID, this);
                    }
                    ExternalCommunityMemberIndexWorker.CreateAccountUnlinked(link);
                    ExternalUserSyncWorker.Create(link.UserID, link.ExternalID, link.Type);
                    break;
            }
        }

        public ExternalAccountContract ToContract(bool eligibleForVanityUrl = false, ExternalAccountMapping mapping = null)
        {
            var contract =  new ExternalAccountContract
            {
                Type = Type,
                ExternalID = ExternalID,
                ExternalName = ExternalUsername,
                ExternalDisplayName = ExternalDisplayName,
                EligibleForVanityUrl = eligibleForVanityUrl,                
                IsPartnered = IsPartnered,
                NeedsReauthentication = NeedsReauthentication
            };

            if (mapping != null)
            {
                contract.DateLinked = mapping.DateMapped.SafeToEpochMilliseconds();
            }

            return contract;
        }

        public string DisplayName
        {
            get
            {
                return string.IsNullOrEmpty(ExternalDisplayName) ? ExternalUsername : ExternalDisplayName;
            }
        }

        public ExternalAccountPublicContract ToPublicContract()
        {
            var contract = new ExternalAccountPublicContract
            {
                Type = Type,
                ExternalID = ExternalID,
                ExternalName = ExternalUsername,
                ExternalDisplayName = ExternalDisplayName,
            };
            
            return contract;
        }

        public static IReadOnlyCollection<ExternalAccount> GetAllForUser(int userID)
        {
            var accountMappings = ExternalAccountMapping.GetAllLocal(m => m.UserID, userID);
            return MultiGetLocal(accountMappings.Where(a => !a.IsDeleted).Select(p => new KeyInfo(p.ExternalID, p.Type)));
        }

        public void MarkForReauthentication()
        {
            if (NeedsReauthentication)
            {
                return;
            }

            NeedsReauthentication = true;
            Update(a => a.NeedsReauthentication);

            NotifyLinkChanged(ExternalAccountChangeType.NeedsRelink);
        }

        private void NotifyLinkChanged(ExternalAccountChangeType changeType)
        {
            var links = ExternalAccountMapping.MultiGetLocal(MappedUsers.Select(id => new KeyInfo(id, Type, ExternalID)));
            foreach (var link in links)
            {
                var notification = new ExternalAccountChangedNotification
                {
                    Account = ToContract(false, link),
                    ChangeType = changeType
                };
                ClientEndpoint.DispatchNotification(link.UserID, ep => ExternalAccountChangedNotifier.Create(notification, ep));
            }
        }

        public static Dictionary<int, ExternalAccount> MultiGetAllTwitchAccountsByUserIDs(int[] userIDs)
        {
            var usersWithExternalAccounts = UserStatistics.MultiGetAllForUserIDs(userIDs);
            return MultiGetAllTwitchAccountsByUserStats(usersWithExternalAccounts);
        }

        public static Dictionary<int, ExternalAccount> MultiGetAllTwitchAccountsByUserStats(UserStatistics[] userStats)
        {
            var syncedAccounts = userStats.Where(u => !string.IsNullOrEmpty(u.TwitchID)).ToArray();
            var externalAccounts = MultiGetLocal(new HashSet<string>(syncedAccounts.Select(u => u.TwitchID)).Select(id => new KeyInfo(id, AccountType.Twitch)))
                .ToDictionary(a => a.ExternalID);

            var dict = new Dictionary<int, ExternalAccount>();
            foreach (var user in syncedAccounts)
            {
                dict[user.UserID] = externalAccounts.GetValueOrDefault(user.TwitchID);
            }
            return dict;
        }

        public static ExternalAccount GetByTwitchUserID(string twitchUserID)
        {
            return GetLocal(twitchUserID, AccountType.Twitch);
        }

        public static IReadOnlyCollection<ExternalAccount> GetByTwitchUserIDs(string[] twitchUserIDs)
        {
            return MultiGetLocal(twitchUserIDs.Select(p => new KeyInfo(p, AccountType.Twitch)));            
        }

        public static Dictionary<string, ExternalAccount> GetDictionaryByTwitchUserIDs(string[] twitchUserIDs)
        {
            var accounts = MultiGetLocal(twitchUserIDs.Select(p => new KeyInfo(p, AccountType.Twitch)));
            return accounts.ToDictionary(p => p.ExternalID);
        }

        public User GetMergedUser()
        {
            if (!IsMerged)
            {
                return null;
            }

            var userRegion = UserRegion.GetByUserID(MergedUserID);
            return userRegion?.GetUser();
        }

        public string GetAvatarEntityID()
        {
            return GetAvatarEntityID(Type, ExternalID);
        }

        public static string GetAvatarEntityID(AccountType type, string syncID)
        {
            return string.Format("{0}/{1}", type, syncID);
        }

        public object GetLogData()
        {
            return new
            {
                ExternalID,
                ExternalUsername,
                IsMerged,
                IsPartnered,             
                MergedUserID   
            };
        }

        public bool ShouldSyncFollows()
        {
            return DateFullFollowSync == DateTime.MinValue || DateTime.UtcNow.Subtract(DateFullFollowSync) >= FullFollowSyncThrottle;
        }
    }
}
