﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "ExternalCommunity", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class ExternalCommunity : BaseTable<ExternalCommunity>, IHostable
    {
        private static readonly long SubSyncThrottle = (long) TimeSpan.FromMinutes(15).TotalMilliseconds;
        private static readonly long ModSyncThrottle = (long) TimeSpan.FromMinutes(15).TotalMilliseconds;

        [Column("ExternalID", KeyOrdinal = 1, IsIndexed = true)]
        public string ExternalID { get; set; }

        [Column("Type", KeyOrdinal = 2)]
        public AccountType Type { get; set; }

        [Column("ExternalName")]
        public string ExternalName { get; set; }

        [Column("ExtDisplayName")]
        public string ExternalDisplayName { get; set; }

        [Column("RegionID", IsIndexed = true)]
        public int RegionID { get; set; }

        [Column("Machine", IsIndexed = true)]
        public string MachineName { get; set; }

        [Column("MappedGroups")]
        public HashSet<Guid> MappedGroups { get; set; }

        [Column("AvatarUrl")]
        public string AvatarUrl { get; set; }

        [Column("IsPartnered")]
        public bool IsPartnered { get; set; }

        [Column("IsAffiliate")]
        public bool IsAffiliate { get; set; }

        [Column("ExtDateCreated")]
        public DateTime ExternalDateCreated { get; set; }

        [Column("Followers")]
        public int Followers { get; set; }

        [Column("Subs")]
        public int Subscribers { get; set; }

        [Column("IsLive")]
        public bool IsLive { get; set; }

        [Column("LiveTimestamp")]
        public long LiveTimestamp { get; set; }

        [Column("ExtStatus")]
        public string ExternalStatus { get; set; }

        [Column("GameID")]
        public int GameID { get; set; }

        [Column("ExtGameName")]
        public string ExternalGameName { get; set; }

        [Column("MembersIndexed")]
        public DateTime DateMembersIndexed { get; set; }

        [Column("DateEmoteSync")]
        public DateTime DateFullEmoticonSync { get; set; }

        public object[] KeyObjects { get { return new object[] { ExternalID, Type }; } }

        public string DisplayName { get { return string.Format("{0} ({1})", ExternalID, Type); } }

        public bool IsHostable { get { return RegionID > 0; } }

        [Column("SyncTimestamp")]
        public long SyncedTimestamp { get; set; }

        [Column("ModSyncTime")]
        public long ModSyncTimestamp { get; set; }

        [Column("SubSyncTime")]
        public long SubSyncTimestamp { get; set; }

        [Column("DateRecentSub")]
        public DateTime DateRecentSub { get; set; }

        [Column("HostableRegion", IsIndexed = true)]
        public int HostableRegion { get; set; }

        public bool CanHaveSubs { get { return IsPartnered || IsAffiliate; } }

        public bool HasMappedGroups
        {
            get
            {
                return MappedGroups != null && MappedGroups.Any();
            }            
        }

        public bool ShouldSyncInfo()
        {
            return SyncedTimestamp.FromEpochMilliconds() < DateTime.UtcNow.AddMinutes(-30);
        }

        public ExternalCommunityRole[] GetRoles()
        {
            return ExternalCommunityRole.GetAll(SourceConfiguration, c => c.ExternalID, ExternalID).Where(c => c.Type == Type).ToArray();
        }

        public IReadOnlyCollection<ExternalCommunityMapping> GetAllMappings()
        {
            if (MappedGroups == null || MappedGroups.Count == 0)
            {
                return new ExternalCommunityMapping[0];
            }
            return ExternalCommunityMapping.MultiGet(SourceConfiguration, MappedGroups.Select(g => new KeyInfo(g, ExternalID, Type)));
        }

        public ExternalCommunityMapping GetMapping(Guid groupID)
        {
            return ExternalCommunityMapping.Get(SourceConfiguration, groupID, ExternalID, Type);
        }

        public ExternalCommunityMapping MapToGroup(GroupMember requestor, Group group)
        {
            var mapping = GetMapping(group.GroupID);
            if (mapping == null)
            {
                mapping = new ExternalCommunityMapping
                {
                    GroupID = group.GroupID,
                    ExternalID = ExternalID,
                    Type = Type,
                    GracePeriodDays = 0,
                    GracePeriodExpireAction = SyncedMemberGracePeriodAction.None,
                    SyncEmotes = true,
                    SyncEvents = true,
                    ExternalName = ExternalName,
                    ExternalDisplayName = ExternalDisplayName
                };
                mapping.Insert(SourceConfiguration);
            }
            else if (mapping.IsDeleted)
            {
                mapping.IsDeleted = false;
                mapping.Update(UpdateMode.Default, SourceConfiguration, m => m.IsDeleted);
            }

            var updates = new List<Expression<Func<ExternalCommunity, object>>>();

            if (MappedGroups == null)
            {
                MappedGroups = new HashSet<Guid>();
            }


            if (MappedGroups.Add(group.GroupID))
            {
                updates.Add(c => c.MappedGroups);
            }

            if (RegionID == 0)
            {
                RegionID = group.RegionID;
                updates.Add(c => c.RegionID);
            }

            if (HostableRegion == 0)
            {
                HostableRegion = RegionID;
                updates.Add(c => c.HostableRegion);
            }

            if (updates.Count>0)
            {
                Update(UpdateMode.Default, SourceConfiguration, updates.ToArray());
            }

            ExternalCommunitySyncWorker.Linked(RegionID, ExternalID, Type, group.GroupID);
            if (string.IsNullOrEmpty(MachineName))
            {
                ExternalCommunityCoordinator.StreamCommissioned(this);
            }
            else
            {
                ExternalCommunityCoordinator.LinksChanged(this);
            }
            GroupEventManager.LogExternalCommunityLinkedEvent(requestor, mapping);
            return mapping;
        }

        public bool AddMemberRole(string externalUserID, GroupRoleTag role, string externalUsername, string externalDisplayName, DateTime roleDate, bool index = true)
        {
            var updated = false;

            var member = ExternalCommunityMembership.Get(SourceConfiguration, externalUserID, ExternalID, Type, role);
            if (member == null)
            {
                var externalRole = ExternalCommunityRole.Get(SourceConfiguration, ExternalID, Type, role);

                if (externalRole == null)
                {
                    TwitchModelHelper.CreateOrUpdateCommunityRoles(ExternalID, ExternalName);
                    externalRole = ExternalCommunityRole.GetLocal(ExternalID, Type, role);
                    if (externalRole == null)
                    {
                        throw new InvalidOperationException();
                    }
                }

                var entryNumber = 0;
                if (externalRole.RoleTag == GroupRoleTag.SyncedSubscriber)
                {
                    entryNumber = externalRole.IncrementCounter(RegionID > 0 ? RegionID : ExternalCommunityRole.LocalConfigID, UpdateMode.Default, r => r.CurrentEntryNumber, 1);
                }

                member = new ExternalCommunityMembership
                {
                    ExternalUserID = externalUserID,
                    ExternalCommunityID = ExternalID,
                    Type = Type,
                    RoleTag = externalRole.RoleTag,
                    ExternalUsername = externalUsername,
                    ExternalUserDisplayName = externalDisplayName,
                    FirstRoleDate = roleDate,
                    CurrentRoleDate = roleDate,
                    Status = ExternalCommunityMembershipStatus.Active,
                    StatusDate = DateTime.UtcNow,
                    UserAndTypeIndex = externalUserID + Type,
                    CommunityAndTypeIndex = ExternalID + Type,
                    UserTypeAndRoleIndex = externalUserID + Type + role,
                    CommunityTypeAndRoleIndex = ExternalID + Type + role,
                    ExternalCommunityName = ExternalName,
                    ExternalCommunityDisplayName = ExternalDisplayName,
                    EntryNumber = entryNumber,
                };
                member.Insert(SourceConfiguration);
                updated = true;
            }
            else
            {
                var updateStatements = new List<Expression<Func<ExternalCommunityMembership, object>>>();
                if (member.Status != ExternalCommunityMembershipStatus.Active)
                {
                    member.Status = ExternalCommunityMembershipStatus.Active;
                    member.StatusDate = DateTime.UtcNow;
                    updateStatements.Add(m => m.Status);
                    updateStatements.Add(m => m.StatusDate);
                }

                if (member.CurrentRoleDate != roleDate)
                {
                    member.CurrentRoleDate = roleDate;
                    updateStatements.Add(m => m.CurrentRoleDate);
                }

                if (member.ExternalUsername != externalUsername)
                {
                    member.ExternalUsername = externalUsername;
                    updateStatements.Add(m => m.ExternalUsername);
                }

                if (member.ExternalCommunityName != ExternalName)
                {
                    member.ExternalCommunityName = ExternalName;
                    updateStatements.Add(m => m.ExternalCommunityName);
                }

                if (member.ExternalCommunityDisplayName != ExternalDisplayName)
                {
                    member.ExternalCommunityDisplayName = ExternalDisplayName;
                    updateStatements.Add(m => m.ExternalCommunityDisplayName);
                }

                if (updateStatements.Any())
                {
                    member.Update(updateStatements.ToArray());
                    updated = true;
                }
            }

            if (updated)
            {
                if (index)
                {
                    ExternalCommunityMemberIndexWorker.CreateExternalMembershipAdded(member);
                }
                foreach (var mapping in GetAllMappings())
                {
                    mapping.AddMemberRoleMapping(member);
                }
            }

            return updated;
        }

        public bool RemoveMemberRole(string externalUserID, GroupRoleTag role, bool index = true)
        {
            var membership = ExternalCommunityMembership.Get(SourceConfiguration, externalUserID, ExternalID, Type, role);
            if (membership == null)
            {
                return false;
            }

            if (membership.Status == ExternalCommunityMembershipStatus.Deleted)
            {
                return false;
            }

            membership.Status = ExternalCommunityMembershipStatus.Deleted;
            membership.StatusDate = DateTime.UtcNow;
            membership.Update(m => m.Status, m => m.StatusDate);

            if (index)
            {
                ExternalCommunityMemberIndexWorker.CreateExternalMembershipRemoved(membership);
            }

            foreach (var communityMapping in GetAllMappings())
            {
                communityMapping.RemoveMemberRoleMapping(membership);
            }

            return true;
        }

        public ExternalCommunityMapping DeleteMapping(GroupMember requestor, Guid groupID)
        {
            var mapping = GetMapping(groupID);
            if (mapping == null)
            {
                return null;
            }

            mapping.IsDeleted = true;
            mapping.Update(m => m.IsDeleted);

            if (MappedGroups == null || !MappedGroups.Contains(groupID))
            {
                return null;
            }

            MappedGroups.Remove(groupID);
            if (MappedGroups.Count == 0)
            {
                HostableRegion = 0;
                Update(s => s.MappedGroups, s => s.HostableRegion);
            }
            else
            {
                Update(s => s.MappedGroups);
            }

            ExternalCommunitySyncWorker.Unlinked(RegionID, ExternalID, Type, groupID);
            if (MappedGroups.Count == 0 && !string.IsNullOrEmpty(MachineName))
            {
                ExternalCommunityCoordinator.StreamDecommissioned(this);
            }
            else
            {
                ExternalCommunityCoordinator.LinksChanged(this);
            }
            GroupEventManager.LogExternalCommunityUnlinkedEvent(requestor, mapping);
            return mapping;
        }

        public bool ResolveMemberships(Dictionary<string, ExternalCommunityMembership> existingMemberships, Dictionary<string, NewCommunityMember> externalUsersWithRole, GroupRoleTag role, bool addOnly = false)
        {
            var changed = false;

            var performFullIndex = DateMembersIndexed == DateTime.MinValue ||
                                   DateTime.UtcNow.Subtract(DateMembersIndexed) >= TimeSpan.FromDays(1);

            var newMemberships = new HashSet<string>();
            var removedMemberships = new HashSet<string>();

            foreach (var user in externalUsersWithRole.Where(u => !existingMemberships.ContainsKey(u.Key) || existingMemberships[u.Key].Status != ExternalCommunityMembershipStatus.Active))
            {
                if (AddMemberRole(user.Key, role, user.Value.ExternalUsername, user.Value.ExternalUserDisplayName, user.Value.RoleDate, false))
                {
                    newMemberships.Add(user.Key);
                }
            }

            changed |= newMemberships.Count > 0;

            if (!performFullIndex)
            {
                ExternalCommunityMemberIndexWorker.CreateExternalMembershipsAdded(this, role, newMemberships.ToArray());
            }

            if (!addOnly)
            {
                foreach (var membership in existingMemberships.Values.Where(m => !externalUsersWithRole.ContainsKey(m.ExternalUserID)))
                {
                    if (RemoveMemberRole(membership.ExternalUserID, role, false))
                    {
                        removedMemberships.Add(membership.ExternalUserID);
                    }
                }

                changed |= removedMemberships.Count > 0;

                if (!performFullIndex)
                {
                    ExternalCommunityMemberIndexWorker.CreateExternalMembershipsRemoved(this, role,
                        removedMemberships.ToArray());
                }
            }

            var membershipsToResolve = existingMemberships.Values
                .Where(m => !newMemberships.Contains(m.ExternalUserID) && !removedMemberships.Contains(m.ExternalUserID))
                .ToArray();

            if (membershipsToResolve.Length > 0)
            {
                foreach (var mapping in GetAllMappings())
                {
                    changed |= mapping.ResolveMemberships(membershipsToResolve);
                }
            }

            if (performFullIndex)
            {
                Logger.Debug("Performing full member re-index: " + ExternalID);
                DateMembersIndexed = DateTime.UtcNow;
                Update(p => p.DateMembersIndexed);                
                ExternalCommunityMemberIndexWorker.CreateReindex(this);
            }

            return changed;
        }

        public void EnsureHost()
        {
            switch (Type)
            {
                case AccountType.Twitch:
                    TwitchHostManager.EnsureServiceHost(this);
                    break;
                default:
                    Logger.Warn("Unsupported external community type for hosting.", Type);
                    throw new InvalidOperationException();
            }
        }

        public string GetExternalUrl()
        {
            switch (Type)
            {
                case AccountType.Twitch:
                    return string.Format(FriendsServiceConfiguration.Instance.TwitchChannelUrlFormat, ExternalName);
                default:
                    return null;
            }
        }

        public ExternalCommunityContract ToContract()
        {
            var contract = new ExternalCommunityContract
            {
                ExternalID = ExternalID,
                Type = Type,
                ExternalName = ExternalName,
                ExternalDisplayName = ExternalDisplayName,
                CreatedTimestamp = ExternalDateCreated.ToEpochMilliseconds(),
                IsPartnered = IsPartnered,
                IsAffiliate = IsAffiliate,
                Followers = Followers,
                GameID = GameID,
                ExternalGameName = ExternalGameName,
                ExternalStatus = ExternalStatus,
                ExternalUrl = GetExternalUrl(),
                Subscribers = Subscribers
            };

            return contract;
        }


        public ExternalCommunityPublicContract ToPublicContract()
        {
            return new ExternalCommunityPublicContract
            {
                IsLive = IsLive,
                ExternalID = ExternalID,
                Type = Type,
                ExternalName = ExternalName,
                ExternalDisplayName = ExternalDisplayName,
                GameID = GameID,
                ExternalGameName = ExternalGameName,
                ExternalStatus = ExternalStatus,
                ExternalUrl = GetExternalUrl(),
            };
        }

        #region Relationships

        public ExternalCommunityMembership[] GetAllMemberships()
        {
            return ExternalCommunityMembership.GetAllByExternalCommunityIDAndType(SourceConfiguration, ExternalID, Type);
        }

        #endregion

        public string GetAvatarEntityID()
        {
            return GetAvatarEntityID(Type, ExternalID);
        }

        public static string GetAvatarEntityID(AccountType type, string syncID)
        {
            return string.Format("{0}/{1}", type, syncID);
        }

        public static ExternalCommunity GetByTwitchID(int twitchID)
        {
            return GetLocal(twitchID.ToString(), AccountType.Twitch);
        }

        public static ExternalCommunity GetByTwitchID(string twitchID)
        {
            return GetLocal(twitchID, AccountType.Twitch);
        }

        public static IReadOnlyCollection<ExternalCommunity> GetAllByTwitchIDs(string[] twitchIDs)
        {
            if (!twitchIDs.Any())
            {
                return new ExternalCommunity[0];
            }

            return MultiGetLocal(twitchIDs.Select(p => new KeyInfo(p, AccountType.Twitch)));
        }

        public static Dictionary<string, ExternalCommunity> GetDictionaryByTwitchIDs(HashSet<string> twitchIDs)
        {
            if (!twitchIDs.Any())
            {
                return new Dictionary<string, ExternalCommunity>();
            }

            var communities = MultiGetLocal(twitchIDs.Select(p => new KeyInfo(p, AccountType.Twitch)));
            return communities.ToDictionary(p => p.ExternalID);
        }

        public bool ShouldSyncSubs()
        {
            return IsPartnered && (SubSyncTimestamp == 0 || DateTime.UtcNow.ToEpochMilliseconds() - SubSyncTimestamp >= SubSyncThrottle);
        }
        public bool ShouldSyncMods()
        {
            return ModSyncTimestamp == 0 || DateTime.UtcNow.ToEpochMilliseconds() - ModSyncTimestamp >= ModSyncThrottle;
        }

        public object GetLogData()
        {
            return new
            {
                Type,
                ExternalID,
                ExternalName,
                ExternalDisplayName,
                IsPartnered,
                MachineName,
                IsLive,
                LiveTimestamp
            };
        }
    }
}
