﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using Curse.Aerospike;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "ExternalCommunityMapping", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class ExternalCommunityMapping : BaseTable<ExternalCommunityMapping>
    {
        [Column("GroupID",KeyOrdinal = 1, IsIndexed = true)]
        public Guid GroupID { get; set; }

        [Column("ExternalID", KeyOrdinal = 2, IsIndexed = true)]
        public string ExternalID { get; set; }

        [Column("Type",KeyOrdinal = 3)]
        public AccountType Type { get; set; }

        [Column("ExternalName")]
        public string ExternalName { get; set; }

        [Column("ExtDisplayName")]
        public string ExternalDisplayName { get; set; }

        [Column("IsDeleted")]
        public bool IsDeleted { get; set; }

        [Column("ExpireAction")]
        public SyncedMemberGracePeriodAction GracePeriodExpireAction { get; set; }

        [Column("SyncEmotes")]
        public bool SyncEmotes { get; set; }

        [Column("SyncEvents")]
        public bool SyncEvents { get; set; }

        [Column("GracePeriod")]
        public int GracePeriodDays { get; set; }

        [Column("SyncedPremiums")]
        public int SyncedPremiumMembers { get; set; }

        private ExternalCommunity _community;
        public ExternalCommunity Community
        {
            get
            {
                _community = _community??ExternalCommunity.GetLocal(ExternalID, Type);
                return _community;
            }
        }

        public bool AddMemberRoleMapping(ExternalCommunityMembership member)
        {
            return DoAddMemberRoleMapping(member, ExternalCommunityMembershipMapping.Get(SourceConfiguration, member.ExternalUserID, ExternalID, Type, member.RoleTag, GroupID));
        }

        private bool DoAddMemberRoleMapping(ExternalCommunityMembership member, ExternalCommunityMembershipMapping mapping)
        {
            var updated = false;
            if (mapping == null)
            {
                mapping = new ExternalCommunityMembershipMapping
                {
                    Type = member.Type,
                    GroupID = GroupID,
                    ExternalUserID = member.ExternalUserID,
                    Status = member.Status,
                    StatusDate = member.StatusDate,
                    ExternalCommunityID = member.ExternalCommunityID,
                    RoleTag = member.RoleTag,
                    ExternalUsername = member.ExternalUsername,

                    CommunityAndTypeIndex = member.CommunityAndTypeIndex,
                    UserAndTypeIndex = member.UserAndTypeIndex,
                    CommunityTypeAndGroupIndex = member.ExternalCommunityID + member.Type + GroupID,
                    UserTypeAndCommunityIndex = member.ExternalUserID + member.Type + member.ExternalCommunityID,
                };
                mapping.Insert(SourceConfiguration);
                updated = true;
            }
            else if (mapping.Status != member.Status)
            {
                mapping.Status = member.Status;
                mapping.StatusDate = member.StatusDate;
                mapping.Update(m => m.Status, m => m.StatusDate);
                updated = true;
            }
            return updated;
        }

        public bool RemoveMemberRoleMapping(ExternalCommunityMembership membership)
        {
            return DoRemoveMemberRoleMapping(ExternalCommunityMembershipMapping.GetLocal(membership.ExternalUserID, ExternalID, Type, membership.RoleTag, GroupID));
        }

        private bool DoRemoveMemberRoleMapping(ExternalCommunityMembershipMapping mapping)
        {
            if (mapping == null || mapping.Status == ExternalCommunityMembershipStatus.Deleted)
            {
                return false;
            }
                      
            mapping.Status = ExternalCommunityMembershipStatus.Deleted;
            mapping.StatusDate = DateTime.UtcNow;
            mapping.Update(m => m.Status, m => m.StatusDate);
            return true;             
        }

        public ExternalCommunityMembershipMapping[] GetMemberships(string externalUserID)
        {
            return ExternalCommunityMembershipMapping.GetAllLocal(m => m.UserTypeAndCommunityIndex, externalUserID + Type + ExternalID)
                .Where(m => m.Status != ExternalCommunityMembershipStatus.Deleted).ToArray();
        }

        public bool ResolveMemberships(ExternalCommunityMembership[] memberships)
        {
            var changed = false;
            var mappings = ExternalCommunityMembershipMapping.MultiGetLocal(memberships.Select(m => new KeyInfo(m.ExternalUserID, ExternalID, Type, m.RoleTag, GroupID)))
                .ToDictionary(m => m.ExternalUserID);
            foreach (var membership in memberships)
            {
                ExternalCommunityMembershipMapping mapping;
                if (!mappings.TryGetValue(membership.ExternalUserID, out mapping))
                {
                    // Only care if an active membership is missing
                    if (membership.Status == ExternalCommunityMembershipStatus.Active)
                    {
                        changed |= DoAddMemberRoleMapping(membership, null);
                    }
                }
                else if (membership.Status == mapping.Status)
                {
                    // No change
                    continue;
                }
                else if (membership.Status == ExternalCommunityMembershipStatus.Active)
                {
                    changed |= DoAddMemberRoleMapping(membership, mapping);
                }
                else
                {
                    changed |= DoRemoveMemberRoleMapping(mapping);
                }
            }
            return changed;
        }

        public ExternalCommunityLinkContract ToContract(ExternalCommunity community)
        {
            return new ExternalCommunityLinkContract
            {
                Community = community.ToContract(),
                GroupID = GroupID,
                GracePeriodAction = GracePeriodExpireAction,
                GracePeriodDays = GracePeriodDays,
                SyncEmotes = SyncEmotes,
                SyncEvents = SyncEvents,
                SyncedPremiumMembers = SyncedPremiumMembers,
                IsLive = community.IsLive
            };
        }

        public void ChangeSettings(bool syncEmotes, bool syncEvents, SyncedMemberGracePeriodAction gracePeriodAction, int gracePeriodDays)
        {
            var updates = new List<Expression<Func<ExternalCommunityMapping, object>>>();

            if (SyncEmotes != syncEmotes)
            {
                SyncEmotes = syncEmotes;
                updates.Add(m=>m.SyncEmotes);
            }

            if (SyncEvents != syncEvents)
            {
                SyncEvents = syncEvents;
                updates.Add(m => m.SyncEvents);
            }

            if (GracePeriodExpireAction != gracePeriodAction)
            {
                GracePeriodExpireAction = gracePeriodAction;
                updates.Add(m => m.GracePeriodExpireAction);
            }

            if (GracePeriodDays != gracePeriodDays)
            {
                GracePeriodDays = gracePeriodDays;
                updates.Add(m => m.GracePeriodDays);
            }

            if (updates.Any())
            {
                Update(updates.ToArray());
            }
        }
    }
}
