﻿using System;
using Curse.Aerospike;
using Curse.Friends.Enums;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "ExternalCommunityRole", KeySpace = "CurseVoice-Global",ReplicationMode = ReplicationMode.Mesh)]
    public class ExternalCommunityRole : BaseTable<ExternalCommunityRole>
    {
        [Column("ExternalID",KeyOrdinal = 1, IsIndexed = true)]
        public string ExternalID { get; set; }

        [Column("Type", KeyOrdinal = 2)]
        public AccountType Type { get; set; }

        [Column("RoleTag", KeyOrdinal = 3)]
        public GroupRoleTag RoleTag { get; set; }

        [Column("VanityBadge")]
        public string VanityBadge { get; set; }

        [Column("VanityColor")]
        public int VanityColor { get; set; }

        [Column("CurrentEntry")]
        public int CurrentEntryNumber { get; set; }

        [Column("IsPremium")]
        public bool IsPremium { get; set; }

        public string GetRoleName()
        {
            return GetRoleName(Type, RoleTag);
        }

        public static string GetRoleName(AccountType type, GroupRoleTag tag)
        {
            switch (type)
            {
                case AccountType.Twitch:
                    return GetTwitchRoleName(tag);
                case AccountType.YouTube:
                    return GetYouTubeRoleName(tag);
                default:
                    throw new InvalidOperationException("Unsupported account type to be treated as an external role!");
            }
        }

        private static string GetTwitchRoleName(GroupRoleTag tag)
        {
            switch (tag)
            {
                case GroupRoleTag.SyncedOwner:
                    return "Streamer";
                case GroupRoleTag.SyncedModerator:
                    return "Moderator";
                case GroupRoleTag.SyncedSubscriber:
                    return "Subscriber";
                case GroupRoleTag.SyncedSubscriberTier2:
                    return "$9.99 Subscriber";
                case GroupRoleTag.SyncedSubscriberTier3:
                    return "$24.99 Subscriber";
                case GroupRoleTag.SyncedFollower:
                    return "Follower";
                default:
                    throw new InvalidOperationException("Unsupported role tag for Twitch role!");
            }
        }

        private static string GetYouTubeRoleName(GroupRoleTag tag)
        {
            switch (tag)
            {
                case GroupRoleTag.SyncedOwner:
                    return "Streamer";
                case GroupRoleTag.SyncedModerator:
                    return "Moderator";
                case GroupRoleTag.SyncedSubscriber:
                    return "Sponsor";
                case GroupRoleTag.SyncedFollower:
                    return "Subscriber";
                default:
                    throw new InvalidOperationException("Unsupported role tag for YouTube role!");
            }
        }

        public int RoleRank
        {
            get
            {
                switch (RoleTag)
                {
                    case GroupRoleTag.GuildMasterRank:
                        return 1000;
                    case GroupRoleTag.GuildRank1:
                        return 1001;
                    case GroupRoleTag.GuildRank2:
                        return 1002;
                    case GroupRoleTag.GuildRank3:
                        return 1003;
                    case GroupRoleTag.GuildRank4:
                        return 1004;
                    case GroupRoleTag.GuildRank5:
                        return 1005;
                    case GroupRoleTag.GuildRank6:
                        return 1006;
                    case GroupRoleTag.GuildRank7:
                        return 1007;
                    case GroupRoleTag.GuildRank8:
                        return 1008;
                    case GroupRoleTag.GuildRank9:
                        return 1009;

                    case GroupRoleTag.SyncedOwner:
                        return 100;
                    case GroupRoleTag.SyncedModerator:
                        return 110;
                    case GroupRoleTag.SyncedSubscriberTier3:
                        return 120;
                    case GroupRoleTag.SyncedSubscriberTier2:
                        return 130;
                    case GroupRoleTag.SyncedSubscriber:
                        return 140;
                    case GroupRoleTag.SyncedFollower:
                        return 150;

                    case GroupRoleTag.None:
                    default:
                        return 10000;
                }
            }
        }

        public bool CheckPremium()
        {
            return IsPremium || (Type == AccountType.Twitch && (RoleTag == GroupRoleTag.SyncedSubscriber || RoleTag == GroupRoleTag.SyncedSubscriberTier2 || RoleTag == GroupRoleTag.SyncedSubscriberTier3));
        }
    }
}
