﻿using System;
using System.Collections.Generic;
using System.Linq.Expressions;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Queues;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "ExternalGuild", ReplicationMode = ReplicationMode.Mesh, KeySpace = "CurseVoice-Global")]
    public class ExternalGuild : BaseTable<ExternalGuild>, IHostable
    {
        [Column("Source", KeyOrdinal = 1)]
        public AccountType Type { get; set; }

        [Column("Region", KeyOrdinal = 2)]
        public int GameRegion { get; set; }

        [Column("Server", KeyOrdinal = 3)]
        public string GameServer { get; set; }

        [Column("Name", KeyOrdinal = 4)]
        public string Name { get; set; }

        [Column("Faction")]
        public int Faction { get; set; }

        [Column("MemberCount")]
        public int MemberCount { get; set; }

        [Column("GuildMaster")]
        public string GuildMasterName { get; set; }

        [Column("GMServer")]
        public string GuildMasterGameServer { get; set; }

        [Column("Level")]
        public int Level { get; set; }

        [Column("Achievements")]
        public int AchievementPoints { get; set; }

        [Column("MappedGroups")]
        public HashSet<Guid> MappedGroups { get; set; }

        [Column("MachineName", IsIndexed = true)]
        public string MachineName { get; set; }

        [Column("RegionID", IsIndexed =  true)]
        public int RegionID { get; set; }

        [Column("LastSync")]
        public DateTime DateLastSynced { get; set; }

        #region Wow Emblem

        [Column("Icon")]
        public int EmblemIcon { get; set; }

        [Column("IconColor")]
        public string EmblemIconColor { get; set; }

        [Column("Border")]
        public int EmblemBorder { get; set; }

        [Column("BorderColor")]
        public string EmblemBorderColor { get; set; }

        [Column("BgColor")]
        public string EmblemBackgroundColor { get; set; }

        #endregion

        public bool IsHostable { get { return RegionID > 0; } }

        public object[] KeyObjects { get { return new object[] {Type, GameRegion, GameServer, Name}; } }

        public string DisplayName { get { return string.Format("{0}-{1}-{2}-{3}", Type, GameRegion, GameServer, Name); } }

        public static ExternalGuild Create(ExternalGuildInfo info)
        {
            var guild = new ExternalGuild
            {
                Type = info.GuildIdentifier.Type,
                GameRegion = info.GuildIdentifier.GameRegion,
                GameServer = info.GuildIdentifier.GameServer,
                Name = info.GuildIdentifier.Name,
                MappedGroups = new HashSet<Guid>(),
                DateLastSynced = DateTime.MinValue,

                Faction = info.Faction,
                GuildMasterName = info.GuildMasterIdentifier.Name ?? string.Empty,
                GuildMasterGameServer = info.GuildMasterIdentifier.GameServer ?? string.Empty,
                Level = info.Level,
                AchievementPoints = info.AchievementPoints,
                MemberCount = info.MemberCount,
            };

            if (info.Emblem != null)
            {
                guild.EmblemBackgroundColor = info.Emblem.BackgroundColor;
                guild.EmblemBorder = info.Emblem.Border;
                guild.EmblemBorderColor = info.Emblem.BorderColor;
                guild.EmblemIcon = info.Emblem.Icon;
                guild.EmblemIconColor = info.Emblem.IconColor;
            }

            guild.InsertLocal();
            return guild;
        }

        public void Update(ExternalGuildInfo info)
        {
            Faction = info.Faction;
            GuildMasterName = info.GuildMasterIdentifier.Name ?? string.Empty;
            GuildMasterGameServer = info.GuildMasterIdentifier.GameServer ?? string.Empty;
            Level = info.Level;
            AchievementPoints = info.AchievementPoints;

            if (info.MemberCount > 0)
            {
                MemberCount = info.MemberCount;
            }

            if (info.Emblem != null)
            {
                EmblemBackgroundColor = info.Emblem.BackgroundColor;
                EmblemBorder = info.Emblem.Border;
                EmblemBorderColor = info.Emblem.BorderColor;
                EmblemIcon = info.Emblem.Icon;
                EmblemIconColor = info.Emblem.IconColor;
            }

            Update(g => g.Faction, g => g.GuildMasterName, g => g.GuildMasterGameServer, g => g.Level, g => g.AchievementPoints,
                g => g.EmblemBackgroundColor, g => g.EmblemBorder, g => g.EmblemBorderColor, g => g.EmblemIcon, g => g.EmblemIconColor,
                g => g.MemberCount);
        }

        public static ExternalGuild Get(ExternalGuildIdentifier identifier)
        {
            return GetLocal(identifier.Type, identifier.GameRegion, identifier.GameServer, identifier.Name);
        }

        public ExternalGuildRole[] GetRoles()
        {
            return ExternalGuildRole.GetAll(SourceConfiguration, g => g.GuildIndex, GetGuildIndex());
        }

        public static string GetGuildIndex(ExternalGuildIdentifier identifier)
        {
            return GetGuildIndex(identifier.Type, identifier.GameRegion, identifier.GameServer, identifier.Name);
        }

        public string GetGuildIndex()
        {
            return GetGuildIndex(Type, GameRegion, GameServer, Name);
        }

        public static string GetGuildIndex(AccountType type, int gameRegion, string gameServer, string name)
        {
            return string.Join("-", type, gameRegion, gameServer, name);
        }

        public ExternalGuildIdentifier GetGuildInfo()
        {
            return GetGuildInfo(Type, GameRegion, GameServer, Name);
        }

        public static ExternalGuildIdentifier GetGuildInfo(AccountType type, int gameRegion, string gameServer, string name)
        {
            return new ExternalGuildIdentifier(type, gameRegion, gameServer, name);
        }

        public ExternalGuildIdentifier GetGuildMasterInfo()
        {
            return new ExternalGuildIdentifier(AccountType.WorldOfWarcraft, GameRegion, GuildMasterGameServer, GuildMasterName);
        }

        public ExternalGuildMapping GetMapping(Guid groupID)
        {
            return ExternalGuildMapping.Get(SourceConfiguration, Type, GameRegion, GameServer, Name, groupID);
        }

        public ExternalGuildMapping MapToGroup(GroupMember requestor, Group group)
        {
            var mapping = GetMapping(group.GroupID);
            if (mapping == null)
            {
                mapping = new ExternalGuildMapping
                {
                    Type = Type,
                    GameRegion = GameRegion,
                    Name = Name,
                    GameServer = GameServer,
                    GroupID = group.GroupID,
                };
                mapping.Insert(SourceConfiguration);
            }
            else if (mapping.IsDeleted)
            {
                mapping.IsDeleted = false;
                mapping.Update(m => m.IsDeleted);
            }

            var updates = new List<Expression<Func<ExternalGuild, object>>>();

            if (MappedGroups == null)
            {
                MappedGroups = new HashSet<Guid>();
            }


            if (MappedGroups.Add(group.GroupID))
            {
                updates.Add(c => c.MappedGroups);
            }

            if (RegionID == 0)
            {
                RegionID = group.RegionID;
                updates.Add(g => g.RegionID);
            }

            if (updates.Count > 0)
            {
                Update(updates.ToArray());
            }

            GroupEventManager.LogExternalGuildLinkedEvent(requestor, mapping);

            ExternalGuildSyncWorker.Linked(GetGuildInfo(), group.GroupID);
            if (string.IsNullOrEmpty(MachineName))
            {
                ExternalGuildCoordinator.CommissionGuild(this);
            }

            return mapping;
        }

        public ExternalGuildMapping DeleteMapping(GroupMember requestor, Guid groupID)
        {
            if (MappedGroups.Remove(groupID))
            {
                Update(s => s.MappedGroups);
            }

            var mapping = GetMapping(groupID);
            if (mapping != null)
            {
                mapping.IsDeleted = true;
                mapping.Update(m => m.IsDeleted);
                GroupEventManager.LogExternalGuildUnlinkedEvent(requestor, mapping);
            }

            ExternalGuildSyncWorker.Unlinked(GetGuildInfo(), groupID);
            if (MappedGroups.Count == 0)
            {
                ExternalGuildCoordinator.DecommissionGuild(this);
            }
            return mapping;
        }

        public GroupRoleTag GetMemberRole(string server, string name)
        {
            var member = ExternalGuildMember.Get(SourceConfiguration, Type, GameRegion, server, name);
            if (member == null)
            {
                return GroupRoleTag.None;
            }
            return member.GuildRole;
        }

        public void CreateRole(GroupRoleTag tag)
        {
            var role = new ExternalGuildRole
            {
                Type = Type,
                GuildGameRegion = GameRegion,
                GuildGameServer = GameServer,
                GuildName = Name,
                RoleTag = tag,
                GuildIndex = GetGuildIndex()
            };
            role.InsertLocal();
        }

        public void EnsureHost()
        {
            switch (Type)
            {
                case AccountType.WorldOfWarcraft:
                    BattleNetHostManager.EnsureServiceHost(this);
                    break;
                default:
                    throw new InvalidOperationException("Unsupported guild type");
            }
        }

        public static int GetGameIDFromType(AccountType type)
        {
            switch (type)
            {
                case AccountType.WorldOfWarcraft:
                    return 1;
                default:
                    throw new InvalidOperationException("Unsupported guild type");
            }
        }

        public ExternalGuildContract ToContract()
        {
            var contract = new ExternalGuildContract
            {
                Type = Type,
                GameRegion = GameRegion,
                GameServer = GameServer,
                Name = Name,
                MemberCount = MemberCount,
                GuildMasterGameServer = GuildMasterGameServer,
                GuildMasterName = GuildMasterName,
                AchievementPoints = AchievementPoints,
                Faction = Faction,
                Level = Level,
            };

            if (!string.IsNullOrEmpty(EmblemBackgroundColor))
            {
                contract.Emblem = new ExternalGuildEmblemContract
                {
                    BackgroundColor = EmblemBackgroundColor,
                    Border = EmblemBorder,
                    BorderColor = EmblemBorderColor,
                    Icon = EmblemIcon,
                    IconColor = EmblemIconColor
                };
            }

            return contract;
        }
    }
}
