﻿using System;
using System.Collections.Generic;
using System.Linq.Expressions;
using Curse.Aerospike;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "ExternalGuildMember", ReplicationMode = ReplicationMode.Mesh, KeySpace = "CurseVoice-Global")]
    public class ExternalGuildMember : BaseTable<ExternalGuildMember>
    {
        [Column("Source", KeyOrdinal = 1)]
        public AccountType Type { get; set; }

        [Column("Region", KeyOrdinal = 2)]
        public int GameRegion { get; set; }

        [Column("Server", KeyOrdinal = 3)]
        public string GameServer { get; set; }

        [Column("Name", KeyOrdinal = 4)]
        public string Name { get; set; }

        [Column("AccountID", IsIndexed = true)]
        public string AccountID { get; set; }

        [Column("GuildRegion")]
        public int GuildGameRegion { get; set; }

        [Column("GuildServer")]
        public string GuildGameServer { get; set; }

        [Column("GuildName")]
        public string GuildName { get; set; }

        [Column("GuildRole")]
        public GroupRoleTag GuildRole { get; set; }

        [Column("GuildIndex", IsIndexed = true)]
        public string GuildIndex { get; set; }

        [Column("Class")]
        public int Class { get; set; }

        [Column("Race")]
        public int Race { get; set; }

        [Column("Gender")]
        public int Gender { get; set; }

        [Column("Level")]
        public int Level { get; set; }


        public static ExternalGuildMember Create(AccountType type, ExternalGuildMemberInfo memberInfo)
        {
            var member = new ExternalGuildMember
            {
                Type = type,
                Name = memberInfo.Name,
                GameRegion = memberInfo.GameRegion,
                GameServer = memberInfo.GameServer,
                GuildRole = memberInfo.GuildRole,
                AccountID = memberInfo.AccountID,

                Class = memberInfo.Class,
                Gender = memberInfo.Gender,
                Race = memberInfo.Race,
                Level = memberInfo.Level
            };

            if (memberInfo.GuildIdentifier != null)
            {
                member.GuildName = memberInfo.GuildIdentifier.Name;
                member.GuildGameRegion = memberInfo.GuildIdentifier.GameRegion;
                member.GuildGameServer = memberInfo.GuildIdentifier.GameServer;
                member.GuildIndex = ExternalGuild.GetGuildIndex(memberInfo.GuildIdentifier);
            }
            member.InsertLocal();

            return member;
        }

        public bool Update(ExternalGuildMemberInfo memberInfo)
        {
            var updates = new List<Expression<Func<ExternalGuildMember, object>>>();

            if (AccountID != memberInfo.AccountID)
            {
                AccountID = memberInfo.AccountID;
                updates.Add(m => m.AccountID);
            }

            if (Class != memberInfo.Class)
            {
                Class = memberInfo.Class;
                updates.Add(m => m.Class);
            }

            if (Level != memberInfo.Level)
            {
                Level = memberInfo.Level;
                updates.Add(m => m.Level);
            }

            if (Gender != memberInfo.Gender)
            {
                Gender = memberInfo.Gender;
                updates.Add(m => m.Gender);
            }

            if (Race != memberInfo.Race)
            {
                Race = memberInfo.Race;
                updates.Add(m => m.Race);
            }

            if (memberInfo.GuildIdentifier != null)
            {
                if (GuildRole != memberInfo.GuildRole)
                {
                    GuildRole = memberInfo.GuildRole;
                    updates.Add(m => m.GuildRole);
                }

                if (GuildName != memberInfo.GuildIdentifier.Name)
                {
                    GuildName = memberInfo.GuildIdentifier.Name;
                    updates.Add(m => m.GuildName);
                }

                if (GuildGameRegion != memberInfo.GuildIdentifier.GameRegion)
                {
                    GuildGameRegion = memberInfo.GuildIdentifier.GameRegion;
                    updates.Add(m => m.GuildGameRegion);
                }

                if (GuildGameServer != memberInfo.GuildIdentifier.GameServer)
                {
                    GuildGameServer = memberInfo.GuildIdentifier.GameServer;
                    updates.Add(m => m.GuildGameServer);
                }

                var newIndex = ExternalGuild.GetGuildIndex(memberInfo.GuildIdentifier);
                if (GuildIndex != newIndex)
                {
                    GuildIndex = newIndex;
                    updates.Add(m => m.GuildIndex);
                }
            }

            if (updates.Count > 0)
            {
                Update(updates.ToArray());
                return true;
            }

            return false;
        }

        public static ExternalGuildMember CreateOrUpdate(AccountType type, ExternalGuildMemberInfo memberInfo)
        {
            var member = GetLocal(type, memberInfo.GameRegion, memberInfo.GameServer, memberInfo.Name);
            if (member == null)
            {
                member = Create(type, memberInfo);
            }
            else
            {
                member.Update(memberInfo);
            }

            return member;
        }

        public ExternalGuildIdentifier GetGuildInfo()
        {
            return new ExternalGuildIdentifier(Type, GuildGameRegion, GuildGameServer, GuildName);
        }
    }
}
