﻿using Curse.Aerospike;
using Curse.Friends.Enums;
using System;

using Curse.Extensions;
using Curse.Friends.NotificationContracts;
using System.Text;
using Curse.MurmurHash;

namespace Curse.Friends.Data
{
    

    [TableDefinition(TableName = "FriendHint", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class FriendHint : BaseTable<FriendHint>
    {

        public const int MaxListsPerUser = 20;
        public const int DefaultMaxStringLength = 64;
        public const int DescriptionMaxLength = 256;
        public const int MaxIdentitiesPerUser = 100;

        [Column("UserID", KeyOrdinal = 1, IsIndexed = true)]
        public int UserID
        {
            get;
            set;
        }

        /// <summary>
        /// The data used to actually search. Ex: SteamID or WoW Character Name
        /// </summary>
        [Column("SearchTerm", KeyOrdinal = 2)]
        public string SearchTerm
        {
            get;
            set;
        }

        [Column("Type", KeyOrdinal = 3)]
        public FriendHintType Type
        {
            get;
            set;
        }

        [Column("GameID", KeyOrdinal = 4)]
        public int GameID
        {
            get;
            set;
        }

        [Column("Region", KeyOrdinal = 5)]
        public string Region
        {
            get;
            set;
        }

        [Column("Server", KeyOrdinal = 6)]
        public string Server
        {
            get;
            set;
        }

        [Column("Site", KeyOrdinal = 7)]
        public string Site
        {
            get;
            set;
        }

        [Column("Platform", KeyOrdinal = 8, IsOptional = true)]
        public FriendPlatform Platform
        {
            get;
            set;
        }

        /// <summary>
        /// The name which appears in the search results. Ex: Steam Profile Name
        /// </summary>
        [Column("DisplayName")]
        public string DisplayName
        {
            get;
            set;
        }

        [Column("AvatarUrl")]
        public string AvatarUrl
        {
            get;
            set;
        }

        [Column("Description")]
        public string Description
        {
            get;
            set;
        }

        [Column("Verification")]
        public FriendHintVerification Verification
        {
            get;
            set;
        }

        [Column("Status")]
        public FriendHintStatus Status
        {
            get;
            set;
        }

        [Column("ProfileVis")]
        public FriendHintVisibility Visibility
        {
            get;
            set;
        }

        [Column("IndexMode")]
        public IndexMode IndexMode
        {
            get;
            set;
        }

        [Column("IsMerged")]
        public bool IsMerged
        {
            get; 
            set;
        }

        public bool HasChanges(FriendHint other)
        {
            if (DisplayName != other.DisplayName)
            {
                return true;
            }

            if (Status != other.Status)
            {
                return true;
            }

            if (Visibility != other.Visibility)
            {
                return true;
            }

            if (AvatarUrl != other.AvatarUrl)
            {
                return true;
            }

            if (Verification != other.Verification)
            {
                return true;
            }

            return false;
        }

        public string GetPublicName()
        {

            if (Type != FriendHintType.Platform)
            {
                return DisplayName ?? SearchTerm;
            }

            switch (Platform)
            {
                case FriendPlatform.BattleNet:
                    var index = SearchTerm.IndexOf('#');
                    return index <= 0 ? SearchTerm : SearchTerm.Substring(0, index);

                case FriendPlatform.Skype:
                    return SearchTerm;

                default:
                    return DisplayName ?? SearchTerm;
            }
        }

        public static string GetSanitizedDisplayName(string fullName)
        {
            try
            {
                var spaceIndex = fullName.IndexOf(' ');

                if (spaceIndex <= 0)
                {
                    return fullName;
                }

                return fullName.Substring(0, spaceIndex) + fullName.Substring(spaceIndex, 2);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to created a truncated name from: " + fullName);
                return fullName;
            }
        }
        
        public bool Validate(out string reason)
        {
            if (string.IsNullOrEmpty(SearchTerm) || SearchTerm.Length > Friendship.OtherUserNicknameMaxLength)
            {
                reason = "SearchTerm is empty or too long";
                return false;
            }

            if (Type == FriendHintType.Platform || Type == FriendHintType.Game)
            {
                if (string.IsNullOrEmpty(DisplayName) || DisplayName.SafeLength() > Friendship.OtherUserNicknameMaxLength)
                {
                    reason = "DisplayName is empty or too long";
                    return false;
                }
            }

            if (Description != null && Description.Length > DescriptionMaxLength)
            {
                reason = "Description is too long";
                return false;
            }

            if (Region != null && Region.Length > DefaultMaxStringLength)
            {
                reason = "Region is too long";
                return false;
            }

            if (Server != null && Server.Length > DefaultMaxStringLength)
            {
                reason = "Server is too long";
                return false;
            }

            if (Site != null && Site.Length > DefaultMaxStringLength)
            {
                reason = "Site is too long";
                return false;
            }

            if (Type == FriendHintType.Platform && Platform == FriendPlatform.Unknown)
            {
                reason = "Platform must be set for platform hints.";
                return false;
            }

            if (Type == FriendHintType.Game && GameID <= 0)
            {
                reason = "Game must be set for game hints.";
                return false;
            }

            reason = null;
            return true;
        }

        protected override void Validate()
        {
            string reason;

            if (!Validate(out reason))
            {
                throw new Exception(reason);
            }
        }

        public uint GetUniqueID()
        {
            var key = GetGeneratedKey();
            var bytes = Encoding.ASCII.GetBytes(key);
            return MurmurHash2.GetHashCode(ref bytes, key.Length, 1);
        }


        public string SanitizedDisplayName
        {
            get
            {
                if (Platform == FriendPlatform.BattleNet && DisplayName != null && !DisplayName.Contains("#"))
                {
                    return GetSanitizedDisplayName(DisplayName);
                }

                return DisplayName;
            }            
        }

        public FriendHintContract ToNotification(bool sanitizeDisplayName = true)
        {
            return new FriendHintContract
            {
                UserID = UserID,
                Status = Status,
                AvatarUrl = AvatarUrl,
                Platform = Platform,
                Description = Description,
                DisplayName = sanitizeDisplayName ? SanitizedDisplayName : DisplayName,
                GameID = GameID,
                Region = Region,
                SearchTerm = SearchTerm,
                Server = Server,
                Site = Site,
                Type = Type,
                Verification = Verification,
                Visibility = Visibility,
                ID = GetUniqueID()
            };
        }

        public FriendHintNotification ToLegacyNotification()
        {
            return new FriendHintNotification
            {
                UserID = UserID,
                Status = Status,
                AvatarUrl = AvatarUrl,
                Platform = Platform,
                Description = Description,
                DisplayName = DisplayName,
                GameID = GameID,
                Region = Region,
                SearchTerm = SearchTerm,
                Server = Server,
                Site = Site,
                Type = Type,
                Verification = Verification,
                Visibility = Visibility,
                ID = GetUniqueID()
            };
        }

        public static FriendHint FromNotification(FriendHintContract notification)
        {
            return new FriendHint
            {
                Type = notification.Type,
                Visibility = notification.Visibility,
                UserID = notification.UserID,
                Status = notification.Status,
                AvatarUrl = notification.AvatarUrl,
                GameID = notification.GameID,
                Verification = notification.Verification,
                Platform = notification.Platform,
                Description = notification.Description,
                Region = notification.Region,
                DisplayName = notification.DisplayName,
                Server = notification.Server,
                SearchTerm = notification.SearchTerm,
                Site = notification.Site, 
            };
        }

        public static FriendHint FromLegacyNotification(FriendHintNotification notification)
        {
            return new FriendHint
            {
                Type = notification.Type,
                Visibility = notification.Visibility,
                UserID = notification.UserID,
                Status = notification.Status,
                AvatarUrl = notification.AvatarUrl,
                GameID = notification.GameID,
                Verification = notification.Verification,
                Platform = notification.Platform,
                Description = notification.Description,
                Region = notification.Region,
                DisplayName = notification.DisplayName,
                Server = notification.Server,
                SearchTerm = notification.SearchTerm,
                Site = notification.Site
            };
        }
    }
}
