﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "FriendSuggestion", KeySpace = "CurseVoice", ReplicationMode = ReplicationMode.None)]
    public class FriendSuggestion : BaseTable<FriendSuggestion>
    {
        [Column("UserID", KeyOrdinal = 1, IsIndexed = true)]
        public int UserID
        {
            get;
            set;
        }

        [Column("OtherUserID", KeyOrdinal = 2, IsIndexed = true)]
        public int OtherUserID
        {
            get;
            set;
        }

        [Column("Username")]
        public string Username
        {
            get;
            set;
        }

        [Column("AvatarUrl")]
        public string AvatarUrl
        {
            get;
            set;
        }

        [Column("FriendCount")]
        public int FriendCount
        {
            get;
            set;
        }

        [Column("Status")]
        public FriendSuggestionStatus Status
        {
            get;
            set;
        }

        [Column("Type")]
        public FriendSuggestionType Type
        {
            get;
            set;
        }

        [Column("GameID")]
        public int GameID
        {
            get;
            set;
        }

        [Column("PlatformID")]
        public FriendPlatform Platform
        {
            get;
            set;
        }

        [Column("ReqUsername")]
        public string RequestUsername
        {
            get;
            set;
        }

        [Column("ReqAvatarUrl")]
        public string RequestAvatarUrl
        {
            get;
            set;
        }

        [Column("IndexMode")]
        public IndexMode IndexMode
        {
            get;
            set;
        }

        [Column("DateSuggested")]
        public DateTime DateSuggested
        {
            get;
            set;
        }

        public bool IsEquivalent(FriendSuggestion other)
        {
            if (Type != other.Type)
            {
                return false;
            }

            if (Username != other.Username)
            {
                return false;
            }

            if (Type == FriendSuggestionType.GameFriend)
            {
                if (GameID != other.GameID)
                {
                    return false;
                }
            }
            else if (Type == FriendSuggestionType.PlatformFriend)
            {
                if (Platform != other.Platform)
                {
                    return false;
                }
            }

            return true;
        }

        public FriendSuggestionContract ToNotification(UserStatistics userStatistics)
        {
            
            var contract = new FriendSuggestionContract
            {
                AvatarUrl = AvatarUrl,
                Username = Username,
                Platform = Platform,
                GameID = GameID,
                Type = Type,
                FriendCount = FriendCount,
                FriendID = OtherUserID,
                SuggestedTimestamp = DateSuggested.SafeToEpochMilliseconds()
            };

            // For Battle.net, ensure that no read ID names are leaked
            if (contract.Platform == FriendPlatform.BattleNet && !contract.Username.Contains("#"))
            {
                contract.Username = FriendHint.GetSanitizedDisplayName(contract.Username);
            }

            if (userStatistics != null)
            {
                contract.FriendIDs = userStatistics.FriendIDs;
            }

            return contract;
        }

        public bool IsValidSuggestion(int userID, UserStatistics userStatistics, UserPrivacySettings privacySettings)
        {
            if (privacySettings != null)
            {
                if (privacySettings.FriendRequestPrivacy == FriendRequestPrivacy.NoOne)
                {
                    // Not accepting friends
                    return false;
                }

                if (privacySettings.FriendRequestPrivacy == FriendRequestPrivacy.FriendsOfFriends &&
                    (userStatistics == null || !userStatistics.FriendIDs.Contains(userID)))
                {
                    // Not a friend of a friend
                    return false;
                }
            }

            return true;
        }

        public static FriendSuggestion[] GetValidFriendSuggestions(int userID, FriendSuggestion[] friendSuggestions, Dictionary<int, UserStatistics> userStatistics = null,
            Dictionary<int, UserPrivacySettings> privacySettings = null)
        {
            var uniqueUserIDs = new HashSet<int>(friendSuggestions.Select(f => f.OtherUserID));
            userStatistics = userStatistics ?? UserStatistics.GetDictionaryByUserIDs(uniqueUserIDs);
            privacySettings = privacySettings ?? UserPrivacySettings.GetDictionaryByUserIDs(uniqueUserIDs);

            var suggestions = new List<FriendSuggestion>();

            foreach (var suggestion in friendSuggestions)
            {
                var statistics = userStatistics.GetValueOrDefault(suggestion.OtherUserID);
                var privacy = privacySettings.GetValueOrDefault(suggestion.OtherUserID);

                if (!suggestion.IsValidSuggestion(userID, statistics, privacy))
                {
                    continue;
                }

                suggestions.Add(suggestion);
            }

            return suggestions.ToArray();
        }


        public static FriendSuggestionContract[] ToValidNotifications(int userID, FriendSuggestion[] friendSuggestions, Dictionary<int, UserStatistics> userStatistics = null,
            Dictionary<int, UserPrivacySettings> privacySettings = null)
        {
            var uniqueUserIDs = new HashSet<int>(friendSuggestions.Select(f => f.OtherUserID));
            userStatistics = userStatistics ?? UserStatistics.GetDictionaryByUserIDs(uniqueUserIDs);
            privacySettings = privacySettings ?? UserPrivacySettings.GetDictionaryByUserIDs(uniqueUserIDs);

            var contracts = new List<FriendSuggestionContract>();

            foreach (var suggestion in friendSuggestions)
            {
                var statistics = userStatistics.GetValueOrDefault(suggestion.OtherUserID);
                var privacy = privacySettings.GetValueOrDefault(suggestion.OtherUserID);

                if (!suggestion.IsValidSuggestion(userID, statistics, privacy))
                {
                    continue;
                }

                contracts.Add(suggestion.ToNotification(statistics));
            }

            return contracts.ToArray();
        }
    }
}
