﻿using System.Collections.Generic;
using Curse.Friends.Enums;
using System;
using System.Linq;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data.Models;
using Curse.Friends.NotificationContracts;
using System.Diagnostics;

namespace Curse.Friends.Data
{

    [TableDefinition(TableName = "Friendship", KeySpace = "CurseVoice", ReplicationMode = ReplicationMode.None)]
    [DebuggerDisplay("{UserID} -> {OtherUsername}: {Status}")]
    public class Friendship : BaseTable<Friendship>
    {

        public const int OtherUserNicknameMaxLength = 64;
        public const int InvitationMaxLength = 256;
        public const int MaxFriendCount = 500;
        public const int CustomStatusMaxLength = 140;

        [Column("UserID", KeyOrdinal = 1, IsIndexed = true)]
        public int UserID
        {
            get;
            set;
        }

        /// <summary>
        /// UserID of the friend
        /// </summary>
        [Column("OtherUserID", KeyOrdinal = 2, IsIndexed = true)]
        public int OtherUserID
        {
            get;
            set;
        }

        /// <summary>
        /// friendshipstatus with otherUserID
        /// </summary>
        [Column("Status")]
        public FriendshipStatus Status
        {
            get;
            set;
        }

        /// <summary>
        /// friends name
        /// </summary>
        [Column("Username")]
        public string OtherUsername
        {
            get;
            set;
        }

        [Column("DisplayName")]
        public string OtherUserDisplayName
        {
            get;
            set;
        }

        [Column("Nickname")]
        public string OtherUserNickname
        {
            get;
            set;
        }

        [Column("RegionID")]
        public int OtherUserRegionID
        {
            get;
            set;
        }


        [Column("InviteMessage")]
        public string InvitationMessage
        {
            get;
            set;
        }

        [Column("IsFavorite")]
        public bool IsFavorite
        {
            get;
            set;
        }

        [Column("AvatarUrl")]
        [Obsolete]
        public string OtherUserAvatarUrl
        {
            get;
            set;
        }

        [Column("DateConfirmed")]
        public DateTime DateConfirmed
        {
            get;
            set;
        }

        [Column("DateMessaged")]
        public DateTime DateMessaged
        {
            get;
            set;
        }

        [Column("UnreadCount")]
        public int UnreadCount
        {
            get;
            set;
        }

        [Column("DateRead")]
        public DateTime DateRead
        {
            get;
            set;
        }

        [Column("IndexMode")]
        public IndexMode IndexMode
        {
            get;
            set;
        }

        [Column("FriendCount")]
        public int MutualFriendCount
        {
            get;
            set;
        }

        [Column("NotifPref")]
        public NotificationPreference NotificationPreference
        {
            get;
            set;
        }

        [Column("NotifFilter")]
        public HashSet<string> NotificationFilters
        {
            get;
            set;
        }

        [Column("DateRequested")]
        public DateTime DateRequested
        {
            get;
            set;
        }

        [Column("DateRemoved")]
        public DateTime DateRemoved
        {
            get;
            set;
        }

        public bool IsSurfaced
        {
            get
            {
                return Status == FriendshipStatus.AwaitingMe
                    || Status == FriendshipStatus.AwaitingThem
                    || Status == FriendshipStatus.Confirmed;
            }
        }

        public string ResolvedName
        {
            get { return string.IsNullOrEmpty(OtherUserDisplayName) ? OtherUsername : OtherUserDisplayName; }
        }

        public bool HasDifferentDisplayName
        {
            get { return !string.IsNullOrEmpty(OtherUserDisplayName) && !OtherUsername.Equals(OtherUserDisplayName, StringComparison.InvariantCultureIgnoreCase); }
        }

        public string FormattedDisplayName
        {
            get
            {
                if (!string.IsNullOrEmpty(OtherUserNickname))
                {
                    return OtherUserNickname;
                }

                if (HasDifferentDisplayName)
                {
                    return OtherUserDisplayName + " (" + OtherUsername + ")";

                }

                return ResolvedName;
            }
        }

        public string DisplayName
        {
            get
            {
                if (!string.IsNullOrEmpty(OtherUserNickname))
                {
                    return OtherUserNickname;
                }

                return OtherUsername;
            }
        }

        public static void UpdateIsFavorite(int regionID, int userID, int otherUserID, bool isFavorite)
        {
            var model = Get(regionID, userID, otherUserID);
            model.IsFavorite = isFavorite;
            model.Update(p => p.IsFavorite);
        }

        public static void UpdateNickname(int regionID, int userID, int otherUserID, string nickname)
        {
            var model = Get(regionID, userID, otherUserID);
            model.OtherUserNickname = nickname.ToEmptyWhenNull();
            model.Update(p => p.OtherUserNickname);
        }

        public static int GetCountByUserID(int regionID, int userID)
        {
            var allFriends = GetAll(regionID, p => p.UserID, userID);
            return allFriends.Count(p => p.Status == FriendshipStatus.Confirmed);
        }

        public void UpdateNotificationPreferences(NotificationPreference preference, HashSet<string> filters)
        {
            NotificationPreference = preference;
            NotificationFilters = filters;
            Update(f => f.NotificationPreference, f => f.NotificationFilters);
        }

        /// <summary>
        /// Returns the number of confirmed or requested friendships, for the purpose of enforcing the limit.
        /// </summary>
        /// <param name="regionID"></param>
        /// <param name="userID"></param>
        /// <returns></returns>
        public static int GetConfirmedOrRequestedCount(int regionID, int userID)
        {
            var allFriends = GetAll(regionID, p => p.UserID, userID);
            return allFriends.Count(p => p.Status == FriendshipStatus.Confirmed || p.Status == FriendshipStatus.AwaitingThem);
        }

        public static void Confirm(User me, UserRegion myRegion, Friendship myFriendship, User them, UserRegion theirRegion, Friendship theirFriendship)
        {

            theirFriendship.Confirm(theirRegion.RegionID, me);
            myFriendship.Confirm(myRegion.RegionID, them);
            UpdateFriendStats(me, myRegion);
            UpdateFriendStats(them, theirRegion);

            // If a invitation message was included with the request, save it out as an offline message
            if (!string.IsNullOrEmpty(myFriendship.InvitationMessage))
            {
                PrivateMessageWorker.Create(them.UserID, null, PrivateConversation.GenerateConversationID(me.UserID, them.UserID), myFriendship.InvitationMessage, Guid.NewGuid(), null);
            }

            // Queue off a notification for each user                
            new FriendshipChangeResolver { InitiatingUserID = me.UserID, UserID = me.UserID, Friend = myFriendship }.Enqueue();
            new FriendshipChangeResolver { InitiatingUserID = me.UserID, UserID = them.UserID, Friend = theirFriendship }.Enqueue();
        }

        private void Confirm(int regionID, User otherUser)
        {
            Status = FriendshipStatus.Confirmed;
            OtherUsername = otherUser.Username;
            OtherUserDisplayName = otherUser.DisplayName;
            DateConfirmed = DateTime.UtcNow;
            Update();

            var conversation = GetConversation();
            conversation.IsFriend = true;
            conversation.Title = FormattedDisplayName;
            conversation.Update(p => p.IsFriend, p => p.Title);

            // See if this came from a suggestion. If so, mark it as accepted.
            var suggestion = FriendSuggestion.Get(regionID, UserID, otherUser.UserID);
            if (suggestion != null)
            {
                suggestion.Status = FriendSuggestionStatus.Accepted;
                suggestion.Update();
            }
        }

        private static void UpdateFriendStats(User user, UserRegion userRegion)
        {
            var friends = GetAllConfirmed(userRegion.RegionID, user.UserID);
            user.UpdateFriendCount(userRegion.RegionID, friends);
            user.UpdateUserStats(friends);
            FriendHintSearchIndexer.CreateForUser(user.UserID);
        }

        public static void Remove(bool blockFutureRequests, User me, Friendship myFriendship, UserRegion myRegion, User them, Friendship theirFriendship, UserRegion theirRegion)
        {

            // See if this came from a suggestion. If so, mark it as declined.
            var suggestion = FriendSuggestion.Get(myRegion.RegionID, me.UserID, them.UserID);
            if (suggestion != null)
            {
                suggestion.Status = FriendSuggestionStatus.Declined;
                suggestion.Update();
            }

            var myConversation = myFriendship.GetConversation();
            myConversation.IsHidden = true;
            myConversation.IsFriend = false;
            myConversation.Update(p => p.IsHidden, p => p.IsFriend);

            var theirConversation = theirFriendship.GetConversation();
            theirConversation.IsHidden = true;
            theirConversation.IsFriend = false;
            theirConversation.Update(p => p.IsHidden, p => p.IsFriend);

            if (blockFutureRequests)
            {
                theirFriendship.Status = FriendshipStatus.DeclinedByThem;
                theirFriendship.Update(p => p.Status);

                myFriendship.Status = FriendshipStatus.DeclinedByMe;
                myFriendship.Update(p => p.Status);
            }
            else
            {
                var now = DateTime.UtcNow;

                // Soft delete my side
                myFriendship.Status = FriendshipStatus.Deleted;
                myFriendship.DateRemoved = now;
                myFriendship.Update(p => p.Status, p => p.DateRemoved);


                // Soft delete their side
                theirFriendship.Status = FriendshipStatus.Deleted;
                theirFriendship.DateRemoved = now;
                theirFriendship.Update(p => p.Status, p => p.DateRemoved);
            }

            UpdateFriendStats(me, myRegion);
            UpdateFriendStats(them, theirRegion);
        }

        private class FriendIdentity
        {
            public string Username { get; private set; }
            public string DisplayName { get; private set; }

            public string Nickname { get; private set; }

            public FriendIdentity(User user, string knownIdentity)
            {

                var isIdentityKnown = knownIdentity == null
                                    || user.Username.Equals(knownIdentity, StringComparison.InvariantCultureIgnoreCase)
                                    || user.DisplayName.ToEmptyWhenNull().Equals(knownIdentity, StringComparison.InvariantCultureIgnoreCase)
                                    || user.EmailAddress.ToEmptyWhenNull().Equals(knownIdentity, StringComparison.InvariantCultureIgnoreCase);

                if (isIdentityKnown) // User knows the targets 
                {
                    Username = user.Username;
                    DisplayName = user.DisplayName;                    
                }
                else
                {
                    Username = knownIdentity;
                    DisplayName = null;
                    Nickname = knownIdentity;
                }

                Username = Username.ToEmptyWhenNull();
                DisplayName = DisplayName.ToEmptyWhenNull();
                Nickname = Nickname.ToEmptyWhenNull();
            }
        }

        public static void SystemCreateFriendship(User requestor, UserRegion requestorRegion, User target, UserRegion targetRegion, FriendshipStatus requestorStatus, FriendshipStatus targetStatus)
        {
            var mutualFriendCount = User.GetMutualFriends(requestor.UserID, target.UserID).Count();

            var requestorStatusChanged = false;
            var requestorFriendship = Get(requestorRegion.RegionID, requestor.UserID, target.UserID);
            if (requestorFriendship == null)
            {
                // Create the friendship request
                requestorFriendship = new Friendship
                {
                    UserID = requestor.UserID,
                    OtherUserID = target.UserID,
                    OtherUsername = target.Username,
                    OtherUserDisplayName = target.DisplayName,
                    OtherUserRegionID = targetRegion.RegionID,
                    Status = requestorStatus,
                    MutualFriendCount = mutualFriendCount,
                };

                requestorFriendship.Insert(requestorRegion.RegionID);
                requestorStatusChanged = true;
            }
            else
            {
                if (requestorStatus != requestorFriendship.Status)
                {
                    requestorFriendship.Status = requestorStatus;
                    requestorStatusChanged = true;
                }
                requestorFriendship.OtherUsername = target.Username;
                requestorFriendship.OtherUserDisplayName = target.DisplayName;
                requestorFriendship.MutualFriendCount = mutualFriendCount;
                requestorFriendship.Update();
            }

            var targetStatusChanged = false;
            var targetFriendship = Get(targetRegion.RegionID, target.UserID, requestor.UserID);
            if (targetFriendship == null)
            {
                targetFriendship = new Friendship
                {
                    UserID = target.UserID,
                    OtherUserID = requestor.UserID,
                    OtherUsername = requestor.Username,
                    OtherUserDisplayName = requestor.DisplayName,
                    OtherUserRegionID = requestorRegion.RegionID,
                    Status = targetStatus,
                    MutualFriendCount = mutualFriendCount,
                };

                targetFriendship.Insert(targetRegion.RegionID);
                targetStatusChanged = true;
            }
            else
            {
                if (targetStatus != targetFriendship.Status)
                {
                    targetStatusChanged = true;
                    targetFriendship.Status = targetStatus;
                }
                targetFriendship.OtherUsername = requestor.Username;
                targetFriendship.OtherUserDisplayName = requestor.DisplayName;
                targetFriendship.MutualFriendCount = mutualFriendCount;
                targetFriendship.Update();
            }

            if (requestorStatusChanged)
            {
                new FriendshipChangeResolver {InitiatingUserID = requestor.UserID, UserID = requestor.UserID, Friend = requestorFriendship}.Enqueue();
            }
            if (targetStatusChanged)
            {
                new FriendshipChangeResolver { InitiatingUserID = requestor.UserID, UserID = target.UserID, Friend = targetFriendship }.Enqueue();
            }
        }

        public static Tuple<Friendship, Friendship> CreateRequest(User requestor, UserRegion requestorRegion, User target, UserRegion targetRegion, string knownIdentity, string invitationMessage,
            bool isFromSuggestion, Friendship requestorFriendship = null, Friendship targetFriendship = null)
        {
            FriendSuggestion suggestion = null;
            if (isFromSuggestion)
            {
                suggestion = FriendSuggestion.Get(requestorRegion.RegionID, requestor.UserID, target.UserID);
                if (suggestion != null && suggestion.Status == FriendSuggestionStatus.Pending)
                {
                    suggestion.Status = FriendSuggestionStatus.Accepted;
                    suggestion.Update();
                }
            }

            // If the requesting user has blocked the target user, automatically unblock them
            UserBlock.ToggleBlock(requestor, target, UserBlockStatus.Inactive);

            // Check if the known identity is 
            
            var mutualFriendCount = User.GetMutualFriends(requestor.UserID, target.UserID).Count();
            var now = DateTime.UtcNow;

            // The identity of the target
            var targetIdentity = new FriendIdentity(target, knownIdentity);
            var requestorIdentity = new FriendIdentity(requestor, !string.IsNullOrWhiteSpace(suggestion?.RequestUsername) ? suggestion.RequestUsername : null);

            if (requestorFriendship == null)
            {
                // Create the friendship request
                requestorFriendship = new Friendship
                {
                    UserID = requestor.UserID,
                    OtherUserID = target.UserID,
                    OtherUsername = targetIdentity.Username,
                    OtherUserDisplayName = targetIdentity.DisplayName,
                    OtherUserNickname = targetIdentity.Nickname,
                    OtherUserRegionID = targetRegion.RegionID,
                    Status = FriendshipStatus.AwaitingThem,                    
                    MutualFriendCount = mutualFriendCount,
                    DateRequested = now
                };

                requestorFriendship.Insert(requestorRegion.RegionID);
            }
            else
            {
                // Update the friendship
                requestorFriendship.Status = FriendshipStatus.AwaitingThem;
                requestorFriendship.OtherUsername = targetIdentity.Username;
                requestorFriendship.OtherUserDisplayName = targetIdentity.DisplayName;
                requestorFriendship.OtherUserNickname = targetIdentity.Nickname;
                requestorFriendship.MutualFriendCount = mutualFriendCount;
                requestorFriendship.DateRequested = now;
                requestorFriendship.Update();

            }
            
            if (targetFriendship == null)
            {
                targetFriendship = new Friendship
                {
                    UserID = target.UserID,
                    OtherUserID = requestor.UserID,
                    OtherUsername = requestorIdentity.Username,
                    OtherUserDisplayName = requestorIdentity.DisplayName,
                    OtherUserNickname = requestorIdentity.Nickname,
                    OtherUserRegionID = requestorRegion.RegionID,
                    Status = FriendshipStatus.AwaitingMe,                    
                    InvitationMessage = invitationMessage,
                    MutualFriendCount = mutualFriendCount,
                    DateRequested = now
                };

                targetFriendship.Insert(targetRegion.RegionID);
            }
            else
            {
                targetFriendship.Status = FriendshipStatus.AwaitingMe;
                targetFriendship.OtherUsername = requestorIdentity.Username;
                targetFriendship.OtherUserDisplayName = requestorIdentity.DisplayName;
                targetFriendship.OtherUserNickname = requestorIdentity.Nickname;
                targetFriendship.MutualFriendCount = mutualFriendCount;
                targetFriendship.InvitationMessage = invitationMessage;
                targetFriendship.DateRequested = now;
                targetFriendship.Update();
            }
            
            // Queue off a notification for each user                
            new FriendshipChangeResolver { InitiatingUserID = requestor.UserID, UserID = requestor.UserID, Friend = requestorFriendship }.Enqueue();
            new FriendshipChangeResolver { InitiatingUserID = requestor.UserID, UserID = target.UserID, Friend = targetFriendship }.Enqueue();

            return new Tuple<Friendship, Friendship>(requestorFriendship, targetFriendship);
        }

        public static Friendship GetConfirmedLocal(int userID, int friendID)
        {
            var friendship = GetLocal(userID, friendID);

            if (friendship == null || friendship.Status != FriendshipStatus.Confirmed)
            {
                return null;
            }

            return friendship;
        }

        public static Friendship[] GetAllConfirmed(int regionID, int userID)
        {
            return GetAll(regionID, p => p.UserID, userID).Where(p => p.Status == FriendshipStatus.Confirmed).ToArray();
        }

        public static bool IsConfirmed(int userID, int friendID)
        {
            var senderRegion = UserRegion.GetByUserID(userID);
            var friendship = Get(senderRegion.RegionID, userID, friendID);
            return friendship != null && friendship.Status == FriendshipStatus.Confirmed;
        }

        public static bool IsConfirmed(int regionID, int userID, int friendID)
        {
            var friendship = Get(regionID, userID, friendID);
            return friendship != null && friendship.Status == FriendshipStatus.Confirmed;
        }

        public static HashSet<int> GetFriendRegionIDs(int regionID, int userID)
        {
            var regionIDs = GetAllBins(regionID, p => p.UserID, userID, p => p.OtherUserRegionID);
            return new HashSet<int>(regionIDs);
        }

        public FriendshipContract ToNotification()
        {
            var userStats = UserStatistics.GetByUserID(OtherUserID);
            var conversation = PrivateConversation.GetByUserIDAndOtherUserID(UserID, OtherUserID);
            var watching = userStats?.GetWatchingCommunity();
            return ToNotification(userStats, conversation, watching);
        }

        public FriendshipContract ToNotification(UserStatistics stats, PrivateConversation privateConversation, ExternalCommunity watchingChannel)
        {
            var contract = new FriendshipContract
            {
                IsFavorite = IsFavorite,
                InvitationMessage = InvitationMessage,
                OtherUserID = OtherUserID,
                OtherUsername = OtherUsername,
                OtherDisplayName = OtherUserDisplayName,
                OtherUserNickname = OtherUserNickname,
                OtherUserRegionID = OtherUserRegionID,
                Status = Status,
                DateConfirmed = DateConfirmed,
                DateMessaged = privateConversation?.DateMessaged ?? DateMessaged,
                UnreadCount = privateConversation?.UnreadCount ?? UnreadCount,
                DateRead = privateConversation?.DateRead ?? DateRead,
                MutualFriendCount = MutualFriendCount,
                RequestedTimestamp = DateRequested.SafeToEpochMilliseconds(),
                OtherUserTwitchID = stats?.TwitchID,
            };

            if (Status == FriendshipStatus.Confirmed && stats != null)
            {
                contract.OtherUserConnectionStatus = stats.ConnectionStatus;
                contract.OtherUserStatusMessage = stats.StatusMessage;
                contract.OtherUserGameID = stats.CurrentGameID;
                contract.OtherUserGameStatusMessage = stats.GameStatusMessage;
                contract.OtherUserGameState = stats.GameState;
                contract.OtherUserGameTimestamp = stats.GameTimestamp.FromEpochMilliconds();
                contract.OtherUserConnectionStatusTimestamp = stats.ConnectStatusTimestamp;
                contract.AvatarTimestamp = stats.AvatarTimestamp;
                contract.OtherBroadcastingGameID = stats.BroadcastingGameID;
                contract.OtherIsBroadcasting = stats.IsBroadcasting;
                contract.Capabilities = stats.Capabilities;

                if (watchingChannel != null)
                {
                    contract.OtherWatchingChannelName = watchingChannel.ExternalDisplayName;
                    contract.OtherWatchingChannelUrl = watchingChannel.GetExternalUrl();
                    contract.OtherWatchingChannelID = watchingChannel.ExternalID;
                }
            }
            else
            {
                contract.OtherUserConnectionStatus = UserConnectionStatus.Offline;
            }

            return contract;
        }

        public static Friendship GetByOtherUserID(int requestingUserID, int otherUserID)
        {
            var userRegion = UserRegion.GetByUserID(requestingUserID);
            if (userRegion == null)
            {
                return null;
            }

            return Get(userRegion.RegionID, requestingUserID, otherUserID);
        }

        public string Title
        {
            get { return string.IsNullOrEmpty(OtherUserNickname) ? OtherUsername : OtherUserNickname; }
        }



        #region IAvatarParent

        public string AvatarUrlSlug
        {
            get { return User.AvatarUrlPath; }
        }

        public string AvatarUrlID
        {
            get { return OtherUserID.ToString(); }
        }

        #endregion

        public PrivateConversation GetConversation()
        {
            return PrivateConversation.GetOrCreateByFriendship(this);
        }
    }
}
