﻿using System;
using System.Linq;
using Curse.Aerospike;
using System.Collections.Generic;
using System.IO;
using System.Linq.Expressions;
using System.Net;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Search;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Newtonsoft.Json;
using Curse.Friends.Data.Messaging;
using Aerospike.Client;

namespace Curse.Friends.Data
{

    [TableDefinition(TableName = "Group", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.HomeRegion)]
    public class Group : BaseTable<Group>, IModelRegion, IHostable, IConversationContainer
    {        
        public const int TitleMaxLength = 256;
        public const int ChannelTitleMaxLength = 64;
        public const int AvatarMaxLength = 256;
        public const int MessageOfTheDayMaxLength = 256;
        public const int MaxNestingDepth = 2;
        public const int MaxPermissions = 64;
        public const int MaxChannelCategoryNameLength = 128;

        public const int MaxServersOwned = 100;
        public const int MaxGroupsOwned = 200;
        public const int MaxUsersInCall = 50;
        public const int MaxUsersForCallNotification = 50;
        public const int MaxUsersInGroup = 1000;
        public const int MaxUsersInLargeGroup = 200000;
        public const int MaxChannelCount = 300;
        public const int MaxImmediateChannels = 50;
        public const int MaxRootChannels = 200;

        public const int MaxNumberFilterWords = 100;
        public const int MaxTotalFilterLength = 32000;

        public const int MemberCountPttThreshold = 10;

        public const int IntegrityCheckIntervalHours = 1;
        public const int MemberIndexSyncIntervalHours = 1;

        public const int ManualSyncThrottleSeconds = 60;

        [Column("GroupID", KeyOrdinal = 1)]
        public Guid GroupID
        {
            get;
            set;
        }

        /// <summary>
        /// This group's home region. After creation, this should never change.
        /// </summary>
        [Column("RegionID")]
        public int RegionID
        {
            get;
            set;
        }

        public object[] KeyObjects { get { return new object[] { GroupID }; } }

        public string DisplayName { get { return Title; } }

        public bool IsHostable { get { return IsRootGroup; } }

        /// <summary>
        /// The name of the machine hosting this groups session.
        /// </summary>
        [Column("MachineName", IsIndexed = true)]
        public string MachineName
        {
            get;
            set;
        }

        [Column("Title")]
        public string Title
        {
            get;
            set;
        }

        [Column("Avatar")]
        public string AvatarUrl
        {
            get;
            set;
        }

        [Column("Status")]
        public GroupStatus Status
        {
            get;
            set;
        }

        [Column("Type")]
        public GroupType Type
        {
            get;
            set;
        }

        [Column("Subtype")]
        public GroupSubType Subtype
        {
            get;
            set;
        }

        [Column("Mode")]
        public GroupMode Mode
        {
            get;
            set;
        }

        [Column("DisplayOrder")]
        public int DisplayOrder
        {
            get;
            set;
        }

        [Column("CategoryID")]
        public Guid DisplayCategoryID
        {
            get;
            set;
        }

        [Column("Category")]
        public string DisplayCategoryName
        {
            get;
            set;
        }

        [Column("CategoryRank")]
        public int DisplayCategoryRank
        {
            get;
            set;
        }

        [Column("DefaultChannel")]
        public bool IsDefaultChannel
        {
            get;
            set;
        }

        [Column("MemberCount")]
        public int MemberCount
        {
            get;
            set;
        }

        [Column("ParentGroupID", IsIndexed = true)]
        public Guid ParentGroupID
        {
            get;
            set;
        }

        [Column("RootGroupID", IsIndexed = true)]
        public Guid RootGroupID
        {
            get;
            set;
        }

        [Column("DefaultGroupID")]
        public Guid DefaultGroupID
        {
            get;
            set;
        }

        /// <summary>
        /// Allow users to create temporary child groups for the purposes of voice calls
        /// </summary>
        [Column("TempChannels")]
        public bool AllowTempChannels
        {
            get;
            set;
        }

        [Column("VoiceSession")]
        public string VoiceSessionCode
        {
            get;
            set;
        }

        [Column("MOTD")]
        public string MessageOfTheDay
        {
            get;
            set;
        }

        [Column("ForcePTT")]
        public bool? ForcePushToTalk
        {
            get;
            set;
        }

        [Column("IndexMode")]
        public IndexMode IndexMode
        {
            get;
            set;
        }

        [Column("DateCreated")]
        public DateTime DateCreated
        {
            get;
            set;
        }

        [Column("CreatorID")]
        public int CreatorID
        {
            get;
            set;
        }

        /// <summary>
        /// Effective owner of the group (there may be multiple owners, but only one is needed to be an effective owner)
        /// </summary>
        [Column("OwnerID")]
        public int OwnerID
        {
            get;
            set;
        }

        [Column("DateMessaged")]
        public DateTime DateMessaged
        {
            get;
            set;
        }

        /// <summary>
        /// The date this group was last integrity checked.
        /// </summary>
        [Column("DateChecked")]
        public DateTime DateChecked
        {
            get;
            set;
        }

        /// <summary>
        /// The date this group last had its member list indexed.
        /// </summary>
        [Column("DateMembersIdx")]
        public DateTime DateMembersIndexed
        {
            get;
            set;
        }

        /// <summary>
        /// The date this group last had its member list indexed.
        /// </summary>
        [Column("MemberActSaved")]
        public DateTime DateMemberActivitySaved
        {
            get;
            set;
        }

        [Column("DefaultRoleID")]
        public int DefaultRoleID
        {
            get;
            set;
        }

        [Column("OwnerRoleID")]
        public int OwnerRoleID
        {
            get;
            set;
        }

        /// <summary>
        /// The effective permissions for a given role ID
        /// </summary>
        [Column("RolePerms")]
        public Dictionary<int, Int64> RolePermissions
        {
            get;
            set;
        }

        [Column("RoleCounter")]
        public int RoleCounter
        {
            get;
            set;
        }

        [Column("ChildGroups")]
        public HashSet<Guid> ChildGroupIDs
        {
            get;
            set;
        }

        [Column("PollNumber")]
        public int LatestPollNumber { get; set; }

        [Column("GiveawayNumber")]
        public int LatestGiveawayNumber { get; set; }

        /// <summary>
        /// The sub channel to move users to, if they go AFK in a voice channel
        /// </summary>
        [Column("AfkChannelID")]
        public Guid AfkChannelID
        {
            get;
            set;
        }

        /// <summary>
        /// How long a user can be idle before being moved to the AFK channel.
        /// </summary>
        [Column("AfkTimer")]
        public int AfkTimerMinutes
        {
            get;
            set;
        }

        /// <summary>
        /// Which region voice servers should be created on, by default.
        /// </summary>
        [Column("VoiceRegionID")]
        public int VoiceRegionID
        {
            get;
            set;
        }

        /// <summary>
        /// Whether or not this group allows people to join without invitations.
        /// </summary>
        [Column("IsPublic")]
        public bool IsPublic { get; set; }

        /// <summary>
        /// Whether or not any of this groups mapped communities are streaming
        /// </summary>
        [Column("IsStreaming")]
        public bool IsStreaming { get; set; }

        /// <summary>
        /// The last time the status was updated.
        /// </summary>
        [Column("StreamTime")]
        public long StreamingTimestamp { get; set; }

        [Column("IsFeatured")]
        public bool IsFeatured { get; set; }

        [Column("FeaturedTime")]
        public long FeaturedTimestamp { get; set; }

        /// <summary>
        /// The vanity url for the group
        /// </summary>
        [Column("Url")]
        public string Url { get; set; }

        /// <summary>
        /// Whether or not the chat throttle is enabled
        /// </summary>
        [Column("ChatThrottled")]
        public bool ChatThrottleEnabled { get; set; }

        /// <summary>
        /// How many seconds must pass between chat messages per user in channels.
        /// </summary>
        [Column("ThrottleSecs")]
        public int ChatThrottleSeconds { get; set; }

        /// <summary>
        /// Whether or not this group is flagged as inappropriate and should be hidden by default
        /// </summary>
        [Column("Inappropriate")]
        public bool FlaggedAsInappropriate { get; set; }

        /// <summary>
        /// The last time the group's avatar was updated
        /// </summary>
        [Column("AvatarTime")]
        public long AvatarTimestamp { get; set; }

        [Column("MembersOnline")]
        public int MembersOnline { get; set; }

        [Column("LastManualSync")]
        public DateTime DateLastManualSync { get; set; }

        /// <summary>
        /// Hide this channel for users that do not have access
        /// </summary>
        [Column("HideNoAccess")]
        public bool HideNoAccess
        {
            get;
            set;
        }

        [Column("ExternalID")]
        public string ExternalChannelID { get; set; }

        /// <summary>
        /// Hide voice users for users that do not have access.
        /// </summary>
        [Column("HideCallNoAxs")]
        public bool HideCallMembersNoAccess
        {
            get;
            set;
        }

        [Column("MappedComms")]
        public HashSet<string> MappedCommunities { get; set; }

        [JsonIgnore]
        public GroupRole DefaultRole
        {
            get
            {
                return GroupRole.GetByGroupIDAndRoleID(SourceConfiguration, RootGroupID, DefaultRoleID);
            }
        }

        [JsonIgnore]
        public GroupRole OwnerRole
        {
            get
            {
                return GroupRole.GetByGroupIDAndRoleID(SourceConfiguration, RootGroupID, OwnerRoleID);
            }
        }

        private GroupEmoticon[] _emotes;

        [JsonIgnore]
        public GroupEmoticon[] Emotes
        {
            get
            {
                _emotes = _emotes ?? GroupEmoticon.GetAllLocal(g => g.GroupID, GroupID).Where(e => e.Status == EmoticonStatus.Active).ToArray();
                return _emotes;
            }
        }

        [JsonIgnore]
        public bool IsRemote
        {
            get { return RegionID != LocalConfigID; }
        }

        public string ConversationID
        {
            get { return GroupID.ToString(); }
        }

        public bool ShouldUpdateMemberPresence
        {
            get { return MemberCount < 1000; }
        }


        protected override void OnRefresh()
        {
            _emotes = null;
        }

        public void AddChildGroupID(Guid groupID)
        {
            if (!IsRootGroup)
            {
                throw new InvalidOperationException("You can only add a child group ID to the root group!");
            }

            if (ChildGroupIDs == null)
            {
                ChildGroupIDs = new HashSet<Guid>();
            }

            ChildGroupIDs.Add(groupID);
            Update(p => p.ChildGroupIDs);
        }

        /// <summary>
        /// Factory Method for constructing groups in a single location to avoid missing new fields.
        /// </summary>
        private static Group Create(AerospikeConfiguration config, int homeRegionID, Group parent, NewGroupMember creator, string title, GroupType type, GroupMode mode, bool allowTempChannels,
            bool isPublic = false, HashSet<int> accessRoles = null, int displayOrder = 0, GroupSubType subtype = GroupSubType.Custom, bool isDefault = false, Guid? groupID = null,
            string externalChannelID = null, bool hideNoAccess = false, bool hideCallNoAccess = false)
        {

            var id = groupID ?? Guid.NewGuid();
            var group = new Group
            {
                RegionID = homeRegionID,
                GroupID = id,
                ParentGroupID = parent == null ? id : parent.GroupID,
                RootGroupID = parent == null ? id : parent.RootGroupID,
                Status = GroupStatus.Normal,
                Title = title,
                Type = type,
                AllowTempChannels = allowTempChannels,
                DisplayOrder = displayOrder,
                DateCreated = DateTime.UtcNow,
                CreatorID = creator.UserID,
                OwnerID = parent == null ? creator.UserID : parent.OwnerID,
                IsDefaultChannel = isDefault,
                Mode = mode,
                Subtype = subtype,
                IsPublic = isPublic,
                HideNoAccess = hideNoAccess,
                ExternalChannelID = externalChannelID,
                HideCallMembersNoAccess = hideCallNoAccess,
            };

            group.Insert(config);

            if (parent != null)
            {
                // Add this to the group hashset for fast lookups
                parent.RootGroup.AddChildGroupID(group.GroupID);

                // Create a permissions instance based on the parent's
                var parentGroupRolePermissions = parent.GetAllRolePermissionsForGroup();
                foreach (var p in parentGroupRolePermissions)
                {
                    var newGroupPermission = new GroupRolePermissions(group.GroupID, p);
                    newGroupPermission.Insert(config);
                }


                if (!string.IsNullOrEmpty(externalChannelID))
                {
                    // Handle twitch chat permissions
                    var defaultRolePerms = group.GetRolePermissions(parent.RootGroup.DefaultRoleID);
                    defaultRolePerms.SetPermission(GroupPermissions.Access, true);
                    defaultRolePerms.SetPermission(GroupPermissions.ChatSendMessages, true);
                }
                else if (isPublic)
                {
                    // Ensure that the default role has access permissions
                    var defaultRolePerms = group.GetRolePermissions(parent.RootGroup.DefaultRoleID);
                    if (!defaultRolePerms.CheckPermission(GroupPermissions.Access))
                    {
                        Logger.Debug("Adding Access permission for the default role to a newly created group.");
                        defaultRolePerms.SetPermission(GroupPermissions.Access, true);
                    }

                    // Also ensure that the default role has invite permissions, since this is a public server
                    defaultRolePerms.SetPermission(GroupPermissions.InviteUsers, true);

                }
                else if (accessRoles != null)
                {
                    var allRoles = group.RootGroup.GetRoles();

                    // Iterate over each role that exists in the server
                    foreach (var role in allRoles)
                    {
                        // Get the permissions for this role and group
                        var rolePermissions = group.GetRolePermissions(role.RoleID);

                        // If te role is part of the supplied access roles, give it access, only if needed.
                        if (accessRoles.Contains(role.RoleID))
                        {
                            if (!rolePermissions.CheckPermission(GroupPermissions.Access))
                            {
                                Logger.Debug("Adding Access permission for the role to a newly created group.");
                                rolePermissions.SetPermission(GroupPermissions.Access, true);
                            }
                        }
                        else if (rolePermissions.CheckPermission(GroupPermissions.Access))
                        {
                            Logger.Debug("Adding Access permission for the role to a newly created group.");
                            rolePermissions.SetPermission(GroupPermissions.Access, false);
                        }
                    }
                }
            }

            return group;
        }

        public GroupRole[] GetRoles(bool returnDeleted = false)
        {
            var roles = GroupRole.GetAll(SourceConfiguration, p => p.GroupID, RootGroupID);

            if (returnDeleted)
            {
                return roles;
            }

            return roles.Where(p => !p.IsDeleted).ToArray();
        }

        public GroupRole GetRole(int roleID, bool returnDeleted = false)
        {
            var role = GroupRole.Get(SourceConfiguration, GroupID, roleID);
            if (role == null || (role.IsDeleted && !returnDeleted))
            {
                return null;
            }

            return role;
        }

        public GroupRolePermissions GetRolePermissions(int roleID)
        {
            return GroupRolePermissions.GetByGroupIDAndRoleID(SourceConfiguration, GroupID, roleID);
        }

        public GroupRolePermissions[] GetAllRolePermissionsForGroup(bool returnDeleted = false)
        {
            // Get all roles for the group
            var allRoles = GetRoles();

            return allRoles.Select(role => GroupRolePermissions.GetByGroupIDAndRoleID(SourceConfiguration, GroupID, role.RoleID))
                                                               .Where(model => model != null && (!model.IsDeleted || returnDeleted)).ToArray();
        }

        public Dictionary<int, GroupRolePermissions> GetPermissionsByRole(bool returnDeleted = false)
        {
            var dict = new Dictionary<int, GroupRolePermissions>();

            // Get all roles for the group
            var allRoles = GetRoles();

            foreach (var role in allRoles)
            {
                var permissions = GroupRolePermissions.GetByGroupIDAndRoleID(SourceConfiguration, GroupID, role.RoleID);
                dict.Add(role.RoleID, permissions);
            }

            return dict;
        }

        public GroupRolePermissions[] GetAllChildRolePermissions()
        {
            if (!IsRootGroup)
            {
                throw new InvalidOperationException("GetAllChildRolePermissions must be called from the root group.");
            }

            return GroupRolePermissions.GetAllByRootGroupID(SourceConfiguration, GroupID).ToArray();
        }

        public GroupRolePermissions[] GetAllPermissionsByRoleID(int roleID)
        {
            // Get the Child Groups
            var allGroups = GetAllChildren(false, true);

            // Multi Get All Role Permissions
            return GroupRolePermissions.MultiGet(SourceConfiguration, allGroups.Select(p => new KeyInfo(p.GroupID, roleID))).ToArray();
        }

        private Group _rootGroup;

        [JsonIgnore]
        public Group RootGroup
        {
            get
            {
                if (_rootGroup == null)
                {
                    if (RootGroupID == GroupID || RootGroupID == Guid.Empty)
                    {
                        _rootGroup = this;
                    }
                    else
                    {
                        _rootGroup = Get(SourceConfiguration, RootGroupID);
                    }
                }

                return _rootGroup;
            }
        }

        private Group _parentGroup;

        [JsonIgnore]
        public Group ParentGroup
        {
            get
            {
                if (_parentGroup == null)
                {
                    if (ParentGroupID == Guid.Empty)
                    {
                        return this;
                    }

                    _parentGroup = Get(SourceConfiguration, ParentGroupID);
                }
                return _parentGroup;
            }
        }

        /// <summary>
        /// returns the array of all Groups in tree. works for root group or a channel
        /// </summary>
        public Dictionary<Guid, Group> GetChildGroupDictionary(bool includeDeleted = false, bool includeSelf = false)
        {
            if (!IsRootGroup)
            {
                throw new InvalidOperationException("GetRootChildren can only be called on a root group!");
            }

            var results = new Dictionary<Guid, Group>();

            if (includeSelf)
            {
                results.Add(GroupID, this);
            }

            if (ChildGroupIDs != null)
            {
                var childKeys = ChildGroupIDs.Select(p => new KeyInfo(p));
                var children = MultiGet(SourceConfiguration, childKeys).Where(p => !p.IsDeleted || includeDeleted).ToArray();

                foreach (var childGroup in children)
                {
                    results[childGroup.GroupID] = childGroup;
                }
            }

            return results;

        }

        /// <summary>
        /// returns the array of all groups from this root group
        /// </summary>
        public Group[] FastGetRootChildren(bool includeDeleted = false, bool includeSelf = false)
        {

            if (!IsRootGroup)
            {
                throw new InvalidOperationException("Cannot call FastGetRootChildren from a non-root group");
            }

            if (!CanHaveChildren)
            {
                return includeSelf ? new[] { this } : new Group[0];
            }

            Group[] children;
            if (ChildGroupIDs == null)
            {
                Logger.Warn("ChildGroupIDs collection is null, using secondary index instead");
                children = GetAll(SourceConfiguration, g => g.RootGroupID, GroupID).Where(p=>!p.IsDeleted || includeDeleted).ToArray();
            }
            else
            {
                var childKeys = ChildGroupIDs.Select(p => new KeyInfo(p));
                children = MultiGet(SourceConfiguration, childKeys).Where(p => !p.IsDeleted || includeDeleted).ToArray();
            }
            if (!includeSelf)
            {
                children = children.Where(p => p.GroupID != GroupID).ToArray();
            }


            return children;
        }

        /// <summary>
        /// returns the array of all Groups in tree. works for root group or a channel
        /// </summary>
        public Group[] GetAllChildren(bool includeDeleted = false, bool includeSelf = false)
        {
            if (IsRootGroup)
            {
                var children = GetAll(SourceConfiguration, p => p.RootGroupID, GroupID);

                if (!includeDeleted)
                {
                    children = children.Where(p => p.Status != GroupStatus.Deleted).ToArray();
                }

                if (!includeSelf)
                {
                    children = children.Where(p => p.GroupID != GroupID).ToArray();
                }

                return children.ToArray();
            }

            var allChildren = new List<Group>();

            if (includeSelf)
            {
                allChildren.Add(this);
            }

            var subChannels = GetImmediateChildren(includeDeleted);
            var currentDepth = 0;

            while (subChannels.Any())
            {
                allChildren.AddRange(subChannels);
                var immediateChildren = new List<Group>();
                foreach (var channel in subChannels)
                {
                    immediateChildren.AddRange(channel.GetImmediateChildren(includeDeleted));
                }

                subChannels = immediateChildren.ToArray();

                // Ensure we don't end up in an endless loop
                if (++currentDepth > MaxNestingDepth + 1)
                {
                    break;
                }
            }

            if (includeDeleted)
            {
                return allChildren.ToArray();
            }

            return allChildren.Where(p => p.Status == GroupStatus.Normal).ToArray();
        }

        public Group[] GetImmediateChildren(bool includeDeleted = false)
        {
            var immediateChildren = GetAll(SourceConfiguration, p => p.ParentGroupID, GroupID)
                                        .Where(p => p.GroupID != GroupID);

            if (!includeDeleted)
            {
                immediateChildren = immediateChildren.Where(p => !p.IsDeleted);
            }

            return immediateChildren.ToArray();
        }

        [JsonIgnore]
        public bool IsDeleted
        {
            get { return (Status == GroupStatus.Deleted); }
        }

        public GroupInvitation CreateGroupInvitation(int requestorUserID, bool autoRemoveMembers, Guid channelID, TimeSpan lifespan, int maxUses, string description, HashSet<string> sourceWords = null)
        {
            if (!CanHaveMembers)
            {
                throw new DataValidationException<CreateGroupInvitationErrorType>("Group cannot have members", CreateGroupInvitationErrorType.NotAServer);
            }

            // Get the channel in question
            var channel = GetByID(channelID == Guid.Empty ? DefaultGroupID : channelID);
            if (channel == null)
            {
                throw new DataNotFoundException();
            }

            if (channel.RootGroupID != GroupID)
            {
                throw new DataValidationException<CreateGroupInvitationErrorType>("The channel must belong to the group requested.", CreateGroupInvitationErrorType.WrongServer);
            }

            if (!channel.RoleHasPermission(GroupPermissions.Access, DefaultRoleID))
            {
                throw new DataValidationException<CreateGroupInvitationErrorType>("The channel is not accessible by guests.", CreateGroupInvitationErrorType.ChannelNotAccessibleByGuests);
            }

            // Ensure the user does not already have an active group invite
            var existingInviteCount = GroupInvitation.GetAllByGroupID(GroupID)
                                                     .Count(p => p.Status == GroupInvitationStatus.Active && !p.IsExpired && p.CreatorID == requestorUserID);

            if (existingInviteCount > GroupInvitation.MaxInvitesPerCreator)
            {
                Logger.Warn("Attempt to create too many invite links by a user", new { GroupID, requestorUserID });
                throw new DataValidationException<CreateGroupInvitationErrorType>("You have too many active invitations.", CreateGroupInvitationErrorType.TooManyInvites);
            }
            
            
            // Ensure that the requesting user has permission to invite guests (Member+ by default)
            var requestor = CheckPermission(IsPublic ? GroupPermissions.Access : GroupPermissions.InviteUsers, requestorUserID);
            
            try
            {
                var dateExpires = lifespan == TimeSpan.Zero ? DateTime.MaxValue : DateTime.UtcNow.Add(lifespan);

                var matchingInvitation = GroupInvitation.GetReusableInvite(this, requestorUserID, maxUses, dateExpires, autoRemoveMembers, channelID, description);

                if (matchingInvitation != null)
                {
                    if (matchingInvitation.Status != GroupInvitationStatus.Active)
                    {
                        var original = matchingInvitation.ShallowClone();

                        // update invite information...                       
                        matchingInvitation.AutoRemoveMembers = autoRemoveMembers;
                        matchingInvitation.ChannelID = channelID;
                        matchingInvitation.CreatorID = requestor.UserID;
                        matchingInvitation.CreatorName = requestor.GetTitleName();
                        matchingInvitation.DateCreated = DateTime.UtcNow;
                        matchingInvitation.DateExpires = dateExpires;
                        matchingInvitation.Description = description;
                        matchingInvitation.InternalUseCounter = 0;
                        matchingInvitation.MaxUses = maxUses;
                        matchingInvitation.Status = GroupInvitationStatus.Active;
                        matchingInvitation.TimesUsed = 0;

                        matchingInvitation.Update();

                        DiagLogger.Debug("GroupInvitation: Reusing existing inactive/defunct code",
                           new
                           {
                               Input = new
                               {
                                   GroupID,
                                   RequestorID = requestorUserID,
                                   AutoRemoveMembers = autoRemoveMembers,
                                   ChannelID = channelID,
                                   DateExpires = dateExpires,
                                   MaxUses = maxUses,
                                   Description = description
                               },
                               Original = original,
                               New = matchingInvitation
                           });
                    }
                    else
                    {
                        DiagLogger.Debug("GroupInvitation: Reusing existing active code",
                            new
                            {
                                Input = new
                                {
                                    GroupID,
                                    RequestorID = requestorUserID,
                                    AutoRemoveMembers = autoRemoveMembers,
                                    ChannelID = channelID,
                                    DateExpires = dateExpires,
                                    MaxUses = maxUses,
                                    Description = description
                                },
                                Match = matchingInvitation
                            });
                    }
                    return matchingInvitation;
                }

                if (sourceWords != null)
                {
                    // Try to make a readable code from the source words
                    var invitation = GroupInvitation.CreateReadable(this, requestor, maxUses, dateExpires, autoRemoveMembers, channel.GroupID, description, sourceWords);
                    if (invitation != null)
                    {
                        return invitation;
                    }
                }

                // Fall back to a random GUID code
                return GroupInvitation.Create(SourceConfiguration, this, requestor, maxUses, dateExpires, autoRemoveMembers, channel.GroupID, description);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to create server invite link!");                
                throw;
            }
            
        }

        public void ProcessInvitation(GroupInvitation invite, User user, UserRegion userRegion, string ipAddress)
        {
            // Ensure the user isn't banned
            if (GroupBannedUser.IsBanned(GroupID, user.UserID, ipAddress))
            {
                throw new GroupPermissionException("User is banned from this group.");
            }

            // Ensure invite is active - an invitation could be expired even with an active state if it 
            // has not been processed yet by the periodic job.
            if (invite.Status != GroupInvitationStatus.Active || invite.IsExpired)
            {
                throw new DataConflictException("Cannot use an invite that is not active.");
            }

            // Ensure the creator still has permission
            var inviter = CheckPermission(IsPublic ? GroupPermissions.Access : GroupPermissions.InviteUsers, invite.CreatorID);

            // Ensure this user isn't already a member
            var existing = GetMember(user.UserID);

            if (existing != null && !existing.IsDeleted)
            {
                return;
            }

            if (invite.MaxUses > 0)
            {
                if (invite.InternalUseCounter >= invite.MaxUses)
                {
                    throw new GroupPermissionException("Invitation has been used too many times.");
                }

                var counter = invite.IncrementCounter(SourceConfiguration, UpdateMode.Default, inv => inv.InternalUseCounter, 1);
                if (counter > invite.MaxUses)
                {
                    throw new GroupPermissionException("Invitation has been used too many times.");
                }
            }

            invite.IncrementCounter(SourceConfiguration, UpdateMode.Default, inv => inv.TimesUsed, 1);

            var newMembers = new[]
            {
                new NewGroupMember
                {
                    UserID = user.UserID,
                    InviteCode = invite.InviteCode,
                    Role = DefaultRole,
                    RegionID = userRegion.RegionID,
                    Username = user.Username
                }
            };
            var addedUsers = DoAddUsers(invite.CreatorID, newMembers);

            if (addedUsers.Contains(user.UserID))
            {
                GroupEventManager.LogAddUsersEvent(this, inviter, newMembers);
            }
            else
            {
                throw new GroupPermissionException("Unable to add user to the group.");
            }
        }

        public void DeleteInvitation(int requestorID, string inviteCode, bool removeMembers)
        {
            var invite = GroupInvitation.GetByInviteCode(inviteCode);

            if (invite == null || invite.Status != GroupInvitationStatus.Active)
            {
                throw new DataNotFoundException();
            }

            if (invite.GroupID != GroupID)
            {
                throw new DataNotFoundException();
            }

            // We need to make the permissoin check contextual, owner of the invite or a manager 
            var requestor = CheckPermission(requestorID == invite.CreatorID
                    ? GroupPermissions.Access
                    : GroupPermissions.ManageInvitations, requestorID);

            // Immediately invalidate the group invite to prevent additional use
            invite.Status = GroupInvitationStatus.Invalid;
            invite.Update(p => p.Status);

            // If the user has chosen to remove members, purge them.
            if (removeMembers)
            {
                PurgeGuests(invite, requestor);
                invite.Status = GroupInvitationStatus.Defunct;
                invite.Update(p => p.Status);
            }
        }
        
        public GroupInvitation[] GetGroupInvitations(int requestorUserID, bool onlyRequestorInvitations, bool onlyActiveInvitations, bool checkPermissions = true)
        {
            if (!CanHaveMembers)
            {
                throw new DataValidationException("Group cannot have members");
            }

            if (checkPermissions)
            {
                if (onlyRequestorInvitations)
                {
                    // Ensure the user has permission to make an invite               
                    CheckPermission(IsPublic ? GroupPermissions.Access : GroupPermissions.InviteUsers, requestorUserID);
                }
                else
                {
                    // Ensure the user has permission to manage group invites
                    CheckPermission(GroupPermissions.ManageInvitations, requestorUserID);
                }

            }

            // Get any active invites they have
            var allGroupInvites = GroupInvitation.GetAllByGroupID(GroupID);

            if (onlyActiveInvitations)
            {
                allGroupInvites = allGroupInvites.Where(p => !p.IsExpired && p.Status != GroupInvitationStatus.Defunct).ToArray();
            }

            if (onlyRequestorInvitations)
            {
                allGroupInvites = allGroupInvites.Where(p => p.CreatorID == requestorUserID).ToArray();
            }

            return allGroupInvites;

        }
        
        /// <summary>
        /// Adds users as members to a top level group. This following methods are requested from webservice and queues Coordination messages to Group Service
        /// </summary>
        /// <param name="requestorUserID"></param>
        /// <param name="participants"></param>
        public void AddUsers(int requestorUserID, NewGroupMember[] participants)
        {
            // Ensure that the requesting user only adds people at a lower role than themselves
            var adder = CheckPermission(IsPublic ? GroupPermissions.Access : GroupPermissions.InviteUsers, requestorUserID, (requestingMember) =>
            {
                return participants.All(participant => participant.Role.Rank >= requestingMember.BestRoleRank);
            });

            var bans = GroupBannedUser.GetAllLocal(b => b.GroupID, GroupID);
            var usersToAdd = participants.Where(p => !GroupBannedUser.IsBanned(p.UserID, p.IPAddress, bans)).ToArray();

            var bannedUsers = participants.Except(usersToAdd).ToArray();
            if (bannedUsers.Any())
            {
                Logger.Info("Detected an attempt by banned user(s) to rejoin a group.", bannedUsers);
            }

            if (!usersToAdd.Any())
            {
                return;
            }

            var added = DoAddUsers(requestorUserID, usersToAdd);
            GroupEventManager.LogAddUsersEvent(this, adder, participants.Where(p => added.Contains(p.UserID)));
        }

        public void JoinPublicServer(NewGroupMember requestor)
        {
            if (!IsRootGroup || Type != GroupType.Large)
            {
                throw new DataValidationException("Not a server");
            }

            if (!IsPublic)
            {
                throw new GroupPermissionException("Server is not public");
            }

            if (GroupBannedUser.IsBanned(GroupID, requestor.UserID, requestor.IPAddress))
            {
                throw new GroupPermissionException("Requestor is banned");
            }          

            var added = DoAddUsers(requestor.UserID, new[] { requestor });
            if (added.Contains(requestor.UserID))
            {
                GroupEventManager.LogAddUsersEvent(this, requestor, new[] { requestor });
            }
        }

        public void SystemJoinServer(NewGroupMember requestor)
        {
            if(!IsRootGroup || Type != GroupType.Large)
            {
                throw new DataValidationException("Not a server");
            }

            if(GroupBannedUser.IsBanned(GroupID, requestor.UserID, requestor.IPAddress))
            {
                throw new GroupPermissionException("User is banned from the server");
            }

            var usersToAdd = new[] { requestor };
            var added = DoAddUsers(requestor.UserID, usersToAdd);
            if (added.Contains(requestor.UserID))
            {
                GroupEventManager.LogAddUsersEvent(this, requestor, usersToAdd);
            }
        }


        /// <summary>
        /// Adds users. This is only meant to be called after permissions checks are performed elsewhere.
        /// </summary>
        private HashSet<int> DoAddUsers(int requestorUserID, NewGroupMember[] participants, bool sendNotification = true)
        {
            if (!CanHaveMembers)
            {
                throw new InvalidOperationException("You cannot add users to this group. It is a child of a main group.");
            }

            // Ensure we don't surpass the user limit
            var maxUsers = GetMaxUsersAllowed(Type);

            if (MemberCount + participants.Length > maxUsers)
            {
                throw new GroupPermissionException("Cannot add more than " + maxUsers + " members to this group!");
            }

            // Get all children
            var childGroups = RootGroup.GetAllChildren();

            var addedUserIDs = new HashSet<int>();

            foreach (var participant in participants)
            {
                // Get the existing member record
                var member = GetMember(participant.UserID);

                if (member != null && !member.IsDeleted)
                {
                    continue;
                }

                // make sure user hasn't joined too many servers. 
                if (GroupMember.HasExceededJoinedServerCount(participant.UserID))
                {
                    DiagLogger.Debug("user has joined too many servers.", new { requestor = participant, group = this });
                    continue;
                }

                if (member == null)
                {
                    member = new GroupMember(this, participant.UserID, participant.Username, participant.DisplayName, participant.RegionID, participant.InviteCode, NotificationPreference.Enabled, new[] { participant.Role });
                    member.Insert(RegionID);
                }
                else
                {
                    member.Restore(this, new[] { participant.Role }, participant.InviteCode);
                }

                // Propagate this out to children (for all regions)
                foreach (var group in childGroups)
                {
                    var childMembership = group.GetMember(member.UserID);
                    if (childMembership != null)
                    {
                        childMembership.Restore(group, new[] { participant.Role }, participant.InviteCode);
                    }
                    else
                    {
                        childMembership = new GroupMember(group, participant.UserID, participant.Username, participant.DisplayName, participant.RegionID, participant.InviteCode, NotificationPreference.Enabled, new[] { participant.Role });
                        childMembership.Insert(RegionID);
                    }
                }

                // Add this as a change for notifications
                addedUserIDs.Add(member.UserID);
            }

            if(addedUserIDs.Count == 0)
            {
                return addedUserIDs;
            }

            try
            {
                // Increment the group member count
                IncrementCounter(RegionID, UpdateMode.Default, p => p.MemberCount, addedUserIDs.Count);
            }
            catch (Exception ex)
            {
                Logger.Warn("Failed to increment group member count.");
            }

            GroupMemberWorker.CreateMembersAdded(this, addedUserIDs);

            // Finally send notifications
            if (sendNotification)
            {
                GroupChangeCoordinator.AddUsers(this, requestorUserID, addedUserIDs);
            }

            return addedUserIDs;
        }

        public bool IsRootGroup
        {
            get
            {
                return RootGroupID == Guid.Empty || RootGroupID == GroupID;
            }
        }

        public bool CanHaveSearchSettings
        {
            get { return IsRootGroup && Type == GroupType.Large; }
        }

        public bool CanHavePrivateMessages
        {
            get { return IsRootGroup && Type == GroupType.Large; }
        }

        public bool CanHaveMembers
        {
            get { return Type == GroupType.Normal || IsRootGroup; }
        }

        public bool CanHaveChildren
        {
            get { return Type == GroupType.Large; }
        }

        /// <summary>
        /// Called from web service, removes users from group 
        /// updates the current member count by queueing off to Group Service
        /// </summary>
        /// <param name="requestorUserID"></param>
        /// <param name="userIDs"></param>
        public void RemoveUsers(int requestorUserID, HashSet<int> userIDs, string kickMessage = null)
        {
            if (!CanHaveMembers)
            {
                throw new InvalidOperationException("This group cannot have direct members!");
            }

            var requestingMember = CheckPermission(GroupPermissions.RemoveUser, requestorUserID, member => CanModerateUsers(member, userIDs));
            DoRemoveUsers(requestingMember, userIDs, GroupMemberRemovedReason.Kicked, kickMessage);
        }

        public void SystemRemoveUsers(HashSet<int> userIDs, string kickMessage = null)
        {
            DoRemoveUsers(GroupMember.CurseSystem, userIDs, GroupMemberRemovedReason.Kicked, kickMessage);
        }

        private bool CanModerateUsers(GroupMember requestingMember, HashSet<int> otherUserIDs)
        {
            // Get each user's roles
            foreach (var userID in otherUserIDs)
            {
                var member = GetMember(userID, true);

                if (member == null)
                {
                    continue;
                }

                if (!CanModerateUser(requestingMember, member))
                {
                    return false;
                }
            }

            return true;
        }

        public bool CanModerateUser(int userID, int otherUserID)
        {
            if (!IsRootGroup)
            {
#if DEBUG
                throw new InvalidOperationException("You must access moderation checks from the root group.");
#endif
                return RootGroup.CanModerateUser(userID, otherUserID);
            }

            var user = GetMember(userID, true);
            if (user == null)
            {
                return false;
            }

            var otherUser = GetMember(otherUserID, true);
            if (otherUser == null)
            {
                return false;
            }

            return CanModerateUser(user, otherUser);
        }

        private bool CanModerateUser(GroupMember user, GroupMember otherUser)
        {
            if (!IsRootGroup)
            {
#if DEBUG
                throw new InvalidOperationException("You must access moderation checks from the root group.");
#endif
                return RootGroup.CanModerateUser(user, otherUser);
            }

            if (user == null || otherUser == null)
            {
                return false;
            }

            // Always let users moderate themselves
            if (user.UserID == otherUser.UserID)
            {
                return true;
            }

            // Never let anyone moderate an owner
            if (otherUser.Roles.Any(p => p == OwnerRoleID))
            {
                return false;
            }
            return user.BestRoleRank <= otherUser.BestRoleRank;
        }

        public void Leave(int userID, int userRegionID)
        {
            if (!CanHaveMembers)
            {
                throw new InvalidOperationException("This group cannot have direct members!");
            }

            // Get the root group membership
            var rootMembership = GetMember(userID);

            if (rootMembership == null || rootMembership.IsDeleted)
            {
                throw new GroupPermissionException("Not in group!");
            }

            // Make the necessary changes
            DoRemoveUsers(rootMembership, new HashSet<int>(new[] { userID }), GroupMemberRemovedReason.Left, null);
        }

        /// <summary>
        /// Internal worker method for handling removing users (after performing permission checks)
        /// </summary>
        /// <param name="requestorUserID"></param>
        /// <param name="userIDs"></param>
        /// <param name="members"></param>
        /// <param name="memberList"></param>
        private IReadOnlyCollection<GroupMember> DoRemoveUsers(GroupMember requestor, HashSet<int> userIDs, GroupMemberRemovedReason reason, string message)
        {
            if (!CanHaveMembers)
            {
                throw new InvalidOperationException("You cannot remove members from a non-root group!");
            }

            var removedUsers = new List<GroupMember>();

            // Remove these users immediately
            foreach (var userID in userIDs)
            {
                var member = GetMember(userID);

                if (member != null && !member.IsDeleted)
                {
                    member.SetDeleted(reason == GroupMemberRemovedReason.Banned);
                    removedUsers.Add(member);
                }
            }

            // Propagate this out to children (for all regions)
            var childGroups = GetAllChildren();

            foreach (var group in childGroups)
            {
                foreach (var userID in userIDs)
                {
                    var membership = group.GetMember(userID);
                    if (membership != null && !membership.IsDeleted)
                    {
                        membership.SetDeleted();
                    }
                }
            }

            // Hide all affected PMs
            foreach (var userID in userIDs)
            {
                try
                {
                    GroupPrivateConversation.HideAll(GroupID, userID);
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, "Failed to hide private message conversations.");
                }

            }

            if (removedUsers.Count > 0)
            {
                var removedUserIDs = new HashSet<int>(removedUsers.Select(u => u.UserID));
                GroupMemberIndexWorker.CreateMembersRemovedWorker(this, removedUserIDs);

                // Notify the group service of these changes
                GroupChangeCoordinator.RemoveUsers(this, requestor.UserID, removedUserIDs, reason, message);
                GroupMemberWorker.CreateMembersRemoved(this, removedUserIDs);
                GroupEventManager.LogRemoveUsersEvent(this, requestor, removedUsers);
            }

            return removedUsers;
        }

        public GroupMemberNicknameStatus ChangeMemberNickname(int userID, string username, string displayName, string nickname)
        {
            if (!CanHaveMembers)
            {
                throw new InvalidOperationException("You cannot change member info on a non-root group.");
            }

            if (!GroupMember.IsValidUserName(nickname))
            {
                return GroupMemberNicknameStatus.Invalid;
            }

            // Get the member
            var member = GetMember(userID);

            if (member == null)
            {
                throw new InvalidOperationException("User is not an active member of the group: " + nickname);
            }

            if (member.Nickname == nickname) // No change
            {
                return GroupMemberNicknameStatus.Success;
            }

            // If the username throttle has elapsed, allow this change, and reset the counter
            if (!HasPermission(GroupPermissions.AccessAdminPanel, userID))
            {
                if (member.UsernameThrottleElapsed())
                {
                    member.NicknameCounter = 0;
                }
                else if (member.NicknameCounter >= 2)
                {
                    return GroupMemberNicknameStatus.Throttled;
                }
                else
                {
                    ++member.NicknameCounter;
                }
            }


            // Get any other users with this exact username or nickname
            var otherMembers = GroupMemberManager.MembersByUsernameOrNickname(GroupID, nickname);

            if (otherMembers.Any())
            {
                // Get all roles that have mod privs
                var modLikeRoles = new HashSet<int>(GetRoles().Where(p => RoleHasPermission(GroupPermissions.AccessAdminPanel, p.RoleID) || RoleHasPermission(GroupPermissions.ChatModerateMessages, p.RoleID)).Select(p => p.RoleID));

                foreach (var otherMemberMatch in otherMembers)
                {
                    var otherMember = GetMember(otherMemberMatch.UserID);
                    if (otherMember == null || otherMember.IsDeleted || otherMember.UserID == userID)
                    {
                        continue;
                    }

                    if (modLikeRoles.Overlaps(otherMember.Roles))
                    {
                        return GroupMemberNicknameStatus.ModeratorName;
                    }
                }
            }

            var formerDisplayName = member.GetTitleName();

            // Update the member record
            member.Nickname = nickname;
            member.DisplayName = displayName;
            member.Username = username;
            member.DateNickname = DateTime.UtcNow;
            member.Update(p => p.Username, p => p.DisplayName, p => p.Nickname, p => p.DateNickname, p => p.NicknameCounter);

            GroupMemberIndexWorker.CreateMemberNicknameWorker(this, userID, member.Username, member.Nickname, member.DisplayName);
            GroupChangeCoordinator.UpdateUsers(this, userID, new HashSet<int> { userID });
            GroupEventManager.LogMemberRenameEvent(this, userID, formerDisplayName, member.Username);
            return GroupMemberNicknameStatus.Success;
        }

        public void TransferOwnership(int requestorID, int newOwnerID)
        {
            if (!CanHaveMembers)
            {
                throw new InvalidOperationException("Cannot transfer ownership of a group that cannot have members!");
            }

            var requestor = CheckPermission(GroupPermissions.Access, requestorID, member => member.BestRole == OwnerRoleID);
            var affectedMember = GetMember(newOwnerID);
            if (affectedMember == null || affectedMember.IsDeleted)
            {
                throw new DataNotFoundException();
            }

            DoAddMemberRole(requestor, OwnerRole, affectedMember);
            DoRemoveMemberRole(requestor, OwnerRole, requestor);
        }

        public void ReconcileOwnership(int formerOwnerID)
        {
            var formerOwner = GetMember(formerOwnerID);

            // Pick the next highest member
            var nextMember = TryGetAllMembers().Where(m => !m.IsDeleted).OrderBy(m => m.BestRoleRank).FirstOrDefault();
            if (nextMember == null)
            {
                // Nobody else left in the group
                // TODO: Decommission group by removing it from searches and such or possibly promote the first joiner to owner?
                return;
            }

            if (nextMember.BestRole == OwnerRoleID)
            {
                // Another owner exists, just update the effective owner ID
                OwnerID = nextMember.UserID;
                Update(g => g.OwnerID);
                return;
            }

            // Treat it as the former owner promoting the new owner
            DoAddMemberRole(formerOwner, OwnerRole, nextMember);
            OwnerID = nextMember.UserID;
            Update(g => g.OwnerID);
        }

        /// <summary>
        /// Updates the title and avatarUrl for a group and also fans this out to groupmembership details
        /// </summary>
        public void ChangeChannelSettings(int requestorUserID, string title, string messageOfTheDay, bool isPublic, HashSet<int> accessRoles, bool? allowTempChannels, bool? forcePushToTalk, bool hideNoAccess, bool hideVoiceMembersNoAccess)
        {
            if (!string.IsNullOrEmpty(ExternalChannelID))
            {
                throw new DataValidationException("Cannot change settings on a Twitch Chat channel");
            }

            if (string.IsNullOrWhiteSpace(title))
            {
                throw new InvalidOperationException("Cannot set title to empty string");
            }

            var url = GenerateAndCheckUrl(ParentGroup, ParentGroup.GetImmediateChildren().Where(g => g.GroupID != GroupID), title);

            var siblings = ParentGroup.GetImmediateChildren().Where(s => s.GroupID != GroupID).ToDictionary(d => d.GroupID);
            if (siblings.Any(c => string.Equals(c.Value.Title, title, StringComparison.InvariantCultureIgnoreCase) || string.Equals(c.Value.Url, url, StringComparison.InvariantCultureIgnoreCase)))
            {
                throw new DataValidationException("Cannot rename a channel with the same title or slug as a sibling");
            }

            if (IsRootGroup)
            {
                throw new InvalidOperationException("Cannot set title to empty string");
            }

            var member = RootGroup.CheckPermission(GroupPermissions.ManageChannels, requestorUserID);

            // Ensure that the requesting user's roles have access
            if (!isPublic)
            {
                accessRoles = ResolveChannelAccessRoles(RootGroup, member, accessRoles, title);
            }

            Title = title;
            IsPublic = isPublic;
            Url = url;
            MessageOfTheDay = messageOfTheDay ?? MessageOfTheDay;
            AllowTempChannels = allowTempChannels ?? AllowTempChannels;
            ForcePushToTalk = forcePushToTalk ?? ForcePushToTalk;

            if (!IsDefaultChannel)
            {
                HideNoAccess = hideNoAccess;
                HideCallMembersNoAccess = hideVoiceMembersNoAccess;
            }

            Update(g => g.Title, g => g.IsPublic, g => g.Url, g => g.MessageOfTheDay, g => g.AllowTempChannels, g => g.ForcePushToTalk, g => g.HideNoAccess, g => g.HideCallMembersNoAccess);

            ReconcileAccessPermissions(isPublic, accessRoles);

            // Notify all members of this change
            GroupChangeCoordinator.ChangeInfo(this, requestorUserID);

        }

        private static HashSet<int> ResolveChannelAccessRoles(Group rootGroup, GroupMember member, HashSet<int> desiredAccessRoles, string channelTitle)
        {
            var resolvedRoles = new HashSet<int>(desiredAccessRoles);
            // Ensure that at least one of the requesting user's roles have access
            if (!desiredAccessRoles.Overlaps(member.Roles))
            {
                // If not, add all of the user's roles thathave channel management
                foreach (var roleID in member.Roles.Where(role => role != rootGroup.DefaultRoleID))
                {
                    // Only add roles that have manage channels permissions
                    if (rootGroup.RoleHasPermission(GroupPermissions.ManageChannels, roleID))
                    {
                        resolvedRoles.Add(roleID);
                    }
                }

                Logger.Info("Channel settings did not include any of the user's roles. They have been added!",
                    new
                    {
                        Server = rootGroup.Title,
                        Channel = channelTitle,
                        desiredAccessRoles,
                        resolvedRoles
                    });
            }
            return resolvedRoles;
        }

        private void ReconcileAccessPermissions(bool isPublic, HashSet<int> accessRoles)
        {
            // Ensure that the default role has access permissions
            if (isPublic)
            {
                var defaultRolePerms = GetRolePermissions(RootGroup.DefaultRoleID);
                if (!defaultRolePerms.CheckPermission(GroupPermissions.Access))
                {
                    Logger.Debug("Adding Access permission for the default role to a newly created group.");
                    defaultRolePerms.SetPermission(GroupPermissions.Access, true);
                }
            }
            else if (accessRoles != null)
            {
                var allRoles = RootGroup.GetRoles();

                // Iterate over each role that exists in the server
                foreach (var role in allRoles)
                {
                    // Get the permissions for this role and group
                    var rolePermissions = GetRolePermissions(role.RoleID);

                    if (rolePermissions == null)
                    {
                        Logger.Warn("Missing role permissions!", new { GroupID, role.RoleID });
                        continue;
                    }

                    // If te role is part of the supplied access roles, give it access, only if needed.
                    if (accessRoles.Contains(role.RoleID))
                    {
                        if (!rolePermissions.CheckPermission(GroupPermissions.Access))
                        {
                            Logger.Debug("Adding Access permission for the role to a newly created group.");
                            rolePermissions.SetPermission(GroupPermissions.Access, true);
                        }
                    }
                    else if (rolePermissions.CheckPermission(GroupPermissions.Access))
                    {
                        Logger.Debug("Adding Access permission for the role to a newly created group.");
                        rolePermissions.SetPermission(GroupPermissions.Access, false);
                    }
                }
            }
        }

        /// <summary>
        /// Updates the title and avatarUrl for a group and also fans this out to groupmembership details
        /// </summary>
        public void ChangeGroupSettings(int requestorUserID, string title, string messageOfTheDay, bool? forcePushToTalk)
        {
            if (!IsRootGroup || Type != GroupType.Normal)
            {
                throw new DataValidationException("You cannot modify the settings of a non-root group.");
            }

            CheckPermission(GroupPermissions.ManageServer, requestorUserID);

            //var oldTitle = Title;
            Title = title;
            ForcePushToTalk = forcePushToTalk;
            MessageOfTheDay = messageOfTheDay;
            Update(p => p.Title, p => p.ForcePushToTalk, p => p.MessageOfTheDay);

            //LogEvent(GroupEvent.CreateChannelTitleChangeEvent(RootGroupID, requestorUserID, oldTitle, title));

            GroupMemberWorker.CreateGroupChangeWorker(RootGroup, this);

            // Notify all members of this change
            GroupChangeCoordinator.ChangeInfo(this, requestorUserID);
        }

        /// <summary>
        /// Updates the title and avatarUrl for a group and also fans this out to groupmembership details
        /// </summary>
        public void ChangeServerSettings(int requestorUserID, string title, Guid afkChannelID, int afkTimerMinutes, int voiceRegionID, bool isPublic, int chatThrottleSeconds, bool chatThrottleEnabled)
        {
            if (!IsRootGroup || Type != GroupType.Large)
            {
                throw new DataValidationException("You cannot modify the settings of a non-root group.");
            }

            CheckPermission(GroupPermissions.ManageServer, requestorUserID);

            var groupUpdates = new List<Expression<Func<Group, object>>>();

            var shouldIndex = false;

            if (title != null && Title != title)
            {
                Title = title;
                groupUpdates.Add(p => p.Title);
                shouldIndex = true;
            }

            if (AfkChannelID != afkChannelID)
            {
                AfkChannelID = afkChannelID;
                groupUpdates.Add(p => p.AfkChannelID);
            }

            if (AfkTimerMinutes != afkTimerMinutes)
            {
                AfkTimerMinutes = afkTimerMinutes;
                groupUpdates.Add(p => p.AfkTimerMinutes);
            }

            if (VoiceRegionID != voiceRegionID)
            {
                VoiceRegionID = voiceRegionID;
                groupUpdates.Add(p => p.VoiceRegionID);
            }

            if (ChatThrottleEnabled != chatThrottleEnabled)
            {
                ChatThrottleEnabled = chatThrottleEnabled;
                groupUpdates.Add(p => p.ChatThrottleEnabled);
            }

            if (ChatThrottleSeconds != chatThrottleSeconds)
            {
                ChatThrottleSeconds = chatThrottleSeconds;
                groupUpdates.Add(p => p.ChatThrottleSeconds);
            }

            if (IsPublic != isPublic)
            {
                IsPublic = isPublic;
                groupUpdates.Add(p => p.IsPublic);
                shouldIndex = true;

                // If the group becomes public, we need to give the Invite permissions to Everyone
                if (isPublic)
                {
                    ToggleRolePermission(DefaultRoleID, GroupPermissions.InviteUsers, true, true);
                }
            }

            if (groupUpdates.Any())
            {
                Update(groupUpdates.ToArray());
                GroupMemberWorker.CreateGroupChangeWorker(RootGroup, this);

                // Notify all members of this change
                GroupChangeCoordinator.ChangeInfo(this, requestorUserID);
            }

            if (shouldIndex)
            {
                GroupSearchIndexWorker.CreateGroupInfoUpdatedWorker(this, GetServerSearchSettingsOrDefault());
            }
        }

        public GroupSearchSettings GetServerSearchSettingsOrDefault()
        {
            var settings = GetServerSearchSettings();


            if (settings != null)
            {
                return settings;
            }

            settings = new GroupSearchSettings
            {
                GroupID = GroupID,
                RegionID = RegionID,
                IsSearchable = true,
                Games = new HashSet<int>(),
                SearchTags = new HashSet<int>(),
                Description = string.Empty,
                MatchAllGames = false
            };

            settings.Insert(RegionID);

            return settings;
        }

        public GroupSearchSettings GetServerSearchSettings()
        {
            if (!IsRootGroup)
            {
                throw new InvalidOperationException("Search settings can only exist on the root group.");
            }
            return GroupSearchSettings.Get(RegionID, GroupID);
        }

        public void ChangeServerSearchSettings(int requestorUserID, bool isSearchable, string description, HashSet<int> searchTags, HashSet<int> games, bool matchAllGames)
        {
            if (!IsRootGroup || Type != GroupType.Large)
            {
                throw new DataValidationException("You cannot modify the settings of a non-root or non-server group.");
            }

            CheckPermission(GroupPermissions.ManageServer, requestorUserID);

            searchTags = searchTags ?? new HashSet<int>();
            games = games ?? new HashSet<int>();

            var index = false;

            var settings = GetServerSearchSettingsOrDefault();

            var updates = new List<Expression<Func<GroupSearchSettings, object>>>();

            if (settings.IsSearchable != isSearchable)
            {
                settings.IsSearchable = isSearchable;
                updates.Add(p => p.IsSearchable);
            }

            if (settings.Description != description)
            {
                settings.Description = description.ToEmptyWhenNull();
                updates.Add(p => p.Description);
            }

            if (!searchTags.SetEquals(settings.SearchTags))
            {
                settings.SearchTags = searchTags;
                updates.Add(p => p.SearchTags);
            }

            if (!games.SetEquals(settings.Games))
            {
                settings.Games = games;
                updates.Add(p => p.Games);
            }

            if (settings.MatchAllGames != matchAllGames)
            {
                settings.MatchAllGames = matchAllGames;
                updates.Add(p => p.MatchAllGames);
            }

            if (updates.Any())
            {
                settings.Update(updates.ToArray());
                index = true;
            }

            if (index)
            {
                GroupSearchIndexWorker.CreateGroupInfoUpdatedWorker(this, settings);
            }
        }

        /// <summary>
        /// Ensures that if a group's member count is above a certain threshold
        /// push to talk is automatically enabled, unless the a group manager has
        /// previously set the preference manually.
        /// </summary>
        public void CheckPushToTalkThreshold()
        {
            if (ForcePushToTalk.HasValue)
            {
                return;
            }

            var writeable = EnsureWritable();
            writeable.ForcePushToTalk = MemberCount >= MemberCountPttThreshold;
            writeable.Update(g => g.ForcePushToTalk);
            GroupChangeCoordinator.ChangeInfo(writeable, 0);
        }

        public void CreateDefaultRoles(string ownerRoleName, string guestRoleName, string moderatorRoleName, bool overwriteExisting = false)
        {
            if (overwriteExisting && (OwnerRoleID > 0 || DefaultRoleID > 0 || RoleCounter > 0))
            {
                RoleCounter = 0;
                RolePermissions = new Dictionary<int, long>();
                Update(p => p.RoleCounter, p => p.RolePermissions);
            }

            // Owner Role
            var ownerRole = CreateRole(ownerRoleName, GroupPermissions.All, true, false, false, GroupRoleTemplate.Owner.DefaultVanityColor);
            OwnerRoleID = ownerRole.RoleID;

            // Moderator Role
            if (!string.IsNullOrWhiteSpace(moderatorRoleName))
            {
                CreateRole(moderatorRoleName, GroupRoleTemplate.Moderator.DefaultPermissions, false, false, false, GroupRoleTemplate.Moderator.DefaultVanityColor);
            }

            // Everyone Role
            var guestRoleTemplate = GroupRoleTemplate.GetDefaultGuestPermissionByGroupType(Type);
            var defaultRole = CreateRole(guestRoleName, guestRoleTemplate.DefaultPermissions, false, true, false, guestRoleTemplate.DefaultVanityColor);
            DefaultRoleID = defaultRole.RoleID;

            Update(p => p.OwnerRoleID, p => p.DefaultRoleID);
        }

        public GroupRole CreateRole(string name, GroupPermissions permissions, bool isOwner, bool isDefault, bool isSynced, int vanityColor = 0, int vanityBadge = 0,
            bool hasCustomVanityBadge = false, string syncID = null, GroupRoleTag tag = GroupRoleTag.None, AccountType roleSource = AccountType.Curse)
        {
            if (IsDeleted)
            {
                throw new ArgumentException("You can only create roles attached to an active group!");
            }

            if (!IsRootGroup)
            {
                throw new ArgumentException("You can only create roles attached to the root group!");
            }

            if (isOwner && OwnerRole != null)
            {
                throw new ArgumentException("You cannot have more than one owner group!");
            }

            if (isDefault && DefaultRole != null)
            {
                throw new ArgumentException("You cannot have more than one default group!");
            }

            // Get the new role counter. Always use the home region for this.
            var newRoleID = IncrementCounter(SourceConfiguration, UpdateMode.Default, p => p.RoleCounter, 1);

            // Create the role
            var role = new GroupRole
            {
                RoleID = newRoleID,
                Name = name,
                Rank = isDefault ? 10000 : newRoleID,
                GroupID = GroupID,
                RootGroupID = RootGroupID,
                IsOwner = isOwner,
                IsDefault = isDefault,
                IsSynced = isSynced,
                SyncID = syncID,
                RegionID = RegionID,
                VanityColor = vanityColor,
                VanityBadge = vanityBadge,
                HasCustomVanityBadge = hasCustomVanityBadge,
                Tag = tag,
                Source = roleSource
            };

            Logger.Debug("Created role for server: " + Title, role);

            role.Insert(SourceConfiguration);

            // Create the permissions set
            var groupPermissions = new GroupRolePermissions(RegionID, GroupID, role.RoleID, RootGroupID, permissions);
            groupPermissions.Insert(SourceConfiguration);

            // Create the permissions cascade
            var allChildren = GetAllChildren(true);
            foreach (var child in allChildren)
            {
                var childPermissions = new GroupRolePermissions(child.GroupID, groupPermissions);

                // If this group isn't public, remove the access permission
                if (!child.IsPublic)
                {
                    childPermissions.PermissionsState[(int)GroupPermissions.Access] = (int)GroupPermissionState.NotAllowed;
                }

                childPermissions.IsDeleted = child.IsDeleted;
                childPermissions.Insert(SourceConfiguration);
            }

            return role;
        }

        private void RestoreRole(GroupRole role)
        {
            if (!role.IsDeleted)
            {
                return;
            }

            role.IsDeleted = false;
            role.Update(r => r.IsDeleted);

            var roleGroupPermissions = GroupRolePermissions.GetByGroupIDAndRoleID(SourceConfiguration, GroupID, role.RoleID);
            if (roleGroupPermissions == null)
            {
                throw new InvalidOperationException("Attempt made to modify role permissions, but they are missing.");
            }

            if (roleGroupPermissions.IsDeleted)
            {
                roleGroupPermissions.IsDeleted = false;
                roleGroupPermissions.Update(r => r.IsDeleted);
            }

            // Cascade permissions to all children
            var children = GetAllChildren();
            var groupPermissions = GroupRolePermissions.MultiGet(SourceConfiguration, children.Select(c => new KeyInfo(c.GroupID, role.RoleID))).ToDictionary(g => g.GroupID);
            foreach (var groupPermission in groupPermissions)
            {
                groupPermission.Value.IsDeleted = false;
                groupPermission.Value.Update(g => g.IsDeleted);
            }
            CascadePermissions(roleGroupPermissions, children.ToLookup(c => c.ParentGroupID), groupPermissions, true);
        }

        public GroupRole ModifyRole(int userID, int roleID, string name, int color, bool hasBadge, int vanityBadge, Dictionary<GroupPermissions, GroupPermissionState> permissions)
        {
            if (!IsRootGroup || IsDeleted)
            {
                throw new GroupPermissionException("You can only modify roles from the root group.");
            }

            CheckPermission(GroupPermissions.ManageServer, userID);

            var role = GroupRole.GetByGroupIDAndRoleID(SourceConfiguration, RootGroupID, roleID);

            if (role == null || role.IsDeleted)
            {
                throw new GroupPermissionException("Role is missing or deleted.");
            }

            role.Name = name;
            role.VanityColor = color;
            role.HasCustomVanityBadge = hasBadge;
            role.VanityBadge = vanityBadge;
            role.Update(p => p.Name, p => p.VanityColor, p => p.HasCustomVanityBadge, p => p.VanityBadge);

            if (!role.IsOwner)
            {
                role = ModifyRolePermissions(userID, roleID, GroupID, permissions);
            }

            return role;
        }

        public GroupRole[] ModifyRoleRanks(int userID, Dictionary<int, int> roleRanks)
        {
            if (!IsRootGroup || IsDeleted)
            {
                throw new GroupPermissionException("You can only modify roles from the root group.");
            }

            CheckPermission(GroupPermissions.ManageServer, userID);

            var allRoles = GetRoles();

            if (roleRanks.Count != allRoles.Length)
            {
                throw new GroupPermissionException("Invalid number of role ranks.");
            }

            foreach (var kvp in roleRanks)
            {
                var roleRank = kvp.Value;
                var role = GetRole(kvp.Key);

                if (role == null)
                {
                    throw new GroupPermissionException("Role is missing or deleted.");
                }

                if (role.IsDefault)
                {
                    roleRank = GroupRole.DefaultRoleRank;
                }
                else if (role.IsOwner)
                {
                    roleRank = 1;
                }
                else if (roleRank <= OwnerRole.Rank)
                {
                    throw new GroupPermissionException("Role rank cannot be better than the owner role.");
                }
                else if (roleRank >= GroupRole.DefaultRoleRank)
                {
                    throw new GroupPermissionException("Role rank cannot be worse than the default role.");
                }

                if (role.Rank == roleRank)
                {
                    continue;
                }

                role.Rank = roleRank;
                role.Update(p => p.Rank);
            }

            // Need to now update all members with this rank
            GroupMemberWorker.CreateRoleRank(this);
            return GetRoles();
        }

        public void DeleteRole(int userID, int roleID)
        {
            if (!IsRootGroup || IsDeleted)
            {
                throw new GroupPermissionException("You can only modify roles from the root group.");
            }

            CheckPermission(GroupPermissions.ManageServer, userID);
            DoDeleteRole(userID, roleID);
            GroupChangeCoordinator.ChangeInfo(this, userID);
        }

        private void DoDeleteRole(int requestorID, int roleID)
        {
            var role = GetRole(roleID);

            if (role == null)
            {
                throw new GroupPermissionException("Role is missing, deleted or part of a different server.");
            }

            if (role.RoleID == OwnerRoleID || role.RoleID == DefaultRoleID)
            {
                throw new GroupPermissionException("You cannot delete the owner or default roles.");
            }

            role.IsDeleted = true;
            role.Update(p => p.IsDeleted);

            GroupMemberWorker.CreateRoleDeleted(this, requestorID, roleID);
        }

        public GroupRole ModifyRolePermissions(int userID, int roleID, Guid groupID, Dictionary<GroupPermissions, GroupPermissionState> permissions)
        {
            if (!IsRootGroup || IsDeleted)
            {
                throw new GroupPermissionException("You can only modify roles from the root group.");
            }

            var affectedGroup = groupID == GroupID ? this : GetChildGroup(groupID);

            // Ensure that the affected group is active
            if (affectedGroup == null)
            {
                throw new GroupPermissionException("Group is missing, deleted or part of a different server.");
            }

            // Ensure that the affected role is active and part of the same tree
            var role = GroupRole.GetByGroupIDAndRoleID(SourceConfiguration, RootGroupID, roleID);
            if (role == null || role.IsDeleted)
            {
                throw new GroupPermissionException("Role is missing or deleted.");
            }

            // The permission check depends on whether this is a root group, or a channel
            if (affectedGroup.IsRootGroup)
            {
                CheckPermission(GroupPermissions.ManageServer, userID);
            }
            else
            {
                CheckPermission(GroupPermissions.ManageChannels, userID, member =>
                {
                    return HasPermission(GroupPermissions.ManageServer, userID) || member.BestRoleRank < role.Rank;
                });
            }

            if (role.IsOwner)
            {
                throw new GroupPermissionException("The owner role permissions cannot be modified.");
            }

            var roleGroupPermissions = GroupRolePermissions.GetByGroupIDAndRoleID(SourceConfiguration, groupID, roleID);
            if (roleGroupPermissions == null)
            {
                throw new InvalidOperationException("Attempt made to modify role permissions, but they are missing.");
            }

            roleGroupPermissions.UpdatePermissionsState(permissions);
            roleGroupPermissions.Update();

            // Cascade permissions to all children
            var children = affectedGroup.GetAllChildren();
            var groupPermissions = GroupRolePermissions.MultiGet(SourceConfiguration, children.Select(c => new KeyInfo(c.GroupID, roleID))).ToDictionary(g => g.GroupID);
            CascadePermissions(roleGroupPermissions, children.ToLookup(c => c.ParentGroupID), groupPermissions);

            if (IsRootGroup)
            {
                VoicePermissionsWorker.CreateRolePermissionsChanged(this);
            }
            else
            {
                VoicePermissionsWorker.CreateChannelPermissionsChanged(affectedGroup);
            }

            return role;
        }

        private void ToggleRolePermission(int roleID, GroupPermissions permission, bool granted, bool shouldCascade)
        {
            if (!IsRootGroup)
            {
                throw new InvalidOperationException("You cannot toggle role permissions for a non root group.");
            }

            var rolePermissions = GetRolePermissions(roleID);
            rolePermissions.SetPermission(permission, granted);

            if (shouldCascade)
            {
                // Cascade permissions to all children
                var children = GetAllChildren();
                var groupPermissions = GroupRolePermissions.MultiGet(SourceConfiguration, children.Select(c => new KeyInfo(c.GroupID, roleID))).ToDictionary(g => g.GroupID);

                CascadePermissions(rolePermissions, children.ToLookup(c => c.ParentGroupID), groupPermissions);
            }
        }

        private static void CascadePermissions(GroupRolePermissions parent, ILookup<Guid, Group> groupsByParent, Dictionary<Guid, GroupRolePermissions> permissionsByGroup, bool createMissing = false)
        {
            foreach (var child in groupsByParent[parent.GroupID])
            {
                if (!string.IsNullOrEmpty(child.ExternalChannelID))
                {
                    // Don't modify a Twitch Chat channel's permissions
                    continue;
                }

                GroupRolePermissions permissions;
                if (permissionsByGroup.TryGetValue(child.GroupID, out permissions))
                {
                    permissions.UpdateInheritance(parent);
                    CascadePermissions(permissions, groupsByParent, permissionsByGroup);
                }
                else if (createMissing)
                {
                    permissions = new GroupRolePermissions(child.GroupID, parent);
                    if (!child.IsPublic)
                    {
                        permissions.PermissionsState[(int)GroupPermissions.Access] = (int)GroupPermissionState.NotAllowed;
                    }
                    permissions.Insert(parent.RegionID);
                }
            }
        }

        public static int GetMaxUsersAllowed(GroupType type)
        {
            return type == GroupType.Large ? MaxUsersInLargeGroup : MaxUsersInGroup;
        }

        public static Group CreateServer(NewGroupMember creator, string title, string textChannelName, string voiceChannelName, string ownerRoleName, string guestRoleName, string moderatorRoleName, bool isPublic, bool notify = true)
        {
            textChannelName.CheckRange(1, TitleMaxLength);
            voiceChannelName.CheckRange(1, TitleMaxLength);
            ownerRoleName.CheckRange(1, GroupRole.NameMaxLength);
            guestRoleName.CheckRange(1, GroupRole.NameMaxLength);

            // Create the server
            var server = Create(LocalConfiguration, LocalConfigID, null, creator, title, GroupType.Large, GroupMode.TextOnly, false, isPublic);

            // Assign a random code for the url
            var vanityUrl = CreateVanityUrl(server, null);

            // Create the default roles (which also create permissions)
            server.CreateDefaultRoles(ownerRoleName, guestRoleName, moderatorRoleName);
            var ownerRole = server.OwnerRole;
            var defaultRole = server.DefaultRole;

            // Log the creation event
            GroupEventManager.LogCreateGroupEvent(server, creator);

            var newMembers = new List<NewGroupMember>();

            // Add the current user (owner of the group)
            newMembers.Add(new NewGroupMember(creator.UserID, creator.RegionID, creator.Username, creator.DisplayName, creator.IPAddress, ownerRole));

            // Add the users to the group
            var addedMembers = server.DoAddUsers(creator.UserID, newMembers.ToArray(), false);
            server.MemberCount = addedMembers.Count;
            server.Update(p => p.MemberCount);

            GroupEventManager.LogAddUsersEvent(server, creator, newMembers);

            // Add the default text channel
            var textChannel = server.CreatePermanentChildGroup(new NewPermanentChildGroupInfo(GroupMode.TextOnly, creator, textChannelName, 0) { IsPublic = true, IsDefault = true });
            server.DefaultGroupID = textChannel.GroupID;
            server.Update(p => p.DefaultGroupID);

            // Add the default voice channel
            var voiceChannel = server.CreatePermanentChildGroup(new NewPermanentChildGroupInfo(GroupMode.TextAndVoice, creator, voiceChannelName, 1) { IsPublic = true });

            // Reload from the database
            server.Refresh();

            if (notify)
            {
                // Queue off a worker job that will get picked up by a group service node, and finalize the group 
                GroupChangeCoordinator.CreateGroup(server, creator.UserID);
            }

            GroupSearchIndexWorker.CreateGroupCreatedWorker(server, server.GetServerSearchSettingsOrDefault());
            return server;
        }

        private static VanityUrl CreateVanityUrl(Group server, string customUrl)
        {
            if (!server.IsRootGroup)
            {
                throw new InvalidOperationException("Vanity URLs can only exist on the root group");
            }

            VanityUrl vanityUrl;
            // Map the server to the url
            if (!string.IsNullOrWhiteSpace(customUrl))
            {
                try
                {
                    vanityUrl = VanityUrl.CreateForGroup(customUrl, server.GroupID);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to create a custom vanity url: " + customUrl);
                    throw;
                }

            }
            else
            {
                try
                {
                    vanityUrl = VanityUrl.RandomForGroup(server.GroupID);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to create a random url: " + customUrl);
                    throw;
                }
            }

            server.Url = vanityUrl.DisplayUrl;
            server.Update(p => p.Url);
            return vanityUrl;
        }

        public static Group CreateSyncedCommunityServer(NewGroupMember creator, string communityName, NewCommunitySync[] syncedCommunities,
            string lobbyName, string voiceChannelName, string ownerRoleName, string guestRoleName, string moderatorRoleName, bool isPublic, string customUrl)
        {
            // Create the server
            var server = Create(LocalConfiguration, LocalConfigID, null, creator, communityName, GroupType.Large, GroupMode.TextOnly, false, isPublic, null, 0, GroupSubType.Stream);

            // Assign the server to a vanity url
            var vanityUrl = CreateVanityUrl(server, customUrl);

            // Create the default roles (which also create permissions)
            server.CreateDefaultRoles(ownerRoleName, guestRoleName, moderatorRoleName);
            var ownerRole = server.OwnerRole;
            var defaultRole = server.DefaultRole;

            // Log the creation event
            GroupEventManager.LogCreateGroupEvent(server, creator);

            // Add the streamer
            var newMembers = new[] { new NewGroupMember(creator.UserID, creator.RegionID, creator.Username, creator.DisplayName, creator.IPAddress, ownerRole) };
            var addedMembers = server.DoAddUsers(creator.UserID, newMembers.ToArray(), false);
            server.MemberCount = addedMembers.Count;
            server.Update(p => p.MemberCount);
            GroupEventManager.LogAddUsersEvent(server, creator, newMembers);
            var creatorMembership = server.GetMember(creator.UserID);

            var channels = new List<Group>();

            // Add the default text channel
            var lobby = server.CreatePermanentChildGroup(new NewPermanentChildGroupInfo(GroupMode.TextOnly, creator, lobbyName, 0) { IsPublic = true, IsDefault = true });
            channels.Add(lobby);
            server.DefaultGroupID = lobby.GroupID;
            server.Update(p => p.DefaultGroupID);

            // Add the default voice channel
            channels.Add(server.CreatePermanentChildGroup(new NewPermanentChildGroupInfo(GroupMode.TextAndVoice, creator, voiceChannelName, 1) { IsPublic = true }));

            // Add Each Community's info
            var currentDisplayOrder = 2;
            foreach (var sync in syncedCommunities)
            {
                ExternalCommunityRole premiumRole = null;
                GroupRole premiumGroupRole = null;
                GroupRole modGroupRole = null;
                GroupRole ownerGroupRole = null;

                // Add roles
                var roles = sync.Community.GetRoles().OrderBy(r => r.RoleRank);
                foreach (var role in roles)
                {
                    if (!sync.Community.CanHaveSubs && role.CheckPremium())
                    {
                        // Don't create sub role for non-partners
                        continue;
                    }

                    var groupRole = CreateSyncedRole(server, role);
                    if(groupRole == null)
                    {
                        continue;
                    }

                    if (role.RoleTag == GroupRoleTag.SyncedOwner)
                    {
                        ownerGroupRole = groupRole;
                        server.SystemAddMemberRole(groupRole, creatorMembership);
                    }
                    else if (role.RoleTag == GroupRoleTag.SyncedSubscriber)
                    {
                        premiumGroupRole = groupRole;
                    }
                    else if (role.RoleTag == GroupRoleTag.SyncedModerator)
                    {
                        modGroupRole = groupRole;
                    }
                }

                // Add syndicated Twitch Chat
                if (sync.Community.Type == AccountType.Twitch)
                {
                    channels.Add(server.CreateSyndicatedTwitchChat(creator.UserID, sync.Community, currentDisplayOrder++, channels.ToArray(), false));
                }

                // Add premium channel if needed
                if (sync.Community.CanHaveSubs && premiumGroupRole != null && sync.PremiumChannelName.SafeLength() > 0)
                {
                    var allowedRoles = new HashSet<int> { ownerRole.RoleID, premiumGroupRole.RoleID };
                    if (modGroupRole != null)
                    {
                        allowedRoles.Add(modGroupRole.RoleID);
                    }
                    if (ownerGroupRole != null)
                    {
                        allowedRoles.Add(ownerGroupRole.RoleID);
                    }

                    channels.Add(server.CreatePermanentChildGroup(
                        new NewPermanentChildGroupInfo(GroupMode.TextOnly, creator, sync.PremiumChannelName, currentDisplayOrder++)
                        {
                            IsPublic = false,
                            AccessRoles = allowedRoles
                        }));
                }

                sync.Community.MapToGroup(creatorMembership, server);
            }

            var avatarCommunity =
                syncedCommunities.Where(p => !string.IsNullOrWhiteSpace(p.Community.AvatarUrl))
                    .OrderByDescending(p => p.Community.Followers)
                    .FirstOrDefault();

            if (avatarCommunity != null)
            {
                var avatarUrl = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, "syncs", "communities/twitch/" + avatarCommunity.Community.ExternalID);
                var avatar = new Avatar
                {
                    AvatarType = (int)AvatarType.Group,
                    Url = avatarUrl,
                    Timestamp = DateTime.UtcNow.ToEpochMilliseconds(),
                    EntityID = server.GroupID.ToString()
                };

                avatar.InsertLocal();
            }

            server.Refresh();

            GroupChangeCoordinator.CreateGroup(server, creator.UserID);
            GroupSearchIndexWorker.CreateGroupCreatedWorker(server, server.GetServerSearchSettingsOrDefault());
            return server;
        }

        public static GroupRole CreateSyncedRole(Group group, ExternalCommunityRole role, GroupRole[] allGroupRoles = null)
        {
            var allRoles = allGroupRoles ?? group.GetRoles(true);
            var groupRole = allRoles.FirstOrDefault(r => r.IsSynced && r.SyncID == role.ExternalID && r.Tag == role.RoleTag);
            if (groupRole != null)
            {
                return groupRole;
            }

            var defaultPermissions = GroupRoleTemplate.LargeGuest.DefaultPermissions;
            switch (role.RoleTag)
            {
                case GroupRoleTag.SyncedModerator:
                    defaultPermissions = GroupRoleTemplate.Moderator.DefaultPermissions;
                    break;
                case GroupRoleTag.SyncedOwner: // This purposefully just has mod permissions, since it is not really used in the server itself
                    defaultPermissions = GroupRoleTemplate.Moderator.DefaultPermissions;
                    break;
                case GroupRoleTag.SyncedSubscriber:
                case GroupRoleTag.SyncedSubscriberTier2:
                case GroupRoleTag.SyncedSubscriberTier3:
                    defaultPermissions = GroupRoleTemplate.Subscriber.DefaultPermissions;
                    break;
            }

            string roleName;
            try
            {
                roleName = role.GetRoleName();
            }
            catch (InvalidOperationException)
            {
                return null;
            }

            var name = roleName;
            var i = 0;
            while (allRoles.Any(r => r.Name == name))
            {
                name = string.Format("{0} {1}", roleName, ++i);
            }

            groupRole = group.CreateRole(name, defaultPermissions, false, false, true, role.VanityColor, 0, !string.IsNullOrEmpty(role.VanityBadge), role.ExternalID,
                role.RoleTag, role.Type);
            return groupRole;
        }

        public static GroupRole CreateGuildRole(Group group, ExternalGuildRole role, string roleName = null, int vanityColor = 0, int vanityBadge = 0, bool hasBadge = false, GroupRole[] allGroupRoles = null)
        {
            var allRoles = allGroupRoles ?? group.GetRoles(true);

            var defaultPermissions = GroupRoleTemplate.GuildMember.DefaultPermissions;
            if (role.RoleTag == GroupRoleTag.GuildMasterRank)
            {
                defaultPermissions = GroupRoleTemplate.GuildMaster.DefaultPermissions;
            }

            roleName = roleName ?? ExternalGuildRole.GetDefaultRoleName(role.Type, role.RoleTag);
            var name = roleName;
            var i = 0;
            while (allRoles.Any(r => r.Name == name))
            {
                name = string.Format("{0} ({1})", roleName, ++i);
            }

            var groupRole = group.CreateRole(name, defaultPermissions, false, false, false, vanityColor, vanityBadge, hasBadge, role.GuildIndex, role.RoleTag, role.Type);
            return groupRole;
        }

        public static Group CreateGuildServer(NewGroupMember creator, NewGuildInfo guildInfo, Guid? rootGroupID = null)
        {
            Group defaultChannel;
            return CreateGuildServer(creator, guildInfo, out defaultChannel, rootGroupID);
        }

        public static Group CreateGuildServer(NewGroupMember creator, NewGuildInfo guildInfo, out Group defaultChannel, Guid? rootGroupID = null)
        {
            // Create the server
            var server = Create(LocalConfiguration, LocalConfigID, null, creator, guildInfo.Title, GroupType.Large, GroupMode.TextOnly, false, guildInfo.IsPublic, null, 0, guildInfo.Subtype, false, rootGroupID);

            // Assign a random code for the url
            CreateVanityUrl(server, null);

            // Create the default roles (which also create permissions)
            var ownerGroupRole = server.CreateRole(guildInfo.OwnerRole.Name, GroupPermissions.All, true, false, false, guildInfo.OwnerRole.VanityColor);
            server.OwnerRoleID = ownerGroupRole.RoleID;

            var guestRoleTemplate = GroupRoleTemplate.GetDefaultGuestPermissionByGroupType(server.Type);
            var defaultGroupRole = server.CreateRole(guildInfo.DefaultRole.Name, guestRoleTemplate.DefaultPermissions, false, true, false, guildInfo.DefaultRole.VanityColor);
            server.DefaultRoleID = defaultGroupRole.RoleID;
            server.Update(p => p.OwnerRoleID, p => p.DefaultRoleID);

            // Create the additional roles if any
            var officerRoles = new HashSet<int>();
            foreach (var role in guildInfo.OtherRoles)
            {
                if (role.IsOfficer)
                {
                    var groupRole = server.CreateRole(role.Name, GroupRoleTemplate.Moderator.DefaultPermissions, false, false, false, role.VanityColor, role.VanityBadge, role.HasBadge);
                    officerRoles.Add(groupRole.RoleID);
                }
                else
                {
                    server.CreateRole(role.Name, GroupRoleTemplate.LargeGuest.DefaultPermissions, false, false, false, role.VanityColor, role.VanityBadge, role.HasBadge);
                }
            }

            // Log the creation event
            GroupEventManager.LogCreateGroupEvent(server, creator);

            var newMembers = new List<NewGroupMember>();

            // Add the current user (owner of the group)
            newMembers.Add(new NewGroupMember(creator.UserID, creator.RegionID, creator.Username, creator.DisplayName, creator.IPAddress, ownerGroupRole));
            var addedMembers = server.DoAddUsers(creator.UserID, newMembers.ToArray(), false);

            // Add the users to the group
            server.MemberCount = addedMembers.Count;
            server.Update(p => p.MemberCount);

            GroupEventManager.LogAddUsersEvent(server, creator, newMembers);

            var displayOrder = 0;
            // Add default text chat
            defaultChannel = server.CreatePermanentChildGroup(
                new NewPermanentChildGroupInfo(GroupMode.TextOnly, creator, guildInfo.DefaultTextChannelName, displayOrder++)
                {
                    IsDefault = true,
                    IsPublic = true
                });
            server.DefaultGroupID = defaultChannel.GroupID;
            server.Update(p => p.DefaultGroupID);

            // Add any additional chat channels
            foreach (var channelInfo in guildInfo.OtherChannels)
            {
                server.CreatePermanentChildGroup(
                    new NewPermanentChildGroupInfo(channelInfo.ChannelMode, creator, channelInfo.ChannelName, displayOrder++)
                    {
                        IsPublic = !channelInfo.RestrictToOfficers,
                        AccessRoles = channelInfo.RestrictToOfficers ? officerRoles : null
                    });
            }

            // Add syncs
            var games = new HashSet<int>();
            if (guildInfo.GuildSyncs.Length > 0)
            {
                var creatorMembership = server.GetMember(creator.UserID);

                // Set up games in the search settings
                foreach (var sync in guildInfo.GuildSyncs)
                {
                    // Add roles
                    var roles = sync.Guild.GetRoles().ToDictionary(r => r.RoleTag);
                    foreach (var role in sync.RolesToCreate)
                    {
                        ExternalGuildRole guildRole;
                        if (!roles.TryGetValue(role.Tag, out guildRole))
                        {
                            continue;
                        }

                        var groupRole = CreateGuildRole(server, guildRole, role.Name, role.VanityColor, role.VanityBadge, role.HasBadge);

                        if (role.IsOfficer)
                        {
                            officerRoles.Add(groupRole.RoleID);
                        }
                    }

                    sync.Guild.MapToGroup(creatorMembership, server);
                    games.Add(sync.GameID);
                }
            }

            // Set up games in the search settings
            if (guildInfo.GameIDs != null)
            {
                foreach (var gameID in guildInfo.GameIDs)
                {
                    games.Add(gameID);
                }
            }

            // Reload from the database
            server.Refresh();

            var searchSettings = server.GetServerSearchSettingsOrDefault();
            searchSettings.IsSearchable = guildInfo.IsSearchable;
            searchSettings.Games = games;
            searchSettings.Update(s => s.Games, s => s.IsSearchable);
            GroupSearchIndexWorker.CreateGroupCreatedWorker(server, searchSettings);

            // Queue off a worker job that will get picked up by a group service node, and finalize the group 
            GroupChangeCoordinator.CreateGroup(server, creator.UserID);

            return server;
        }

        public static Group CreateNormalGroup(NewGroupMember creator, string title, NewGroupMember[] invitees, string ownerRoleName, string guestRoleName, bool notify = true)
        {

            // Data Validation
            invitees.CheckRange(0, MaxUsersInGroup);
            title.CheckRange(1, TitleMaxLength);
            ownerRoleName.CheckRange(1, GroupRole.NameMaxLength, "Owner role name must be between {0} and {1} characters");
            guestRoleName.CheckRange(1, GroupRole.NameMaxLength, "Guest role name must be between {0} and {1} characters");

            var newGroup = Create(LocalConfiguration, LocalConfigID, null, creator, title, GroupType.Normal, GroupMode.TextAndVoice, false);

            // Create the default roles (which also create permissions)
            newGroup.CreateDefaultRoles(ownerRoleName, guestRoleName, null);
            var ownerRole = newGroup.OwnerRole;
            var defaultRole = newGroup.DefaultRole;

            // Log the creation event
            GroupEventManager.LogCreateGroupEvent(newGroup, creator);

            var newMembers = new List<NewGroupMember>();

            // Add the current user (owner of the group)
            newMembers.Add(new NewGroupMember(creator.UserID, creator.RegionID, creator.Username, creator.DisplayName, creator.IPAddress, ownerRole));

            // Add each of the requested users with the default role
            foreach (var invitee in invitees)
            {
                newMembers.Add(new NewGroupMember(invitee.UserID, invitee.RegionID, invitee.Username, invitee.DisplayName, invitee.IPAddress, defaultRole));
            }

            // Add the users to the group
            var addedUsers = newGroup.DoAddUsers(creator.UserID, newMembers.ToArray(), false);
            newGroup.Refresh();
            newGroup.MemberCount = addedUsers.Count;
            newGroup.Update(g => g.MemberCount);

            GroupEventManager.LogAddUsersEvent(newGroup, creator, newMembers);

            if (notify)
            {
                // Queue off a worker job that will get picked up by a group service node, and finalize the group 
                GroupChangeCoordinator.CreateGroup(newGroup, creator.UserID);
            }

            return newGroup;
        }

        public Group CreateChannel(NewGroupMember creator, GroupType type, GroupMode mode, bool isPublic, HashSet<int> accessRoles, string title, GroupCategoryInfo category, bool hideNoAccess, bool hideCallNoAccess)
        {
            if (Type != GroupType.Large)
            {
                throw new InvalidOperationException("You can only create nested groups beneath large groups");
            }

            // Check the permission
            var requestor = CheckPermission(GroupPermissions.ManageChannels, creator.UserID);
            if (!isPublic)
            {
                accessRoles = ResolveChannelAccessRoles(IsRootGroup ? this : RootGroup, requestor, accessRoles, title);
            }

            // Ensure the parent's depth is below the limit
            var depth = GetDepth();
            if (depth >= MaxNestingDepth)
            {
                throw new InvalidOperationException(string.Format("You cannot nest groups deeper than {0} levels.",
                    MaxNestingDepth));
            }

            var allChildren = GetAllChildren();
            var immediateChildren = GetImmediateChildren();

            if (allChildren.Count() > MaxChannelCount)
            {
                throw new GroupPermissionException(string.Format("You cannot have more than {0} channels.", MaxChannelCount));
            }

            var maxChannels = IsRootGroup ? MaxRootChannels : MaxImmediateChannels;

            if (immediateChildren.Count() > maxChannels)
            {
                throw new GroupPermissionException(string.Format("You cannot have more than {0} channels attached to a single parent.", MaxImmediateChannels));
            }

            if (immediateChildren.Any(p => p.Title.Equals(title, StringComparison.InvariantCultureIgnoreCase)))
            {
                throw new DataValidationException("Channel names must be unique.");
            }

            var displayOrder = GetImmediateChildren().Aggregate(-1, (current, child) => Math.Max(current, child.DisplayOrder)) + 1;

            Group group;
            switch (type)
            {
                case GroupType.Temporary:
                    group = CreateTemporaryChildGroup(creator, title, displayOrder);
                    break;
                case GroupType.Large:
                    group = CreatePermanentChildGroup(
                        new NewPermanentChildGroupInfo(mode, creator, title, displayOrder)
                        {
                            IsPublic = isPublic,
                            AccessRoles = accessRoles,
                            HideNoAccess = hideNoAccess,
                            HideCallNoAccess = hideCallNoAccess
                        });
                    break;
                default:
                    throw new InvalidOperationException("You can only create large or temporary groups beneath large groups");
            }

            group.Refresh();

            if (category != null)
            {
                group.DisplayCategoryID = category.ID;
                group.DisplayCategoryName = category.Name;
                group.DisplayCategoryRank = category.DisplayRank;
                group.Update(g => g.DisplayCategoryID, g => g.DisplayCategoryName, g => g.DisplayCategoryRank);
            }

            GroupEventManager.LogCreateSubgroup(this, requestor, group, accessRoles);
            GroupChangeCoordinator.CreateGroup(group, creator.UserID);

            return group;
        }

        private static string GenerateAndCheckUrl(Group parent, IEnumerable<Group> groups, string title)
        {
            var slug = title.ToUnicodeSlug(TitleMaxLength);
            if (FriendsServiceConfiguration.Instance.ReservedSlugs.Contains(slug))
            {
                slug = slug + "-channel";
            }

            var url = string.Format("{0}/{1}", parent.Url, slug);

            if (groups.Any(g => g.Url == url))
            {
                throw new DataValidationException("Channel has a duplicate url");
            }

            return url;
        }

        private static string GetUrlPath(Group parent)
        {
            var parentUrlPath = string.Empty;
            var parentGroup = parent;
            var visited = new HashSet<Group>();

            while (parentGroup != null)
            {
                if (visited.Contains(parentGroup))
                {
                    throw new InvalidOperationException("");
                }

                parentUrlPath += parentGroup.Url;
                if (parentGroup.IsRootGroup)
                {
                    break;
                }
                parentGroup = parentGroup.ParentGroup;
                visited.Add(parentGroup);
            }

            return parentUrlPath;
        }

        private Group CreateTemporaryChildGroup(NewGroupMember creator, string title, int displayOrder)
        {
            if (!AllowTempChannels)
            {
                throw new GroupPermissionException("This group does not permit the creation of temporary channels");
            }

            CheckPermission(GroupPermissions.CreateTemporaryGroup, creator.UserID);

            var group = Create(SourceConfiguration, RegionID, this, creator, title, GroupType.Temporary, GroupMode.TextAndVoice, false, false, null, displayOrder);

            GroupMemberWorker.CreateNewGroupWorker(group);

            return group;
        }

        public Group CreateSyndicatedTwitchChat(int requestorID, ExternalCommunity community, int displayOrder = 0, Group[] allChannels = null, bool notify = true)
        {
            allChannels = allChannels ?? GetAllChildren(true);
            if (displayOrder == 0)
            {
                displayOrder = allChannels.Length == 0 ? 1 : allChannels.Max(c => c.DisplayOrder);
            }

            var existing = allChannels.FirstOrDefault(c => c.ExternalChannelID == community.ExternalID);
            if (existing != null)
            {
                if (existing.IsDeleted)
                {
                    existing.DoRestoreChannel(GroupMember.CurseSystem);
                }
                return existing;
            }

            var title = community.ExternalDisplayName;
            var name = title;
            var i = 0;
            while (allChannels.Any(r => r.Title.ToUnicodeSlug(ChannelTitleMaxLength) == name.ToUnicodeSlug(ChannelTitleMaxLength)))
            {
                name = string.Format("{0} {1}", title, ++i);
            }

            var channel = CreatePermanentChildGroup(
                new NewPermanentChildGroupInfo(GroupMode.TextOnly, NewGroupMember.CurseSystem, name, displayOrder)
                {
                    IsPublic = true,
                    ExternalChannelID = community.ExternalID,
                    BypassPermissions = true
                });

            if (notify)
            {
                GroupChangeCoordinator.CreateGroup(channel, requestorID);
            }
            return channel;
        }

        private Group CreatePermanentChildGroup(NewPermanentChildGroupInfo creationInfo)
        {
            if (!creationInfo.BypassPermissions)
            {
                CheckPermission(GroupPermissions.ManageChannels, creationInfo.Creator.UserID);
            }

            if (string.IsNullOrEmpty(creationInfo.Title))
            {
                throw new DataValidationException("title cannot be empty");
            }

            var url = GenerateAndCheckUrl(ParentGroup, ParentGroup.GetImmediateChildren(), creationInfo.Title);

            var group = Create(SourceConfiguration, RegionID, this, creationInfo.Creator, creationInfo.Title, GroupType.Large, creationInfo.Mode, false, creationInfo.IsPublic, creationInfo.AccessRoles,
                creationInfo.DisplayOrder, GroupSubType.Custom, creationInfo.IsDefault, null, creationInfo.ExternalChannelID, creationInfo.HideNoAccess, creationInfo.HideCallNoAccess);

            group.Url = url;
            group.Update(p => p.Url);

            if (creationInfo.IsDefault) // For the default channel, give the default role permission to view channel history
            {
                var defaultRolePermissions = group.GetRolePermissions(DefaultRoleID);
                defaultRolePermissions.SetPermission(GroupPermissions.ChatReadHistory, true);
            }

            // Force create the creator's membership to the child group
            if (creationInfo.Creator.UserID > 0)
            {
                var subgroupMembership = new GroupMember(group, creationInfo.Creator.UserID, creationInfo.Creator.Username, creationInfo.Creator.DisplayName, creationInfo.Creator.RegionID, null);
                subgroupMembership.Insert(SourceConfiguration);
            }

            GroupMemberWorker.CreateNewGroupWorker(group);

            return group;
        }

        public bool SyncMessages
        {
            get { return Type != GroupType.Temporary; }
        }

        public bool SupportsOfflineMessaging
        {
            get
            {
                switch (Type)
                {
                    case GroupType.Temporary:
                        return false;
                    case GroupType.Normal:
                        return true;
                    case GroupType.Large:
                        return !IsRootGroup;
                    default:
                        throw new InvalidOperationException("Unknown group type: " + Type);
                }
            }
        }

        public void AddMemberRole(int requestorUserID, GroupRole newRole, int affectingUserID)
        {
            if (!CanHaveMembers)
            {
                throw new GroupPermissionException("This group cannot have direct members!");
            }

            if (newRole.RoleID == DefaultRoleID)
            {
                throw new InvalidOperationException("You cannot add the default to to a member. It is implicit!");
            }

            var otherMember = GetMember(affectingUserID);
            var requestingMember = CheckPermission(GroupPermissions.ChangeUserRole, requestorUserID, rm => CanModerateUser(rm, otherMember));

            if (newRole.Rank < requestingMember.BestRoleRank)
            {
                throw new GroupPermissionException("Cannot promote someone to a rank higher than your own!");
            }

            if (otherMember.Roles.Contains(newRole.RoleID))
            {
                throw new GroupPermissionException("Cannot promote someone to a role they are already a member of!");
            }

            DoAddMemberRole(requestingMember, newRole, otherMember);
        }

        public void SystemAddMemberRole(GroupRole newRole, GroupMember member, GroupRole[] allRoles = null)
        {
            DoAddMemberRole(GroupMember.CurseSystem, newRole, member, allRoles);
        }

        private void DoAddMemberRole(GroupMember requestor, GroupRole role, GroupMember affectedMember, GroupRole[] allRoles = null)
        {
            allRoles = allRoles ?? GetRoles();

            // Update the member
            affectedMember.AddRole(role);

            // Remove the default role ID
            if (affectedMember.Roles.Contains(DefaultRoleID))
            {
                affectedMember.RemoveRole(DefaultRole, DefaultRole, allRoles);
                GroupMemberIndexWorker.CreateRoleRemovedWorker(this, affectedMember.UserID, DefaultRole);
            }

            // Log it
            GroupEventManager.LogAddUserRoleEvent(this, requestor, affectedMember, role);

            // Let the group service know
            GroupMemberIndexWorker.CreateRoleAddedWorker(this, affectedMember.UserID, role);
            GroupChangeCoordinator.UpdateUsers(this, requestor.UserID, new HashSet<int> { affectedMember.UserID });
            VoicePermissionsWorker.CreateMemberRoleChanged(this, affectedMember.UserID);
        }

        public void RemoveMemberRole(int requestorUserID, GroupRole role, int affectingUserID)
        {
            if (!CanHaveMembers)
            {
                throw new GroupPermissionException("This group cannot have direct members!");
            }

            if (role.RoleID == DefaultRoleID)
            {
                throw new GroupPermissionException("You cannot add the default to to a member. It is implicit!");
            }

            var otherMember = GetMember(affectingUserID);

            var requestor = CheckPermission(GroupPermissions.ChangeUserRole, requestorUserID, rm => CanModerateUser(rm, otherMember));

            if (!otherMember.Roles.Contains(role.RoleID))
            {
                throw new GroupPermissionException("Cannot remove a role that does not exist!");
            }

            DoRemoveMemberRole(requestor, role, otherMember);
        }

        public void SystemRemoveMemberRole(GroupRole role, GroupMember member, GroupRole[] allRoles = null)
        {
            DoRemoveMemberRole(GroupMember.CurseSystem, role, member, allRoles);
        }

        public void ReconcileRemovedRole(int requestorUserID, int roleID)
        {
            var requestingMember = requestorUserID == 0 ? GroupMember.CurseSystem : GetMember(requestorUserID);
            if (requestingMember == null)
            {
                return;
            }

            var role = GetRole(roleID, true);
            if (role == null)
            {
                return;
            }

            // Get all of the non-deleted roles, for best role calculation
            var allRoles = GetRoles();

            foreach (var member in TryGetAllMembers().Where(m => m.Roles.Contains(roleID)))
            {
                DoRemoveMemberRole(role.IsSynced ? GroupMember.CurseSystem : requestingMember, role, member, allRoles);
            }
        }

        private void DoRemoveMemberRole(GroupMember requestor, GroupRole role, GroupMember affectedMember, GroupRole[] roles = null)
        {
            roles = roles ?? GetRoles();

            // Update the member
            affectedMember.RemoveRole(role, DefaultRole, roles);

            // Only coordinate/log if the member isn't deleted
            if (!affectedMember.IsDeleted)
            {
                GroupEventManager.LogRemoveUserRoleEvent(this, requestor, affectedMember, role);
                GroupChangeCoordinator.UpdateUsers(this, requestor.UserID, new HashSet<int> { affectedMember.UserID });
                VoicePermissionsWorker.CreateMemberRoleChanged(this, affectedMember.UserID);
            }

            // Always index
            GroupMemberIndexWorker.CreateRoleRemovedWorker(this, affectedMember.UserID, role);
        }

        public int GetDepth()
        {
            int depth = 0;
            var current = this;
            while (true)
            {
                if (depth > 3)
                {
                    throw new InvalidOperationException("Group depth exceeds limits! Group ID: " + GroupID);
                }

                if (current.GroupID == current.ParentGroupID)
                {
                    break;
                }

                current = current.ParentGroup;
                ++depth;
            }

            return depth;
        }

        public bool IsOwner(int userID)
        {
            if (!IsRootGroup)
            {
                throw new InvalidOperationException("Attempt to access owner data from a non root group.");
            }

            var member = GetMember(userID);
            if (member == null)
            {
                return false;
            }

            return member.Roles.Contains(OwnerRoleID);
        }

        public bool RoleHasPermission(GroupPermissions requiredPermission, int roleID)
        {
            var rolePermissions = GroupRolePermissions.GetByGroupIDAndRoleID(SourceConfiguration, GroupID, roleID);
            if (rolePermissions == null)
            {
                return false;
            }

            return rolePermissions.CheckPermission(requiredPermission);
        }

        public bool HasPermission(GroupPermissions requiredPermission, int userID, out GroupMember groupMember, bool returnLocalMembership = false)
        {
            try
            {
                groupMember = CheckPermission(requiredPermission, userID, null, returnLocalMembership);
                return true;
            }
            catch (GroupPermissionException)
            {
                groupMember = null;
                return false;
            }
        }

        public bool HasPermission(GroupPermissions requiredPermission, int userID, Func<GroupMember, bool> additionalCheck = null)
        {
            try
            {
                CheckPermission(requiredPermission, userID, additionalCheck);
                return true;
            }
            catch (GroupPermissionException)
            {
                return false;
            }
        }

        public bool HasPermission(GroupPermissions requiredPermission, GroupMember membership, IEnumerable<GroupRolePermissions> groupRolePermissions, Func<GroupMember, bool> additionalCheck = null)
        {

            // If it is missing or deleted, the user has no roles
            if (membership == null || membership.IsDeleted)
            {
                return false;
            }

            var hasPermission = false;

            // If they do have roles, get the permissions for the role specifically for this group
            foreach (var rolePermissions in groupRolePermissions)
            {
                // Always check for a deleted role permission
                if (rolePermissions == null || rolePermissions.IsDeleted)
                {
                    continue;
                }


                if (rolePermissions.CheckPermission(GroupPermissions.Access) // Regardless of the permission requested, always check for Access permissions 
                    && rolePermissions.CheckPermission(requiredPermission)) // And check the actual permission requested as well
                {
                    hasPermission = true;
                    break;
                }

            }

            if (!hasPermission)
            {
                return false;
            }

            if (additionalCheck != null)
            {
                return additionalCheck(membership);
            }

            return true;
        }


        /// <summary>
        /// This is an optimized method to quickly determine if a member has access to a server/channel. 
        /// It should not be used for anything other than the display of data, due to the potential for data consistency issues.
        /// </summary>
        /// <param name="member"></param>
        /// <returns></returns>
        public bool HasAccess(GroupMember member)
        {
            if (member == null || member.IsDeleted)
            {
                return false;
            }

            foreach (var role in member.Roles)
            {
                // If this is the owner role, always return true
                if (role == OwnerRoleID)
                {
                    return true;
                }

                long permissionsFlags;
                if (!RolePermissions.TryGetValue(role, out permissionsFlags))
                {
                    continue;
                }

                var permissions = (GroupPermissions)permissionsFlags;
                if (permissions.HasFlag(GroupPermissions.Access))
                {
                    return true;
                }
            }

            return false;
        }


        public GroupMember CheckPermission(GroupPermissions requiredPermission, int userID, Func<GroupMember, bool> additionalCheck = null, bool returnLocalMembership = false)
        {

            // Get the user's membership to the root group
            var rootMembership = RootGroup.GetMember(userID, true);

            // Get the user's local membership
            var localMembership = returnLocalMembership && !IsRootGroup ? GetMember(userID, true) : rootMembership;

            // If it is missing or deleted, the user has no roles
            if (rootMembership == null || rootMembership.IsDeleted)
            {
                throw new GroupPermissionException("No membership to group");
            }

            // If it is missing or deleted, the user has no roles
            if (localMembership == null || localMembership.IsDeleted)
            {
                throw new GroupPermissionException("No membership to group");
            }

            // Get the roles for the server
            var effectiveRoles = rootMembership.Roles;

            // Get the default role ID from the root group
            var defaultRoleID = IsRootGroup ? DefaultRoleID : RootGroup.DefaultRoleID;

            // Always ensure that the member has the default role
            effectiveRoles.Add(defaultRoleID);

            // Gather all of the role permissions for which this user is a member of (including the default role)
            var memberRolePermissions = effectiveRoles.Select(roleID => GroupRolePermissions.GetByGroupIDAndRoleID(SourceConfiguration, GroupID, roleID))
                                                      .Where(rolePermissions => rolePermissions != null)
                                                      .ToArray();

            // Call the check permissions method which makes no data access
            var hasPermission = HasPermission(requiredPermission, rootMembership, memberRolePermissions, additionalCheck);

            if (!hasPermission)
            {
                throw new GroupPermissionException(requiredPermission);
            }

            return localMembership;
        }

        /// <summary>
        /// Permanently deletes the group, all children. Group can only be deleted by the creator of the group
        /// </summary>
        /// <param name="userID"></param>
        public void DeleteRoot(int userID)
        {
            if (!IsRootGroup)
            {
                throw new InvalidOperationException("This is not a root group: " + GroupID);
            }

            CheckPermission(GroupPermissions.ManageServer, userID, member => member.Roles.Contains(OwnerRoleID));

            var subChannels = GetAllChildren(false, true);

            // Soft delete groups in the channel hierarchy
            foreach (var child in subChannels)
            {
                child.Status = GroupStatus.Deleted;
                child.Update(p => p.Status);
            }


            // Soft delete all group member records
            GroupMemberWorker.CreateDeleteGroupWorker(this);

            // Delete this root and notify others                      
            GroupChangeCoordinator.RemoveGroup(this, userID);

            if (Type == GroupType.Large)
            {
                GroupSearchIndexWorker.CreateGroupDeletedWorker(this, GetServerSearchSettingsOrDefault());
            }
        }

        /// <summary>
        /// Deletes the sub group and all the groupmemberships related to the channel.
        /// Can be deleted only by admin, owner of the root group
        /// </summary>        
        public void DeleteChannel(int userID, Group channel)
        {
            if (!IsRootGroup)
            {
                throw new InvalidOperationException("You must call this API from a root goup.");
            }

            if (channel.IsRootGroup)
            {
                throw new InvalidOperationException("You cannot delete a root goup with this API.");
            }

            if (channel.IsDefaultChannel)
            {
                throw new InvalidOperationException("Cannot delete the  default channel : " + GroupID);
            }

            GroupMember requestor;
            if (Type == GroupType.Large)
            {
                requestor = CheckPermission(GroupPermissions.ManageChannels, userID);
            }
            else if (Type == GroupType.Temporary)
            {
                requestor = CreatorID != userID ? CheckPermission(GroupPermissions.ManageChannels, userID) : GetMember(userID, true);
            }
            else
            {
                throw new InvalidOperationException("Invalid call made on this group: " + GroupID);
            }

            DoDeleteChannel(requestor, channel);
        }

        private void DoDeleteChannel(GroupMember requestor, Group channel)
        {
            var channels = channel.GetAllChildren(false, true);

            // Soft delete all sub channels
            foreach (var child in channels)
            {
                child.Status = GroupStatus.Deleted;
                child.Update(p => p.Status);
            }

            // Soft delete all group members
            GroupMemberWorker.CreateDeleteChannelWorker(this, channels.Select(p => p.GroupID).ToArray());

            // Notify all members of group
            GroupEventManager.LogRemoveSubgroup(ParentGroup, requestor, channel);

            // Send out notification
            GroupChangeCoordinator.RemoveGroup(channel, requestor.UserID);
        }

        /// <summary>
        /// Restores the soft deleted Large child group for a root group
        /// </summary>
        public void RestoreChannel(int requestingUserID)
        {
            if (Type != GroupType.Large)
            {
                //can only restore large sub groups
                throw new InvalidOperationException("cannot restore non-large groups : " + GroupID);
            }

            if (IsRootGroup)
            {
                //cannot restore deleted root groups
                throw new InvalidOperationException("Cannot restore a deleted root group" + GroupID);
            }

            var requestor = CheckPermission(GroupPermissions.ManageChannels, requestingUserID);
            DoRestoreChannel(requestor);
        }

        private void DoRestoreChannel(GroupMember requestor)
        {
            var subChannels = GetAllChildren();

            // Restore the group and all its children
            Status = GroupStatus.Normal;
            Update(p => p.Status);

            foreach (var child in subChannels)
            {
                child.Status = GroupStatus.Normal;
                child.Update(p => p.Status);
            }

            // Propagate this change out to member records
            GroupMemberWorker.CreateRestoreChannelWorker(this, subChannels.Select(p => p.GroupID).ToArray());

            // Let users know
            GroupChangeCoordinator.CreateGroup(this, requestor.UserID);
        }

        public void ReorganizeChildren(GroupCategoryInfo[] categoryInfos, int requestorUserID)
        {
            if (!IsRootGroup || Type != GroupType.Large)
            {
                throw new InvalidOperationException("Only root groups of Large type can have their children reorganized.");
            }

            CheckPermission(GroupPermissions.ManageChannels, requestorUserID);

            var allGroups = GetAllChildren().ToDictionary(g => g.GroupID);

            var groupsToUpdate = new Dictionary<Guid, Group>();
            var isRestructured = false;
            foreach (var categoryChange in categoryInfos)
            {
                var categoryGroups = allGroups.Values.Where(g => g.DisplayCategoryID == categoryChange.ID);

                foreach (var group in categoryGroups.Where(g => g.DisplayCategoryRank != categoryChange.DisplayRank))
                {
                    group.DisplayCategoryRank = categoryChange.DisplayRank;
                    groupsToUpdate[group.GroupID] = group;
                }

                foreach (var groupChange in categoryChange.Groups)
                {
                    Group group;
                    if (!allGroups.TryGetValue(groupChange.GroupID, out group))
                    {
                        throw new DataValidationException("One or more channels do not exist");
                    }

                    if (group.IsRootGroup)
                    {
                        throw new DataValidationException("Cannot modify the root group");
                    }

                    if (group.IsDefaultChannel && group.ParentGroupID != groupChange.ParentID)
                    {
                        throw new DataValidationException("Cannot change the parent of the default channel");
                    }

                    isRestructured = isRestructured || group.ParentGroupID != groupChange.ParentID;

                    group.DisplayCategoryID = categoryChange.ID;
                    group.DisplayCategoryName = categoryChange.Name;
                    group.DisplayCategoryRank = categoryChange.DisplayRank;
                    group.DisplayOrder = groupChange.DisplayOrder;
                    group.ParentGroupID = groupChange.ParentID;

                    groupsToUpdate[group.GroupID] = group;
                }
            }

            if (!GroupTree.IsValidTree(this, allGroups.GroupBy(g => g.Value.ParentGroupID).ToDictionary(g => g.Key, g => g.Select(v => v.Value))))
            {
                throw new DataValidationException("Requested changes form an invalid tree");
            }

            if (groupsToUpdate.Count > 0)
            {
                foreach (var group in groupsToUpdate.Values)
                {
                    group.Update(g => g.DisplayCategoryID, g => g.DisplayCategoryName, g => g.DisplayOrder, g => g.DisplayCategoryRank, g => g.ParentGroupID);
                }

                if (isRestructured)
                {
                    GroupMemberWorker.CreateGroupRestructuredWorker(this);
                }

                GroupChangeCoordinator.ChangeInfo(this, requestorUserID);
            }
        }

        public void VoiceSessionChanged(string code, int[] userIDs, GroupChangeType changeType)
        {
            // A new code has been assigned, ignore this change
            if (!string.IsNullOrEmpty(VoiceSessionCode) && VoiceSessionCode != code)
            {
                // Only log this issue for calls starting and ending
                if (changeType == GroupChangeType.VoiceSessionStarted || changeType == GroupChangeType.VoiceSessionEnded)
                {
                    Logger.Warn("Received a group voice session change request for a stale voice session code. This is not necessarily an error, but could indicate one.", new { GroupID, ChangeType = changeType, RequestCode = code, CurrentCode = VoiceSessionCode });
                }

                return;
            }

            switch (changeType)
            {
                case GroupChangeType.VoiceSessionStarted:
                    VoiceSessionCode = code;
                    Update(p => p.VoiceSessionCode);
                    AddMembersToCall(userIDs);
                    break;
                case GroupChangeType.VoiceSessionEnded:
                    VoiceSessionCode = string.Empty;
                    Update(p => p.VoiceSessionCode);
                    RemoveMembersFromCall(userIDs, true);
                    break;
                case GroupChangeType.VoiceSessionUserJoined:
                    AddMembersToCall(userIDs);
                    break;
                case GroupChangeType.VoiceSessionUserLeft:
                    RemoveMembersFromCall(userIDs, false);
                    break;
                default:
                    throw new DataValidationException("Invalid voice session change type " + changeType);
            }

            GroupChangeCoordinator.VoiceSessionChanged(this, new HashSet<int>(userIDs), VoiceSessionCode, changeType);
        }

        private void PurgeGuests(GroupInvitation invite, GroupMember requestor)
        {

            var expiredGuests = GroupMember.GetAllLocal(p => p.InviteCode, invite.InviteCode)
                                           .Where(p => p.GroupID == RootGroupID && p.BestRole == DefaultRoleID)
                                           .ToArray();

            DoRemoveUsers(requestor, new HashSet<int>(expiredGuests.Select(g => g.UserID)), GroupMemberRemovedReason.Kicked, invite.KickMembersMessage);
        }

        public void InvalidateExpiredInvitations()
        {
            var invites = GroupInvitation.GetAllByGroupID(GroupID)
                .Where(invite =>
                    invite.Status == GroupInvitationStatus.Active &&
                    invite.IsExpired);

            foreach (var invite in invites)
            {
                if (invite.AutoRemoveMembers)
                {
                    GroupMember requestor;
                    if (!HasPermission(GroupPermissions.InviteUsers, invite.CreatorID, out requestor))
                    {
                        requestor = GroupMember.CurseSystem;
                    }
                    // Treating -1 as the autoremove requestor since nobody can have that ID.
                    PurgeGuests(invite, requestor);
                    invite.Status = GroupInvitationStatus.Defunct;
                }
                else
                {
                    // Members may still exist, let periodic task handle making the invite defunct
                    invite.Status = GroupInvitationStatus.Invalid;
                }
                invite.Update(p => p.Status);
            }
        }

        public void ProcessInvalidatedInvitations()
        {
            var invites = GroupInvitation.GetAllByGroupID(GroupID)
                .Where(invite =>
                    invite.Status == GroupInvitationStatus.Invalid);

            foreach (var invite in invites)
            {
                if (GroupMember.GetAllLocal(p => p.InviteCode, invite.InviteCode).Any(x => !x.IsDeleted))
                {
                    continue;
                }
                // No active guests use this code, disuse this invitation                
                invite.Status = GroupInvitationStatus.Defunct;
                invite.Update(i => i.Status);
            }
        }



        private ExternalCommunityMapping[] GetExternalCommunityLinks(bool includeDeleted = false)
        {
            var mappings = ExternalCommunityMapping.GetAllLocal(m => m.GroupID, GroupID);

            if (!includeDeleted)
            {

                mappings = mappings.Where(m => !m.IsDeleted).ToArray();
            }

            return mappings;
        }

        private ExternalGuildMapping[] GetExternalGuildLinks(bool includeDeleted = false)
        {
            var mappings = ExternalGuildMapping.GetAllLocal(m => m.GroupID, GroupID);

            if (!includeDeleted)
            {
                mappings = mappings.Where(m => !m.IsDeleted).ToArray();
            }

            return mappings;
        }

        #region Notifications

        /// <summary>
        /// Create a notification from this data model.
        /// </summary>
        /// <param name="siteUrl">The URL of the root host of the group, ex: https://www.curse.com</param>
        /// <param name="metaDataOnly"></param>
        /// <param name="includeRoles"></param>
        /// <param name="includePermissions"></param>
        /// <param name="includeMembers"></param>
        /// <returns></returns>
        public GroupNotification ToNotification(string siteUrl, bool metaDataOnly = false, bool includeRoles = false, bool includePermissions = false, bool includeMembers = false, bool includeEmoticons = false, bool includeLinks = false)
        {
            var notification = new GroupNotification
            {
                GroupID = GroupID,
                ParentGroupID = ParentGroupID,
                RootGroupID = RootGroupID,
                DisplayOrder = DisplayOrder,
                GroupTitle = Title,
                MemberCount = MemberCount,
                IsDefaultChannel = IsDefaultChannel,
                GroupMode = Mode,
                GroupType = Type,
                GroupSubtype = Subtype,
                UrlPath = Url,
                UrlHost = siteUrl,
                IsPublic = IsPublic,
                AvatarTimestamp = AvatarTimestamp,
                MembersOnline = MembersOnline,
                HideNoAccess = HideNoAccess,
                HideCallMembersNoAccess = HideCallMembersNoAccess,
                ExternalChannelID = ExternalChannelID
            };

            if (metaDataOnly)
            {
                notification.MetaDataOnly = true;
            }
            else
            {
                notification.MetaDataOnly = false;
                notification.GroupType = Type;
                notification.MessageOfTheDay = MessageOfTheDay;
                notification.VoiceSessionCode = VoiceSessionCode == string.Empty ? null : VoiceSessionCode; // Temp fix                                
                notification.AllowTemporaryChildGroups = AllowTempChannels;
                notification.ForcePushToTalk = ForcePushToTalk.HasValue && ForcePushToTalk.Value;
                notification.Status = Status;
                notification.AfkTimerMins = AfkTimerMinutes;


                if (includeEmoticons)
                {
                    notification.Emotes = Emotes.Select(e => e.ToNotification()).ToArray();
                }

                notification.RolePermissions = RolePermissions;
                notification.IsStreaming = IsStreaming;
                notification.FlaggedAsInappropriate = FlaggedAsInappropriate;

                if (IsRootGroup)
                {
                    notification.ChatThrottleEnabled = ChatThrottleEnabled;
                    notification.ChatThrottleSeconds = ChatThrottleSeconds;
                    notification.HomeRegionID = RegionID;
                    notification.HomeRegionKey = GetRegionKey(RegionID);
                }

                if (includeRoles && IsRootGroup)
                {
                    var roles = GetRoles();
                    notification.Roles = roles.Select(r => r.ToNotification()).ToArray();
                }

                if (includeMembers && IsRootGroup && Type == GroupType.Normal)
                {
                    var allGroupMembers = TryGetAllMembers(true).Where(p => !p.IsDeleted).ToArray();
                    var groupMemberStats = UserStatistics.GetAllByUserIDs(allGroupMembers.Select(p => p.UserID)).ToDictionary(p => p.UserID);
                    notification.Members = allGroupMembers.Select(p => p.ToNotification(groupMemberStats.GetValueOrDefault(p.UserID))).ToArray();
                }

                if (includeLinks && IsRootGroup && Type == GroupType.Large)
                {
                    var communityLinks = GetExternalCommunityLinks();
                    var communities = ExternalCommunity.MultiGetLocal(communityLinks.Select(l => new KeyInfo(l.ExternalID, l.Type)));
                    notification.LinkedCommunities = communities.Select(c => c.ToPublicContract()).ToArray();

                    var guildLinks = GetExternalGuildLinks();
                    var guilds = ExternalGuild.MultiGetLocal(guildLinks.Select(g => new KeyInfo(g.Type, g.GameRegion, g.GameServer, g.Name)));
                    notification.LinkedGuilds = guilds.Select(g => g.ToContract()).ToArray();
                }
            }

            return notification;
        }

        public GroupNotification ToContactNotification(string siteUrl, Dictionary<Guid, Group> groupDictionary, Dictionary<Guid, GroupMember> membershipDictionary)
        {
            var notification = ToNotification(siteUrl);

            // Get the membership for this group from the supplied dictionary
            var membership = membershipDictionary.GetValueOrDefault(GroupID);
            if (membership != null)
            {
                notification.Membership = membership.ToMembershipNotification(this);
            }

            // Now get channels
            notification.Channels = groupDictionary.Values.Where(p => !p.IsRootGroup && p.RootGroupID == GroupID) // && (p.IsPublic || !p.HideNoAccess || p.HasAccess(membershipDictionary.GetValueOrDefault(p.RootGroupID))))
                                                          .Select(p => p.ToChannelNotification(membershipDictionary.GetValueOrDefault(p.GroupID), null))
                                                          .ToArray();

            return notification;
        }

        public void ValidateUrlPath(Dictionary<Guid, Group> groupDictionary)
        {
            if (!IsRootGroup)
            {
                throw new InvalidOperationException("Invalid group!");
            }


            var groupsByParent = groupDictionary.Select(g => g.Value).ToLookup(g => g.ParentGroupID);
            foreach (var kvp in groupDictionary)
            {
                var newUrl = kvp.Value.IsRootGroup ? string.IsNullOrWhiteSpace(kvp.Value.Url) ? VanityUrl.RandomForGroup(GroupID).Url : kvp.Value.Url
                                                   : GenerateAndCheckUrl(kvp.Value.ParentGroup, groupsByParent[kvp.Value.ParentGroupID].Where(g => g.GroupID != kvp.Key), kvp.Value.Title);
                if (kvp.Value.Url != newUrl)
                {
                    kvp.Value.Url = newUrl;
                    kvp.Value.Update(g => g.Url);
                }
            }

        }

        public GroupNotification ToServerNotification(string groupsRootUrl, int userID, bool showDeletedChannels = false)
        {
            var notification = ToNotification(groupsRootUrl, false, true, true, false, true, true);

            var groupDictionary = GetChildGroupDictionary(showDeletedChannels, true);

#if DEBUG
            ValidateUrlPath(groupDictionary);
#endif

            // Get the user's membership to these groups
            var membershipDictionary = GroupMember.MultiGetLocal(groupDictionary.Keys.Select(channelID => new KeyInfo(channelID, userID)))
                                                 .Where(p => !p.IsDeleted)
                                                 .ToDictionary(p => p.GroupID);

            // Get the membership for this group from the supplied dictionary
            var membership = membershipDictionary.GetValueOrDefault(GroupID);
            if (membership != null)
            {
                notification.Membership = membership.ToMembershipNotification(this);
            }

            // Get the voice users
            var voiceMembersLists = GetServerCallMembers(userID);
            var allVoiceMemberIDs = voiceMembersLists.SelectMany(p => p.Value.Members);
            var allVoiceMembers = GetMembers(allVoiceMemberIDs, true).GroupBy(m => m.UserID).ToDictionary(p => p.Key, p => p.First());
            var voiceMemberDictionary = voiceMembersLists.ToDictionary(kvp => kvp.Key, kvp => kvp.Value.Members.Select(p => allVoiceMembers.GetValueOrDefault(p)).ToArray());

            // Now get channels
            notification.Channels = groupDictionary.Values
                .Where(p => !p.IsRootGroup && p.RootGroupID == GroupID) // && (p.IsPublic || !p.HideNoAccess || p.HasAccess(membershipDictionary.GetValueOrDefault(p.RootGroupID))))
                .Select(p => p.ToChannelNotification(membershipDictionary.GetValueOrDefault(p.GroupID),
                    !p.HideCallMembersNoAccess || p.HasAccess(membershipDictionary.GetValueOrDefault(p.RootGroupID)) ? voiceMemberDictionary.GetValueOrDefault(p.GroupID) : null)
                )
                .ToArray();

            return notification;
        }

        public GroupNotification ToGroupNotification(string groupsRootUrl, int userID)
        {
            // Get the user's membership to these groups
            var groupMembership = GetMember(userID, true);
            var notification = ToNotification(groupsRootUrl, false, true, true, true);
            notification.Membership = groupMembership.ToMembershipNotification(this);
            return notification;
        }

        public ChannelContract ToChannelNotification(GroupMember member, GroupMember[] voiceMembers)
        {
            var contract = new ChannelContract
            {
                GroupMode = Mode,
                GroupType = Type,
                GroupID = GroupID,
                RootGroupID = RootGroupID,
                ParentGroupID = ParentGroupID,
                MessageOfTheDay = MessageOfTheDay,
                ForcePushToTalk = ForcePushToTalk.HasValue && ForcePushToTalk.Value,
                AllowTemporaryChildGroups = AllowTempChannels,
                DisplayOrder = DisplayOrder,
                DisplayCategoryID = DisplayCategoryID,
                DisplayCategory = DisplayCategoryName,
                DisplayCategoryRank = DisplayCategoryRank,
                GroupTitle = Title,
                RolePermissions = RolePermissions,
                IsDefaultChannel = IsDefaultChannel,
                VoiceSessionCode = VoiceSessionCode,
                IsPublic = IsPublic,
                UrlPath = Url,
                HideNoAccess = HideNoAccess,
                HideCallMembersNoAccess = HideCallMembersNoAccess,
                ExternalChannelID = ExternalChannelID
            };

            if (voiceMembers != null)
            {
                try
                {
                    contract.VoiceMembers = voiceMembers.Select(p => p.ToNotification(null)).ToArray();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to serialize voice members for channel", new { GroupID, RootGroup, voiceMembers });
                }

            }

            if (member != null)
            {
                contract.Membership = member.ToChannelMembershipContract(this);
            }

            return contract;

        }


        #endregion

        /// <summary>
        /// Upgrades temporary channel of clan to permanent.
        /// </summary>
        public void UpgradeToPermanent(int requestorID)
        {
            if (Type != GroupType.Temporary)
            {
                throw new InvalidOperationException("Only temporary groups can be upgraded.");
            }

            CheckPermission(GroupPermissions.ManageChannels, requestorID);
            Type = GroupType.Large;
            Update(p => p.Type);

            // TODO: Create/edit group member

            GroupChangeCoordinator.ChangeInfo(this, requestorID);
        }

        public static Group GetByID(Guid groupID, bool returnDeleted = false)
        {
            return GetByID(groupID.ToString(), returnDeleted);
        }

        public static Group GetWritableByID(Guid groupID, bool returnDeleted = false)
        {
            var group = GetWritable(groupID);

            if (group == null)
            {
                return null;
            }

            if (returnDeleted || !group.IsDeleted)
            {
                return group;
            }

            return null;
        }

        public static Group GetByID(string groupID, bool returnDeleted = false)
        {
            var group = GetLocal(groupID);
            if (group == null)
            {
                return null;
            }

            if (returnDeleted || !group.IsDeleted)
            {
                return group;
            }

            return null;
        }

        public GroupCallMemberList GetGroupCallMemberList(int requestorID)
        {
            return GroupCallMemberList.GetLocal(GroupID);
        }

        public Dictionary<Guid, GroupCallMemberList> GetServerCallMembers(int requestorID)
        {
            if (Type != GroupType.Large || !IsRootGroup)
            {
                throw new DataValidationException("Group must be a server");
            }
            var children = ChildGroupIDs ?? new HashSet<Guid>();
            children.Remove(GroupID);
            return GroupCallMemberList.MultiGetLocal(children.Select(id => new KeyInfo(id))).Where(p => p.Members != null).ToDictionary(p => p.GroupID);
        }

        public void AddMembersToCall(int[] userIDs)
        {
            var members = GroupMember.MultiGetLocal(userIDs.Select(id => new KeyInfo(GroupID, id)));

            var attempt = 1;
            do
            {
                try
                {
                    var list = GroupCallMemberList.Get(RegionID, GroupID);
                    if (list == null)
                    {
                        list = new GroupCallMemberList
                        {
                            GroupID = GroupID,
                            RegionID = RegionID,
                            Members = new HashSet<int>(userIDs)
                        };

                        list.Insert(RegionID);
                        return;
                    }

                    list.Members = list.Members ?? new HashSet<int>();

                    var tryUpdate = false;
                    foreach (var member in members.Where(m => !list.Members.Contains(m.UserID)))
                    {
                        list.Members.Add(member.UserID);
                        tryUpdate = true;
                    }

                    if (tryUpdate)
                    {
                        list.Update(UpdateMode.Concurrent, g => g.Members);
                    }

                    return;
                }
                catch (AerospikeException ex)
                {
                    if (ex.Message.Contains("Generation error"))
                    {
                        Logger.Debug(ex, "Failed to update voice members, due to a concurrency collision.");
                    }
                    else
                    {
                        Logger.Error(ex, "Failed to update voice members, due to an Aerospike error!");
                        break;
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to update voice members.");
                    break;
                }
            } while (attempt++ <= 10);

            Logger.Warn(string.Format("Failed to add members to call list after {0} attempts", attempt), new { GroupID, userIDs });
        }

        public void RemoveMembersFromCall(int[] userIDs, bool purgeAll)
        {
            var attempt = 1;
            do
            {
                try
                {
                    var list = GroupCallMemberList.Get(RegionID, GroupID);
                    if (list == null)
                    {
                        return;
                    }

                    if (purgeAll)
                    {
                        if (list.Members.Count == 0)
                        {
                            return;
                        }

                        list.Members = new HashSet<int>();
                        list.Update(UpdateMode.Concurrent, g => g.Members);
                        return;
                    }

                    list.Members = list.Members ?? new HashSet<int>();

                    var update = false;
                    foreach (var userID in userIDs)
                    {
                        update |= list.Members.Remove(userID);
                    }

                    if (update)
                    {
                        list.Update(UpdateMode.Concurrent, g => g.Members);
                    }

                    return;
                }
                catch (AerospikeException ex)
                {
                    if (ex.Message.Contains("Generation error"))
                    {
                        Logger.Debug(ex, "Failed to update voice members, due to a concurrency collision.");
                    }
                    else
                    {
                        Logger.Error(ex, "Failed to update voice members, due to an Aerospike error!");
                        break;
                    }

                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to update voice members.");
                    break;
                }
            } while (attempt++ <= 10);

            Logger.Warn(string.Format("Failed to remove members from call list after {0} attempts", attempt), new { GroupID, userIDs, purgeAll });
        }

        #region Relationships

        public GroupMember GetMember(int userID, bool allowDirtyRead = false, bool returnDeleted = true)
        {
            var member = allowDirtyRead ? GroupMember.Get(SourceConfiguration, GroupID, userID) : GroupMember.Get(RegionID, GroupID, userID);

            // If the member couldn't be found from the local region, try the remote one
            if (member == null && allowDirtyRead && SourceConfiguration.RegionIdentifier != RegionID)
            {
                member = GroupMember.Get(RegionID, GroupID, userID);
            }

            if (member == null || (member.IsDeleted && !returnDeleted))
            {
                return null;
            }

            return member;
        }

        public bool IsMember(int userID)
        {
            var member = GetMember(userID);
            return member != null && !member.IsDeleted;
        }

        public GroupMember[] TryGetAllMembers(bool allowDirtyRead = false, int maxAttempts = 3, int attempt = 1)
        {
            try
            {
                return GetAllMembers(allowDirtyRead);
            }
            catch (Exception ex)
            {
                if (attempt == maxAttempts)
                {
                    Logger.Error(ex, "Failed to retrieve group members after " + attempt + " attempts", new { LocalRegion = LocalConfiguration.RegionKey, AccessRegion = SourceConfiguration.RegionKey });
                    return null;
                }

                return TryGetAllMembers(allowDirtyRead, maxAttempts, ++attempt);
            }
        }

        public GroupMember[] GetAllMembers(bool allowDirtyRead = false)
        {
            return allowDirtyRead ? GroupMember.GetAllLocal(p => p.GroupID, GroupID)
                                  : GroupMember.GetAll(RegionID, p => p.GroupID, GroupID);
        }

        public GroupMember[] GetAllRootMembers(bool allowDirtyRead = false)
        {
            return allowDirtyRead ? GroupMember.GetAllLocal(p => p.GroupID, RootGroupID)
                                  : GroupMember.GetAll(RegionID, p => p.GroupID, RootGroupID);
        }

        public IReadOnlyCollection<GroupMember> GetMembers(IEnumerable<int> userIDs, bool allowDirtyRead = false)
        {
            return allowDirtyRead ? GroupMember.MultiGetLocal(userIDs.Select(p => new KeyInfo(GroupID, p)))
                                  : GroupMember.MultiGet(RegionID, userIDs.Select(p => new KeyInfo(GroupID, p)));
        }


        public Group GetChildGroup(Guid groupID, bool allowDeleted = false)
        {
            var childGroup = Get(RegionID, groupID);

            if (childGroup == null)
            {
                Logger.Warn("Failed to retrieve child group from database.", new { RootGroupID, RegionID, ChildGroupID = groupID });
                return null;
            }

            if (childGroup.RootGroupID != RootGroupID)
            {
                Logger.Warn("Attempt to access a group from another root.", new { RootGroupID, ChildRootGroupID = childGroup.RootGroupID, ChildGroupID = childGroup.RootGroupID });
                return null;
            }

            if (childGroup.IsDeleted && !allowDeleted)
            {
                return null;
            }

            return childGroup;
        }

        public static IReadOnlyCollection<Group> GetAllFromIDs(IEnumerable<Guid> groupIDs)
        {
            return MultiGetLocal(groupIDs.Select(p => new KeyInfo(p)));
        }

        public static Dictionary<Guid, Group> GetDictionaryFromIDs(IEnumerable<Guid> groupIDs)
        {
            var results = new Dictionary<Guid, Group>();
            var groups = MultiGetLocal(groupIDs.Select(p => new KeyInfo(p)));
            foreach (var group in groups)
            {
                results[group.GroupID] = group;
            }

            return results;
        }

        #endregion

        public void LinkExternalCommunity(int requestorID, ExternalCommunity community)
        {
            if (!IsRootGroup)
            {
                throw new DataValidationException("Not a root group");
            }

            var accountLink = ExternalAccountMapping.GetLocal(requestorID, community.Type, community.ExternalID);
            if (accountLink == null || accountLink.IsDeleted)
            {
                throw new GroupPermissionException("Not the owner of the external account.");
            }

            var requestor = CheckPermission(GroupPermissions.ManageServer, requestorID);

            if(MappedCommunities == null)
            {
                MappedCommunities = new HashSet<string>(GetExternalCommunityLinks().Select(c => c.ExternalID));
            }

            MappedCommunities.Add(community.ExternalID);
            var mapping = community.MapToGroup(requestor, this);
            Update(g => g.MappedCommunities);

            var roles = GetRoles(true).Where(r => r.IsSynced && r.SyncID == community.ExternalID).ToArray();
            foreach (var role in ExternalCommunityRole.GetAllLocal(r => r.ExternalID, community.ExternalID).Where(r => r.Type == community.Type).OrderBy(r=>r.RoleRank))
            {
                if (!community.CanHaveSubs && role.CheckPremium())
                {
                    // Don't create sub role for non-partners
                    continue;
                }

                var existing = roles.FirstOrDefault(r => r.Source == role.Type && r.Tag == role.RoleTag);
                if (existing == null)
                {
                    CreateSyncedRole(this, role);
                }
                else if (existing.IsDeleted)
                {
                    RestoreRole(existing);
                }
            }

            CreateSyndicatedTwitchChat(requestorID, community);

            if (!IsStreaming && community.IsLive)
            {
                UpdateIsStreaming(community);
            }

            GroupChangeCoordinator.ChangeInfo(this, requestorID);
        }

        public void UnlinkExternalCommunity(int requestorID, string externalCommunityID, AccountType communityType)
        {
            if (!IsRootGroup)
            {
                throw new DataValidationException("Not a root group");
            }

            var accountLink = ExternalAccountMapping.GetLocal(requestorID, communityType, externalCommunityID);
            if (accountLink == null || accountLink.IsDeleted)
            {
                throw new GroupPermissionException("Not the owner of the external account");
            }

            var requestor = CheckPermission(GroupPermissions.ManageServer, requestorID);
            DoUnlinkExternalCommunity(requestor, externalCommunityID, communityType);
        }

        public void SystemUnlinkExternalCommunity(string externalCommunityID, AccountType communityType)
        {
            if (!IsRootGroup)
            {
                throw new DataValidationException("Not a root group");
            }

            DoUnlinkExternalCommunity(GroupMember.CurseSystem, externalCommunityID, communityType);
        }

        private void DoUnlinkExternalCommunity(GroupMember requestor, string externalCommunityID, AccountType communityType)
        {
            if(MappedCommunities == null)
            {
                var links = GetExternalCommunityLinks();
                MappedCommunities = new HashSet<string>(links.Select(l=>l.ExternalID));
            }

            if (!MappedCommunities.Remove(externalCommunityID))
            {
                // No effective change
                return;
            }

            Update(g => g.MappedCommunities);

            var affectedCommunity = ExternalCommunity.GetLocal(externalCommunityID, AccountType.Twitch);
            if (affectedCommunity == null)
            {
                return;
            }

            var mapping = affectedCommunity.DeleteMapping(requestor, GroupID);

            if (mapping == null)
            {
                return;
            }

            foreach (var groupRole in GetRoles().Where(r => r.IsSynced && r.SyncID == affectedCommunity.ExternalID && r.Source == affectedCommunity.Type))
            {
                DoDeleteRole(requestor.UserID, groupRole.RoleID);
            }

            var children = FastGetRootChildren();
            var affectedChild = children.FirstOrDefault(c => c.ExternalChannelID == affectedCommunity.ExternalID);
            if (affectedChild != null)
            {
                DoDeleteChannel(GroupMember.CurseSystem, affectedChild);
                GroupChangeCoordinator.RemoveGroup(affectedChild, requestor.UserID);
            }

            if (IsStreaming && affectedCommunity.IsLive)
            {
                var communities = GetMappedCommunities();
                if (communities.Where(c => c.ExternalID != affectedCommunity.ExternalID).All(c => !c.IsLive))
                {
                    // If this was the last mapping that was streaming, don't leave the group in the streaming state
                    // Fake IsLive = false since it is no longer linked
                    affectedCommunity.IsLive = false;
                    UpdateIsStreaming(affectedCommunity);
                }
            }

            GroupChangeCoordinator.ChangeInfo(this, requestor.UserID);
        }

        public ExternalAccount[] GetExternalAccounts()
        {
            var mappings = ExternalCommunityMapping.GetAllLocal(c => c.GroupID, GroupID).Where(c => !c.IsDeleted);
            return ExternalAccount.MultiGetLocal(mappings.Select(m => new KeyInfo(m.ExternalID, m.Type))).ToArray();
        }

        public void ReconcileExternalCommunity(int userID)
        {
            var mappings = ExternalCommunityMapping.GetAllLocal(c => c.GroupID, GroupID).Where(c => !c.IsDeleted).ToArray();
            var externalAccounts = ExternalAccount.MultiGetLocal(mappings.Select(m => new KeyInfo(m.ExternalID, m.Type)));
            var links = ExternalAccountMapping.MultiGetLocal(externalAccounts.SelectMany(a => a.MappedUsers.Select(u => new KeyInfo(u, a.Type, a.ExternalID))));
            var members = GroupMember.MultiGetLocal(links.Select(l => l.UserID).Distinct().Select(id => new KeyInfo(GroupID, id))).Where(m => !m.IsDeleted).ToDictionary(m => m.UserID);
            var linksByCommunity = links.ToLookup(l => new Tuple<string, AccountType>(l.ExternalID, l.Type));

            foreach (var mapping in mappings)
            {
                var communityTuple = new Tuple<string, AccountType>(mapping.ExternalID, mapping.Type);
                if (linksByCommunity[communityTuple].Count(l => !l.IsDeleted && members.ContainsKey(l.UserID)) == 0)
                {
                    DoUnlinkExternalCommunity(GroupMember.CurseSystem, mapping.ExternalID, mapping.Type);
                }
            }
        }

        public void ReconcileExternalGuild(int userID)
        {
            // For now, just clean up if there are no group members left
            if (MemberCount > 5)
            {
                // Break early to not have to request all members
                return;
            }

            var trueMemberCount = GetAllMembers().Count(m => !m.IsDeleted);
            if (trueMemberCount > 0)
            {
                return;
            }


            var mappings = GetExternalGuildLinks();
            var allGuildDetails = mappings.Select(m => new { GuildInfo = m.GetGuildInfo(), GuildIndex = m.GetGuildIndex() }).ToArray();

            var guilds = ExternalGuild.MultiGetLocal(allGuildDetails.Select(g => new KeyInfo(g.GuildInfo.Type, g.GuildInfo.GameRegion, g.GuildInfo.GameServer, g.GuildInfo.Name)))
                .ToDictionary(g => g.GetGuildIndex());

            foreach (var mapping in mappings)
            {
                var guildIndex = mapping.GetGuildIndex();

                ExternalGuild guild;
                if (!guilds.TryGetValue(guildIndex, out guild))
                {
                    continue;
                }

                DoUnlinkExternalGuild(GroupMember.CurseSystem, guild);
            }
        }

        public bool AddEmoticon(int requestorID, string regex, string url, EmoticonSource source, HashSet<int> requiredRoles, bool notify = true, string syncID = null)
        {
            var requestor = CheckPermission(GroupPermissions.ManageServer, requestorID);
            return DoAddEmoticon(requestor, regex, source, url, requiredRoles, notify, syncID);
        }

        public bool SystemAddEmoticon(string regex, string url, EmoticonSource source, HashSet<int> requiredRoles, bool notify = true, string syncID = null)
        {
            return DoAddEmoticon(GroupMember.CurseSystem, regex, source, url, requiredRoles, notify, syncID);
        }

        private bool DoAddEmoticon(GroupMember requestor, string regex, EmoticonSource source, string url, HashSet<int> requiredRoles, bool notify, string syncID)
        {
            var emote = GroupEmoticon.GetLocal(GroupID, regex, source);
            var conflict = Emotes.FirstOrDefault(e => e.RegexString == regex && e.Status == EmoticonStatus.Active);

            EmoticonStatus status;
            if (conflict == null || conflict.Source == source)
            {
                status = EmoticonStatus.Active;
            }
            else if (source == EmoticonSource.Curse)
            {
                status = EmoticonStatus.Active;
                conflict.Status = EmoticonStatus.Conflict;
                conflict.Update(e => e.Status);
            }
            else
            {
                status = EmoticonStatus.Conflict;
            }

            if (emote == null)
            {
                emote = new GroupEmoticon
                {
                    GroupID = GroupID,
                    RegexString = regex,
                    Url = url,
                    Source = source,
                    SyncID = syncID,
                    Status = status,
                    RequiredRoles = requiredRoles
                };
                emote.Insert(SourceConfiguration);

                if (status == EmoticonStatus.Active)
                {
                    if (notify)
                    {
                        GroupChangeCoordinator.UpdateEmoticons(this, requestor.UserID);
                    }
                    return true;
                }
            }
            else
            {
                var expressions = new List<Expression<Func<GroupEmoticon, object>>>();
                if (emote.Status != status)
                {
                    emote.Status = status;
                    expressions.Add(e => e.Status);
                }

                if (emote.Url != url)
                {
                    emote.Url = url;
                    expressions.Add(e => e.Url);
                }

                if (!emote.RequiredRoles.SetEquals(requiredRoles))
                {
                    emote.RequiredRoles = requiredRoles;
                    expressions.Add(e => e.RequiredRoles);
                }

                if (emote.SyncID != syncID)
                {
                    emote.SyncID = syncID;
                    expressions.Add(e => e.SyncID);
                }

                if (expressions.Any())
                {
                    emote.Update(expressions.ToArray());
                    if (status == EmoticonStatus.Active)
                    {
                        if (notify)
                        {
                            GroupChangeCoordinator.UpdateEmoticons(this, requestor.UserID);
                        }
                        return true;
                    }
                }
            }
            return false;
        }

        public bool RemoveEmoticon(int requestorID, GroupEmoticon emoticon, bool notify = true)
        {
            var requestor = CheckPermission(GroupPermissions.ManageServer, requestorID);
            return DoRemoveEmoticon(requestor, emoticon, notify);
        }

        public bool SystemRemoveEmoticon(GroupEmoticon emoticon, bool notify = true)
        {
            return DoRemoveEmoticon(GroupMember.CurseSystem, emoticon, notify);
        }

        public bool DoRemoveEmoticon(GroupMember requestor, GroupEmoticon emoticon, bool notify = true)
        {

            var originalStatus = emoticon.Status;
            if (originalStatus == EmoticonStatus.Deleted)
            {
                return false;
            }

            emoticon.Status = EmoticonStatus.Deleted;
            emoticon.Update(e => e.Status);

            if (originalStatus == EmoticonStatus.Active)
            {
                // Reactivate another emote that is conflicted if possible
                var conflicted = GroupEmoticon.GetAllLocal(g => g.GroupID, GroupID).FirstOrDefault(e => e.RegexString == emoticon.RegexString && e.Status == EmoticonStatus.Conflict);
                if (conflicted != null)
                {
                    conflicted.Status = EmoticonStatus.Active;
                    conflicted.Update(e => e.Status);
                }

                if (notify)
                {
                    GroupChangeCoordinator.UpdateEmoticons(this, requestor.UserID);
                }

                return true;
            }

            return false;
        }

        public GroupGiveaway CreateGiveaway(int requestorID, string title, HashSet<int> requiredRoles, Dictionary<int, int> roleBonuses,
            int sharingBonus, TimeSpan responseWindow, HashSet<int> autoEnterRoles, HashSet<int> autoClaimRoles, int rollsBeforeWinner,
            bool autoEnterActiveUsers, bool allowRepeatWinners, HashSet<int> ignoredUsers, bool includeOfflineMembers)
        {
            // Check permission
            var member = CheckPermission(GroupPermissions.ManageGiveaways, requestorID);

            // Check if a giveaway is in progress
            var latest = GetGiveaway(LatestGiveawayNumber);
            if (latest != null && latest.Status != GroupGiveawayStatus.Inactive)
            {
                throw new DataConflictException();
            }

            var nextGiveaway = LatestGiveawayNumber + 1;

            // Create the giveaway
            var giveaway = new GroupGiveaway
            {
                GroupID = GroupID,
                GiveawayID = nextGiveaway,
                CreatorID = member.UserID,
                Title = title,
                ResponseWindow = responseWindow,
                SharingBonus = sharingBonus,
                Status = GroupGiveawayStatus.Active,
                DateStatusChanged = DateTime.UtcNow,
                DateStarted = DateTime.UtcNow,
                RegionID = RegionID,
                AllowRepeatWinners = allowRepeatWinners,
                AutoEnterActiveUsers = autoEnterActiveUsers,
                FakeRollsBeforeWinner = rollsBeforeWinner,
                EligibleEntries = 0,
                RequiredRoles = requiredRoles ?? new HashSet<int>(),
                RoleBonuses = roleBonuses ?? new Dictionary<int, int>(),
                AutoClaimRoles = autoClaimRoles ?? new HashSet<int>(),
                AutoEnterRoles = autoEnterRoles ?? new HashSet<int>(),
                IgnoredUsers = ignoredUsers ?? new HashSet<int>(),
                FakeRollsLeft = rollsBeforeWinner,
                CurrentRoll = 0,
                PendingWinnerUserID = 0,
                IncludeOfflineMembers = includeOfflineMembers
            };

            var attempts = 0;
            var maxAttempts = 100;

            while (true)
            {
                attempts++;

                try
                {
                    giveaway.Insert(RegionID, UpdateMode.Concurrent);
                    break;
                }
                catch (AerospikeException ex)
                {
                    if (!ex.Message.StartsWith("Error Code 5: Key already exists"))
                    {
                        Logger.Error(ex, "Unable to insert giveaway due to general database error.", new { attempts, GroupID, Title });
                        throw;
                    }

                    if (attempts >= maxAttempts)
                    {
                        Logger.Error(ex, "Unable to insert giveaway after maximum attempts.", new { attempts, GroupID, Title });
                        throw;
                    }

                    Logger.Warn("Giveaway failed to insert. Incrementing giveaway counter...", new { attempts, GroupID, Title });
                    giveaway.GiveawayID = giveaway.GiveawayID + 1;
                }
            }

            var settings = GroupGiveawaySettings.Get(RegionID, GroupID);
            if (settings == null)
            {
                settings = new GroupGiveawaySettings
                {
                    AutoEnterActiveUsers = giveaway.AutoEnterActiveUsers,
                    RequiredRoles = giveaway.RequiredRoles,
                    RollsBeforeWinner = giveaway.FakeRollsBeforeWinner,
                    IgnoredUsers = giveaway.IgnoredUsers,
                    AllowRepeatWinners = giveaway.AllowRepeatWinners,
                    GroupID = giveaway.GroupID,
                    AutoEnterRoles = giveaway.AutoEnterRoles,
                    AutoClaimRoles = giveaway.AutoClaimRoles,
                    RegionID = giveaway.RegionID,
                    SharingBonus = giveaway.SharingBonus,
                    RoleBonuses = giveaway.RoleBonuses,
                    ResponseWindow = giveaway.ResponseWindow,
                    IncludeOfflineMembers = giveaway.IncludeOfflineMembers
                };
                settings.Insert(RegionID);
            }
            else
            {
                settings.AutoEnterActiveUsers = giveaway.AutoEnterActiveUsers;
                settings.RequiredRoles = giveaway.RequiredRoles;
                settings.RollsBeforeWinner = giveaway.FakeRollsBeforeWinner;
                settings.IgnoredUsers = giveaway.IgnoredUsers;
                settings.AllowRepeatWinners = giveaway.AllowRepeatWinners;
                settings.AutoEnterRoles = giveaway.AutoEnterRoles;
                settings.AutoClaimRoles = giveaway.AutoClaimRoles;
                settings.SharingBonus = giveaway.SharingBonus;
                settings.RoleBonuses = giveaway.RoleBonuses;
                settings.ResponseWindow = giveaway.ResponseWindow;
                settings.IncludeOfflineMembers = giveaway.IncludeOfflineMembers;
                settings.Update();
            }

            LatestGiveawayNumber = giveaway.GiveawayID;
            Update(g => g.LatestGiveawayNumber);

            GroupGiveawayCoordinator.StartGiveaway(giveaway, member, autoEnterRoles);
            return giveaway;
        }

        public GroupGiveaway GetGiveaway(int giveawayID, bool allowDirtyRead = false)
        {
            return GroupGiveaway.Get(allowDirtyRead ? LocalConfigID : RegionID, GroupID, giveawayID);
        }

        public GroupGiveawaySettings GetGiveawaySettingsOrDefault(int requestorID)
        {
            CheckPermission(GroupPermissions.ManageGiveaways, requestorID);

            var settings = GroupGiveawaySettings.GetLocal(GroupID);
            if (settings != null)
            {
                return settings;
            }

            settings = new GroupGiveawaySettings
            {
                AllowRepeatWinners = false,
                AutoClaimRoles = new HashSet<int>(),
                AutoEnterActiveUsers = false,
                AutoEnterRoles = new HashSet<int>(),
                GroupID = GroupID,
                RegionID = RegionID,
                IgnoredUsers = new HashSet<int>(),
                RollsBeforeWinner = 0,
                SharingBonus = 0,
                RoleBonuses = new Dictionary<int, int>(),
                ResponseWindow = TimeSpan.FromSeconds(60),
                RequiredRoles = new HashSet<int>()
            };
            settings.Insert(RegionID);

            return settings;
        }

        public GroupPoll CreatePoll(int requestorID, string title, Dictionary<int, string> options, int durationMinutes, GroupPollSettings newSettings)
        {
            var requestor = CheckPermission(GroupPermissions.ManagePolls, requestorID);

            var polls = GroupPoll.GetAllLocal(p => p.GroupID, GroupID);
            if (polls.Count(p => p.Status != GroupPollStatus.Inactive) >= GroupPoll.MaxActivePolls)
            {
                throw new DataConflictException();
            }

            var newPollID = polls.Length == 0 ? 1 : polls.Max(p => p.PollID) + 1;
            var now = DateTime.UtcNow;
            var poll = new GroupPoll
            {
                GroupID = GroupID,
                PollID = newPollID,
                StartDate = now,
                EndDate = durationMinutes > 0 ? now + TimeSpan.FromMinutes(durationMinutes) : now,
                DurationMinutes = durationMinutes,
                Status = GroupPollStatus.Starting,
                StatusChangeDate = now,
                Title = title,
                RegionID = RegionID,
                CreatorUserID = requestorID,

                DisplayType = newSettings.DisplayType,
                AllowMultipleSelections = newSettings.AllowMultipleSelections,
                RequiredRoles = newSettings.RequiredRoles,
                Code = newSettings.IsPublic ? GroupPoll.CreatePollCode() : string.Empty,
                AllowRevotes = newSettings.AllowRevotes,
                DuplicateMode = newSettings.DuplicateMode,
                OptionIDs = new HashSet<int>(options.Keys)
            };
            poll.Insert(RegionID, UpdateMode.Concurrent);
            poll.CreateOptions(options);
            poll.UpdateSettings();

            GroupPollCoordinator.Create(requestor, poll, GroupPollChangeType.Started);
            return poll;
        }

        public GroupPoll[] GetActivePolls()
        {
            return GroupPoll.GetAllLocal(p => p.GroupID, GroupID).Where(p => p.Status != GroupPollStatus.Inactive).ToArray();
        }

        public GroupPoll GetPoll(int pollID, bool allowDirtyRead = false)
        {
            return GroupPoll.Get(allowDirtyRead ? LocalConfigID : RegionID, GroupID, pollID);
        }

        public GroupPollSettings GetPollSettingsOrDefault()
        {
            return GroupPoll.GetSettingsOrDefault(this);
        }

        public GroupBannedUser BanUser(int requestorID, int userID, string reason, string ipAddress, DateTime? deleteMessagesStartDate)
        {
            var member = GetMember(userID, true);
            if (member == null)
            {
                throw new DataNotFoundException();
            }

            if (requestorID == userID)
            {
                throw new DataValidationException("Can't ban yourself!");
            }

            var requestor = CheckPermission(GroupPermissions.BanUser, requestorID, m => CanModerateUser(m, member));

            var ip = ipAddress ?? string.Empty;

            var changed = false;
            var bannedUser = GroupBannedUser.Get(SourceConfiguration, GroupID, userID);
            if (bannedUser == null)
            {
                bannedUser = new GroupBannedUser
                {
                    GroupID = GroupID,
                    UserID = userID,
                    Username = member.GetTitleName(),
                    DateStatusChanged = DateTime.UtcNow,
                    IsDeleted = false,
                    Reason = reason,
                    RequestorID = requestorID,
                    RequestorUsername = requestor.GetTitleName(),
                    IPAddress = ip
                };
                bannedUser.Insert(SourceConfiguration);
                changed = true;
            }
            else if (bannedUser.IsDeleted)
            {
                bannedUser.IsDeleted = false;
                bannedUser.DateStatusChanged = DateTime.UtcNow;
                bannedUser.Reason = reason;
                bannedUser.RequestorID = requestorID;
                bannedUser.RequestorUsername = requestor.GetTitleName();
                bannedUser.IPAddress = ip;
                bannedUser.Update();
                changed = true;
            }
            else if (bannedUser.IPAddress != ip)
            {
                bannedUser.IPAddress = ip;
                bannedUser.Update(b => b.IPAddress);
                changed = true;
            }

            if (changed)
            {
                DoRemoveUsers(requestor, new HashSet<int> { userID }, GroupMemberRemovedReason.Banned, reason);
                GroupBannedUserIndexWorker.Create(new GroupBannedUserSearchModel(bannedUser));

                if (deleteMessagesStartDate.HasValue)
                {
                    ConversationMessageBulkWorker.CreateBulkDelete(GroupID.ToString(), requestor.UserID, requestor.GetTitleName(), DateTime.UtcNow, userID, deleteMessagesStartDate.Value);
                }
            }

            return bannedUser;
        }

        public void UnbanUser(int requestorID, int userID)
        {
            CheckPermission(GroupPermissions.BanUser, requestorID);

            var bannedUser = GroupBannedUser.GetLocal(GroupID, userID);
            if (bannedUser == null || bannedUser.IsDeleted)
            {
                return;
            }

            bannedUser.IsDeleted = true;
            bannedUser.DateStatusChanged = DateTime.UtcNow;
            bannedUser.Update(b => b.IsDeleted, b => b.DateStatusChanged);

            var member = GetMember(userID);
            if (member != null)
            {
                member.IsBanned = false;
                member.Update(u => u.IsBanned);
            }

            GroupBannedUserIndexWorker.Create(new GroupBannedUserSearchModel(bannedUser));
        }

        #region IConversationContainer

        bool IConversationContainer.CanAccess(int userID)
        {
            GroupMember member;
            return HasPermission(GroupPermissions.Access, userID, out member, true);
        }

        bool IConversationContainer.CanView(int userID, out DateTime latestDate, out DateTime earliestDate)
        {
            GroupMember member;

            if (HasPermission(GroupPermissions.ChatReadHistory, userID, out member, true))
            {

                latestDate = DateMessaged > member.DateMessaged ? DateMessaged : member.DateMessaged;
                earliestDate = DateCreated;
                return true;
            }

            if (HasPermission(GroupPermissions.Access, userID, out member, true))
            {
                latestDate = member.DateMessaged;
                earliestDate = member.DateJoined;
                if (earliestDate < DateCreated)
                {
                    earliestDate = DateCreated;
                }
                return true;
            }

            latestDate = DateTime.MaxValue;
            earliestDate = DateTime.MaxValue;
            return false;

        }

        bool IConversationContainer.CanEditAttachment(int userID, Attachment attachment)
        {
            return attachment.UploaderUserID == userID ||
                   HasPermission(GroupPermissions.ChatModerateMessages, userID, (member) => RootGroup.CanModerateUser(member.UserID, attachment.UploaderUserID));
        }

        bool IConversationContainer.CanEditMessage(int userID, ConversationMessage message)
        {
            // Ensures the message is part of this conversation
            if (message.RootConversationID != RootGroupID.ToString())
            {
                return false;
            }

            // If the requesting user has moderator privs let them moderate the message
            if (Type == GroupType.Large && HasPermission(GroupPermissions.ChatEditOtherMessages, userID, member => RootGroup.CanModerateUser(userID, message.SenderID)))
            {
                return true;
            }

            // If the requesting user is the message author, check the edit message grace period
            if (message.SenderID == userID)
            {
                return HasPermission(GroupPermissions.ChatSendMessages, userID) && message.Timestamp.FromEpochMilliconds() >= DateTime.UtcNow.AddMinutes(-ConversationConstants.EditMessageGracePeriodMinutes);
            }

            if (message.Mentions != null)
            {
                var mentions = new HashSet<int>(message.Mentions);

                if (mentions.Any() && !HasPermission(GroupPermissions.ChatMentionEveryone, userID))
                {
                    return false;
                }

                if (mentions.Contains(1) && !HasPermission(GroupPermissions.ChatMentionEveryone, userID))
                {
                    return false;
                }
            }

            return false;
        }

        bool IConversationContainer.CanDeleteMessage(int userID, ConversationMessage message)
        {
            // Ensures the message is part of this conversation
            if (message.RootConversationID != RootGroupID.ToString())
            {
                return false;
            }

            // If the requesting user has moderator privs let them moderate the message
            if (HasPermission(GroupPermissions.ChatModerateMessages, userID, member => RootGroup.CanModerateUser(userID, message.SenderID)))
            {
                return true;
            }

            // If the requesting user is the message author, check the edit message grace period
            if (message.SenderID == userID)
            {
                return HasPermission(GroupPermissions.ChatSendMessages, userID);
            }

            return false;
        }

        bool IConversationContainer.CanSendMessage(int userID)
        {
            return HasPermission(GroupPermissions.ChatSendMessages, userID);
        }

        bool IConversationContainer.CanMention(int userID, ConversationMessage message)
        {
            return HasPermission(GroupPermissions.ChatMentionUsers, userID);
        }

        bool IConversationContainer.CanMentionEveryone(int userID, ConversationMessage message)
        {
            return HasPermission(GroupPermissions.ChatMentionEveryone, userID);
        }

        void IConversationContainer.OnChatMessageChanged(ConversationMessage message, ConversationNotificationType changeType)
        {
            GroupMessageChangeCoordinator.Create(this, message.ToNotification(0, null, ConversationType.Group, changeType));
        }

        bool IConversationContainer.CanSearch(int userID, out DateTime? minSearchDate)
        {
            minSearchDate = null;
            GroupMember member;
            // If the requesting user has moderator privs let them moderate the message
            if (!HasPermission(GroupPermissions.Access, userID, out member, true))
            {
                return false;
            }

            if (!HasPermission(GroupPermissions.ChatReadHistory, userID))
            {
                minSearchDate = member.DateJoined;
            }

            return true;
        }

        bool IConversationContainer.CanLikeMessage(int userID, ConversationMessage message)
        {
            return !message.IsDeleted && HasPermission(GroupPermissions.Access, userID);
        }

        bool IConversationContainer.CanCall(int userID)
        {
            return Mode == GroupMode.TextAndVoice && HasPermission(GroupPermissions.Access, userID);
        }

        bool IConversationContainer.CanUnlockCall(int userID)
        {
            return Type != GroupType.Large && Mode == GroupMode.TextAndVoice && HasPermission(GroupPermissions.Access, userID);
        }

        ConversationType IConversationContainer.ConversationType
        {
            get
            {
                return ConversationType.Group;
            }
        }

        void IConversationContainer.OnChatMessageLike(int userID, string username, ConversationMessage message, Like like)
        {
            // Coordinates this to the group session to avoid race conditions.
            GroupMessageLikeCoordinator.Create(this, userID, message.ConversationID, message.ID, message.Timestamp);
        }

        IConversationParent IConversationContainer.GetConversationParent(int userID)
        {
            return GetMember(userID, true);
        }

        bool IConversationContainer.CanHide()
        {
            return false;
        }

        #endregion

        #region IAvatarParent

        string IAvatarParent.AvatarUrlSlug
        {
            get { return "groups"; }
        }

        string IAvatarParent.AvatarUrlID
        {
            get { return GroupID.ToString(); }
        }

        #endregion

        public void UpdateIsFeatured(bool isFeatured, DateTime date)
        {
            if (IsFeatured == isFeatured)
            {
                return;
            }

            IsFeatured = isFeatured;
            FeaturedTimestamp = date.ToEpochMilliseconds();
            Update(g => g.IsFeatured, g => g.FeaturedTimestamp);

            GroupSearchIndexWorker.CreateGroupFeaturedStatus(this, GetServerSearchSettingsOrDefault());
        }

        public void UpdateIsStreaming(ExternalCommunity stream)
        {
            if (stream.IsLive && !IsStreaming)
            {
                IsStreaming = true;
                StreamingTimestamp = stream.LiveTimestamp;
                Update(p => p.IsStreaming, p => p.StreamingTimestamp);
                GroupSearchIndexWorker.CreateGroupStreamingStatus(this, GetServerSearchSettingsOrDefault());
                ExternalCommunityLinkChangedCoordinator.Create(this, ExternalCommunityLinkChangeType.LiveStatus, stream);
            }
            else if (!stream.IsLive && IsStreaming)
            {
                var otherCommunities = GetMappedCommunities().Where(c => c.ExternalID != stream.ExternalID).ToArray();
                if (otherCommunities.Length == 0 || otherCommunities.All(c => !c.IsLive))
                {
                    IsStreaming = false;
                    StreamingTimestamp = stream.LiveTimestamp;
                    Update(p => p.IsStreaming, p => p.StreamingTimestamp);
                    GroupSearchIndexWorker.CreateGroupStreamingStatus(this, GetServerSearchSettingsOrDefault());
                    ExternalCommunityLinkChangedCoordinator.Create(this, ExternalCommunityLinkChangeType.LiveStatus, stream);
                }
            }
        }

        public Guid? GetCurrentCallGroup(int userId)
        {
            if (!IsRootGroup)
            {
                throw new DataValidationException("Must provide the root group.");
            }

            if (Type == GroupType.Normal)
            {
                var callList = GroupCallMemberList.GetLocal(GroupID);
                if (callList == null || callList.Members == null || !callList.Members.Contains(userId))
                {
                    return null;
                }
                return GroupID;
            }

            var callLists = GroupCallMemberList.MultiGetLocal(ChildGroupIDs.Select(id => new KeyInfo(id)));
            var currentCallGroup = callLists.FirstOrDefault(l => l.Members != null && l.Members.Contains(userId));
            return currentCallGroup == null ? (Guid?)null : currentCallGroup.GroupID;
        }

        public void SetInappropriateFlag(bool inappropriate)
        {
            if (!IsRootGroup || Type != GroupType.Large)
            {
                throw new DataValidationException("Specified group is not a server!");
            }

            FlaggedAsInappropriate = inappropriate;
            Update(f => f.FlaggedAsInappropriate);

            GroupSearchIndexWorker.CreateGroupQuarantineWorker(this, GetServerSearchSettingsOrDefault());
        }

        public void UpdateVanityUrl(string url)
        {
            if (Url == url)
            {
                throw new DataConflictException();
            }

            var vanity = VanityUrl.CreateForGroup(url, GroupID);

            Url = vanity.Url;
            Update(g => g.Url);

            var allChildren = GetAllChildren();
            var childrenByParent = allChildren.ToLookup(c => c.ParentGroupID);

            foreach (var child in allChildren)
            {
                try
                {
                    child.Url = GenerateAndCheckUrl(child.ParentGroup, childrenByParent[child.ParentGroupID], child.Title);
                    child.Update(c => c.Url);
                }
                catch (Exception ex)
                {
                    Logger.Warn(ex, "Failed to change a child URL", new { Root = this, Child = child });
                }
            }

            GroupChangeCoordinator.ChangeInfo(this, 0);
        }

        public string GetRegionKey()
        {
            return GetRegionKey(RegionID);
        }

        public IReadOnlyCollection<ExternalCommunity> GetMappedCommunities()
        {
            if(MappedCommunities == null)
            {
                var mappings = ExternalCommunityMapping.GetAllLocal(m => m.GroupID, GroupID).Where(m => !m.IsDeleted);
                return ExternalCommunity.MultiGetLocal(mappings.Select(m => new KeyInfo(m.ExternalID, m.Type)));
            }
            else
            {
                return ExternalCommunity.MultiGetLocal(MappedCommunities.Select(id => new KeyInfo(id, AccountType.Twitch)));
            }
        }

        #region External Guilds

        public void LinkExternalGuild(int requestorID, ExternalGuild guild, NewGuildSyncRole[] syncRoles)
        {
            if (!IsRootGroup)
            {
                throw new DataValidationException("Not a root group");
            }

            var accounts = ExternalAccountMapping.GetAllLocal(m => m.UserID, requestorID).Where(m => !m.IsDeleted && m.Type == guild.Type);

            var characters = accounts.SelectMany(a => ExternalGuildMember.GetAllLocal(g => g.AccountID, a.ExternalID));
            if (characters.All(c => c.GuildGameRegion != guild.GameRegion || c.GuildName != guild.Name || c.GuildGameServer != guild.GameServer))
            {
                throw new GroupPermissionException("Not the owner of the external account.");
            }

            var requestor = CheckPermission(GroupPermissions.ManageServer, requestorID);

            guild.MapToGroup(requestor, this);

            var guildIndex = guild.GetGuildIndex();
            var roles = GetRoles(true).Where(r => r.IsSynced && r.SyncID == guildIndex).ToArray();
            var syncRolesDictionary = syncRoles.ToDictionary(r => r.Tag);
            foreach (var role in ExternalGuildRole.GetAllLocal(r => r.GuildIndex, guildIndex).OrderBy(r => r.RoleTag))
            {
                var existing = roles.FirstOrDefault(r => r.Source == role.Type && r.Tag == role.RoleTag);
                if (existing == null)
                {
                    var newRole = syncRolesDictionary.GetValueOrDefault(role.RoleTag);
                    CreateGuildRole(this, role, newRole == null ? null : newRole.Name);
                }
                else if (existing.IsDeleted)
                {
                    RestoreRole(existing);
                }
            }

            GroupChangeCoordinator.ChangeInfo(this, requestorID);
        }

        public void UnlinkExternalGuild(int requestorID, ExternalGuild guild)
        {
            if (!IsRootGroup)
            {
                throw new DataValidationException("Not a root group");
            }

            var requestor = GetMember(requestorID, true);
            if (requestor.BestRole != OwnerRoleID)
            {
                throw new GroupPermissionException("Only the owner of the server can unlink a guild");
            }
            DoUnlinkExternalGuild(requestor, guild);
        }

        private void DoUnlinkExternalGuild(GroupMember requestor, ExternalGuild guild)
        {
            var mapping = guild.DeleteMapping(requestor, GroupID);

            if (mapping == null)
            {
                return;
            }

            GroupChangeCoordinator.ChangeInfo(this, requestor.UserID);
        }



        #endregion
    }
}
