﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Enums;
using Newtonsoft.Json;

namespace Curse.Friends.Data.Models
{
    public class GroupEvent : ILargeListElement
    {
        public LegacyGroupEventType Type { get; set; }

        public Guid EventID { get; set; }

        public Guid RootGroupID { get; set; }

        public DateTime Timestamp { get; set; }

        public int InitiatingUserID { get; set; }

        public string EventData { get; set; }

        public long GetKey()
        {
            return Timestamp.ToEpochMilliseconds();
        }

        public Dictionary<object, object> ToHash()
        {
            return new Dictionary<object, object>
            {
                {"type", Type},
                {"id", EventID},
                {"groupID", RootGroupID},
                {"timestamp", Timestamp},
                {"initiatingUserID", InitiatingUserID},
                {"eventData", EventData}
            };
        }

        public void Hydrate(Dictionary<object, object> hash)
        {
            Type = (LegacyGroupEventType) hash["type"];
            EventID = (Guid) hash["id"];
            RootGroupID = (Guid) hash["groupID"];
            Timestamp = (DateTime) hash["timestamp"];
            InitiatingUserID = Convert.ToInt32(hash["initiatingUserID"]);
            EventData = (string) hash["eventData"];
        }

        private static GroupEvent CreateEvent(Guid rootGroupID, int initiatingUserID, LegacyGroupEventType type, object eventData)
        {
            return new GroupEvent
            {
                EventID = Guid.NewGuid(),
                Timestamp = DateTime.UtcNow,
                RootGroupID = rootGroupID,
                InitiatingUserID = initiatingUserID,
                Type = type,
                EventData = JsonConvert.SerializeObject(eventData)
            };
        }

        public static GroupEvent CreateRootGroupCreatedEvent(Guid rootGroupID, int initiatingUserID)
        {
            return CreateEvent(rootGroupID, initiatingUserID, LegacyGroupEventType.RootGroupCreated, null);
        }

        public static GroupEvent CreateUsersAddedEvent(Guid rootGroupID, int initiatingUserID, int[] affectedUserIDs)
        {
            return CreateEvent(rootGroupID, initiatingUserID, LegacyGroupEventType.UsersAdded,
                new {AddedUsers = affectedUserIDs});
        }

        public static GroupEvent CreateUsersRemovedEvent(Guid rootGroupID, int initiatingUserID, int[] removedUserIDs)
        {
            return CreateEvent(rootGroupID, initiatingUserID, LegacyGroupEventType.UsersRemoved,
                new {RemovedUsers = removedUserIDs});
        }
                

        public static GroupEvent CreateUserRoleChanged(Guid rootGroupID, bool isAdding, int initiatingUserID, int affectedUserID, GroupRole role)
        {
            return CreateEvent(rootGroupID, initiatingUserID, isAdding ? LegacyGroupEventType.UserPromoted : LegacyGroupEventType.UserDemoted,
                new
                {
                    DemotedUser = affectedUserID,
                    RoleID = role.RoleID,
                    RoleName = role.Name
                });
        }

        public static GroupEvent CreateSubgroupCreatedEvent(Guid rootGroupID, int initiatingUserID, string title)
        {
            return CreateEvent(rootGroupID, initiatingUserID, LegacyGroupEventType.SubgroupCreated,
                new {GroupTitle = title});
        }

        public static GroupEvent CreateSubgroupRemovedEvent(Guid rootGroupID, int initiatingUserID, string title)
        {
            return CreateEvent(rootGroupID, initiatingUserID, LegacyGroupEventType.SubgroupRemoved,
                new {GroupTitle = title});
        }

        public static GroupEvent CreateChannelTitleChangeEvent(Guid rootGroupID, int initiatingUserID, string oldTitle,
            string newTitle)
        {
            return CreateEvent(rootGroupID, initiatingUserID, LegacyGroupEventType.GroupTitleChanged,
                new
                {
                    FormerTitle = oldTitle,
                    NewTitle = newTitle
                });
        }

        public static GroupEvent CreateGroupAvatarChangeEvent(Guid rootGroupID, int initiatingUserID,
            string affectedGroupTitle, string oldAvatarUrl, string newAvatarUrl)
        {
            return CreateEvent(rootGroupID, initiatingUserID, LegacyGroupEventType.GroupAvatarChanged,
                new
                {
                    GroupTitle = affectedGroupTitle,
                    FormerAvatarUrl = oldAvatarUrl,
                    NewAvatarUrl = newAvatarUrl
                });
        }

        public static GroupEvent CreateUserLeftEvent(Guid rootGroupID, int initiatingUserID)
        {
            return CreateEvent(rootGroupID, initiatingUserID, LegacyGroupEventType.UserLeft, null);
        }

        public static GroupEvent CreateUserInvitedEvent(Guid rootGroupID, int initiatingUserID, int invitedUser)
        {
            return CreateEvent(rootGroupID, initiatingUserID, LegacyGroupEventType.UserInvited,
                new {InvitedUser = invitedUser});
        }
    }
}
