﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "GroupGiveaway", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.HomeRegion)]
    public class GroupGiveaway : BaseTable<GroupGiveaway>, IHostable, IModelRegion
    {
        #region Metadata and Routing

        [Column("GroupID", KeyOrdinal = 1, IsIndexed = true)]
        public Guid GroupID { get; set; }

        [Column("GiveawayID", KeyOrdinal = 2)]
        public int GiveawayID { get; set; }

        [Column("MachineName", IsIndexed = true)]
        public string MachineName { get; set; }

        [Column("RegionID", IsIndexed = true)]
        public int RegionID { get; set; }

        #endregion

        #region Settings

        [Column("RequiredRoles")]
        public HashSet<int> RequiredRoles { get; set; }

        [Column("RoleBonuses")]
        public Dictionary<int, int> RoleBonuses { get; set; }

        [Column("ResponseWindow")]
        public TimeSpan ResponseWindow { get; set; }

        [Column("SharingBonus")]
        public int SharingBonus { get; set; }

        [Column("IgnoredUsers")]
        public HashSet<int> IgnoredUsers { get; set; }

        [Column("AutoEnterActiv")]
        public bool AutoEnterActiveUsers { get; set; }

        [Column("AutoEnterRoles")]
        public HashSet<int> AutoEnterRoles { get; set; }

        [Column("AutoClaimRoles")]
        public HashSet<int> AutoClaimRoles { get; set; }

        [Column("RollsPreWinner")]
        public int FakeRollsBeforeWinner { get; set; }

        [Column("AllowRepeatWin")]
        public bool AllowRepeatWinners { get; set; }

        [Column("IncludeOffline")]
        public bool IncludeOfflineMembers { get; set; }

        #endregion

        #region State

        [Column("CreatorID")]
        public int CreatorID { get; set; }

        [Column("Title")]
        public string Title { get; set; }

        [Column("Status")]
        public GroupGiveawayStatus Status { get; set; }

        [Column("StatusDate")]
        public DateTime DateStatusChanged { get; set; }

        [Column("DateStarted")]
        public DateTime DateStarted { get; set; }

        [Column("DateEnded")]
        public DateTime DateEnded { get; set; }

        [Column("Entries")]
        public int EligibleEntries { get; set; }

        [Column("TotalEntries")]
        public int TotalEntries { get; set; }


        [Column("CurrentRoll")]
        public int CurrentRoll { get; set; }

        [Column("FakeRollsLeft")]
        public int FakeRollsLeft { get; set; }

        [Column("TopShares")]
        public Dictionary<int, int> TopShares { get; set; }

        [Column("PendWinnerID")]
        public int PendingWinnerUserID { get; set; }

        #endregion

        public bool IsHostable { get { return true; } }

        public object[] KeyObjects { get { return new object[] {GroupID, GiveawayID}; } }
        
        public string DisplayName { get{return Title;} }

        public string GetLookupIndex()
        {
            return string.Format("{0}:{1}", GroupID, GiveawayID);
        }

        public void Continue(GroupMember requestor)
        {
            if (Status != GroupGiveawayStatus.Claimed)
            {
                throw new DataConflictException();
            }

            GroupGiveawayCoordinator.ContinueGiveaway(this, requestor);
        }

        public void End(GroupMember requestor)
        {
            if (Status == GroupGiveawayStatus.Inactive)
            {
                throw new DataConflictException();
            }

            GroupGiveawayCoordinator.EndGiveaway(this, requestor);
        }

        public void Deactivate(GroupMember requestor)
        {
            if (Status == GroupGiveawayStatus.Inactive)
            {
                throw new DataConflictException();
            }

            GroupGiveawayCoordinator.DeactivateGiveaway(this, requestor);
        }

        public bool AddParticipant(GroupMember requestor, int referrerID)
        {
            if (Status == GroupGiveawayStatus.Inactive || Status == GroupGiveawayStatus.Ended)
            {
                throw new DataConflictException();
            }

            if (IgnoredUsers.Contains(requestor.UserID))
            {
                throw new DataConflictException();
            }

            var participant = GroupGiveawayParticipant.Get(RegionID, requestor.UserID, GroupID, GiveawayID);

            if (participant == null)
            {
                participant = new GroupGiveawayParticipant
                {
                    GroupID = GroupID,
                    UserID = requestor.UserID,
                    Username = requestor.GetTitleName(),
                    DateEntered = DateTime.UtcNow,
                    Status = GroupGiveawayParticipantStatus.Entered,
                    DateStatusChanged = DateTime.UtcNow,
                    GiveawayID = GiveawayID,
                    GroupAndGiveawayIdIndex = GetLookupIndex(),
                    ReferrerUserID = referrerID
                };
                participant.Insert(RegionID);
            }
            else if (participant.Status == GroupGiveawayParticipantStatus.OptedOut)
            {
                participant.Status = GroupGiveawayParticipantStatus.Entered;
                participant.DateStatusChanged = DateTime.UtcNow;
                participant.Update(p => p.Status, p => p.DateStatusChanged);
            }
            else
            {
                throw new DataConflictException();
            }


            GroupGiveawayCoordinator.AddParticipant(this, requestor, participant);
            return RequiredRoles.Count == 0 || requestor.Roles.Overlaps(RequiredRoles);
        }

        public void RemoveParticipant(GroupMember requestor)
        {
            if (Status == GroupGiveawayStatus.Inactive || Status == GroupGiveawayStatus.Ended)
            {
                throw new DataConflictException();
            }

            var participant = GroupGiveawayParticipant.Get(RegionID, requestor.UserID, GroupID, GiveawayID);
            if (participant == null || participant.Status != GroupGiveawayParticipantStatus.Entered)
            {
                return;
            }

            participant.Status = GroupGiveawayParticipantStatus.OptedOut;
            participant.DateStatusChanged = DateTime.UtcNow;
            participant.Update(p => p.Status, p => p.DateStatusChanged);

            GroupGiveawayCoordinator.RemoveParticipant(this, requestor, participant);
        }

        public GroupGiveawayParticipant[] GetParticipants()
        {
            return GroupGiveawayParticipant.GetAllLocal(p => p.GroupAndGiveawayIdIndex, GetLookupIndex());
        }

        public GroupGiveawayParticipant GetParticipant(int userID)
        {
            return GroupGiveawayParticipant.GetLocal(userID, GroupID, GiveawayID);
        }

        public void Roll(GroupMember requestor)
        {
            if (Status != GroupGiveawayStatus.Active)
            {
                throw new DataConflictException();
            }

            // Get participants and their memberships
            var participants = GroupGiveawayParticipant.GetAll(RegionID, p => p.GroupAndGiveawayIdIndex, GetLookupIndex())
                .Where(p => p.Status == GroupGiveawayParticipantStatus.Entered).ToDictionary(p => p.UserID);

            if (participants.Count == 0)
            {
                throw new DataConflictException();
            }

            GroupGiveawayCoordinator.Roll(this, requestor);
        }

        public void Claim(GroupMember requestor)
        {
            if (PendingWinnerUserID != requestor.UserID)
            {
                throw new DataConflictException();
            }

            var participant = GroupGiveawayParticipant.Get(RegionID, requestor.UserID, GroupID, GiveawayID);
            if (participant.Status != GroupGiveawayParticipantStatus.Winner)
            {
                throw new DataConflictException();
            }

            participant.Status = GroupGiveawayParticipantStatus.Claimed;
            participant.DateStatusChanged = DateTime.UtcNow;
            participant.Update(p => p.Status, p => p.DateStatusChanged);

            GroupGiveawayCoordinator.Claim(this, requestor, participant);
        }

        public GroupGiveawayNotification ToNotification()
        {
            return new GroupGiveawayNotification
            {
                GroupID = GroupID,
                GiveawayID = GiveawayID,

                Title = Title,
                Status = Status,
                CreatorID = CreatorID,
                RequiredRoles = RequiredRoles,
                RoleBonuses = RoleBonuses,
                ResponseWindowSeconds = (int) ResponseWindow.TotalSeconds,
                SharingBonus = SharingBonus,
                AllowRepeatWinners = AllowRepeatWinners,
                AutoEnterActiveUsers = AutoEnterActiveUsers,
                RollsBeforeWinner = FakeRollsBeforeWinner,
                AutoClaimRoles = AutoClaimRoles,
                AutoEnterRoles = AutoEnterRoles,
                IgnoredUsers = IgnoredUsers,
                IncludeOfflineMembers = IncludeOfflineMembers,

                Entries = EligibleEntries,
                FakeRollsLeft = FakeRollsLeft,
                TopShares = TopShares,
                CurrentRollNumber = CurrentRoll
            };
        }
    }
}
