﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data.Utils;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Aerospike.Client;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "GroupInvitation", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class GroupInvitation : BaseTable<GroupInvitation>
    {
        /// <summary>
        /// This length should be 22 or 24 (22 for stripped padding, 24 if left) since it represents a base-64 GUID string.
        /// </summary>
        public const int InviteCodeLength = 22;

        public const int MaxInvitesPerCreator = 100;

        public const int DescriptionMaxLength = 128;

        /// <summary>
        /// The unique code representing this invitation
        /// </summary>
        [Column("InviteCode", KeyOrdinal = 1)]
        public string InviteCode { get; set; }

        /// <summary>
        /// A display version of a readable code
        /// </summary>
        [Column("DisplayCode")]
        public string DisplayCode { get; set; }

        /// <summary>
        /// The user ID of the person who created the invitation
        /// </summary>
        [Column("CreatorID")]
        public int CreatorID { get; set; }

        /// <summary>
        /// The username of the person who created the invitation
        /// </summary>
        [Column("CreatorName")]
        public string CreatorName { get; set; }

        /// <summary>
        /// The root group ID this invitation belongs to
        /// </summary>
        [Column("GroupID", IsIndexed = true)]
        public Guid GroupID { get; set; }

        /// <summary>
        /// The channel ID this invitation is for
        /// </summary>
        [Column("ChannelID")]
        public Guid ChannelID { get; set; }

        /// <summary>
        /// The date this invitation was created
        /// </summary>
        [Column("DateCreated")]
        public DateTime DateCreated { get; set; }

        /// <summary>
        /// The date this invitation expires
        /// </summary>
        [Column("DateExpires")]
        public DateTime DateExpires { get; set; }

        /// <summary>
        /// Whether or not to automatically remove members with this invite code, after expiration
        /// </summary>
        [Column("RemoveMembers")]
        public bool AutoRemoveMembers { get; set; }

        [Column("Status")]
        public GroupInvitationStatus Status { get; set; }

        /// <summary>
        /// The maximum amount of times this invitation can be used unless set to 0.
        /// </summary>
        [Column("MaxUses")]
        public int MaxUses { get; set; }

        /// <summary>
        /// The (display) number of times this invitation has been used (should never actually exceed MaxUses).
        /// </summary>
        [Column("TimesUsed")]
        public int TimesUsed { get; set; }

        /// <summary>
        /// The internally used counter to determine if an invitation can be used - will potentially exceed MaxUses, but logic should prevent claiming when it does.
        /// </summary>
        [Column("IntUseCounter")]
        public int InternalUseCounter { get; set; }

        /// <summary>
        /// The description of the group invitation, visible only to the creator and admins.
        /// </summary>
        [Column("Description")]
        public string Description { get; set; }

        /// <summary>
        /// The message to send users that are removed via purging guests.
        /// </summary>
        [Column("KickMessage")]
        public string KickMembersMessage { get; set; }

        [Column("ExpirationTS", IsIndexed = true)]
        public long ExpirationTimestamp { get; set; }

        public bool IsExpired
        {
            get { return DateExpires < DateTime.UtcNow; }
        }

        public static GroupInvitation GetByInviteCode(string code, bool returnInactive = false)
        {
            if (string.IsNullOrEmpty(code))
            {
                throw new ArgumentException("Invalid invite code: " + code);
            }

            // Get code as-is as well as lower invariant
            var keys = new[] { new KeyInfo(code), new KeyInfo(code.ToLowerInvariant()) };
            var invite = MultiGetLocal(keys).FirstOrDefault();
            if (invite == null)
            {
                return null;
            }

            if (invite.Status == GroupInvitationStatus.Active || (invite.Status != GroupInvitationStatus.Defunct && !invite.IsExpired))
            {
                return invite;
            }

            return returnInactive ? invite : null;
        }

        public static GroupInvitation[] GetAllByGroupID(Guid groupID)
        {
            return GetAllLocal(p => p.GroupID, groupID);
        }

        /// <summary>
        /// Creates a more URL friendly version of a GUID
        /// </summary>
        /// <returns></returns>
        public static string CreateID()
        {
            var enc = Convert.ToBase64String(Guid.NewGuid().ToByteArray());
            enc = enc.Replace("/", "a");
            enc = enc.Replace("+", "z");
            return enc.Substring(0, InviteCodeLength);
        }

        public GroupInvitationNotification ToNotification(string groupsRootUrl, Group group, Group channel, string inviteUrlFormatter, bool includeDescription = false)
        {
            return new GroupInvitationNotification
            {
                InviteCode = InviteCode,
                CreatorID = CreatorID,
                GroupID = GroupID,
                DateCreated = DateCreated.ToEpochMilliseconds(),
                CreatorName = CreatorName,
                Group = group.ToNotification(groupsRootUrl, false, false, false),
                Channel = channel.ToNotification(groupsRootUrl, false, false, false),
                DateExpires = DateExpires == DateTime.MaxValue ? null : (long?)DateExpires.ToEpochMilliseconds(),
                ChannelID = ChannelID,
                MaxUses = MaxUses > 0 ? (int?)MaxUses : null,
                TimesUsed = TimesUsed,
                IsRedeemable = Status == GroupInvitationStatus.Active && !IsExpired && (MaxUses == 0 || TimesUsed < MaxUses),
                InviteUrl = string.Format(inviteUrlFormatter, String.IsNullOrEmpty(DisplayCode) ? InviteCode : DisplayCode),
                AdminDescription = includeDescription ? Description : null                
            };
        }

        protected override void Validate()
        {
            if (!Description.SafeRange(0, DescriptionMaxLength))
            {
                throw new Exception("Description out of range: " + Description.SafeLength());
            }

            if (!CreatorName.SafeRange(0, User.NameMaxLength))
            {
                throw new Exception("Creator name length out of range.");
            }
        }

        public static GroupInvitation CreateReadable(Group group, GroupMember requestor, int maxUses, DateTime dateExpires, bool autoRemoveMembers, Guid channelID, string description, HashSet<string> words)
        {
            var combos = new HashSet<string>();
            var wordsArray = words.ToArray();
            for (var i = 0; i < 500; ++i)
            {
                var word1 = wordsArray[ThreadSafeRandom.Next(wordsArray.Length)];
                var word2 = wordsArray[ThreadSafeRandom.Next(wordsArray.Length)];
                var word3 = wordsArray[ThreadSafeRandom.Next(wordsArray.Length)];

                combos.Add(string.Join(string.Empty, word1, word2, word3));
            }

            var existingInvites = new HashSet<string>(MultiGetLocal(combos.Select(c => new KeyInfo(c.ToLowerInvariant()))).Select(i => i.DisplayCode));
            var available = combos.Where(c => !existingInvites.Contains(c));
            var codesToTry = new HashSet<string>(available.Take(20));
            if (codesToTry.Count == 0)
            {
                return null;
            }

            foreach (var codeToTry in codesToTry)
            {
                try
                {
                    var invite = new GroupInvitation
                    {
                        GroupID = group.GroupID,
                        InviteCode = codeToTry.ToLowerInvariant(),
                        DisplayCode = codeToTry,
                        DateCreated = DateTime.UtcNow,
                        CreatorID = requestor.UserID,
                        CreatorName = requestor.GetTitleName(),
                        AutoRemoveMembers = autoRemoveMembers,
                        DateExpires = dateExpires,
                        ChannelID = channelID,
                        MaxUses = maxUses,
                        Description = description,
                        Status = GroupInvitationStatus.Active,
                        InternalUseCounter = 0,
                        TimesUsed = 0,
                        ExpirationTimestamp = dateExpires.ToEpochMilliseconds(),
                    };
                    invite.Insert(1, UpdateMode.Concurrent);
                    return invite;
                }
                catch (AerospikeUniqueKeyViolation)
                {
                }
            }
            return null;
        }

        public static GroupInvitation Create(AerospikeConfiguration sourceConfiguration, Group group, GroupMember requestor, int maxUses, DateTime dateExpires, bool autoRemoveMembers, Guid channelID, string description)
        {
            var potentialIDs = new HashSet<string>();
            // if the link expires start with a shorter length. 
            var length = dateExpires == DateTime.MaxValue ? 6 : 4;
            for (int curLength = length; curLength < 8; curLength++)
            {                
                for (int i = 0; i < 2000; i++)
                {
                    var id = CreateID();
                    potentialIDs.Add(id.Substring(0, curLength));
                }

                // get existing invites that are in the generated codes to excluded from attempt to create a new invite. 
                var existingInvites = new HashSet<string>(MultiGetLocal(potentialIDs.Select(id => new KeyInfo(id.ToLower()))).Select(x => x.DisplayCode));
                // only codes that aren't used. 
                var available = potentialIDs.Where(x => !existingInvites.Contains(x));
                if(!available.Any())
                {
                    // none availabe, just go to a higher length. 
                    continue;
                }

                foreach (var code in available)
                {
                    try
                    {
                        var invite = new GroupInvitation
                        {
                            GroupID = group.GroupID,
                            InviteCode = code,
                            DisplayCode = string.Empty,
                            DateCreated = DateTime.UtcNow,
                            CreatorID = requestor.UserID,
                            CreatorName = requestor.GetTitleName(),
                            AutoRemoveMembers = autoRemoveMembers,
                            DateExpires = dateExpires,
                            ChannelID = channelID,
                            MaxUses = maxUses,
                            Description = description,
                            Status = GroupInvitationStatus.Active,
                            InternalUseCounter = 0,
                            TimesUsed = 0,
                            ExpirationTimestamp = dateExpires.ToEpochMilliseconds(),
                        };

                        invite.Insert(1, UpdateMode.Concurrent);
                        return invite;
                    }
                    catch (AerospikeException ex)
                    {
                        if (ex.Result != 5)
                        {
                            throw;
                        }
                    }
                }
                potentialIDs.Clear();
            }

            return null;                      
        }

        public static GroupInvitation GetReusableInvite(Group group, int requestorID, int maxUses, DateTime dateExpires, bool autoRemoveMembers, Guid channelID, string description)
        {
            var sourceList = group.GetGroupInvitations(requestorID, false, false, false);

            if (!sourceList.Any())
            {
                return null;
            }

            var fifteenMinutesAgo = DateTime.UtcNow.AddMinutes(-15);
           
            // find an active invite that matches base criteria, has open slots available, has a matching expiration (which is probably only items that never expire)
            var match = sourceList.FirstOrDefault(x => x.Status == GroupInvitationStatus.Active &&
                !x.IsExpired &&
                x.CreatorID == requestorID &&
                x.DateCreated >= fifteenMinutesAgo &&                
                (x.DateExpires == dateExpires || Math.Abs((x.DateExpires - dateExpires).TotalSeconds) <= 60) && // never expires or expiration is within a minute. 
                x.MaxUses == maxUses &&
                (x.MaxUses == 0 || x.TimesUsed == 0) && // unlimited use or links that haven't been used. 
                x.AutoRemoveMembers == autoRemoveMembers &&
                x.ChannelID == channelID &&
                x.Description == description);

            if (match == null)
            {
                // check defunct invite links first to be sure the purge for autodelete has happened. 
                match = sourceList.FirstOrDefault(x => x.Status == GroupInvitationStatus.Defunct);
            }

            return match;
        }

        public static int DeleteExpiredInvitations(DateTime expiredBefore)
        {
            // must provide a value and not be a non-expiring date. 
            if(expiredBefore == DateTime.MinValue || expiredBefore == DateTime.MaxValue)
            {
                return 0;
            }
            
            // only defunct and make sure we don't accidentally delete links before the new indexable timestamp field is built. 
            var invitations = GroupInvitation.GetAllInRangeLocal(x => x.ExpirationTimestamp, 0, expiredBefore.ToEpochMilliseconds())
                .Where(x => x.Status == Enums.GroupInvitationStatus.Defunct && x.DateExpires != DateTime.MaxValue && x.ExpirationTimestamp > 0);
            var deleteCount = 0;

            foreach (var invite in invitations)
            {
                try
                {
                    invite.UnassociateGroupMembers();
                    invite.DurableDelete();
                    deleteCount++;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Could not delete old invitation for re-use.", invite);
                }
            }

            return deleteCount;
        }

        public void UnassociateGroupMembers()
        {
            var invitedMembers = GroupMember.GetAllLocal(x => x.InviteCode, InviteCode);            
            foreach (var member in invitedMembers)
            {
                member.InviteCode = string.Empty;
                member.Update(x => x.InviteCode);
            }

        }

    }
}
