﻿using Curse.Aerospike;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Text.RegularExpressions;
using Curse.Extensions;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "GroupMember", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class GroupMember : BaseTable<GroupMember>, IConversationParent, IUserIdentity
    {
        /// <summary>
        /// A constant for determining if a member is considered 'active'
        /// </summary>
        public static readonly long ActivityMilliseconds = (long)TimeSpan.FromMinutes(15).TotalMilliseconds;
        public static readonly TimeSpan ActivityTimespan = TimeSpan.FromMinutes(15);

        public const int MaxJoinedServerCount = 1000;

        public GroupMember()
        {

        }

        public GroupMember(Group group, int userID, string username, string displayName, int regionID, string inviteCode = null, NotificationPreference preference = NotificationPreference.Enabled, GroupRole[] roles = null)
        {
            GroupID = group.GroupID;
            RootGroupID = group.RootGroupID;
            ParentGroupID = group.ParentGroupID;
            UserID = userID;            
            IsFavorite = false;
            DateJoined = DateTime.UtcNow;            
            NotificationPreference = preference;
            DateLastActive = DateTime.UtcNow;                
            
            if (group.IsRootGroup)
            {                
                RegionID = regionID;
                Username = username;
                DisplayName = displayName;
                InviteCode = inviteCode ?? string.Empty;
                UpdateRoleInfo(roles);
            }
        }


        private void UpdateRoleInfo(GroupRole[] roles)
        {
            Roles = new HashSet<int>(roles.Select(r => r.RoleID));
            var bestRole = roles.OrderBy(p => p.Rank).First();
            BestRole = bestRole.RoleID;
            BestRoleRank = bestRole.Rank;
        }

        public void Restore(Group group, GroupRole[] roles, string inviteCode = null)
        {
            DateJoined = DateTime.UtcNow;
            DateLastActive = DateTime.UtcNow;
            IsDeleted = false;
            UnreadCount = 0;

            if (group.IsRootGroup)
            {                
                InviteCode = inviteCode ?? string.Empty;
                UpdateRoleInfo(roles);
            }

            Update();
        }

        public void AddRole(GroupRole role)
        {
            if (GroupID != RootGroupID)
            {
                throw new InvalidOperationException("Cannot alter roles on the non-root group!");
            }

            Roles.Add(role.RoleID);
            if (role.Rank < BestRoleRank)
            {
                BestRole = role.RoleID;
                BestRoleRank = role.Rank;
            }
            Update(p => p.Roles, p => p.BestRole, p => p.BestRoleRank);
        }

        public void RemoveRole(GroupRole role, GroupRole defaultRole, GroupRole[] allRoles)
        {
            if (GroupID != RootGroupID)
            {
                throw new InvalidOperationException("Cannot alter roles on the non-root group!");
            }

            Roles.Remove(role.RoleID);
            var roles = Roles.Select(p => GroupRole.GetByGroupIDAndRoleID(SourceConfiguration, GroupID, p)).Where(p => p != null).ToArray();
            if (roles.Length == 0)
            {
                roles = new[] {defaultRole};
            }

            // If we're removing the user's current best role
            if (role.RoleID == BestRole)
            {
                if (roles.Any() && allRoles != null && allRoles.Any())
                {
                    var newBestRole = allRoles.Where(p => Roles.Contains(p.RoleID)).OrderBy(p => p.Rank).FirstOrDefault();
                    if (newBestRole != null)
                    {
                        BestRole = newBestRole.RoleID;
                        BestRoleRank = newBestRole.Rank;
                    }
                }                
            }

            if (role.RoleID == BestRole)
            {
                BestRole = defaultRole.RoleID;
                BestRoleRank = defaultRole.Rank;
            }

            UpdateRoleInfo(roles);
            Update(p => p.Roles, p => p.BestRole, p => p.BestRoleRank);
        }

        #region Metadata

        [Column("GroupID", KeyOrdinal = 1, IsIndexed = true)]
        public Guid GroupID { get; set; }

        [Column("UserID", KeyOrdinal = 2, IsIndexed = true)]
        public int UserID { get; set; }

        [Column("Username")]
        public string Username { get; set; }

        [Column("DisplayName")]
        public string DisplayName { get; set; }

        [Column("Nickname")]
        public string Nickname { get; set; }

        [Column("UsernameDate")]
        public DateTime DateNickname { get; set; }

        [Column("UsernameCount")]
        public int NicknameCounter { get; set; }

        private static readonly TimeSpan UsernameChangeFrequency = TimeSpan.FromHours(1);

        public bool UsernameThrottleElapsed()
        {
            return DateTime.UtcNow.Subtract(DateNickname) > UsernameChangeFrequency;
        }

        /// <summary>
        /// The region of the user
        /// </summary>
        [Column("RegionID")]
        public int RegionID { get; set; }

        /// <summary>
        /// The root group this membership belongs to
        /// </summary>
        [Column("RootGroupID", IsIndexed = true)]
        public Guid RootGroupID { get; set; }

        [JsonIgnore]
        public bool IsRootGroup
        {
            get { return GroupID == RootGroupID; }
        }

        /// <summary>
        /// The parent group this membership belongs to
        /// </summary>
        [Column("ParentGroupID")]
        public Guid ParentGroupID { get; set; }

        /// <summary>
        /// Whether or not this membership is still active
        /// </summary>
        [Column("IsDeleted")]
        public bool IsDeleted { get; set; }

        /// <summary>
        /// When this membership was removed/deleted.
        /// </summary>
        [Column("DateRemoved")]
        public DateTime DateRemoved { get; set; }

        /// <summary>
        /// Was this user Banned when removed?
        /// </summary>
        [Column("IsBanned")]
        public bool IsBanned { get; set; }

        #endregion

       
        #region Root Properties

    
        /// <summary>
        /// The highest ranked role this user has
        /// </summary>
        [Column("BestRole")]
        public int BestRole { get; set; }

        /// <summary>
        /// The rank of the highest role
        /// </summary>
        [Column("BestRoleRank")]
        public int BestRoleRank { get; set; }

        /// <summary>
        /// The roles this user has in the group
        /// </summary>
        [Column("Roles")]
        public HashSet<int> Roles { get; set; }

        /// <summary>
        /// The invite code that granted this user access
        /// </summary>
        [Column("InviteCode", IsIndexed = true)]
        public string InviteCode { get; set; }

        #endregion

        #region State Data

        /// <summary>
        /// This is used to determine from which point in a time a user should be able to read a group's chat history.
        /// </summary>
        [Column("DateJoined")]
        public DateTime DateJoined { get; set; }

        /// <summary>
        /// This is used to determine from which point in a time messages should be considered as unread
        /// </summary>
        [Column("DateRead")]
        public DateTime DateRead { get; set; }

        [Column("UnreadCount")]
        public int UnreadCount { get; set; }

        /// <summary>
        /// This stores the last time that a group message was sent
        /// </summary>
        [Column("DateMessaged")]
        public DateTime DateMessaged { get; set; }

        /// <summary>
        /// This stores the last time that a group member did anything with a group
        /// </summary>
        [Column("DateLastActive")]
        public DateTime DateLastActive { get; set; }

        /// <summary>
        /// Whether or not the user has been muted by a mod.
        /// </summary>
        [Column("IsVoiceMuted")]
        public bool IsVoiceMuted { get; set; }

        /// <summary>
        /// Whether or not the user has been deafened by a mod.
        /// </summary>
        [Column("IsVoiceDeaf")]
        public bool IsVoiceDeafened { get; set; }

        #endregion

        #region User Preferences

        /// <summary>
        /// The user can favorite a group, to appear more prominently in the client UI
        /// </summary>
        [Column("IsFavorite")]
        public bool IsFavorite { get; set; }

        [Column("NotifPref")]
        public NotificationPreference NotificationPreference { get; set; }

        [Column("NotifFilters")]
        public HashSet<string> NotificationFilters { get; set; }

        [Column("MutePushEnd")]
        public DateTime NotificationMuteDate { get; set; }
        
        protected override void Validate()
        {
            if (GroupID == Guid.Empty)
            {
                throw new DataValidationException("Unable to save group member. The GroupID is empty.");
            }

            if (UserID <= 0)
            {
                throw new DataValidationException("Unable to save group member. The UserID is invalid.");
            }
        }

        public void UpdateUserPreference(NotificationPreference preference, HashSet<string> filterSet)
        {
            NotificationPreference = preference;
            if (filterSet != null && filterSet.Any())
            {
                NotificationPreference = NotificationPreference.Filtered;
                NotificationFilters = filterSet;
            }
            Update(p => p.NotificationPreference, p => p.NotificationFilters);
        }

        #endregion

        public static readonly GroupMember CurseSystem = new GroupMember {UserID = 0, Username = "Twitch", Nickname = "Twitch" };

        public GroupMemberContract ToNotification(UserStatistics userStatistics, IReadOnlyCollection<ExternalAccount> externalAccounts = null)
        {
            var contract = new GroupMemberContract
            {                
                BestRole = BestRole,
                Roles = Roles.ToArray(),
                UserID = UserID,
                Nickname = Nickname,                 
                DateLastActive = DateLastActive.ToEpochMilliseconds(),
                DateJoined = DateJoined.ToEpochMilliseconds(),                           
                IsVoiceMuted = IsVoiceMuted,
                IsVoiceDeafened = IsVoiceDeafened,
                Username = Username,
                DateRemoved = DateRemoved.ToEpochMilliseconds(),
                IsBanned = IsBanned
            };
                        
            if (userStatistics != null)
            {
                contract.ConnectionStatus = userStatistics.ConnectionStatus;
                contract.DateLastSeen = userStatistics.DateLastSeen;
                contract.CurrentGameID = userStatistics.CurrentGameID;
                contract.IsActive = DetermineActive(DateLastActive, contract.ConnectionStatus);
                contract.AvatarTimestamp = userStatistics.AvatarTimestamp;
                contract.DisplayName = userStatistics.DisplayName;
            }
            else
            {
                contract.IsActive = DetermineActive(DateLastActive);
            }

            if (externalAccounts != null)
            {
                contract.ExternalAccounts = externalAccounts.Select(p => p.ToPublicContract()).ToArray();
                contract.IsVerified = externalAccounts.Any(p => p.IsPartnered &&
                    (p.DisplayName.Replace("_", string.Empty).Equals(Username.Replace("_", string.Empty), StringComparison.OrdinalIgnoreCase)
                    || p.DisplayName.Replace("_", string.Empty).Equals(Nickname.ToEmptyWhenNull().Replace("_", string.Empty), StringComparison.OrdinalIgnoreCase)));
            }
          
            
            return contract;
        }

        public GroupPresenceContract ToPresenceContract(UserStatistics userStatistics)
        {
            if (userStatistics == null)
            {
                return null;
            }

            var contract = new GroupPresenceContract
            {
                UserID = UserID,
                ConnectionStatus = userStatistics.ConnectionStatus,
                GameID = userStatistics.CurrentGameID,
                DateLastSeen = userStatistics.DateLastSeen,
                IsActive = DetermineActive(DateLastActive, userStatistics.ConnectionStatus),
            };

            return contract;
        }

        public static bool DetermineActive(DateTime dateLastActive, UserConnectionStatus? status = null)
        {
            if (status.HasValue && status.Value != UserConnectionStatus.Offline)
            {
                return true;
            }

            return DateTime.UtcNow.Subtract(dateLastActive) < ActivityTimespan;
        }
        

        public GroupMembershipNotification ToMembershipNotification(Group group)
        {
            var bestRole = BestRole;
            var roles = Roles;

            if ((Roles == null || Roles.Count == 0) && group != null && group.IsRootGroup)
            {
                bestRole = group.DefaultRoleID;
                roles = new HashSet<int> { group.DefaultRoleID };
            }

            return new GroupMembershipNotification
            {
                Nickname = this.GetTitleName(),
                CanChangeNickname = CanChangeNickname(),
                BestRole = bestRole,
                Roles = roles,
                IsFavorite = IsFavorite,
                DateJoined = DateJoined,
                DateMessaged = DateMessaged,
                DateRemoved = DateRemoved,
                DateRead = DateRead,
                NotificationFilters = NotificationFilters,
                NotificationMuteDate = NotificationMuteDate,
                NotificationPreference = NotificationPreference,
                UnreadCount = UnreadCount,
                IsVoiceMuted = IsVoiceMuted,
                IsVoiceDeafened = IsVoiceDeafened,
                IsBanned = IsBanned
            };
        }

        public bool CanChangeNickname()
        {
            return UsernameThrottleElapsed() || NicknameCounter < 2;            
        }

        public ChannelMembershipContract ToChannelMembershipContract(Group group)
        {  
            return new ChannelMembershipContract
            {                
                IsFavorite = IsFavorite,                
                DateMessaged = DateMessaged,
                DateRead = DateRead,
                NotificationFilters = NotificationFilters,
                NotificationMuteDate = NotificationMuteDate,
                NotificationPreference = NotificationPreference,
                UnreadCount = UnreadCount
            };
        }

        public static GroupMember[] GetAllByUserID(int userID, bool returnDeleted = false)
        {
            var results = GetAllLocal(p => p.UserID, userID);
            if (!returnDeleted)
            {
                results = results.Where(p => !p.IsDeleted).ToArray();
            }

            return results;
        }

        private static readonly LogCategory ThrottledLogger = new LogCategory("GroupMember") { Throttle = TimeSpan.FromSeconds(30) };

        public static Dictionary<Guid, GroupMember> GetDictionaryByUserID(int userID, bool returnDeleted = false, int limit = 0)
        {
            var results = new Dictionary<Guid, GroupMember>();
            var allMemberships = GetAllLocal(p => p.UserID, userID).Where(m=>!m.IsDeleted || returnDeleted).ToArray();

            if (limit > 0 && allMemberships.Length > limit)
            {
                ThrottledLogger.Warn("Attempt to retrieve more than the max number of group memberships. The client will be returned the top root groups.", new { userID, groupMembershipCount = allMemberships.Length, limit });
                allMemberships = allMemberships.Where(p => p.IsRootGroup).Take(limit).ToArray();
            }

            foreach (var membership in allMemberships)
            {
                results[membership.GroupID] = membership;
            }

            return results;
        }

        public void MarkAsRead(DateTime timestamp)
        {

            if (timestamp < DateRead && UnreadCount == 0)
            {
                return;
            }

            // Let the group session know that we've marked this as read
            var group = Group.GetByID(GroupID);

            if (group == null)
            {
                Logger.Warn("Unable to coordinate group read state. The group could not be retrieved!", GroupID);
                return;
            }

            GroupMessageReadCoordinator.Create(group, UserID, timestamp);
        }

        public void MarkAsUnread(DateTime timestamp, int numberOfMessages)
        {
            throw new InvalidOperationException("You must use the static method.");
        }

        public static void MarkAsUnread(Guid groupID, int userID, DateTime timestamp, int numberOfMessages)
        {
            IncrementCounterAndSetValue(LocalConfiguration, UpdateMode.Fast,  p => p.UnreadCount, numberOfMessages, p => p.DateMessaged, timestamp, groupID, userID);
        }

        public static void MarkAsRead(Guid groupID, int userID, DateTime? dateMessaged, DateTime dateRead)
        {
            if (dateMessaged.HasValue)
            {
                ResetCounterAndSetValues(LocalConfiguration, UpdateMode.Fast, new KeyInfo(groupID, userID), p => p.UnreadCount,
                new Tuple<Expression<Func<GroupMember, object>>, object>(p => p.DateMessaged, dateMessaged),
                new Tuple<Expression<Func<GroupMember, object>>, object>(p => p.DateRead, dateRead));
            }
            else
            {
                ResetCounterAndSetValues(LocalConfiguration, UpdateMode.Fast, new KeyInfo(groupID, userID), p => p.UnreadCount,                    
                    new Tuple<Expression<Func<GroupMember, object>>, object>(p => p.DateRead, dateRead));    
            }            
        }

        public void SetDeleted(bool isBan = false)
        {
            IsDeleted = true;
            DateRemoved = DateTime.UtcNow;
            if (isBan)
            {
                IsBanned = true;
            }

            Update(m => m.IsDeleted, m => m.DateRemoved, m => m.IsBanned);
        }

        string IConversationParent.GetSenderName(int senderID, string fallback)
        {
            // If this user has a friendship, use the nickname
            var friendship = Friendship.GetLocal(UserID, senderID);
            if (friendship != null && friendship.Status == FriendshipStatus.Confirmed)
            {
                return friendship.FormattedDisplayName;
            }

            return fallback;
        }

        bool IConversationParent.ShouldSendPushNotification(User user, string messageBody, HashSet<int> mentionedUserIDs)
        {
            // If the user is mentioned, and they have mention push enabled globally, notify them!
            if (user.MentionsPushEnabled == true && (mentionedUserIDs.Contains(UserID) || mentionedUserIDs.Contains(0)))
            {
                return true;
            }
            
            // If they have muted this convo, do not notify them!
            if (NotificationPreference == NotificationPreference.Disabled)
            {
                return false;
            }


            var globalPreference = user.GroupMessagePushPreference;

            // If this is a user's favorite group, and they have favorite push enabled, notify them!
            if (globalPreference == PushNotificationPreference.Favorites && IsFavorite)
            {
                return true;
            }

            // If a filtered word is included in the message body, notify them
            if (NotificationPreference == NotificationPreference.Filtered && NotificationFilters.Any(messageBody.Contains))
            {
                return true;
            }

            // Check the root membership to inherit preferences and filters too prevent filtered/muted notifications
            if (!IsRootGroup)
            {
                var rootConversation = GetLocal(RootGroupID, UserID) as IConversationParent;
                if (rootConversation != null && !rootConversation.ShouldSendPushNotification(user, messageBody, mentionedUserIDs))
                {
                    return false;
                }
            }

            // If the user has push enabled, notify them
            return globalPreference == PushNotificationPreference.All;
        }

        string IConversationParent.Title
        {
            get
            {
                var group = Group.GetLocal(GroupID);
                if (group == null)
                {
                    Logger.Warn("Failed to get title for group member.");
                    return "";
                }

                return group.Title;                    
            }
        }        

        void IConversationParent.ToggleHidden(bool isHidden)
        {
            throw new NotImplementedException("Group chats cannot currently be hidden.");
        }

        void IConversationParent.ToggleMuted(bool isMuted)
        {
            NotificationPreference = isMuted ? NotificationPreference.Disabled : NotificationPreference.Enabled;       
            Update(p => p.NotificationPreference);
        }

        static readonly Regex ValidAlphaNumUnderscore = new Regex("^[a-zA-Z0-9_]+$");

        public static bool IsValidUserName(string name)
        {
            if (name.Length < 2 || name.Length > 32)
            {
                return false;
            }

            return ValidAlphaNumUnderscore.IsMatch(name);
        }

        public static bool HasExceededJoinedServerCount(int userID)
        {
            return GroupMember.GetAllByUserID(userID).Count(g=>g.IsRootGroup) >= MaxJoinedServerCount;
        }
    }
}
