﻿using System;
using System.Collections.Generic;
using System.Linq;
using Aerospike.Client;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;

namespace Curse.Friends.Data
{


    [TableDefinition(TableName = "GroupPoll", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.HomeRegion)]
    public class GroupPoll : BaseTable<GroupPoll>, IHostable, IModelRegion
    {
        public static readonly int MaxActivePolls = 2;
        public static readonly int MaxPollTitleLength = 128;

        [Column("GroupID", KeyOrdinal = 1, IsIndexed = true)]
        public Guid GroupID { get; set; }

        [Column("PollID", KeyOrdinal = 2)]
        public int PollID { get; set; }

        [Column("Title")]
        public string Title { get; set; }

        [Column("CreatorID")]
        public int CreatorUserID { get; set; }

        [Column("Status")]
        public GroupPollStatus Status { get; set; }

        [Column("StatusDate")]
        public DateTime StatusChangeDate { get; set; }

        [Column("StartDate")]
        public DateTime StartDate { get; set; }

        [Column("EndDate")]
        public DateTime EndDate { get; set; }

        [Column("DurationMins")]
        public int DurationMinutes { get; set; }

        [Column("MultiSelect")]
        public bool AllowMultipleSelections { get; set; }

        [Column("DisplayType")]
        public GroupPollDisplayType DisplayType { get; set; }

        [Column("RequiredRoles")]
        public HashSet<int> RequiredRoles { get; set; }

        [Column("MachineName", IsIndexed = true)]
        public string MachineName { get; set; }

        [Column("RegionID", IsIndexed = true)]
        public int RegionID { get; set; }

        [Column("Code", IsIndexed = true)]
        public string Code { get; set; }

        [Column("AllowRevotes")]
        public bool AllowRevotes { get; set; }

        [Column("DuplicateMode")]
        public GroupPollDuplicateMode DuplicateMode { get; set; }

        [Column("OptionIDs")]
        public HashSet<int> OptionIDs { get; set; }

        #region Legacy

        [Column("Options")]
        public Dictionary<int, string> Options { get; set; }

        [Column("Votes")]
        public Dictionary<int, int> Votes { get; set; }

        #endregion

        public object[] KeyObjects { get { return new object[] { GroupID, PollID }; } }

        public string DisplayName { get { return Title; } }

        public bool IsHostable { get { return true; } }

        public string GetLookupIndex()
        {
            return GroupID.ToString() + PollID;
        }

        public static GroupPollSettings GetSettingsOrDefault(Group group)
        {
            // Get the settings or return defaults
            var settings = GroupPollSettings.GetLocal(group.GroupID) ??
                           new GroupPollSettings
                           {
                               GroupID = group.GroupID,
                               RegionID = group.RegionID,

                               RequiredRoles = new HashSet<int>(),
                               AllowMultipleSelections = false,
                               DisplayType = GroupPollDisplayType.BarGraph,
                               IsPublic = false,
                               AllowRevotes = true
                           };
            return settings;
        }

        public void End(int requestorID)
        {
            var group = Group.Get(RegionID, GroupID);
            if (group == null || group.IsDeleted)
            {
                throw new DataNotFoundException();
            }

            if (Status != GroupPollStatus.Running && Status != GroupPollStatus.Starting)
            {
                throw new DataConflictException();
            }

            var requestor = group.CheckPermission(GroupPermissions.ManagePolls, requestorID);

            GroupPollCoordinator.Create(requestor, this, GroupPollChangeType.Ended);
        }

        public void Deactivate(int requestorID)
        {
            var group = Group.Get(RegionID, GroupID);
            if (group == null || group.IsDeleted)
            {
                throw new DataNotFoundException();
            }

            if (Status == GroupPollStatus.Inactive)
            {
                throw new DataConflictException();
            }

            var requestor = group.CheckPermission(GroupPermissions.ManagePolls, requestorID);

            GroupPollCoordinator.Create(requestor, this, GroupPollChangeType.Deactivated);
        }

        public void Vote(string identifier, HashSet<int> vote)
        {
            var group = Group.Get(RegionID, GroupID);
            if (group == null || group.IsDeleted)
            {
                throw new DataNotFoundException();
            }

            if (Status != GroupPollStatus.Running)
            {
                throw new DataConflictException();
            }

            if (!AllowMultipleSelections && vote.Count > 1)
            {
                throw new DataConflictException();
            }

            var changes = vote.ToDictionary(v => v, v => 1);
            if (DuplicateMode != GroupPollDuplicateMode.AllowDuplicates)
            {
                var participant = GroupPollParticipant.Get(RegionID, identifier, GroupID, PollID);
                if (participant == null)
                {
                    participant = new GroupPollParticipant
                    {
                        ParticipantID = identifier,
                        GroupID = GroupID,
                        PollID = PollID,
                        DateVoted = DateTime.UtcNow,
                        Vote = vote,
                        LookupIndex = GetLookupIndex()
                    };
                    participant.Insert(RegionID);
                }
                else if (!AllowRevotes)
                {
                    throw new DataConflictException("Not allowed to change your vote on this poll");
                }
                else if (!participant.Vote.SetEquals(vote))
                {
                    foreach (var v in participant.Vote)
                    {
                        int change;
                        if (!changes.TryGetValue(v, out change))
                        {
                            change = 0;
                        }

                        changes[v] = change - 1;
                    }

                    if (participant.ParticipantID == null)
                    {
                        // Legacy to ensure the key is populated
                        participant.ParticipantID = participant.UserID.ToString();
                    }
                    participant.Vote = vote;
                    participant.DateVoted = DateTime.UtcNow;
                    participant.Update(p => p.Vote, p => p.DateVoted);
                }
                else
                {
                    // No change
                    return;
                }
            }

            foreach (var option in GroupPollOption.MultiGet(RegionID, changes.Keys.Select(id => new KeyInfo(GroupID, PollID, id))))
            {
                int diff;
                if (changes.TryGetValue(option.OptionID, out diff) && diff!=0)
                {
                    // Only hit the DB on a net change
                    option.IncrementCounter(RegionID, UpdateMode.Default,  p => p.VoteCount, diff);
                }
            }

            if (string.IsNullOrEmpty(MachineName))
            {
                GroupPollCoordinator.VoteChanged(this);
            }
        }

        public GroupPollParticipant GetParticipant(string requestorID)
        {
            return GroupPollParticipant.GetLocal(requestorID, GroupID, PollID);
        }

        public static GroupPoll GetByPollCode(string pollCode)
        {
            return GetAllLocal(p => p.Code, pollCode).FirstOrDefault();
        }

        public static string CreatePollCode()
        {
            var attempt = 0;
            do
            {
                var enc = Convert.ToBase64String(Guid.NewGuid().ToByteArray());
                enc = enc.Replace("/", "a");
                enc = enc.Replace("+", "z");
                var code = enc.Substring(0, 22);

                if (GetByPollCode(code) == null)
                {
                    return code;
                }
            } while (attempt++ < 10);

            Logger.Warn(string.Format("Could not generate a unique poll code after {0} attempts, treating as private", attempt));
            return string.Empty;
        }


        public void UpdateSettings()
        {
            var settings = GroupPollSettings.GetLocal(GroupID);
            if (settings == null)
            {
                settings = CreateSettings(GroupID, RegionID, AllowMultipleSelections, AllowRevotes, RequiredRoles, Code != null, DisplayType);
                settings.InsertLocal();
            }
            else
            {
                settings.AllowRevotes = AllowRevotes;
                settings.AllowMultipleSelections = AllowMultipleSelections;
                settings.DisplayType = DisplayType;
                settings.IsPublic = !string.IsNullOrEmpty(Code);
                settings.RequiredRoles = RequiredRoles;
                settings.Update();
            }
        }

        public static GroupPollSettings CreateSettings(Guid groupID, int regionID, bool allowMultipleSelections, bool allowRevotes, HashSet<int> requiredRoles, bool isPublic,
            GroupPollDisplayType displayType)
        {
            var settings = new GroupPollSettings
            {
                GroupID = groupID,
                RegionID = regionID,
                AllowMultipleSelections = allowMultipleSelections,
                DisplayType = displayType,
                RequiredRoles = requiredRoles,
                IsPublic = isPublic,
                AllowRevotes = allowRevotes
            };
            return settings;
        }

        public GroupPollNotification ToNotification(string publicPollUrlFormat, bool calculateVotes=false)
        {
            Dictionary<int, string> options;
            Dictionary<int, int> votes;

            if (OptionIDs != null)
            {
                var optionModels = GetOptions();
                options = optionModels.ToDictionary(o => o.OptionID, o => o.OptionText);
                votes = optionModels.ToDictionary(o => o.OptionID, o => o.VoteCount);
            }
            else
            {
                // Legacy
                options = Options;
                votes = calculateVotes ? CalculateVotesLegacy() : Votes;
            }

            var notification = new GroupPollNotification
            {
                GroupID = GroupID,
                PollID = PollID,
                Title = Title,
                Options = options,
                Votes = votes,
                StartDate = StartDate.ToEpochMilliseconds(),
                RequiredRoles = RequiredRoles,
                AllowMultipleSelections = AllowMultipleSelections,
                DisplayType = DisplayType,
                Status = Status,
                AllowRevotes = AllowRevotes,
                DuplicationMode = DuplicateMode,

                MillisecondsLeft = DurationMinutes == 0 ? null : (long?) Math.Max(0, (EndDate - DateTime.UtcNow).TotalMilliseconds)
            };

            if (!string.IsNullOrEmpty(Code))
            {
                notification.PublicCode = Code;
                notification.PublicUrl = string.Format(publicPollUrlFormat, Code);
            }

            return notification;
        }

        public void CreateOptions(Dictionary<int, string> options)
        {
            foreach (var option in options)
            {
                var pollOption = new GroupPollOption
                {
                    GroupID = GroupID,
                    PollID = PollID,
                    OptionID = option.Key,
                    RegionID = RegionID,
                    VoteCount = 0,
                    OptionText = option.Value,
                };
                pollOption.Insert(RegionID);
            }
        }

        public GroupPollOption[] GetOptions(bool allowDirtyRead = true)
        {
            return GroupPollOption.MultiGet(allowDirtyRead ? LocalConfigID : RegionID, OptionIDs.Select(id => new KeyInfo(GroupID, PollID, id))).ToArray();
        }

        public Dictionary<int,int> CalculateVotesLegacy()
        {
            var votes = Options.ToDictionary(o => o.Key, o => 0);
            var participants = GroupPollParticipant.GetAllLocal(p => p.LookupIndex, GetLookupIndex());
            foreach (var participant in participants)
            {
                foreach (var vote in participant.Vote)
                {
                    votes[vote]++;
                }
            }
            return votes;
        }
    }
}
