﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;

namespace Curse.Friends.Data.Models
{
    [TableDefinition(TableName = "GroupPrivateConversation", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class GroupPrivateConversation : BaseTable<GroupPrivateConversation>, IConversationParent, IConversationContainer
    {
        [Column("UserID", KeyOrdinal = 1)]
        public int UserID
        {
            get;
            set;
        }

        /// <summary>
        /// UserID of the friend
        /// </summary>
        [Column("OtherUserID", KeyOrdinal = 2)]
        public int OtherUserID
        {
            get;
            set;
        }

        /// <summary>
        /// The group ID of the community this private message belongs to
        /// </summary>
        [Column("GroupID", KeyOrdinal = 3)]
        public Guid GroupID
        {
            get;
            set;
        }

        private Group _group;

        public Group Group
        {
            get
            {
                if (_group == null)
                {
                    _group = Group.GetLocal(GroupID);
                }
                return _group;
            }
        }

        /// <summary>
        /// The group ID of the community this private message belongs to
        /// </summary>
        [Column("GroupUserIdx", IsIndexed = true)]
        public string GroupAndUserIndex
        {
            get;
            set;
        }

        /// <summary>
        /// The username of the other party
        /// </summary>
        [Column("Username")]
        public string OtherUsername
        {
            get;
            set;
        }
        
        [Column("DateCreated")]
        public DateTime DateCreated
        {
            get;
            set;
        }

        [Column("DateMessaged")]
        public DateTime DateMessaged
        {
            get;
            set;
        }

        [Column("UnreadCount")]
        public int UnreadCount
        {
            get;
            set;
        }

        [Column("DateRead")]
        public DateTime DateRead
        {
            get;
            set;
        }

        [Column("IsBlocked")]
        public bool IsBlockedByThem
        {
            get;
            set;
        }

        [Column("OtherIsBlocked")]
        public bool IsBlockedByMe
        {
            get; 
            set;
        }

        [Column("IsHidden")]
        public bool IsHidden
        {
            get;
            set;
        }
      
        public static string GenerateGroupAndUserIndex(Guid groupID, int userID)
        {
            return groupID + ":" + userID;
        }

        protected override void Validate()
        {
            if (string.IsNullOrEmpty(OtherUsername))
            {
                throw new Exception("Username cannot be null or empty");
            }

            if (string.IsNullOrEmpty(GroupAndUserIndex))
            {
                throw new Exception("Username cannot be null or empty");
            }

            if (UserID < 1)
            {
                throw new Exception("UserID must be 1 or higher");
            }

            if (OtherUserID < 1)
            {
                throw new Exception("OtherUserID must be 1 or higher");
            }

            if (GroupID == Guid.Empty)
            {
                throw new Exception("GroupID must not be empty");
            }
        }

        public string ConversationID
        {
            get { return GenerateConversationID(GroupID, UserID, OtherUserID); }
        }

        public static string GenerateConversationID(Guid groupID, int userID, int otherUserID)
        {
            return string.Format("{0}:{1}:{2}", groupID, Math.Min(userID, otherUserID), Math.Max(userID, otherUserID));
        }
        
        public static GroupPrivateConversation GetOrCreateBySenderIDAndRecipientIDAndGroupID(int senderID, int recipientID, Guid groupID)
        {
            var conversation = GetLocal(senderID, recipientID, groupID);
            if (conversation == null)
            {

                conversation = new GroupPrivateConversation
                {
                    GroupID = groupID,
                    UserID = senderID,
                    OtherUserID = recipientID
                };
            }

            return conversation;
        }

        public static GroupPrivateConversation GetBySenderIDAndRecipientIDAndGroupID(int senderID, int recipientID, Guid groupID)
        {
            return GetLocal(senderID, recipientID, groupID);            
        }

        public bool IsProvisional
        {
            get { return DateCreated == default(DateTime); }
        }

        public void MakePermanent(DateTime timestamp)
        {
            DateCreated = timestamp;
            GroupAndUserIndex = GenerateGroupAndUserIndex(GroupID, UserID);
            var otherUserRegion = UserRegion.GetByUserID(OtherUserID);
            var otherUser = otherUserRegion.GetUser();
            OtherUsername = otherUser.Username;
            InsertLocal();
        }
        
        public static GroupPrivateConversation Create(Guid groupID, int senderID, int recipientID)
        {
            var userAndRegion = UserRegion.GetByUserID(recipientID);

            if (userAndRegion == null)
            {
                throw new InvalidOperationException("Unable to get user region for user: " + recipientID);
            }

            var user = userAndRegion.GetUser();

            if (user == null)
            {
                throw new InvalidOperationException("Unable to get user: " + recipientID);
            }

             // Create a record for both sides of the convo
            var senderConvo = new GroupPrivateConversation
            {
                UserID = senderID,
                GroupID = groupID,
                OtherUserID = recipientID, 
                DateCreated = DateTime.UtcNow,
                GroupAndUserIndex = GenerateGroupAndUserIndex(groupID, senderID),
                OtherUsername = user.Username, 
                UnreadCount = 0                      
            };

            senderConvo.InsertLocal();

            return senderConvo;
        }

        public GroupPrivateConversationContract ToContract(string displayName = null)
        {
            return new GroupPrivateConversationContract
            {
                UserID = UserID,
                OtherUserID = OtherUserID,
                DateCreated = DateCreated,
                DateMessaged = DateMessaged,
                OtherUsername = OtherUsername,
                OtherUserDisplayName = displayName ?? OtherUsername,
                DateRead = DateRead,
                UnreadCount = UnreadCount,
                IsBlockedByThem = IsBlockedByThem,
                IsBlockedByMe = IsBlockedByMe,
            };
        }

        IConversationParent IConversationContainer.GetConversationParent(int userID)
        {
            return this;
        }

        bool IConversationParent.ShouldSendPushNotification(User user, string messageBody, HashSet<int> mentionedUserIDs)
        {
            return false;
        }

        string IConversationParent.GetSenderName(int senderID, string fallback)
        {
            return OtherUsername;
        }

        void IConversationParent.ToggleHidden(bool isHidden)
        {
            IsHidden = isHidden;
            Update(p => p.IsHidden);
        }

        void IConversationParent.ToggleMuted(bool isMuted)
        {
            throw new NotImplementedException("Ad hoc chats cannot currently be hidden.");
        }

        public void MarkAsRead(DateTime timestamp)
        {

            if (timestamp < DateRead)
            {
                return;
            }

            ResetCounterAndSetValue(SourceConfiguration, UpdateMode.Fast, p => p.UnreadCount, p => p.DateRead, timestamp);
            DateRead = timestamp;
            UnreadCount = 0;
        }

        public void MarkAsUnread(DateTime timestamp, int numberOfMessages)
        {
            if (timestamp <= DateMessaged)
            {
                return;
            }

            if (numberOfMessages > 0)
            {
                UnreadCount = IncrementCounterAndSetValue(SourceConfiguration, UpdateMode.Fast,  p => p.UnreadCount, numberOfMessages, p => p.DateMessaged, timestamp);
                DateMessaged = timestamp;
            }
            else
            {
                DateMessaged = timestamp;
                Update(p => p.DateMessaged);
            }
        }

        public static void HideAll(Guid groupID, int userID)
        {
            var convos = GetAllByGroupIDAndUserID(groupID, userID);
            var otherUserIDs = convos.Select(p => p.OtherUserID);
            var otherConvos = MultiGetLocal(otherUserIDs.Select(p => new KeyInfo(groupID, p)));
            var allConvos = convos.Concat(otherConvos);

            foreach (var convo in allConvos)
            {
                if (!convo.IsHidden)
                {
                    convo.IsHidden = true;
                    convo.Update(p => p.IsHidden);
                }
            }
        }

        public static GroupPrivateConversation[] GetAllByGroupIDAndUserID(Guid groupID, int userID)
        {
            var conversations = GetAllLocal(p => p.GroupAndUserIndex, GenerateGroupAndUserIndex(groupID, userID)).ToArray();
            return conversations;
        }

        #region IConversationContainer

        public string Title
        {
            get { return OtherUsername; }
        }

        bool IConversationContainer.CanAccess(int userID)
        {        
            return true;
        }

        bool IConversationContainer.CanView(int userID, out DateTime latestDate, out DateTime earliestDate)
        {
            earliestDate = DateTime.MinValue;
            latestDate = DateMessaged;
            return true;
        }

        bool IConversationContainer.CanEditAttachment(int userID, Attachment attachment)
        {
            return attachment.UploaderUserID == userID;
        }

        bool IConversationContainer.CanEditMessage(int userID, ConversationMessage message)
        {

            // If the requesting user is the message author, check the edit message grace period
            return message.SenderID == userID && message.Timestamp.FromEpochMilliconds() >= DateTime.UtcNow.AddMinutes(-ConversationConstants.EditMessageGracePeriodMinutes);
        }

        bool IConversationContainer.CanDeleteMessage(int userID, ConversationMessage message)
        {
            // If the requesting user is the message author, check the edit message grace period
            return message.SenderID == userID;
        }

        bool IConversationContainer.CanMention(int userID, ConversationMessage message)
        {
            return false;
        }

        bool IConversationContainer.CanMentionEveryone(int userID, ConversationMessage message)
        {
            return false;
        }

        bool IConversationContainer.CanSearch(int userID, out DateTime? minSearchDate)
        {
            minSearchDate = null;
            return true;
        }

        bool IConversationContainer.CanLikeMessage(int userID, ConversationMessage message)
        {
            return !message.IsDeleted;
        }

        bool IConversationContainer.CanCall(int userID)
        {
            return false;
        }

        bool IConversationContainer.CanUnlockCall(int userID)
        {
            return false;
        }

        bool IConversationContainer.CanSendMessage(int userID)
        {
            if (IsBlockedByThem || IsBlockedByMe)
            {
                return false;
            }

            // Ensure the recipient allows private messages
            var userPrivacySettings = UserPrivacySettings.GetByUserOrDefault(OtherUserID);

            // Always allowed between friends
            if (Friendship.IsConfirmed(UserID, OtherUserID))
            {
                return true;
            }

            // If they don't allow PMs from anyone, check if this is either an existiong convo, or that tehy are friends
            return !IsProvisional && !userPrivacySettings.BlockStrangerPMs;
        }

        void IConversationContainer.OnChatMessageChanged(ConversationMessage message, ConversationNotificationType changeType)
        {
            // Create the notification
            var senderNotification = message.ToNotification(message.SenderID, null, ConversationType.GroupPrivateConversation, changeType);

            // Notify the Sender
            ClientEndpoint.DispatchNotification(message.SenderID, ep =>
            {
                FriendMessageNotifier.Create(ep, senderNotification);
            });

            // Create the recipient notification
            var recipientNotification = message.ToNotification(message.RecipientID, null, ConversationType.GroupPrivateConversation, changeType);

            // Notify the Recipient
            ClientEndpoint.DispatchNotification(message.RecipientID, ep =>
            {
                FriendMessageNotifier.Create(ep, recipientNotification);
            });            
        }

        ConversationType IConversationContainer.ConversationType
        {
            get
            {
                return ConversationType.GroupPrivateConversation;
            }
        }

        void IConversationContainer.OnChatMessageLike(int userID, string username, ConversationMessage message, Like like)
        {
            var userIDs = new HashSet<int>(message.LikeUserIDs);
            var userNames = new HashSet<string>(message.LikeUsernames);
            var likeCount = 0;

            if (like.Unlike)
            {
                userIDs.Remove(userID);
                userNames.Remove(username);
                likeCount = userIDs.Count;
            }
            else
            {
                userIDs.Add(userID);
                userNames.Add(username);
                likeCount = userIDs.Count;
            }

            message.LikeCount = likeCount;
            message.LikeUserIDs = userIDs.ToArray();
            message.LikeUsernames = userNames.ToArray();

            (this as IConversationContainer).OnChatMessageChanged(message, ConversationNotificationType.Liked);

            // Queue off a worker item to persist the change
            ConversationMessageWorker.CreateLikeMessage(message.ConversationID, message.ID, message.Timestamp.FromEpochMilliconds(), userID, username, like.Unlike);
        }

        bool IConversationContainer.CanHide()
        {
            return true;
        }

        #endregion

        #region IAvatarParent

        public string AvatarUrlSlug
        {
            get { return "users"; }
        }

        public string AvatarUrlID
        {
            get { return OtherUserID.ToString(); }
        }

        #endregion
    }
}
