﻿using Curse.Aerospike;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using System;
using System.Collections.Generic;

namespace Curse.Friends.Data
{    
    [TableDefinition(TableName = "GroupRole", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.HomeRegion)]
    public class GroupRole : BaseTable<GroupRole>, IModelRegion
    {
        public const int NameMaxLength = 32;
        public const int MaxPermissionCount = 64;
        public const int MaxVanityColors = 32;
        public const int MaxVanityBadges = 64;
        public const int DefaultRoleRank = 10000;

        [Column("GroupID", KeyOrdinal = 1, IsIndexed = true)]
        public Guid GroupID
        {
            get;
            set;
        }

        [Column("RoleID", KeyOrdinal = 2)]
        public int RoleID
        {
            get;
            set;
        }

        [Column("RootGroupID", IsIndexed = true)]
        public Guid RootGroupID
        {
            get;
            set;
        }

        [Column("RegionID")]
        public int RegionID
        {
            get;
            set;
        }

        [Column("Name")]
        public string Name
        {
            get;
            set;
        }

        [Column("Rank")]
        public int Rank
        {
            get;
            set;
        }

        /// <summary>
        /// If this the group owner role. It automatically has all permissions, and cannot be deleted.
        /// </summary>
        [Column("IsOwner")]
        public bool IsOwner
        {
            get;
            set;
        }

        /// <summary>
        /// If this the default role that all guests get.
        /// </summary>
        [Column("IsDefault")]
        public bool IsDefault
        {
            get;
            set;
        }

        /// <summary>
        /// Tag for easy identification of a special role type.
        /// </summary>
        [Column("Tag")]
        public GroupRoleTag Tag
        {
            get;
            set;
        }

        /// <summary>
        /// The source for easy identification of the external account type
        /// </summary>
        [Column("RoleSource")]
        public AccountType Source
        {
            get;
            set;
        }

        /// <summary>
        /// If this group role is synced (Twitch, YouTube, etc). Synced roles cannot be deleted.
        /// </summary>
        [Column("IsSynced")]
        public bool IsSynced
        {
            get;
            set;
        }

        /// <summary>
        /// If synced, an ID for easily looking up the role later. Ex: Twitch-Subscribers-SivHD
        /// </summary>
        [Column("SyncID")]
        public string SyncID
        {
            get;
            set;
        }

        /// <summary>
        /// If this group role has been deleted (soft deletes only!)
        /// </summary>
        [Column("IsDeleted")]
        public bool IsDeleted
        {
            get;
            set;
        }

        /// <summary>
        /// The color to use for members of this role
        /// </summary>
        [Column("VanityColor")]
        public int VanityColor
        {
            get;
            set;
        }

        /// <summary>
        /// Whether or not this has a vanity badge
        /// </summary>
        [Column("VanityBadge")]
        public int VanityBadge
        {
            get;
            set;
        }

        /// <summary>
        /// Whether or not this has a vanity badge
        /// </summary>
        [Column("CustVanity")]
        public bool HasCustomVanityBadge
        {
            get;
            set;
        }

        public bool CanBeDeleted
        {
            get { return !IsOwner && !IsDefault && !IsSynced; }
        }


        public bool Validate(out string reason)
        {
            if (string.IsNullOrEmpty(Name) || Name.Length > NameMaxLength)
            {
                reason = "Name is empty or too long";
                return false;
            }

            if (RoleID < 1)
            {
                reason = "RoleID must be 1 or higher";
                return false;
            }

            if (Rank < 1)
            {
                reason = "Rank must be 1 or higher";
                return false;
            }

            if (RegionID < 1)
            {
                reason = "Region ID must be 1 or higher";
                return false;
            }

            reason = null;
            return true;
        }

        protected override void Validate()
        {
            string reason;

            if (!Validate(out reason))
            {
                throw new Exception(reason);
            }
        }

        public static GroupRole GetByGroupIDAndRoleID(AerospikeConfiguration config, Guid groupID, int roleID, bool returnDeleted = false)
        {
            var role = Get(config, groupID, roleID);
            if (role == null)
            {
                return null;
            }

            if (role.IsDeleted && !returnDeleted)
            {
                return null;
            }

            return role;
        }

        public GroupRoleNotification ToNotification()
        {
            return new GroupRoleNotification
            {
                RoleID = RoleID,
                Name = Name,
                Rank = Rank,
                HasCustomVanityBadge = HasCustomVanityBadge,
                VanityBadge = VanityBadge,
                VanityColor = VanityColor,
                IsDefault = IsDefault,
                IsOwner = IsOwner,
                Tag = Tag,
                Source = Source,
                IsHidden = Tag == GroupRoleTag.SyncedOwner,
                ExternalID = SyncID,
            };
        }
    }


}
