﻿using Curse.Aerospike;
using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Enums;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "GroupRolePermissions", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.HomeRegion)]
    public class GroupRolePermissions : BaseTable<GroupRolePermissions>, IModelRegion
    {
        public GroupRolePermissions() { }

        public GroupRolePermissions(Guid groupID, GroupRolePermissions parent)
        {
            GroupID = groupID;
            RoleID = parent.RoleID;
            RegionID = parent.RegionID;
            RootGroupID = parent.RootGroupID;
            IsDeleted = parent.IsDeleted;
            PermissionsState = new Dictionary<int, int>();

            foreach(var kvp in parent.PermissionsState)
            {
                var state = (GroupPermissionState)kvp.Value;

                if(state == GroupPermissionState.Allowed || state == GroupPermissionState.AllowedInherited)
                {
                    state = GroupPermissionState.AllowedInherited;
                }
                else
                {
                    state = GroupPermissionState.NotAllowedInherited;
                }
                PermissionsState[kvp.Key] = (int)state;
            }
        }

        public GroupRolePermissions(int homeRegionID, Guid groupID, int roleID, Guid rootGroupID, GroupPermissions permissions)
        {
            GroupID = groupID;
            RoleID = roleID;
            RegionID = homeRegionID;
            RootGroupID = rootGroupID;
            PermissionsState = new Dictionary<int, int>();
            var allPermissions = Enum.GetValues(typeof(GroupPermissions));
            foreach (GroupPermissions permission in allPermissions)
            {
                if(permissions.HasFlag(permission))
                {
                    PermissionsState[(int)permission] = (int)GroupPermissionState.Allowed;
                }
                else
                {
                    PermissionsState[(int)permission] = (int)GroupPermissionState.NotAllowed;
                }
            }
        }

        [Column("GroupID", KeyOrdinal = 1, IsIndexed = true)]
        public Guid GroupID
        {
            get;
            set;
        }

        [Column("RoleID", KeyOrdinal = 2)]
        public int RoleID
        {
            get;
            set;
        }

        [Column("RootGroupID", IsIndexed = true)]
        public Guid RootGroupID
        {
            get;
            set;
        }

        [Column("RegionID")]
        public int RegionID
        {
            get;
            set;
        }

        /// <summary>
        /// A dictionary of permission states (Allowed, Inherited, etc)
        /// </summary>
        [Column("PermState")]
        public Dictionary<int, int> PermissionsState
        {
            get;
            set;
        }

        /// <summary>
        /// A bitmask of all enabled permissions for this role
        /// </summary>
        [Column("Permissions")]
        public GroupPermissions Permissions
        {
            get;
            set;
        }


        [Column("IsDeleted")]
        public bool IsDeleted
        {
            get;
            set;
        }

        protected override void OnBeforeInsertOrUpdate()
        {
            Permissions = GeneratePermissions();

            var group = Group.Get(SourceConfiguration, GroupID);

            if(group.RolePermissions == null)
            {
                group.RolePermissions = new Dictionary<int, long>();
            }
            var permissions = (long)Permissions;
            if (IsDeleted && group.RolePermissions.ContainsKey(RoleID))
            {
                group.RolePermissions.Remove(RoleID);
                group.Update(p => p.RolePermissions);
            }

            else if (!IsDeleted && (!group.RolePermissions.ContainsKey(RoleID) || group.RolePermissions[RoleID] != permissions))
            {
                group.RolePermissions[RoleID] = permissions;
                group.Update(p => p.RolePermissions);
            }
        }
        
        public bool Validate(out string reason)
        {           
            if (RoleID < 1)
            {
                reason = "RoleID must be 1 or higher";
                return false;
            }          

            if (RegionID < 1)
            {
                reason = "Region ID must be 1 or higher";
                return false;
            }

            if (PermissionsState == null)
            {
                reason = "PermissionsState cannot be null";
                return false;
            }

            reason = null;
            return true;
        }

        protected override void Validate()
        {
            string reason;

            if (!Validate(out reason))
            {
                throw new Exception(reason);
            }
        }

        public void UpdatePermissionsState(Dictionary<GroupPermissions, GroupPermissionState> newState)
        {
            PermissionsState = new Dictionary<int, int>();

            foreach (var state in newState)
            {
                PermissionsState[(int)state.Key] = (int)state.Value;
            }
        }

        private GroupPermissions GeneratePermissions()
        {
            var role = GroupRole.GetByGroupIDAndRoleID(SourceConfiguration, RootGroupID, RoleID);

            if(role == null)
            {
                throw new InvalidOperationException(string.Format("Root group role not found: {0} on {1}", RootGroupID, RoleID));
            }

            if (role.IsOwner)
            {
                return GroupPermissions.All;
            }

            var permissions = GroupPermissions.None;
            // Rebuild the permissions bitmask, from the dictionary
            foreach (var kvp in PermissionsState)
            {
                var perm = (GroupPermissions)kvp.Key;
                var value = (GroupPermissionState)kvp.Value;

                if (value == GroupPermissionState.Allowed || value == GroupPermissionState.AllowedInherited)
                {
                    permissions = permissions | perm;
                }
            }

            return permissions;
        }

        public static GroupRolePermissions GetByGroupIDAndRoleID(AerospikeConfiguration configuration, Guid groupID, int roleID)
        {
            return Get(configuration, groupID, roleID);
        }

        public static GroupRolePermissions[] GetAllByRootGroupID(AerospikeConfiguration configuration, Guid rootGroupID)
        {
            return GetAll(configuration, p => p.RootGroupID, rootGroupID);
        }

        public bool CheckPermission(GroupPermissions permission)
        {
            return Permissions.HasFlag(permission);
        }

        public void SetPermission(GroupPermissions permission, bool allowed)
        {
            PermissionsState[(int)permission] = allowed ? (int)GroupPermissionState.Allowed : (int)GroupPermissionState.NotAllowed;
            Update();
        }

        public void UpdateInheritance(GroupRolePermissions parent)
        {
            var changed = false;

            foreach (var parentPermission in parent.PermissionsState)
            {
                var parentState = (GroupPermissionState)parentPermission.Value;
                GroupPermissionState? childState = null;
                int state;
                if (PermissionsState.TryGetValue(parentPermission.Key, out state))
                {
                    childState = (GroupPermissionState?)state;
                }
                
                if ((parentState == GroupPermissionState.AllowedInherited || parentState == GroupPermissionState.Allowed) && (!childState.HasValue || childState.Value == GroupPermissionState.NotAllowedInherited))
                {
                    PermissionsState[parentPermission.Key] = (int)GroupPermissionState.AllowedInherited;
                    changed = true;
                }

                if ((parentState == GroupPermissionState.NotAllowedInherited || parentState == GroupPermissionState.NotAllowed) && (!childState.HasValue || childState.Value == GroupPermissionState.AllowedInherited))
                {
                    PermissionsState[parentPermission.Key] = (int)GroupPermissionState.NotAllowedInherited;
                    changed = true;
                }
            }

            if (changed)
            {
                Update(p => p.PermissionsState, p => p.Permissions);
            }
        }
    }
}
