﻿using System;
using System.Collections.Generic;
using System.Linq;
using Aerospike.Client;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Logging;

namespace Curse.Friends.Data
{
    [TableDefinition(TableName = "PrivateConversation", KeySpace = "CurseVoice-Global", ReplicationMode = ReplicationMode.Mesh)]
    public class PrivateConversation : BaseTable<PrivateConversation>, IConversationContainer, IConversationParent
    {
        private static readonly LogCategory ThrottledLogger = new LogCategory("PrivateConversation") { ReleaseLevel = LogLevel.Debug, Throttle = TimeSpan.FromMinutes(1) };

        private static readonly TimeSpan FamiliaryCacheTime = TimeSpan.FromHours(1);

        [Column("UserID", KeyOrdinal = 1, IsIndexed = true)]
        public int UserID
        {
            get;
            set;
        }

        /// <summary>
        /// UserID of the friend
        /// </summary>
        [Column("OtherUserID", KeyOrdinal = 2, IsIndexed = true)]
        public int OtherUserID
        {
            get;
            set;
        }

        /// <summary>
        /// The title of the conversation (the other user's username or nickname)
        /// </summary>
        [Column("Title")]
        public string Title
        {
            get;
            set;
        }

        /// <summary>
        /// The date the conversation was started
        /// </summary>
        [Column("DateCreated")]
        public DateTime DateCreated
        {
            get;
            set;
        }

        /// <summary>
        /// The date of the most recent message
        /// </summary>
        [Column("DateMessaged")]
        public DateTime DateMessaged
        {
            get;
            set;
        }

        /// <summary>
        /// How many messages are unread
        /// </summary>
        [Column("UnreadCount")]
        public int UnreadCount
        {
            get;
            set;
        }

        /// <summary>
        /// The date of the last message the user has read
        /// </summary>
        [Column("DateRead")]
        public DateTime DateRead
        {
            get;
            set;
        }

        /// <summary>
        /// Whether or not the user has elected to hide this conversation
        /// </summary>
        [Column("IsHidden")]
        public bool IsHidden
        {
            get;
            set;
        }

        /// <summary>
        /// Whether or not the recipient is familiar to the sender
        /// </summary>
        [Column("IsFamiliar")]
        public bool IsFamiliar
        {
            get;
            set;
        }

        /// <summary>
        /// Whether or not the user has elected to hide this conversation
        /// </summary>
        [Column("IsMuted")]
        public bool IsMuted
        {
            get;
            set;
        }
        

        public bool HasFamiliarityExpired
        {
            get
            {
                return FamiliarityTimestamp < DateTime.UtcNow.Subtract(FamiliaryCacheTime).ToEpochMilliseconds();
            }         
        }

        /// <summary>
        /// The timestamp from when familiarity was last cached
        /// </summary>
        [Column("FamilarityTs")]
        public long FamiliarityTimestamp
        {
            get;
            set;
        }

        /// <summary>
        /// Whether or not this conversation is with a friend
        /// </summary>
        [Column("IsFriend")]
        public bool IsFriend
        {
            get;
            set;
        }

        /// <summary>
        /// Whether or not the user has ever sent a message as part of this conversation.
        /// </summary>
        [Column("HasSent")]
        public bool HasSent
        {
            get;
            set;
        }

        [Column("ConversationID")]
        public string ConversationID { get; set; }

        public DateTime Recency 
        {
            get
            {
                if (DateRead > DateTime.MinValue || DateMessaged > DateTime.MinValue)
                {
                    return DateRead > DateMessaged ? DateRead : DateMessaged;    
                }

                return DateCreated;
            }
        }

        public bool HasMessaged 
        {
            get { return DateMessaged > DateTime.MinValue; }
        }

        public PrivateConversationContract ToContract()
        {
            return new PrivateConversationContract
            {
                DateMessaged = DateMessaged,
                Title = Title,
                OtherUserID = OtherUserID,
                ConversationID = ConversationID,
                UnreadCount = UnreadCount,
                DateRead = DateRead
            };
        }
        
        public static string GenerateConversationID(int userID, int otherUserID)
        {
            return string.Format("{0}:{1}", Math.Min(userID, otherUserID), Math.Max(userID, otherUserID));
        }

        public void MarkAsRead(DateTime timestamp)
        {

            if (timestamp < DateRead)
            {
                return;
            }

            ResetCounterAndSetValue(SourceConfiguration, UpdateMode.Fast, p => p.UnreadCount, p => p.DateRead, timestamp);
            DateRead = timestamp;
            UnreadCount = 0;
        }

        public void MarkAsSent(DateTime timestamp)
        {
            DateMessaged = timestamp;
            IsHidden = false;
            HasSent = true;
            Update(p => p.DateMessaged, p => p.IsHidden, p => p.HasSent);                        
        }

        public void MarkAsUnread(DateTime timestamp, int numberOfMessages)
        {
            // Ensure that the convo isn't hidden
            if (IsHidden)
            {
                IsHidden = false;
                Update(p => p.IsHidden);
            }

            // Validate the data, ensuring DataMessages only increases
            if (timestamp < DateMessaged)
            {
                timestamp = DateMessaged;
            }
            
            UnreadCount = IncrementCounterAndSetValue(SourceConfiguration, UpdateMode.Fast, p => p.UnreadCount, numberOfMessages, p => p.DateMessaged, timestamp);
            DateMessaged = timestamp;           
        }
      
        bool IConversationContainer.CanAccess(int userID)
        {
            return true;
        }

        bool IConversationContainer.CanView(int userID, out DateTime latestDate, out DateTime earliestDate)
        {
            earliestDate = DateCreated;
            latestDate = DateMessaged > DateTime.UtcNow ? DateMessaged : DateTime.UtcNow;
            return true;
        }

        bool IConversationContainer.CanEditAttachment(int userID, Attachment attachment)
        {
            return attachment.UploaderUserID == userID;
        }

        bool IConversationContainer.CanEditMessage(int userID, ConversationMessage message)
        {
            // If the requesting user is the message author, check the edit message grace period
            return message.SenderID == userID && message.Timestamp.FromEpochMilliconds() >= DateTime.UtcNow.AddMinutes(-ConversationConstants.EditMessageGracePeriodMinutes);
        }

        bool IConversationContainer.CanDeleteMessage(int userID, ConversationMessage message)
        {
            // If the requesting user is the message author, check the edit message grace period
            return message.SenderID == userID;
        }

        bool IConversationContainer.CanMention(int userID, ConversationMessage message)
        {
            return false;
        }

        bool IConversationContainer.CanMentionEveryone(int userID, ConversationMessage message)
        {
            return false;
        }

        bool IConversationContainer.CanSearch(int userID, out DateTime? minSearchDate)
        {
            minSearchDate = null;
            return true;
        }

        bool IConversationContainer.CanLikeMessage(int userID, ConversationMessage message)
        {
            return !message.IsDeleted;
        }

        bool IConversationContainer.CanCall(int userID)
        {
            if (UserBlock.IsBlocked(userID, OtherUserID))
            {
                ThrottledLogger.Debug("Attempt to call a blocked user.", new { userID, OtherUserID});
                return false;
            }

            return IsFriend;
        }

        bool IConversationContainer.CanUnlockCall(int userID)
        {            
            return IsFriend;
        }

        bool IConversationContainer.CanSendMessage(int userID)
        {
            if (UserBlock.IsBlocked(userID, OtherUserID))
            {
                ThrottledLogger.Debug("Attempt to whisper a blocked user.", new { userID, OtherUserID });
                return false;
            }

            return true; // Defer to logic worker
        }

        bool IConversationContainer.CanHide()
        {
            return true;
        }

        void IConversationContainer.OnChatMessageChanged(ConversationMessage message, ConversationNotificationType changeType)
        {
            // Create the notification
            var senderNotification = message.ToNotification(message.SenderID, null, ConversationType.Friendship, changeType);
            
            // Notify the Sender
            ClientEndpoint.DispatchNotification(message.SenderID, ep =>
            {
                FriendMessageNotifier.Create(ep, senderNotification);
            });

            // Create the recipient notification
            var recipientNotification = message.ToNotification(message.RecipientID, null, ConversationType.Friendship, changeType);

            // Notify the Recipient
            ClientEndpoint.DispatchNotification(message.RecipientID, ep =>
            {
                FriendMessageNotifier.Create(ep, recipientNotification);
            });
        }

        ConversationType IConversationContainer.ConversationType
        {
            get
            {
                return ConversationType.Friendship;
            }
        }

        void IConversationContainer.OnChatMessageLike(int userID, string username, ConversationMessage message, Like like)
        {
            var userIDs = new HashSet<int>(message.LikeUserIDs);
            var userNames = new HashSet<string>(message.LikeUsernames);
            var likeCount = 0;

            if (like.Unlike)
            {
                userIDs.Remove(userID);
                userNames.Remove(username);
                likeCount = userIDs.Count;
            }
            else
            {
                userIDs.Add(userID);
                userNames.Add(username);
                likeCount = userIDs.Count;
            }

            message.LikeCount = likeCount;
            message.LikeUserIDs = userIDs.ToArray();
            message.LikeUsernames = userNames.ToArray();

            (this as IConversationContainer).OnChatMessageChanged(message, ConversationNotificationType.Liked);

            // Queue off a worker item to persist the change
            ConversationMessageWorker.CreateLikeMessage(message.ConversationID, message.ID, message.Timestamp.FromEpochMilliconds(), userID, username, like.Unlike);
        }

        IConversationParent IConversationContainer.GetConversationParent(int userID)
        {
            return this;
        }

        bool IConversationParent.ShouldSendPushNotification(User user, string messageBody, HashSet<int> mentionedUserIDs)
        {
            if (IsMuted)
            {
                return false;
            }

            return user.FriendMessagePushPreference != PushNotificationPreference.None;
        }

        string IConversationParent.GetSenderName(int senderID, string fallback)
        {
            return Title ?? fallback;
        }

        void IConversationParent.ToggleHidden(bool isHidden)
        {
            IsHidden = isHidden;
            if (UnreadCount > 0)
            {
                UnreadCount = 0;
                DateRead = DateMessaged;
                Update(p => p.IsHidden, p => p.DateRead, p => p.UnreadCount);
            }
            else
            {
                Update(p => p.IsHidden);    
            }            
        }

        void IConversationParent.ToggleMuted(bool isMuted)
        {
            IsMuted = isMuted;
            Update(p => p.IsMuted);
        }

        #region IAvatarParent

        public string AvatarUrlSlug
        {
            get { return User.AvatarUrlPath; }
        }

        public string AvatarUrlID
        {
            get { return OtherUserID.ToString(); }
        }

        #endregion

        public static PrivateConversation[] GetAllByUserID(int userID)
        {
            return GetAllLocal(p => p.UserID, userID);
        }

        public static PrivateConversation GetByUserIDAndOtherUserID(int userID, int otherUserID)
        {
            return GetLocal(userID, otherUserID);
        }


        public static PrivateConversation GetOrCreateByUserIDAndOtherUserID(DateTime timestamp, int userID, int otherUserID)
        {
            var conversation = GetLocal(userID, otherUserID);
            if (conversation != null)
            {
                return conversation;
            }

            // Look for a friendship record
            var friendship = Friendship.GetByOtherUserID(userID, otherUserID);
            
            if (friendship != null)
            {
                try
                {
                    return CreateByFriendship(friendship);
                }
                catch (AerospikeException ex)
                {
                    if (ex.Result == 5) // Key already exists
                    {
                        return GetLocal(userID, otherUserID);
                    }

                    throw;
                }
                
            }

            // Fallback to stats
            var userStats = UserStatistics.GetByUser(otherUserID);

            if (userStats != null)
            {
                try
                {
                    var preferredName = userStats.HasDifferentDisplayName() 
                        ? $"{userStats.DisplayName} ({userStats.Username})" 
                        : string.IsNullOrWhiteSpace(userStats.DisplayName) 
                            ? userStats.Username 
                            : userStats.DisplayName;
                    return CreateByUserIDAndOtherUserID(timestamp, userID, otherUserID, preferredName, false);  
                }
                catch (AerospikeException ex)
                {
                    if (ex.Result == 5) // Key already exists
                    {
                        return GetLocal(userID, otherUserID);
                    }

                    throw;
                }         
            }

            return null;
        }

        public static PrivateConversation GetOrCreateByFriendship(Friendship friendship)
        {
            var conversation = GetLocal(friendship.UserID, friendship.OtherUserID);

            if (conversation != null)
            {
                return conversation;
            }

            try
            {
                return CreateByFriendship(friendship);
            }
            catch (AerospikeException ex)
            {
                if (ex.Result == 5) // Key already exists
                {
                    return GetLocal(friendship.UserID, friendship.OtherUserID);
                }

                throw;
            }
        }
        
        private static PrivateConversation CreateByFriendship(Friendship friendship)
        {
            var conversation = new PrivateConversation
            {
                UserID = friendship.UserID,
                OtherUserID = friendship.OtherUserID,
                DateCreated = friendship.DateConfirmed,
                Title = friendship.FormattedDisplayName,
                IsFriend = friendship.Status == FriendshipStatus.Confirmed,
                ConversationID = GenerateConversationID(friendship.UserID, friendship.OtherUserID),
                DateMessaged = friendship.DateMessaged,
                DateRead = friendship.DateRead,
                UnreadCount = friendship.UnreadCount
            };

            conversation.InsertLocal(UpdateMode.Concurrent);
            return conversation;

            
        }
      
        private static PrivateConversation CreateByUserIDAndOtherUserID(DateTime timestamp, int userID, int otherUserID, string otherDisplayName, bool isFriend)
        {
            
            var conversation = new PrivateConversation
            {
                UserID = userID,
                OtherUserID = otherUserID,
                DateCreated = timestamp,
                Title = string.IsNullOrEmpty(otherDisplayName) ? otherUserID.ToString() : otherDisplayName,
                IsFriend = isFriend, 
                ConversationID = GenerateConversationID(userID, otherUserID)
            };

            conversation.InsertLocal(UpdateMode.Concurrent);
            
            
            return conversation;
        }
        

        public bool RepairFriendship()
        {
            if (!Friendship.IsConfirmed(UserID, OtherUserID))
            {
                return false;
            }

            IsFriend = true;
            Update(p => p.IsFriend);

            var other = GetByUserIDAndOtherUserID(OtherUserID, UserID);
            other.IsFriend = true;
            other.Update(p => p.IsFriend);
            return true;
        }
    }
}
